<?php

use App\Models\Tournament;
use App\Models\Team;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;

uses(RefreshDatabase::class);

test('team registration page shows tournament entry fee', function () {
    // Create an active tournament with entry fee
    $tournament = Tournament::factory()->create([
        'status' => 'active',
        'entry_fee' => 2500.00,
        'currency' => 'KES'
    ]);

    $response = $this->get('/register-team');

    $response->assertStatus(200);
    $response->assertInertia(fn ($page) => 
        $page->component('frontend/auth/RegisterTeam')
            ->has('registration_fee', 2500)
            ->has('tournament', fn ($tournamentData) =>
                $tournamentData['id'] === $tournament->id &&
                $tournamentData['name'] === $tournament->name &&
                $tournamentData['entry_fee'] === 2500.00 &&
                $tournamentData['currency'] === 'KES'
            )
    );
});

test('team registration uses config fee when no active tournament', function () {
    // Create a tournament that is not active
    Tournament::factory()->create([
        'status' => 'draft',
        'entry_fee' => 2500.00
    ]);

    $response = $this->get('/register-team');

    $response->assertStatus(200);
    $response->assertInertia(fn ($page) => 
        $page->component('frontend/auth/RegisterTeam')
            ->has('registration_fee', 1000) // Default from config
            ->has('tournament', null)
    );
});

test('team registration with tournament entry fee', function () {
    // Create an active tournament with entry fee
    $tournament = Tournament::factory()->create([
        'status' => 'active',
        'entry_fee' => 1500.00,
        'currency' => 'KES'
    ]);

    $teamData = [
        'team' => [
            'name' => 'Test Team',
            'email' => 'team@test.com',
            'phone' => '+254700000000',
            'location' => 'Nairobi, Kenya',
            'coach' => 'John Coach',
            'year' => 2020
        ],
        'captain' => [
            'name' => 'Captain Name',
            'email' => 'captain@test.com',
            'phone' => '+254700000001',
            'password' => 'password123',
            'password_confirmation' => 'password123'
        ],
        'players' => [
            [
                'name' => 'Player 1',
                'email' => 'player1@test.com',
                'phone' => '+254700000002',
                'position' => 'Goalkeeper',
                'jersey_number' => '1'
            ],
            [
                'name' => 'Player 2',
                'email' => 'player2@test.com',
                'phone' => '+254700000003',
                'position' => 'Defender',
                'jersey_number' => '2'
            ]
        ],
        'officials' => [
            [
                'name' => 'Official 1',
                'email' => 'official1@test.com',
                'phone' => '+254700000004',
                'role' => 'Coach',
                'qualifications' => 'UEFA License'
            ]
        ],
        'terms_accepted' => true
    ];

    $response = $this->post('/register-team', $teamData);

    $response->assertRedirect(route('profile.complete'));
    
    // Verify team was created
    $this->assertDatabaseHas('teams', [
        'name' => 'Test Team'
    ]);
    
    // Verify captain was created
    $this->assertDatabaseHas('users', [
        'email' => 'captain@test.com',
        'user_type' => 'team'
    ]);
    
    // Verify team is registered for tournament
    $team = Team::where('name', 'Test Team')->first();
    $this->assertTrue($team->isRegisteredForTournament($tournament));
});

test('payment required page shows tournament entry fee', function () {
    // Create an active tournament with entry fee
    $tournament = Tournament::factory()->create([
        'status' => 'active',
        'entry_fee' => 3000.00,
        'currency' => 'KES'
    ]);

    // Create a user
    $user = User::factory()->create([
        'user_type' => 'team',
        'payment_status' => 'pending'
    ]);

    $this->actingAs($user);

    $response = $this->get('/profile/payment-required');

    $response->assertStatus(200);
    $response->assertInertia(fn ($page) => 
        $page->component('frontend/payment/Required')
            ->has('registration_fee', 3000)
            ->has('tournament', fn ($tournamentData) =>
                $tournamentData['id'] === $tournament->id &&
                $tournamentData['entry_fee'] === 3000.00 &&
                $tournamentData['currency'] === 'KES'
            )
    );
}); 