import React, { useState } from 'react';
import { Head, router, usePage } from '@inertiajs/react';
import AppLayout from '@/layouts/app-layout';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Switch } from '@/components/ui/switch';
import { Alert, AlertDescription } from '@/components/ui/alert';
import { 
    Bell, 
    CheckCircle2, 
    XCircle, 
    ChevronDown, 
    ChevronRight,
    Save,
    AlertTriangle,
    Info,
    Lock
} from 'lucide-react';
import { cn } from '@/lib/utils';

interface NotificationSetting {
    id: number;
    notification_type: string;
    category: string;
    enabled: boolean;
    description: string;
    channels: string[];
}

interface UserPreference {
    id: number;
    notification_type: string;
    preference_type: 'category' | 'individual';
    enabled: boolean;
}

interface Props {
    systemNotifications: Record<string, NotificationSetting[]>;
    userPreferences: {
        categories: Record<string, UserPreference>;
        individual: Record<string, UserPreference>;
    };
    categoryCounts: Record<string, number>;
}

export default function NotificationPreferences({ systemNotifications, userPreferences, categoryCounts }: Props) {
    const { auth } = usePage().props as any;
    const [expandedCategories, setExpandedCategories] = useState<Set<string>>(new Set());
    const [pendingChanges, setPendingChanges] = useState<Map<string, { type: string; preferenceType: string; enabled: boolean }>>(new Map());

    // Category metadata with role relevance
    const categoryInfo: Record<string, { 
        icon: string; 
        color: string; 
        description: string;
        relevantRoles: string[];
    }> = {
        auth: { 
            icon: '🔐', 
            color: 'bg-blue-500', 
            description: 'Login alerts, password changes, security notifications',
            relevantRoles: ['all']
        },
        team: { 
            icon: '👥', 
            color: 'bg-green-500', 
            description: 'Team updates, member changes, team events',
            relevantRoles: ['team_manager', 'player', 'admin', 'super_admin']
        },
        player: { 
            icon: '⚽', 
            color: 'bg-purple-500', 
            description: 'Player profiles, ratings, digital IDs',
            relevantRoles: ['player', 'team_manager', 'admin', 'super_admin']
        },
        payment: { 
            icon: '💳', 
            color: 'bg-yellow-500', 
            description: 'Payment status, receipts, transactions',
            relevantRoles: ['team_manager', 'player', 'admin', 'super_admin']
        },
        tournament: { 
            icon: '🏆', 
            color: 'bg-orange-500', 
            description: 'Tournament updates, registrations, schedules',
            relevantRoles: ['all']
        },
        match: { 
            icon: '🎮', 
            color: 'bg-red-500', 
            description: 'Match schedules, results, streaming',
            relevantRoles: ['player', 'team_manager', 'referee', 'admin', 'super_admin']
        },
        official: { 
            icon: '👔', 
            color: 'bg-indigo-500', 
            description: 'Official assignments, verifications',
            relevantRoles: ['referee', 'admin', 'super_admin']
        },
        user: { 
            icon: '👤', 
            color: 'bg-gray-500', 
            description: 'Profile updates, account changes',
            relevantRoles: ['all']
        },
        admin: { 
            icon: '⚙️', 
            color: 'bg-slate-500', 
            description: 'Admin & organizer notifications',
            relevantRoles: ['admin', 'super_admin', 'organizer']
        },
        certificate: { 
            icon: '🎓', 
            color: 'bg-teal-500', 
            description: 'Achievement certificates, awards',
            relevantRoles: ['player', 'team_manager', 'admin', 'super_admin']
        },
        invitation: { 
            icon: '✉️', 
            color: 'bg-pink-500', 
            description: 'Team and event invitations',
            relevantRoles: ['all']
        },
        system: { 
            icon: '🔔', 
            color: 'bg-cyan-500', 
            description: 'System announcements, updates',
            relevantRoles: ['all']
        },
    };

    // Check if user has relevant role for category
    const isRelevantCategory = (category: string): boolean => {
        const info = categoryInfo[category];
        if (!info) return true;
        if (info.relevantRoles.includes('all')) return true;
        
        // Check user's roles
        const userRoles = auth.user?.roles?.map((r: any) => r.name) || [];
        return info.relevantRoles.some(role => userRoles.includes(role));
    };

    // Filter notifications to show only relevant ones
    const relevantNotifications = Object.entries(systemNotifications).reduce((acc, [category, items]) => {
        if (!isRelevantCategory(category)) {
            return acc;
        }
        acc[category] = items;
        return acc;
    }, {} as Record<string, NotificationSetting[]>);

    // Toggle category expansion
    const toggleCategory = (category: string) => {
        const newExpanded = new Set(expandedCategories);
        if (newExpanded.has(category)) {
            newExpanded.delete(category);
        } else {
            newExpanded.add(category);
        }
        setExpandedCategories(newExpanded);
    };

    // Get current user preference for category
    const getCategoryPreference = (category: string): boolean => {
        // Check pending changes first
        const pendingKey = `category_${category}`;
        if (pendingChanges.has(pendingKey)) {
            return pendingChanges.get(pendingKey)!.enabled;
        }

        // Check existing preference
        const pref = userPreferences.categories[category];
        return pref ? pref.enabled : true; // Default to enabled
    };

    // Get current user preference for individual notification
    const getNotificationPreference = (notificationType: string, category: string): boolean => {
        // Check pending changes first
        const pendingKey = `individual_${notificationType}`;
        if (pendingChanges.has(pendingKey)) {
            return pendingChanges.get(pendingKey)!.enabled;
        }

        // Check existing individual preference
        const individualPref = userPreferences.individual[notificationType];
        if (individualPref) {
            return individualPref.enabled;
        }

        // Fall back to category preference
        return getCategoryPreference(category);
    };

    // Toggle category preference
    const toggleCategoryPreference = (category: string) => {
        const currentValue = getCategoryPreference(category);
        const newChanges = new Map(pendingChanges);
        newChanges.set(`category_${category}`, {
            type: category,
            preferenceType: 'category',
            enabled: !currentValue
        });
        setPendingChanges(newChanges);
    };

    // Toggle individual notification preference
    const toggleNotificationPreference = (notificationType: string, category: string) => {
        const currentValue = getNotificationPreference(notificationType, category);
        const newChanges = new Map(pendingChanges);
        newChanges.set(`individual_${notificationType}`, {
            type: notificationType,
            preferenceType: 'individual',
            enabled: !currentValue
        });
        setPendingChanges(newChanges);
    };

    // Save all pending changes
    const saveChanges = () => {
        const preferences = Array.from(pendingChanges.values()).map(change => ({
            notification_type: change.type,
            preference_type: change.preferenceType,
            enabled: change.enabled
        }));

        router.post('/settings/notifications', {
            preferences
        }, {
            preserveScroll: true,
            onSuccess: () => {
                setPendingChanges(new Map());
            }
        });
    };

    // Reset to defaults
    const resetToDefaults = () => {
        if (confirm('Are you sure you want to reset all notification preferences to default? All notifications will be enabled.')) {
            router.post('/settings/notifications/reset', {}, {
                preserveScroll: true,
                onSuccess: () => {
                    setPendingChanges(new Map());
                }
            });
        }
    };

    // Calculate statistics
    const stats = {
        total: Object.values(relevantNotifications).flat().length,
        enabled: 0,
        categories: Object.keys(relevantNotifications).length
    };

    Object.entries(relevantNotifications).forEach(([category, items]) => {
        const categoryEnabled = getCategoryPreference(category);
        items.forEach(notification => {
            const notifEnabled = getNotificationPreference(notification.notification_type, category);
            if (notifEnabled) stats.enabled++;
        });
    });

    stats.disabled = stats.total - stats.enabled;

    return (
        <AppLayout>
            <Head title="Notification Preferences" />

            <div className="max-w-5xl mx-auto space-y-6 p-6">
                {/* Header */}
                <div>
                    <h1 className="text-3xl font-bold">Notification Preferences</h1>
                    <p className="text-muted-foreground mt-1">
                        Customize which notifications you want to receive
                    </p>
                </div>

                {/* Alert for pending changes */}
                {pendingChanges.size > 0 && (
                    <Alert>
                        <AlertTriangle className="h-4 w-4" />
                        <AlertDescription className="flex items-center justify-between">
                            <span>You have {pendingChanges.size} unsaved change{pendingChanges.size !== 1 ? 's' : ''}.</span>
                            <div className="flex gap-2">
                                <Button size="sm" onClick={saveChanges}>Save Changes</Button>
                                <Button size="sm" variant="outline" onClick={() => setPendingChanges(new Map())}>Cancel</Button>
                            </div>
                        </AlertDescription>
                    </Alert>
                )}

                {/* Info Alert */}
                <Alert>
                    <Info className="h-4 w-4" />
                    <AlertDescription>
                        You'll only see notifications relevant to your role. Individual notification settings override category settings.
                        Notifications marked with <Lock className="w-3 h-3 inline" /> are controlled by system administrators.
                    </AlertDescription>
                </Alert>

                {/* Statistics Cards */}
                <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                    <Card>
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium">Available</CardTitle>
                            <Bell className="h-4 w-4 text-muted-foreground" />
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold">{stats.total}</div>
                            <p className="text-xs text-muted-foreground">Notifications for your role</p>
                        </CardContent>
                    </Card>
                    <Card>
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium">Enabled</CardTitle>
                            <CheckCircle2 className="h-4 w-4 text-green-600" />
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold text-green-600">{stats.enabled}</div>
                            <p className="text-xs text-muted-foreground">You'll receive these</p>
                        </CardContent>
                    </Card>
                    <Card>
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium">Disabled</CardTitle>
                            <XCircle className="h-4 w-4 text-red-600" />
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold text-red-600">{stats.disabled}</div>
                            <p className="text-xs text-muted-foreground">You won't receive these</p>
                        </CardContent>
                    </Card>
                </div>

                {/* Action Buttons */}
                <div className="flex justify-between items-center">
                    <Button variant="outline" onClick={resetToDefaults}>
                        Reset to Defaults
                    </Button>
                    {pendingChanges.size > 0 && (
                        <Button onClick={saveChanges} className="gap-2">
                            <Save className="w-4 h-4" />
                            Save {pendingChanges.size} Changes
                        </Button>
                    )}
                </div>

                {/* Notifications by Category */}
                <div className="space-y-4">
                    {Object.entries(relevantNotifications).map(([category, items]) => {
                        const isExpanded = expandedCategories.has(category);
                        const info = categoryInfo[category] || { icon: '🔔', color: 'bg-gray-500', description: 'Notifications', relevantRoles: ['all'] };
                        const categoryEnabled = getCategoryPreference(category);
                        const enabledCount = items.filter(n => getNotificationPreference(n.notification_type, category)).length;

                        return (
                            <Card key={category}>
                                <CardHeader>
                                    <div className="flex items-center justify-between">
                                        <div className="flex items-center gap-3 flex-1 cursor-pointer" onClick={() => toggleCategory(category)}>
                                            <div className={cn("w-10 h-10 rounded-lg flex items-center justify-center text-white text-xl", info.color)}>
                                                {info.icon}
                                            </div>
                                            <div className="flex-1">
                                                <CardTitle className="flex items-center gap-2">
                                                    {category.charAt(0).toUpperCase() + category.slice(1)}
                                                    <Badge variant="secondary">{items.length}</Badge>
                                                    <Badge variant={categoryEnabled ? "default" : "destructive"}>
                                                        {enabledCount}/{items.length} enabled
                                                    </Badge>
                                                </CardTitle>
                                                <CardDescription>{info.description}</CardDescription>
                                            </div>
                                            {isExpanded ? <ChevronDown className="w-5 h-5" /> : <ChevronRight className="w-5 h-5" />}
                                        </div>
                                        <div className="flex items-center gap-2">
                                            <span className="text-sm text-muted-foreground">
                                                {categoryEnabled ? 'Enabled' : 'Disabled'}
                                            </span>
                                            <Switch
                                                checked={categoryEnabled}
                                                onCheckedChange={() => toggleCategoryPreference(category)}
                                            />
                                        </div>
                                    </div>
                                </CardHeader>

                                {isExpanded && (
                                    <CardContent>
                                        <div className="space-y-2">
                                            {items.map((notification) => {
                                                const isEnabled = getNotificationPreference(notification.notification_type, category);
                                                const pendingKey = `individual_${notification.notification_type}`;
                                                const hasChanges = pendingChanges.has(pendingKey);

                                                return (
                                                    <div
                                                        key={notification.id}
                                                        className={cn(
                                                            "flex items-center justify-between p-3 rounded-lg border transition-colors",
                                                            hasChanges && "bg-yellow-50 border-yellow-300"
                                                        )}
                                                    >
                                                        <div className="flex-1">
                                                            <div className="flex items-center gap-2">
                                                                <code className="text-sm font-mono bg-muted px-2 py-1 rounded">
                                                                    {notification.notification_type}
                                                                </code>
                                                                {hasChanges && (
                                                                    <Badge variant="outline" className="text-xs">
                                                                        Pending
                                                                    </Badge>
                                                                )}
                                                            </div>
                                                            {notification.description && (
                                                                <p className="text-sm text-muted-foreground mt-1">
                                                                    {notification.description}
                                                                </p>
                                                            )}
                                                            <div className="flex items-center gap-2 mt-1">
                                                                {notification.channels?.map((channel) => (
                                                                    <Badge key={channel} variant="secondary" className="text-xs">
                                                                        {channel}
                                                                    </Badge>
                                                                ))}
                                                            </div>
                                                        </div>
                                                        <div className="flex items-center gap-2">
                                                            <span className={cn("text-sm font-medium", isEnabled ? "text-green-600" : "text-red-600")}>
                                                                {isEnabled ? 'Enabled' : 'Disabled'}
                                                            </span>
                                                            <Switch
                                                                checked={isEnabled}
                                                                onCheckedChange={() => toggleNotificationPreference(notification.notification_type, category)}
                                                            />
                                                        </div>
                                                    </div>
                                                );
                                            })}
                                        </div>
                                    </CardContent>
                                )}
                            </Card>
                        );
                    })}
                </div>

                {Object.keys(relevantNotifications).length === 0 && (
                    <Card>
                        <CardContent className="py-12 text-center">
                            <Bell className="w-12 h-12 text-muted-foreground mx-auto mb-4" />
                            <p className="text-lg font-medium">No notifications available</p>
                            <p className="text-sm text-muted-foreground">
                                No notifications are currently available for your role
                            </p>
                        </CardContent>
                    </Card>
                )}
            </div>
        </AppLayout>
    );
}

