import React, { useMemo, useState } from "react";
import { router, useForm } from "@inertiajs/react";
import { Input } from "@/components/ui/input";
import { Button } from "@/components/ui/button";
import { Label } from "@/components/ui/label";
import Header from "@/components/frontend/layout/Header";
import Footer from "@/components/frontend/sections/Footer";
import { LoaderCircle } from "lucide-react";

const CompleteProfile = ({ user, team, player, countries, positions, experienceLevels }: { user: any, team: any, player: any, countries: any, positions: any, experienceLevels: any }) => {
  const { data, setData, post, processing, errors } = useForm({
    name: user.name || '',
    phone: user.phone || '',
    date_of_birth: '',
    gender: '',
    country: '',
    city: '',
    bio: '',
    avatar: null as File | null,
    national_id_front: null as File | null,
    national_id_back: null as File | null,
  });

  const [avatarPreview, setAvatarPreview] = useState<string | null>(null);
  const [idPreview, setIdPreview] = useState<string | null>(null);
  const [idBackPreview, setIdBackPreview] = useState<string | null>(null);
  const handleSubmit = (e: React.FormEvent<HTMLFormElement>) => {
    e.preventDefault();
    post(route('profile.complete.update'), { forceFormData: true });
  };

  return (
    <>
      <Header />
      <div className="min-h-screen bg-gradient-to-br from-green-50 to-gray-50 py-8">
        <div className="container mx-auto px-4 max-w-2xl">
          <div className="text-center mb-8">
            <h1 className="text-3xl font-bold text-gray-900">Complete Your Profile</h1>
            <p className="text-gray-600 mt-2">Please provide your information to continue</p>
          </div>

          <div className="bg-white rounded-lg shadow-xl border border-gray-200 p-8">
            <form onSubmit={handleSubmit} className="space-y-6" encType="multipart/form-data">
              <div>
                <Label htmlFor="name">Full Name *</Label>
                <Input
                  id="name"
                  value={data.name}
                  onChange={(e) => setData('name', e.target.value)}
                  placeholder="Enter your full name"
                  required
                />
                {errors.name && <p className="text-red-600 text-sm mt-1">{errors.name}</p>}
              </div>
              
              <div>
                <Label htmlFor="phone">Phone Number *</Label>
                <Input
                  id="phone"
                  value={data.phone}
                  onChange={(e) => setData('phone', e.target.value)}
                  placeholder="+254 700 000 000"
                  required
                />
                {errors.phone && <p className="text-red-600 text-sm mt-1">{errors.phone}</p>}
              </div>
              
              <div>
                <Label htmlFor="date_of_birth">Date of Birth *</Label>
                <Input
                  id="date_of_birth"
                  type="date"
                  value={data.date_of_birth}
                  onChange={(e) => setData('date_of_birth', e.target.value)}
                  required
                />
                {errors.date_of_birth && <p className="text-red-600 text-sm mt-1">{errors.date_of_birth}</p>}
              </div>
              
              <div>
                <Label htmlFor="gender">Gender *</Label>
                <select
                  id="gender"
                  value={data.gender}
                  onChange={(e) => setData('gender', e.target.value)}
                  className="w-full p-2 border border-gray-300 rounded-lg"
                  required
                >
                  <option value="">Select Gender</option>
                  <option value="male">Male</option>
                  <option value="female">Female</option>
                  <option value="other">Other</option>
                  <option value="prefer_not_to_say">Prefer not to say</option>
                </select>
                {errors.gender && <p className="text-red-600 text-sm mt-1">{errors.gender}</p>}
              </div>
              
              <div>
                <Label htmlFor="country">Country *</Label>
                <select
                  id="country"
                  value={data.country}
                  onChange={(e) => setData('country', e.target.value)}
                  className="w-full p-2 border border-gray-300 rounded-lg"
                  required
                >
                  <option value="">Select Country</option>
                  {Object.entries(countries).map(([key, value]) => (
                    <option key={key} value={key}>{value as string}</option>
                  ))}
                </select>
                {errors.country && <p className="text-red-600 text-sm mt-1">{errors.country}</p>}
              </div>
              
              <div>
                <Label htmlFor="city">City *</Label>
                <Input
                  id="city"
                  value={data.city}
                  onChange={(e) => setData('city', e.target.value)}
                  placeholder="Enter your city"
                  required
                />
                {errors.city && <p className="text-red-600 text-sm mt-1">{errors.city}</p>}
              </div>

              {/* Passport Photo (Avatar) */}
              <div>
                <Label htmlFor="avatar">Passport Photo (Headshot) *</Label>
                <Input
                  id="avatar"
                  type="file"
                  accept="image/jpeg,image/png,image/webp"
                  onChange={(e) => {
                    const file = e.currentTarget.files?.[0] || null;
                    setData('avatar', file);
                    setAvatarPreview(file ? URL.createObjectURL(file) : null);
                  }}
                  required
                />
                <p className="text-xs text-gray-500 mt-1">JPEG/PNG/WEBP, max 4MB. Clear front-facing headshot on plain background.</p>
                {avatarPreview && (
                  <img src={avatarPreview} alt="Avatar preview" className="mt-2 h-24 w-24 object-cover border" />
                )}
                {errors.avatar && <p className="text-red-600 text-sm mt-1">{errors.avatar}</p>}
              </div>

              {/* National ID (Front) */}
              <div>
                <Label htmlFor="national_id_front">National ID (Front) *</Label>
                <Input
                  id="national_id_front"
                  type="file"
                  accept="image/jpeg,image/png,image/webp,application/pdf"
                  onChange={(e) => {
                    const file = e.currentTarget.files?.[0] || null;
                    setData('national_id_front', file);
                    setIdPreview(file && file.type.startsWith('image/') ? URL.createObjectURL(file) : null);
                  }}
                  required
                />
                <p className="text-xs text-gray-500 mt-1">Image or PDF, max 4MB. Ensure all details are readable.</p>
                {idPreview && (
                  <img src={idPreview} alt="ID preview" className="mt-2 h-28 rounded border object-contain" />
                )}
                {errors.national_id_front && <p className="text-red-600 text-sm mt-1">{errors.national_id_front}</p>}
              </div>

              {/* National ID (Back) */}

              <div>
                <Label htmlFor="national_id_back">National ID (Back) *</Label>
                <Input
                  id="national_id_back"
                  type="file"
                  accept="image/jpeg,image/png,image/webp,application/pdf"
                  onChange={(e) => {
                    const file = e.currentTarget.files?.[0] || null;
                    setData('national_id_back', file);
                    setIdBackPreview(file && file.type.startsWith('image/') ? URL.createObjectURL(file) : null);
                  }}
                  required
                />
                <p className="text-xs text-gray-500 mt-1">Image or PDF, max 4MB. Ensure all details are readable.</p>
                {idBackPreview && (
                  <img src={idBackPreview} alt="ID preview" className="mt-2 h-28 rounded border object-contain" />
                )}
                {errors.national_id_back && <p className="text-red-600 text-sm mt-1">{errors.national_id_back}</p>}
              </div>

              <div className="flex justify-end pt-6 border-t border-gray-200">
                <Button
                  type="submit"
                  disabled={processing}
                  className="bg-green-600 hover:bg-green-700 text-white"
                >
                  {processing ? (
                    <>
                      <LoaderCircle className="animate-spin h-4 w-4 mr-2" />
                      Saving...
                    </>
                  ) : (
                    'Complete Profile'
                  )}
                </Button>
              </div>
            </form>
          </div>
        </div>
      </div>
      <Footer />
    </>
  );
};

export default CompleteProfile; 