import React, { useState, useEffect } from "react";
import { router } from "@inertiajs/react";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Alert, AlertDescription } from "@/components/ui/alert";
import Header from "@/components/frontend/layout/Header";
import Footer from "@/components/frontend/sections/Footer";
import { LoaderCircle, CreditCard, CheckCircle, AlertCircle, Smartphone, Clock, Phone } from "lucide-react";
import axios from "axios";

interface PaymentState {
  status: 'idle' | 'initiating' | 'processing' | 'waiting' | 'completed' | 'failed';
  message: string;
  checkoutRequestId?: string;
  error?: string;
}

interface Payment {
  id: number;
  amount: number;
  description: string;
  status: string;
  created_at: string;
}

interface PaymentRequiredProps {
  user: {
    phone?: string;
  };
  team?: {
    id: number;
    name?: string;
    category?: string;
  };
  registration_fee: number;
  existing_payments: Payment[];
  payment_methods: any[];
  tournament?: {
    id: number;
    name: string;
    entry_fee: number;
    currency: string;
    category_fees?: Record<string, number>;
  };
}

const PaymentRequired: React.FC<PaymentRequiredProps> = ({ user, team, registration_fee, existing_payments, payment_methods, tournament }) => {
  const [paymentState, setPaymentState] = useState<PaymentState>({
    status: 'idle',
    message: ''
  });
  const [pollingInterval, setPollingInterval] = useState<NodeJS.Timeout | null>(null);
  const [countdownInterval, setCountdownInterval] = useState<NodeJS.Timeout | null>(null);
  const [timeRemaining, setTimeRemaining] = useState(0);
  const [phoneNumber, setPhoneNumber] = useState(user?.phone || '');
  const [phoneError, setPhoneError] = useState('');
  const [pollingDelay, setPollingDelay] = useState(0);
  const [isPolling, setIsPolling] = useState(false);

  // Cleanup intervals on unmount
  useEffect(() => {
    return () => {
      if (pollingInterval) {
        clearInterval(pollingInterval);
      }
      if (countdownInterval) {
        clearInterval(countdownInterval);
      }
    };
  }, [pollingInterval, countdownInterval]);

  const validatePhoneNumber = (phone: string): boolean => {
    // Remove any spaces, dashes, or plus signs
    const cleanPhone = phone.replace(/[\s\-\+]/g, '');
    
    // Check if it matches 254XXXXXXXXX format (Kenyan M-Pesa format)
    const phoneRegex = /^254[0-9]{9}$/;
    
    if (!phoneRegex.test(cleanPhone)) {
      setPhoneError('Please enter a valid Kenyan phone number (e.g., 254712345678)');
      return false;
    }
    
    setPhoneError('');
    return true;
  };

  const formatPhoneNumber = (phone: string): string => {
    // Remove any spaces, dashes, or plus signs
    let cleanPhone = phone.replace(/[\s\-\+]/g, '');
    
    // If it starts with 0, replace with 254
    if (cleanPhone.startsWith('0')) {
      cleanPhone = '254' + cleanPhone.substring(1);
    }
    // If it starts with 7 and is 9 digits, add 254
    else if (cleanPhone.startsWith('7') && cleanPhone.length === 9) {
      cleanPhone = '254' + cleanPhone;
    }
    
    return cleanPhone;
  };

  const handlePayment = async () => {
    // Validate phone number first
    const formattedPhone = formatPhoneNumber(phoneNumber);
    if (!validatePhoneNumber(formattedPhone)) {
      return;
    }

    setPaymentState({
      status: 'initiating',
      message: 'Initiating payment request...'
    });

    // Prepare payment description with category info
    const categoryName = team?.category ? team.category.replace('_', ' ').replace(/\b\w/g, (l: string) => l.toUpperCase()) : '';
    const paymentDescription = team?.category && tournament?.category_fees 
      ? `${tournament.name} - ${categoryName} Category Fee`
      : 'Tournament Registration Fee';

    console.log('Initiating payment:', {
      amount: registration_fee,
      team: team?.name,
      category: team?.category,
      tournament: tournament?.name,
      description: paymentDescription,
      phone: formattedPhone.substring(0, 6) + 'XXX' // Masked for security
    });

    try {
      const response = await axios.post(route('payment.initiate'), {
        amount: registration_fee,
        phone_number: formattedPhone,
        description: paymentDescription,
        team_id: team?.id
      });

      if (response.data.success) {
        setPaymentState({
          status: 'processing',
          message: 'Payment initiated. Complete payment on your phone.',
          checkoutRequestId: response.data.checkout_request_id
        });

        // Start polling delay countdown
        setPollingDelay(5); // 5 seconds delay
        const delayInterval = setInterval(() => {
          setPollingDelay((prev) => {
            if (prev <= 1) {
              clearInterval(delayInterval);
              setIsPolling(true);
              startStatusPolling(response.data.checkout_request_id);
              return 0;
            }
            return prev - 1;
          });
        }, 1000);

        // Start countdown timer
        setTimeRemaining(60); // 1 minute timeout
        const newCountdownInterval = setInterval(() => {
          setTimeRemaining((prev) => {
            if (prev <= 1) {
              clearInterval(newCountdownInterval);
              setCountdownInterval(null);
              // Trigger final API status check when timer hits 0
              triggerFinalStatusCheck(response.data.checkout_request_id);
              return 0;
            }
            return prev - 1;
          });
        }, 1000);
        setCountdownInterval(newCountdownInterval);
      } else {
        setPaymentState({
          status: 'failed',
          message: response.data.message,
          error: 'Payment initiation failed'
        });
      }
    } catch (error: any) {
      setPaymentState({
        status: 'failed',
        message: error.response?.data?.message || 'Payment initiation failed',
        error: 'Network error occurred'
      });
    }
  };

  const triggerFinalStatusCheck = async (checkoutRequestId: string) => {
    try {
      const response = await axios.post(route('payment.poll-status'), {
        checkout_request_id: checkoutRequestId
      });

      if (response.data.success) {
        const { status, message, should_redirect, redirect_url } = response.data;

        setPaymentState({
          status: status,
          message: message,
          checkoutRequestId: checkoutRequestId
        });

        if (status === 'completed') {
          // Stop polling and timer
          if (pollingInterval) {
            clearInterval(pollingInterval);
            setPollingInterval(null);
          }
          if (countdownInterval) {
            clearInterval(countdownInterval);
            setCountdownInterval(null);
          }
          setIsPolling(false);
          
          // Redirect after showing success message
          setTimeout(() => {
            if (should_redirect && redirect_url) {
              router.visit(redirect_url);
            }
          }, 2000);
        } else if (status === 'failed') {
          // Stop polling and timer
          if (pollingInterval) {
            clearInterval(pollingInterval);
            setPollingInterval(null);
          }
          if (countdownInterval) {
            clearInterval(countdownInterval);
            setCountdownInterval(null);
          }
          setIsPolling(false);
        } else {
          // Still processing, continue polling
        }
      }
    } catch (error: any) {
      // Handle rate limit or other errors
      if (error.response?.status === 429 || error.response?.data?.errorCode === '500.001.1001') {
        setPaymentState({
          status: 'failed',
          message: 'Rate limit reached. Please try again later.',
          error: 'Rate limit error'
        });
      } else {
        setPaymentState({
          status: 'failed',
          message: 'Unable to check payment status. Please contact support.',
          error: 'API error'
        });
      }
      
      // Stop polling and timer on error
      if (pollingInterval) {
        clearInterval(pollingInterval);
        setPollingInterval(null);
      }
      if (countdownInterval) {
        clearInterval(countdownInterval);
        setCountdownInterval(null);
      }
      setIsPolling(false);
    }
  };

  const startStatusPolling = (checkoutRequestId: string) => {
    let pollCount = 0;
    const interval = setInterval(async () => {
      try {
        pollCount++;
        const response = await axios.post(route('payment.poll-status'), {
          checkout_request_id: checkoutRequestId
        });

        if (response.data.success) {
          const { status, message, should_redirect, redirect_url } = response.data;

          setPaymentState({
            status: status,
            message: message,
            checkoutRequestId: checkoutRequestId
          });

          if (status === 'completed') {
            clearInterval(interval);
            setPollingInterval(null);
            // Stop timer as well
            if (countdownInterval) {
              clearInterval(countdownInterval);
              setCountdownInterval(null);
            }
            setIsPolling(false);
            
            // Redirect after showing success message
            setTimeout(() => {
              if (should_redirect && redirect_url) {
                router.visit(redirect_url);
              }
            }, 2000);
          } else if (status === 'failed') {
            clearInterval(interval);
            setPollingInterval(null);
            // Stop timer as well
            if (countdownInterval) {
              clearInterval(countdownInterval);
              setCountdownInterval(null);
            }
            setIsPolling(false);
          }
        }
      } catch (error: any) {
        // If we get rate limit errors, increase polling interval
        if (error.response?.status === 429 || error.response?.data?.errorCode === '500.001.1001') {
          clearInterval(interval);
          // Retry with longer interval
          setTimeout(() => {
            startStatusPolling(checkoutRequestId);
          }, 10000); // Wait 10 seconds before retrying
        }
      }
    }, 2000); // Poll every 2 seconds (faster for database checks)

    setPollingInterval(interval);

    // Stop polling after 5 minutes
    setTimeout(() => {
      clearInterval(interval);
      setPollingInterval(null);
      // Stop timer as well
      if (countdownInterval) {
        clearInterval(countdownInterval);
        setCountdownInterval(null);
      }
      setIsPolling(false);
      if (paymentState.status === 'processing') {
        setPaymentState({
          status: 'failed',
          message: 'Payment timeout. Please try again.',
          error: 'Timeout occurred'
        });
      }
    }, 300000); // 5 minutes
  };

  const getStatusIcon = () => {
    switch (paymentState.status) {
      case 'initiating':
        return <LoaderCircle className="animate-spin h-8 w-8 text-blue-600" />;
      case 'processing':
        return <Smartphone className="h-8 w-8 text-orange-600 animate-pulse" />;
      case 'waiting':
        return <Clock className="h-8 w-8 text-yellow-600 animate-pulse" />;
      case 'completed':
        return <CheckCircle className="h-8 w-8 text-green-600" />;
      case 'failed':
        return <AlertCircle className="h-8 w-8 text-red-600" />;
      default:
        return <CreditCard className="h-8 w-8 text-gray-600" />;
    }
  };

  const getStatusColor = () => {
    switch (paymentState.status) {
      case 'initiating':
        return 'border-blue-200 bg-blue-50';
      case 'processing':
        return 'border-orange-200 bg-orange-50';
      case 'waiting':
        return 'border-yellow-200 bg-yellow-50';
      case 'completed':
        return 'border-green-200 bg-green-50';
      case 'failed':
        return 'border-red-200 bg-red-50';
      default:
        return 'border-gray-200 bg-gray-50';
    }
  };

  const formatTime = (seconds: number) => {
    const minutes = Math.floor(seconds / 60);
    const remainingSeconds = seconds % 60;
    return `${minutes}:${remainingSeconds.toString().padStart(2, '0')}`;
  };

  return (
    <>
      <Header />
      <div className="min-h-screen bg-gradient-to-br from-green-50 to-gray-50 py-8">
        <div className="container mx-auto px-4 max-w-2xl">
          <div className="text-center mb-8">
            <div className="w-20 h-20 bg-green-600 rounded-full flex items-center justify-center mx-auto mb-4">
              <CreditCard className="h-10 w-10 text-white" />
            </div>
            <h1 className="text-3xl font-bold text-gray-900">Payment Required</h1>
            <p className="text-gray-600 mt-2">Complete your registration by making the payment</p>
          </div>

          <div className="bg-white rounded-lg shadow-xl border border-gray-200 p-8">
            {/* Payment Status Section */}
            {paymentState.status !== 'idle' && (
              <div className={`mb-6 border rounded-lg p-6 ${getStatusColor()}`}>
                <div className="flex items-center space-x-4">
                  {getStatusIcon()}
                  <div className="flex-1">
                    <h3 className="font-semibold text-lg capitalize">{paymentState.status}</h3>
                    <p className="text-sm mt-1">{paymentState.message}</p>
                    
                                         {(paymentState.status === 'processing' || paymentState.status === 'waiting') && (
                       <div className="mt-3 space-y-2">
                         {pollingDelay > 0 && (
                           <div className="flex items-center space-x-2 text-sm text-blue-600">
                             <Clock className="h-4 w-4" />
                             <span>Polling starts in {pollingDelay} seconds</span>
                           </div>
                         )}
                         {isPolling && (
                           <div className="flex items-center space-x-2 text-sm text-orange-600">
                             <LoaderCircle className="animate-spin h-4 w-4" />
                             <span>Checking payment status...</span>
                           </div>
                         )}
                         {timeRemaining > 0 && (
                           <div className="flex items-center space-x-2 text-sm text-gray-600">
                             <Clock className="h-4 w-4" />
                             <span>Time remaining: {formatTime(timeRemaining)}</span>
                           </div>
                         )}
                       </div>
                     )}
                    
                                         {(paymentState.status === 'processing' || paymentState.status === 'waiting') && (
                       <div className="mt-3">
                         <div className="bg-white/50 rounded-full p-2 text-center">
                           <p className="text-sm font-medium">
                             {paymentState.status === 'waiting' 
                               ? 'Complete the payment on your phone' 
                               : 'Check your phone for M-Pesa prompt'
                             }
                           </p>
                         </div>
                       </div>
                     )}

                    {paymentState.status === 'completed' && (
                      <div className="mt-3">
                        <p className="text-sm text-green-700">Redirecting to dashboard...</p>
                      </div>
                    )}

                    {paymentState.error && (
                      <Alert className="mt-3 border-red-200 bg-red-50">
                        <AlertCircle className="h-4 w-4 text-red-600" />
                        <AlertDescription className="text-red-800">
                          {paymentState.error}
                        </AlertDescription>
                      </Alert>
                    )}
                  </div>
                </div>
              </div>
            )}
            
            <div className="mb-6">
              <h2 className="text-xl font-semibold text-gray-900 mb-4">Registration Fee</h2>
              <div className="bg-green-50 border border-green-200 rounded-lg p-4">
                {tournament && (
                  <div className="mb-3 pb-3 border-b border-green-200">
                    <p className="text-sm text-green-800">
                      <strong>Tournament:</strong> {tournament.name}
                    </p>
                    {team?.category && (
                      <div className="mt-2 space-y-1">
                        <p className="text-sm text-green-800">
                          <strong>Team Category:</strong> {team.category.replace('_', ' ').replace(/\b\w/g, l => l.toUpperCase())}
                        </p>
                        
                      </div>
                    )}
                  </div>
                )}
                <div className="flex items-center justify-between">
                  <span className="text-lg font-medium text-green-900">Amount Due:</span>
                  <span className="text-2xl font-bold text-green-600">
                    {tournament?.currency || 'KES'} {(registration_fee || 0).toLocaleString()}
                  </span>
                </div>
                {team?.category && tournament?.category_fees && (
                  <div className="mt-3 pt-3 border-t border-green-200">
                    <p className="text-xs text-green-700">
                      <strong>Fee Breakdown:</strong> This is the registration fee for the <strong>{team.category.replace('_', ' ')}</strong> category.
                    </p>
                  </div>
                )}
              </div>
            </div>

            <div className="mb-6">
              <h3 className="text-lg font-semibold text-gray-900 mb-4 flex items-center space-x-2 hidden" >
                <Phone className="h-5 w-5 text-green-600" />
                <span>M-Pesa Phone Number</span>
                <span className="text-red-500 text-sm font-normal">*</span>
              </h3>
              <div className="space-y-3">
                <div className="bg-blue-50 border border-blue-200 rounded-lg p-4 hidden">
                  <div className="flex items-start space-x-3">
                    <div className="flex-shrink-0 w-8 h-8 bg-blue-100 rounded-full flex items-center justify-center">
                      <Phone className="h-4 w-4 text-blue-600" />
                    </div>
                    <div className="flex-1">
                      <p className="text-sm font-medium text-blue-900 mb-1">
                        Enter your M-Pesa registered phone number
                      </p>
                      <p className="text-xs text-blue-700">
                        This is where you'll receive the payment prompt
                      </p>
                    </div>
                  </div>
                </div>
                
                <div className="relative">
                  <Label htmlFor="phone" className="text-sm font-medium text-gray-700 mb-2 block">
                    Phone Number
                  </Label>
                  <div className="relative">
                    <div className="absolute inset-y-0 left-0 pl-4 flex items-center pointer-events-none">
                      <Phone className="h-5 w-5 text-gray-400" />
                    </div>
                    <Input
                      id="phone"
                      type="tel"
                      value={phoneNumber}
                      onChange={(e) => {
                        setPhoneNumber(e.target.value);
                        if (phoneError) setPhoneError(''); // Clear error on input change
                      }}
                      placeholder="07XXXXXXXX"
                      className={`pl-12 h-12 text-lg font-mono ${
                        phoneError 
                          ? 'border-red-300 focus:border-red-500 focus:ring-red-500 bg-red-50' 
                          : phoneNumber.trim() && !phoneError
                          ? 'border-green-300 focus:border-green-500 focus:ring-green-500 bg-green-50'
                          : 'border-gray-300 focus:border-blue-500 focus:ring-blue-500'
                      }`}
                      disabled={paymentState.status === 'initiating' || paymentState.status === 'processing'}
                    />
                    {phoneNumber.trim() && !phoneError && (
                      <div className="absolute inset-y-0 right-0 pr-3 flex items-center">
                        <CheckCircle className="h-5 w-5 text-green-500" />
                      </div>
                    )}
                  </div>
                </div>
                
                {phoneError && (
                  <div className="flex items-start space-x-2 p-3 bg-red-50 border border-red-200 rounded-lg">
                    <AlertCircle className="h-4 w-4 text-red-500 mt-0.5 flex-shrink-0" />
                    <div>
                      <p className="text-sm text-red-800 font-medium">Invalid phone number</p>
                      <p className="text-xs text-red-700 mt-1">{phoneError}</p>
                    </div>
                  </div>
                )}
                
                <div className="bg-gray-50 border border-gray-200 rounded-lg p-3 hidden">
                  <p className="text-xs text-gray-600 mb-2 font-medium">Accepted formats:</p>
                  <div className="grid grid-cols-1 sm:grid-cols-3 gap-2 text-xs">
                    <div className="bg-white p-2 rounded border">
                      <span className="font-mono text-gray-700">0712345678</span>
                    </div>
                    <div className="bg-white p-2 rounded border">
                      <span className="font-mono text-gray-700">712345678</span>
                    </div>
                    <div className="bg-white p-2 rounded border">
                      <span className="font-mono text-gray-700">254712345678</span>
                    </div>
                  </div>
                </div>
              </div>
            </div>

            {existing_payments && existing_payments.length > 0 && (
              <div className="mb-6 hidden">
                <h3 className="text-lg font-semibold text-gray-900 mb-4">Previous Payments</h3>
                <div className="space-y-3">
                  {existing_payments.map((payment: Payment) => (
                    <div key={payment.id} className="border border-gray-200 rounded-lg p-4">
                      <div className="flex items-center justify-between">
                        <div>
                          <p className="font-medium text-gray-900">KES {payment.amount}</p>
                          <p className="text-sm text-gray-600">{payment.description}</p>
                          <p className="text-xs text-gray-500">{new Date(payment.created_at).toLocaleDateString()}</p>
                        </div>
                        <div className="flex items-center">
                          {payment.status === 'completed' ? (
                            <CheckCircle className="h-5 w-5 text-green-600" />
                          ) : payment.status === 'failed' ? (
                            <AlertCircle className="h-5 w-5 text-red-600" />
                          ) : (
                            <div className="animate-spin h-5 w-5 border-2 border-blue-600 border-t-transparent rounded-full"></div>
                          )}
                          <span className="ml-2 text-sm font-medium capitalize">{payment.status}</span>
                        </div>
                      </div>
                    </div>
                  ))}
                </div>
              </div>
            )}

            <div className="mb-6">
              <h3 className="text-lg font-semibold text-gray-900 mb-4">Payment Method</h3>
              <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
                <div className="flex items-center space-x-3">
                  <CreditCard className="h-6 w-6 text-blue-600" />
                  <div>
                    <p className="font-medium text-blue-900">M-Pesa STK Push</p>
                    <p className="text-sm text-blue-700">Pay securely using your M-Pesa PIN</p>
                  </div>
                </div>
              </div>
            </div>

            <div className="mb-6">
              <h3 className="text-lg font-semibold text-gray-900 mb-4">Payment Process</h3>
              <ol className="space-y-2 text-sm text-gray-600">
                <li className="flex items-start space-x-2">
                  <span className="flex-shrink-0 w-5 h-5 bg-green-600 text-white rounded-full flex items-center justify-center text-xs font-medium">1</span>
                  <span>Click "Pay Now" to initiate the payment</span>
                </li>
                <li className="flex items-start space-x-2">
                  <span className="flex-shrink-0 w-5 h-5 bg-green-600 text-white rounded-full flex items-center justify-center text-xs font-medium">2</span>
                  <span>You'll receive an M-Pesa prompt on your phone ({phoneNumber || 'your entered number'})</span>
                </li>
                <li className="flex items-start space-x-2">
                  <span className="flex-shrink-0 w-5 h-5 bg-green-600 text-white rounded-full flex items-center justify-center text-xs font-medium">3</span>
                  <span>Enter your M-Pesa PIN to complete the payment</span>
                </li>
                <li className="flex items-start space-x-2">
                  <span className="flex-shrink-0 w-5 h-5 bg-green-600 text-white rounded-full flex items-center justify-center text-xs font-medium">4</span>
                  <span>You'll be automatically redirected upon successful payment</span>
                </li>
              </ol>
            </div>

            <div className="flex justify-end pt-6 border-t border-gray-200">
                             <Button
                 onClick={handlePayment}
                 disabled={paymentState.status === 'initiating' || paymentState.status === 'processing' || paymentState.status === 'waiting' || !phoneNumber.trim()}
                 className="bg-green-600 hover:bg-green-700 text-white px-8 py-3 text-lg font-medium disabled:opacity-50"
               >
                {paymentState.status === 'initiating' ? (
                  <>
                    <LoaderCircle className="animate-spin h-5 w-5 mr-2" />
                    Initiating...
                  </>
                                 ) : (paymentState.status === 'processing' || paymentState.status === 'waiting') ? (
                   <>
                     <Smartphone className="animate-pulse h-5 w-5 mr-2" />
                     {pollingDelay > 0 ? (
                       <>
                         Starting in {pollingDelay}s
                       </>
                     ) : isPolling ? (
                       <>
                         Checking Status...
                       </>
                     ) : paymentState.status === 'waiting' ? (
                       <>
                         Waiting...
                       </>
                     ) : (
                       <>
                         Check Your Phone
                       </>
                     )}
                   </>
                 ) : (
                  <>
                    <CreditCard className="h-5 w-5 mr-2" />
                    Pay Now
                  </>
                )}
              </Button>
            </div>
          </div>
        </div>
      </div>
      <Footer />
    </>
  );
};

export default PaymentRequired; 