import React, { useState } from "react";
import { router, useForm } from "@inertiajs/react";
import { Input } from "@/components/ui/input";
import { Button } from "@/components/ui/button";
import { Label } from "@/components/ui/label";
import Header from "@/components/frontend/layout/Header";
import Footer from "@/components/frontend/sections/Footer";
import { LoaderCircle, Mail, Shield, Users } from "lucide-react";

const PlayerVerification = ({ player, team, email }) => {
  const [otp, setOtp] = useState('');
  const [isVerifying, setIsVerifying] = useState(false);
  const [error, setError] = useState('');
  const [success, setSuccess] = useState('');

  const { post, processing } = useForm();

  const handleVerify = (e) => {
    e.preventDefault();
    if (!otp || otp.length !== 6) {
      setError('Please enter a valid 6-digit OTP');
      return;
    }

    setIsVerifying(true);
    setError('');

    post(route('otp.player.verify.submit'), {
      data: { email, otp },
      onSuccess: (response) => {
        setSuccess('Email verified successfully! You will receive login credentials soon.');
        setTimeout(() => {
          router.visit(route('otp.success', { type: 'player' }));
        }, 2000);
      },
      onError: (errors) => {
        setError(errors.otp || 'Verification failed. Please try again.');
        setIsVerifying(false);
      }
    });
  };

  const handleResend = () => {
    post(route('otp.player.resend'), {
      data: { email },
      onSuccess: () => {
        setSuccess('OTP resent successfully! Check your email.');
        setError('');
      },
      onError: (errors) => {
        setError(errors.email || 'Failed to resend OTP. Please try again.');
      }
    });
  };

  return (
    <>
      <Header />
      <div className="min-h-screen bg-gradient-to-br from-green-50 to-gray-50 py-8">
        <div className="container mx-auto px-4 max-w-md">
          <div className="text-center mb-8">
            <div className="w-20 h-20 bg-green-600 rounded-full flex items-center justify-center mx-auto mb-4">
              <Shield className="h-10 w-10 text-white" />
            </div>
            <h1 className="text-3xl font-bold text-gray-900">Verify Your Email</h1>
            <p className="text-gray-600 mt-2">Complete your team registration</p>
          </div>

          <div className="bg-white rounded-lg shadow-xl border border-gray-200 p-8">
            {/* Player Info */}
            <div className="mb-6 p-4 bg-blue-50 border border-blue-200 rounded-lg">
              <div className="flex items-center space-x-3 mb-3">
                <Users className="h-5 w-5 text-blue-600" />
                <h3 className="font-medium text-blue-900">Player Information</h3>
              </div>
              <div className="text-sm text-blue-800">
                <p><strong>Name:</strong> {player.display_name}</p>
                <p><strong>Team:</strong> {team.name}</p>
                <p><strong>Position:</strong> {player.football_position}</p>
                <p><strong>Email:</strong> {email}</p>
              </div>
            </div>

            {/* OTP Form */}
            <form onSubmit={handleVerify} className="space-y-6">
              <div>
                <Label htmlFor="otp">Enter OTP Code *</Label>
                <Input
                  id="otp"
                  type="text"
                  value={otp}
                  onChange={(e) => setOtp(e.target.value.replace(/\D/g, '').slice(0, 6))}
                  placeholder="Enter 6-digit code"
                  maxLength={6}
                  className="text-center text-2xl font-mono tracking-widest"
                  required
                />
                <p className="text-sm text-gray-600 mt-2">
                  Enter the 6-digit code sent to your email
                </p>
              </div>

              {error && (
                <div className="p-3 bg-red-50 border border-red-200 rounded-lg">
                  <p className="text-red-800 text-sm">{error}</p>
                </div>
              )}

              {success && (
                <div className="p-3 bg-green-50 border border-green-200 rounded-lg">
                  <p className="text-green-800 text-sm">{success}</p>
                </div>
              )}

              <div className="space-y-3">
                <Button
                  type="submit"
                  disabled={processing || isVerifying || !otp || otp.length !== 6}
                  className="w-full bg-green-600 hover:bg-green-700 text-white"
                >
                  {(processing || isVerifying) ? (
                    <>
                      <LoaderCircle className="animate-spin h-4 w-4 mr-2" />
                      Verifying...
                    </>
                  ) : (
                    'Verify Email'
                  )}
                </Button>

                <Button
                  type="button"
                  variant="outline"
                  onClick={handleResend}
                  disabled={processing}
                  className="w-full"
                >
                  <Mail className="h-4 w-4 mr-2" />
                  Resend OTP
                </Button>
              </div>
            </form>

            {/* Help Text */}
            <div className="mt-6 p-4 bg-yellow-50 border border-yellow-200 rounded-lg">
              <h4 className="text-sm font-medium text-yellow-900 mb-2">Need help?</h4>
              <ul className="text-xs text-yellow-800 space-y-1">
                <li>• Check your email spam folder</li>
                <li>• Ensure you entered the correct email address</li>
                <li>• Contact your team captain if you haven't received the email</li>
                <li>• OTP codes expire after 24 hours</li>
              </ul>
            </div>
          </div>
        </div>
      </div>
      <Footer />
    </>
  );
};

export default PlayerVerification; 