import React, { useState } from 'react';
import { Head, useForm } from '@inertiajs/react';
import { Users, Crown, UserCheck, Briefcase, FileCheck } from 'lucide-react';
import Header from '@/frontend/components/layout/Header'; // Comment out for now
import RegistrationSteps from '@/components/frontend/registration-steps';
import TeamInfoStep from '@/components/frontend/registration-form-steps/team-info-step';
import CaptainStep from '@/components/frontend/registration-form-steps/captain-step';
import PlayersStep from '@/components/frontend/registration-form-steps/players-step';
import OfficialsStep from '@/components/frontend/registration-form-steps/officials-step';
import ReviewStep from '@/components/frontend/registration-form-steps/review-step';

interface Player {
  name: string;
  email: string;
  phone: string;
  position: string;
  jersey_number: string;
}

interface Official {
  name: string;
  email: string;
  phone: string;
  role: string;
  qualifications: string;
}

const defaultPlayer: Player = {
  name: "",
  email: "",
  phone: "",
  position: "Midfielder",
  jersey_number: ""
};

const defaultOfficial: Official = {
  name: "",
  email: "",
  phone: "",
  role: "Coach",
  qualifications: ""
};

const RegisterTeamNew = () => {
  const [currentStep, setCurrentStep] = useState("team");
  const [termsAccepted, setTermsAccepted] = useState(false);

  // Mock existing users data - in real app, this would come from props or API
  const existingUsers = [
    { id: 1, name: "John Smith", email: "john@example.com", phone: "+1 555-0101" },
    { id: 2, name: "Sarah Johnson", email: "sarah@example.com", phone: "+1 555-0102" },
    { id: 3, name: "Mike Davis", email: "mike@example.com", phone: "+1 555-0103" },
    { id: 4, name: "Lisa Wilson", email: "lisa@example.com", phone: "+1 555-0104" },
    { id: 5, name: "Tom Brown", email: "tom@example.com", phone: "+1 555-0105" },
  ];

  const { data, setData, post, processing, errors, reset } = useForm({
    team: {
      name: "",
      email: "",
      phone: "",
      location: "",
      coach: "",
      year: new Date().getFullYear()
    },
    captain: {
      name: "",
      email: "",
      phone: "",
      password: "",
      password_confirmation: "",
      existing_user_id: null
    },
    players: Array.from({ length: 11 }, () => ({ 
      ...defaultPlayer, 
      existing_user_id: null 
    })),
    officials: [{ 
      ...defaultOfficial, 
      existing_user_id: null 
    }],
    terms_accepted: false
  });

  const steps = [
    {
      id: 'team',
      title: 'Team Info',
      description: 'Basic team information',
      icon: <Users className="h-6 w-6 text-green-600" />,
    },
    {
      id: 'captain',
      title: 'Team Captain',
      description: 'Captain account setup',
      icon: <Crown className="h-6 w-6 text-green-600" />,
    },
    {
      id: 'players',
      title: 'Players',
      description: 'Player roster details',
      icon: <UserCheck className="h-6 w-6 text-green-600" />,
    },
    {
      id: 'officials',
      title: 'Officials',
      description: 'Coaching staff info',
      icon: <Briefcase className="h-6 w-6 text-green-600" />,
    },
    {
      id: 'review',
      title: 'Review',
      description: 'Final confirmation',
      icon: <FileCheck className="h-6 w-6 text-green-600" />,
    },
  ];

  const canProceedToNext = () => {
    switch (currentStep) {
      case "team":
        return data.team.name && data.team.email && data.team.phone && data.team.location && data.team.coach;
      case "captain":
        if (data.captain.existing_user_id) {
          return data.captain.name && data.captain.email && data.captain.phone;
        }
        return data.captain.name && data.captain.email && data.captain.phone && 
               data.captain.password && data.captain.password_confirmation;
      case "players":
        const validPlayers = data.players.filter((p: any) => p.name && p.email && p.phone && p.position);
        return validPlayers.length >= 11;
      case "officials":
        const validOfficials = data.officials.filter((o: any) => o.name && o.email && o.phone && o.role);
        return validOfficials.length >= 1;
      case "review":
        return termsAccepted;
      default:
        return true;
    }
  };

  const nextStep = () => {
    const stepOrder = ["team", "captain", "players", "officials", "review"];
    const currentIndex = stepOrder.indexOf(currentStep);
    if (currentIndex < stepOrder.length - 1 && canProceedToNext()) {
      setCurrentStep(stepOrder[currentIndex + 1]);
    }
  };

  const prevStep = () => {
    const stepOrder = ["team", "captain", "players", "officials", "review"];
    const currentIndex = stepOrder.indexOf(currentStep);
    if (currentIndex > 0) {
      setCurrentStep(stepOrder[currentIndex - 1]);
    }
  };

  const handleSubmit = () => {
    if (!canProceedToNext()) return;
    
    // Set terms_accepted in form data
    setData('terms_accepted', termsAccepted as any);
    
    // Submit the form
    post(route('register'), {
      onSuccess: () => {
        // Handle success - will redirect to profile completion
      },
      onError: (errors) => {
        // Navigate back to step with errors if needed
        if (errors['team.name'] || errors['team.email']) setCurrentStep('team');
        else if (errors['captain.name'] || errors['captain.email']) setCurrentStep('captain');
        else if (errors['players']) setCurrentStep('players');
        else if (errors['officials']) setCurrentStep('officials');
      }
    });
  };

  const renderCurrentStep = () => {
    switch (currentStep) {
      case 'team':
        return (
          <TeamInfoStep
            data={data}
            setData={setData}
            errors={errors}
          />
        );
      case 'captain':
        return (
          <CaptainStep
            data={data}
            setData={setData}
            errors={errors}
            existingUsers={existingUsers}
          />
        );
      case 'players':
        return (
          <PlayersStep
            data={data}
            setData={setData}
            errors={errors}
            existingUsers={existingUsers}
          />
        );
      case 'officials':
        return (
          <OfficialsStep
            data={data}
            setData={setData}
            errors={errors}
            existingUsers={existingUsers}
          />
        );
      case 'review':
        return (
          <ReviewStep
            data={data}
            setData={setData}
            errors={errors}
            termsAccepted={termsAccepted}
            setTermsAccepted={setTermsAccepted}
            registration_fee={0}
          />
        );
      default:
        return null;
    }
  };

  return (
    <>
      <Head title="Team Registration" />
      {/* <Header /> */}
      
      <RegistrationSteps
        steps={steps}
        currentStep={currentStep}
        onStepChange={setCurrentStep}
        onNext={nextStep}
        onPrevious={prevStep}
        onSubmit={handleSubmit}
        isSubmitting={processing}
        canProceed={!!canProceedToNext()}
        isLastStep={currentStep === 'review'}
      >
        {renderCurrentStep()}
      </RegistrationSteps>
    </>
  );
};

export default RegisterTeamNew;