import React, { useState } from 'react';
import { useForm, router, Head } from '@inertiajs/react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Badge } from '@/components/ui/badge';
import { Users, User, Shield, FileText, CheckCircle, AlertCircle, LoaderCircle, Trophy } from 'lucide-react';
import TournamentSelectionStep from '@/components/frontend/registration-form-steps/tournament-selection-step';
import CategorySelector from '@/components/CategorySelector';
import Footer from '@/components/frontend/sections/Footer';
import Header from '@/components/frontend/layout/Header';

interface Player {
  name: string;
  email: string;
  phone: string;
  position: string;
  id_number: string;
  country: string;
  county: string;
  location: string;
  date_of_birth: string;
  gender: 'male' | 'female' | 'other';
}

interface Official {
  name: string;
  email: string;
  phone: string;
  role: string;
  qualifications: string;
}

interface Tournament {
  id: number;
  name: string;
  description: string;
  entry_fee: number;
  currency: string;
  start_date?: string;
  end_date?: string;
  registration_end?: string;
  is_free: boolean;
  min_players?: number;
  max_players?: number;
  enable_categories?: boolean;
  category_fees?: Record<string, number>;
  available_categories?: string[];
}

const createDefaultPlayer = (): Player => ({
  name: "",
  email: "",
  phone: "",
  position: "Midfielder",
  id_number: "",
  country: "Kenya",
  county: "",
  location: "",
  date_of_birth: "",
  gender: "male"
});

const defaultOfficial: Official = {
  name: "",
  email: "",
  phone: "",
  role: "Coach",
  qualifications: ""
};

interface FormData {
  team: {
    name: string;
    email: string;
    phone: string;
    country: string;
    county: string;
    location: string;
    year: number;
    category: string;
  };
  captain: {
    name: string;
    email: string;
    phone: string;
    password: string;
    password_confirmation: string;
  };
  players: Player[];
  officials: Official[];
  terms_accepted: boolean;
  tournament_id: number | null;
  [key: string]: any;
}

interface CategoryLocation {
  restrict_locations: boolean;
  locations: string[];
}

interface RegisterTeamProps {
  registration_fee: number;
  tournaments: Tournament[];
  default_tournament: Tournament | null;
  max_players: number;
  min_players: number;
  max_officials: number;
  categories: Record<string, any>;
  category_locations: Record<string, CategoryLocation>;
  countries: string[];
  counties: string[];
}

const RegisterTeam: React.FC<RegisterTeamProps> = ({ registration_fee, tournaments, default_tournament, max_players, min_players, max_officials, categories, category_locations, countries, counties }) => {
  const [currentStep, setCurrentStep] = useState("tournament-selection");
  const [termsAccepted, setTermsAccepted] = useState(false);
  const [showTermsModal, setShowTermsModal] = useState(false);
  const [selectedTournament, setSelectedTournament] = useState<Tournament | null>(default_tournament);
  const [useExistingTeam, setUseExistingTeam] = useState(false);
  const [stepsWithErrors, setStepsWithErrors] = useState<string[]>([]);

  // Get min/max players from selected tournament or fallback to config
  const minPlayers = selectedTournament?.min_players || min_players;
  const maxPlayers = selectedTournament?.max_players || max_players;

  const { data, setData, post, processing, errors: formErrors, reset } = useForm({
    team: {
      name: "",
      email: "",
      phone: "",
      country: countries[0] || "Kenya",
      county: counties[0] || "",
      location: "",
      year: new Date().getFullYear(),
      category: "men"
    },
    captain: {
      name: "",
      email: "",
      phone: "",
      password: "",
      password_confirmation: ""
    },
    players: Array.from({ length: selectedTournament?.min_players || min_players }, () => createDefaultPlayer()),
    officials: [{ ...defaultOfficial }],
    terms_accepted: false,
    tournament_id: selectedTournament?.id || null
  } as any);

  // Handle tournament selection
  const handleSelectTournament = (tournament: Tournament) => {
    setSelectedTournament(tournament);
    setData('tournament_id', tournament.id);
  };

  const handleCreateNewTeam = () => {
    setUseExistingTeam(false);
    setData('team', { ...data.team, name: "" });
  };

  // Email validation function
  const isValidEmail = (email: string) => {
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    return emailRegex.test(email);
  };

  // Phone validation function
  const isValidPhone = (phone: string) => {
    const phoneRegex = /^[\+]?[0-9][\d]{0,15}$/;
    return phoneRegex.test(phone.replace(/\s/g, ''));
  };

  // Name validation function
  // Name is letter and spaces and integers
  const isValidName = (name: string) => {
    return name.trim().length >= 2 && /^[a-zA-Z\s\d]*$/.test(name.trim());
  };

  const handlePlayerChange = (idx: number, field: keyof Player, value: string) => {
    setData('players', data.players.map((p, i) => i === idx ? { ...p, [field]: value } : p));
  };

  const addPlayer = () => {
    if (data.players.length < maxPlayers) {
      setData('players', [...data.players, createDefaultPlayer()]);
    }
  };

  const removePlayer = (idx: number) => {
    if (data.players.length > minPlayers) {
      setData('players', data.players.filter((_, i) => i !== idx));
    }
  };

  const handleOfficialChange = (idx: number, field: keyof Official, value: string) => {
    setData('officials', data.officials.map((o, i) => i === idx ? { ...o, [field]: value } : o));
  };

  const addOfficial = () => {
    if (data.officials.length < 10) {
      setData('officials', [...data.officials, { ...defaultOfficial }]);
    }
  };

  const removeOfficial = (idx: number) => {
    if (data.officials.length > 1) {
      setData('officials', data.officials.filter((_, i) => i !== idx));
    }
  };

  const getStepForError = (errorKey: string): string => {
    // Map error keys to steps
    if (errorKey.startsWith('team.')) return 'team';
    if (errorKey.startsWith('captain.')) return 'captain';
    if (errorKey.startsWith('players.')) return 'players';
    if (errorKey.startsWith('officials.')) return 'officials';
    if (errorKey === 'tournament_id') return 'tournament-selection';
    return 'review';
  };

  // Cast errors to allow dynamic indexing
  const errors = formErrors as Record<string, any>;

  // Get errors for specific step
  const getStepErrors = (step: string): Record<string, string> => {
    const stepErrors: Record<string, string> = {};
    Object.entries(errors).forEach(([key, value]) => {
      if (getStepForError(key) === step) {
        stepErrors[key] = Array.isArray(value) ? value[0] : value;
      }
    });
    return stepErrors;
  };

  // Render error summary for a step
  const renderErrorSummary = (step: string) => {
    const stepErrors = getStepErrors(step);
    const errorCount = Object.keys(stepErrors).length;
    
    if (errorCount === 0) return null;

    return (
      <div className="bg-red-50 border-l-4 border-red-500 p-4 mb-6 rounded-r-lg">
        <div className="flex items-start gap-3">
          <AlertCircle className="h-5 w-5 text-red-600 mt-0.5 flex-shrink-0" />
          <div className="flex-1">
            <h3 className="text-red-900 font-semibold mb-2">
              {errorCount} Error{errorCount > 1 ? 's' : ''} Found
            </h3>
            <ul className="space-y-1 text-sm text-red-800">
              {Object.entries(stepErrors).map(([key, message]) => (
                <li key={key} className="flex items-start gap-2">
                  <span className="text-red-600">•</span>
                  <span>{message}</span>
                </li>
              ))}
            </ul>
          </div>
        </div>
      </div>
    );
  };

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    if (!termsAccepted) {
      alert('Please accept the terms and conditions to continue.');
      return;
    }
    
    // Clear previous error states
    setStepsWithErrors([]);
    
    // Set terms_accepted and then submit
    setData('terms_accepted', true as any);
    
    // Submit after state update
    setTimeout(() => {
      post(route('register.team.submit'), {
        onSuccess: () => {
          // Handle success - will redirect to profile completion
          setStepsWithErrors([]);
        },
        onError: (errors) => {
          // Identify which steps have errors
          const errorSteps = new Set<string>();
          Object.keys(errors).forEach(errorKey => {
            const step = getStepForError(errorKey);
            errorSteps.add(step);
          });
          
          const errorStepsArray = Array.from(errorSteps);
          setStepsWithErrors(errorStepsArray);
          
          // Navigate to first step with error
          const stepOrder = ["tournament-selection", "team", "captain", "players", "officials", "review"];
          const firstErrorStep = stepOrder.find(step => errorStepsArray.includes(step));
          
          if (firstErrorStep) {
            setCurrentStep(firstErrorStep);
            // Scroll to top
            window.scrollTo({ top: 0, behavior: 'smooth' });
          }
        }
      });
    }, 0);
  };

  // Helper to show completion indicators without blocking navigation
  const getStepCompletionStatus = (step: string) => {
    switch (step) {
      case "tournament-selection":
        return selectedTournament !== null;
      case "team":
        return data.team.name && data.team.email && data.team.phone && data.team.country && data.team.county && data.team.location && data.team.category;
      case "captain":
        return data.captain.name && data.captain.email && data.captain.password;
      case "players":
        // Check minimum count AND that all required fields are filled for each player
        return data.players.length >= minPlayers &&
               data.players.every((p: any) => 
                 p.name && p.email && p.position && p.id_number && 
                 p.country && p.county && p.location && 
                 p.date_of_birth && p.gender
               );
      case "officials":
        // Check minimum count AND that all required fields are filled for each official
        return data.officials.length >= 1 &&
               data.officials.every((o: any) => 
                 o.name && o.email && o.phone && o.role
               );
      case "review":
        // Review step is complete when all other steps are complete
        const playersComplete = data.players.length >= minPlayers &&
                                data.players.every((p: any) => 
                                  p.name && p.email && p.position && p.id_number && 
                                  p.country && p.county && p.location && 
                                  p.date_of_birth && p.gender
                                );
        const officialsComplete = data.officials.length >= 1 &&
                                  data.officials.every((o: any) => 
                                    o.name && o.email && o.phone && o.role
                                  );
        
        return selectedTournament !== null &&
               data.team.name && data.team.email && data.team.phone && data.team.country && data.team.county && data.team.location && data.team.category &&
               data.captain.name && data.captain.email && data.captain.password &&
               playersComplete &&
               officialsComplete;
      default:
        return false;
    }
  };

  const canProceedToNext = () => {
    // Allow free navigation - only validate on final submit
    // Show visual warnings but don't block navigation
    return true;
  };

  const nextStep = () => {
    const steps = ["tournament-selection", "team", "captain", "players", "officials", "review"];
    const currentIndex = steps.indexOf(currentStep);
    if (currentIndex < steps.length - 1) {
      setCurrentStep(steps[currentIndex + 1]);
    }
  };

  const prevStep = () => {
    const steps = ["tournament-selection", "team", "captain", "players", "officials", "review"];
    const currentIndex = steps.indexOf(currentStep);
    if (currentIndex > 0) {
      setCurrentStep(steps[currentIndex - 1]);
    }
  };

  return (
    <>
    <Header />
      <div className="min-h-screen bg-gray-50 py-12">
        <div className="max-w-7xl mx-auto px-4">
          <div className="text-center mb-8">
            <h1 className="text-3xl font-bold text-gray-900 mb-2">Team Registration</h1>
            <p className="text-gray-600">
              Register your team for a tournament
              {selectedTournament && (
                <span className="block mt-2 text-sm text-green-600">
                  Selected Tournament: {selectedTournament.name}
                </span>
              )}
            </p>
          </div>

          <form onSubmit={handleSubmit}>
            {/* Step Progress Indicator */}
            {selectedTournament && (
              <div className="bg-white rounded-lg shadow-md border border-gray-200 p-4 mb-6">
                <div className="flex items-center justify-between mb-2">
                  <div className="flex items-center gap-2">
                    <Trophy className="h-5 w-5 text-green-600" />
                    <span className="font-semibold text-gray-900">Registration Steps</span>
                  </div>
                  <span className="text-sm text-gray-600">
                    {selectedTournament.name}
                  </span>
                </div>
                <p className="text-xs text-gray-600 mb-3">
                  💡 Tip: Click any step to jump directly. Fill information at your own pace.
                </p>
                <div className="mt-4 grid grid-cols-2 md:grid-cols-6 gap-2">
                  {[
                    { key: 'tournament-selection', label: '1. Tournament', icon: '🏆' },
                    { key: 'team', label: '2. Team & Category', icon: '👥' },
                    { key: 'captain', label: '3. Manager', icon: '👤' },
                    { key: 'players', label: '4. Players', icon: '⚽' },
                    { key: 'officials', label: '5. Officials', icon: '🎓' },
                    { key: 'review', label: '6. Review', icon: '✅' },
                  ].map((step) => {
                    const isCompleted = getStepCompletionStatus(step.key);
                    const hasError = stepsWithErrors.includes(step.key);
                    return (
                      <button
                        key={step.key}
                        type="button"
                        onClick={() => {
                          setCurrentStep(step.key);
                          window.scrollTo({ top: 0, behavior: 'smooth' });
                        }}
                        className={`text-center p-2 rounded-lg text-xs font-medium transition-all cursor-pointer hover:shadow-md ${
                          currentStep === step.key
                            ? 'bg-green-600 text-white shadow-lg'
                            : hasError
                            ? 'bg-red-100 text-red-800 border-2 border-red-400 animate-pulse'
                            : isCompleted
                            ? 'bg-green-100 text-green-800 border-2 border-green-300'
                            : 'bg-gray-100 text-gray-600 hover:bg-gray-200'
                        }`}
                      >
                        <div className="text-lg mb-1">
                          {step.icon}
                          {hasError && currentStep !== step.key && <span className="ml-1 text-red-600">✗</span>}
                          {!hasError && isCompleted && currentStep !== step.key && <span className="ml-1 text-green-600">✓</span>}
                        </div>
                        <div>{step.label}</div>
                        {hasError && <div className="text-[10px] text-red-600 font-bold">Error!</div>}
                      </button>
                    );
                  })}
                </div>
              </div>
            )}

            <div className="bg-white rounded-lg shadow-xl border border-gray-200 p-8">
              <Tabs value={currentStep} onValueChange={setCurrentStep}>
                {/* Tournament Selection Step */}
                <TabsContent value="tournament-selection" className="space-y-6">
                  {/* Error Summary */}
                  {renderErrorSummary('tournament-selection')}
                  
                  <TournamentSelectionStep
                    tournaments={tournaments}
                    onSelectTournament={handleSelectTournament}
                    onCreateNewTeam={handleCreateNewTeam}
                    selectedTournament={selectedTournament || undefined}
                    onNext={() => setCurrentStep('team')}
                  />
                </TabsContent>

                {/* Team Information */}
                <TabsContent value="team" className="space-y-6">
                  <div className="flex items-center space-x-3 mb-6">
                    <Users className="h-6 w-6 text-green-600" />
                    <h2 className="text-2xl font-bold text-gray-900">Team Information & Category</h2>
                  </div>

                  {/* Error Summary */}
                  {renderErrorSummary('team')}

                  {/* Category Selection - First in Team Information */}
                  {selectedTournament?.enable_categories && (
                    <div className="mb-6">
                      <div className="bg-gradient-to-r from-blue-50 to-green-50 border-l-4 border-green-500 p-4 mb-4">
                        <h3 className="font-semibold text-gray-900 mb-1 flex items-center gap-2">
                          <Trophy className="h-4 w-4 text-green-600" />
                          Step 1: Select Your Team Category
                        </h3>
                        <p className="text-sm text-gray-700">
                          Choose the category that matches your team. All players must meet the age and gender requirements.
                        </p>
                      </div>
                      
                      <CategorySelector
                        categories={(() => {
                          // Filter categories to show only enabled ones for this tournament
                          const availableCategories = selectedTournament?.available_categories || [];
                          
                          return Object.fromEntries(
                            Object.entries(categories)
                              .filter(([key]) => {
                                // Only show categories that are in the tournament's available_categories
                                return availableCategories.includes(key);
                              })
                              .map(([key, cat]: [string, any]) => [
                                key,
                                {
                                  ...cat,
                                  key,
                                  // Use tournament's category fee directly
                                  fee: selectedTournament?.category_fees?.[key] ?? cat.default_fee ?? 0
                                }
                              ])
                          );
                        })()}
                        selectedCategory={data.team.category}
                        onCategoryChange={(category) => setData('team', { ...data.team, category })}
                        currency={selectedTournament?.currency || 'KES'}
                      />

                      {errors['team.category'] && (
                        <p className="text-red-600 text-sm mt-2">{errors['team.category']}</p>
                      )}
                    </div>
                  )}

                  {/* Team Details - Second part */}
                  <div className="bg-gradient-to-r from-blue-50 to-green-50 border-l-4 border-blue-500 p-4 mb-4">
                    <h3 className="font-semibold text-gray-900 mb-1">
                      {selectedTournament?.enable_categories ? 'Step 2: Enter Team Details' : 'Team Details'}
                    </h3>
                    <p className="text-sm text-gray-700">
                      Provide your team's basic information and contact details.
                    </p>
                  </div>
                  
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <div>
                      <Label htmlFor="team_name">Team Name *</Label>
                      <Input
                        id="team_name"
                        value={data.team.name}
                        onChange={(e) => setData('team', { ...data.team, name: e.target.value })}
                        placeholder="Enter team name"
                        required
                        disabled={useExistingTeam}
                        className={data.team.name && !isValidName(data.team.name) ? 'border-red-500' : ''}
                      />
                      {errors['team.name'] && <p className="text-red-600 text-sm mt-1">{errors['team.name']}</p>}
                      {data.team.name && !isValidName(data.team.name) && (
                        <p className="text-red-600 text-sm mt-1">Please enter a valid team name (letters and spaces only, minimum 2 characters)</p>
                      )}
                    </div>
                    
                    <div>
                      <Label htmlFor="team_email">Team Email *</Label>
                      <Input
                        id="team_email"
                        type="email"
                        value={data.team.email}
                        onChange={(e) => setData('team', { ...data.team, email: e.target.value })}
                        placeholder="team@example.com"
                        required
                        className={data.team.email && !isValidEmail(data.team.email) ? 'border-red-500' : ''}
                      />
                      {errors['team.email'] && <p className="text-red-600 text-sm mt-1">{errors['team.email']}</p>}
                      {data.team.email && !isValidEmail(data.team.email) && (
                        <p className="text-red-600 text-sm mt-1">Please enter a valid email address</p>
                      )}
                    </div>
                    
                    <div>
                      <Label htmlFor="team_phone">Team Phone *</Label>
                      <Input
                        id="team_phone"
                        value={data.team.phone}
                        onChange={(e) => setData('team', { ...data.team, phone: e.target.value })}
                        placeholder="+254 700 000 000"
                        required
                        className={data.team.phone && !isValidPhone(data.team.phone) ? 'border-red-500' : ''}
                      />
                      {errors['team.phone'] && <p className="text-red-600 text-sm mt-1">{errors['team.phone']}</p>}
                      {data.team.phone && !isValidPhone(data.team.phone) && (
                        <p className="text-red-600 text-sm mt-1">Please enter a valid phone number</p>
                      )}
                    </div>

                    <div>
                      <Label htmlFor="team_country">Country *</Label>
                      <select
                        id="team_country"
                        value={data.team.country || countries[0]}
                        onChange={(e) => setData('team', { ...data.team, country: e.target.value })}
                        className={`w-full p-2 border rounded-lg ${errors['team.country'] ? 'border-red-500' : 'border-gray-300'}`}
                        required
                      >
                        {countries.map((country) => (
                          <option key={country} value={country}>
                            {country}
                          </option>
                        ))}
                      </select>
                      {errors['team.country'] && <p className="text-red-600 text-sm mt-1">{errors['team.country']}</p>}
                    </div>

                    <div>
                      <Label htmlFor="team_county">County *</Label>
                      <select
                        id="team_county"
                        value={data.team.county || counties[0]}
                        onChange={(e) => setData('team', { ...data.team, county: e.target.value })}
                        className={`w-full p-2 border rounded-lg ${errors['team.county'] ? 'border-red-500' : 'border-gray-300'}`}
                        required
                      >
                        {counties.map((county) => (
                          <option key={county} value={county}>
                            {county}
                          </option>
                        ))}
                      </select>
                      {errors['team.county'] && <p className="text-red-600 text-sm mt-1">{errors['team.county']}</p>}
                    </div>
                    
                    <div>
                      <Label htmlFor="team_location">Location *</Label>
                      {(() => {
                        const categoryLoc = category_locations[data.team.category];
                        const isRestricted = categoryLoc?.restrict_locations && categoryLoc?.locations?.length > 0;
                        
                        if (isRestricted) {
                          return (
                            <>
                              <select
                                id="team_location"
                                value={data.team.location}
                                onChange={(e) => setData('team', { ...data.team, location: e.target.value })}
                                className="w-full p-2 border border-gray-300 rounded-lg"
                                required
                              >
                                <option value="">Select a location</option>
                                {categoryLoc.locations.map((location) => (
                                  <option key={location} value={location}>
                                    {location}
                                  </option>
                                ))}
                              </select>
                              <p className="text-xs text-muted-foreground mt-1">
                                Select from allowed locations for {categories[data.team.category]?.name} category
                              </p>
                            </>
                          );
                        } else {
                          return (
                            <>
                              <Input
                                id="team_location"
                                value={data.team.location}
                                onChange={(e) => setData('team', { ...data.team, location: e.target.value })}
                                placeholder="Enter your team's location"
                                required
                              />
                              <p className="text-xs text-muted-foreground mt-1">
                                Enter any location - no restrictions for {categories[data.team.category]?.name} category
                              </p>
                            </>
                          );
                        }
                      })()}
                      {errors['team.location'] && <p className="text-red-600 text-sm mt-1">{errors['team.location']}</p>}
                    </div>
                  
                    
                    <div>
                      <Label htmlFor="team_year">Founded Year *</Label>
                      <Input
                        id="team_year"
                        type="number"
                        value={data.team.year}
                        onChange={(e) => setData('team', { ...data.team, year: parseInt(e.target.value) })}
                        placeholder="2020"
                        min="1900"
                        max={new Date().getFullYear()}
                        required
                      />
                      {errors['team.year'] && <p className="text-red-600 text-sm mt-1">{errors['team.year']}</p>}
                    </div>
                  </div>
                </TabsContent>

                {/* Captain Information */}
                <TabsContent value="captain" className="space-y-6">
                  <div className="flex items-center space-x-3 mb-6">
                    <User className="h-6 w-6 text-green-600" />
                    <h2 className="text-2xl font-bold text-gray-900">Team Manager</h2>
                  </div>

                  {/* Error Summary */}
                  {renderErrorSummary('captain')}
                  
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <div>
                      <Label htmlFor="captain_name">Manager Name *</Label>
                      <Input
                        id="captain_name"
                        value={data.captain.name}
                        onChange={(e) => setData('captain', { ...data.captain, name: e.target.value })}
                        placeholder="Enter captain's full name"
                        required
                        className={data.captain.name && !isValidName(data.captain.name) ? 'border-red-500' : ''}
                      />
                      {errors['captain.name'] && <p className="text-red-600 text-sm mt-1">{errors['captain.name']}</p>}
                      {data.captain.name && !isValidName(data.captain.name) && (
                        <p className="text-red-600 text-sm mt-1">Please enter a valid name (letters and spaces only, minimum 2 characters)</p>
                      )}
                    </div>
                    
                    <div>
                      <Label htmlFor="captain_email">Manager Email *</Label>
                      <Input
                        id="captain_email"
                        type="email"
                        value={data.captain.email}
                        onChange={(e) => setData('captain', { ...data.captain, email: e.target.value })}
                        placeholder="captain@example.com"
                        required
                        className={data.captain.email && !isValidEmail(data.captain.email) ? 'border-red-500' : ''}
                      />
                      {errors['captain.email'] && <p className="text-red-600 text-sm mt-1">{errors['captain.email']}</p>}
                      {data.captain.email && !isValidEmail(data.captain.email) && (
                        <p className="text-red-600 text-sm mt-1">Please enter a valid email address</p>
                      )}
                    </div>
                    
                    <div>
                      <Label htmlFor="captain_phone">Manager Phone *</Label>
                      <Input
                        id="captain_phone"
                        value={data.captain.phone}
                        onChange={(e) => setData('captain', { ...data.captain, phone: e.target.value })}
                        placeholder="+254 700 000 000"
                        required
                        className={data.captain.phone && !isValidPhone(data.captain.phone) ? 'border-red-500' : ''}
                      />
                      {errors['captain.phone'] && <p className="text-red-600 text-sm mt-1">{errors['captain.phone']}</p>}
                      {data.captain.phone && !isValidPhone(data.captain.phone) && (
                        <p className="text-red-600 text-sm mt-1">Please enter a valid phone number</p>
                      )}
                    </div>
                    
                    <div>
                      <Label htmlFor="captain_password">Password *</Label>
                      <Input
                        id="captain_password"
                        type="password"
                        value={data.captain.password}
                        onChange={(e) => setData('captain', { ...data.captain, password: e.target.value })}
                        placeholder="Create a strong password"
                        required
                        className={data.captain.password && data.captain.password.length < 8 ? 'border-red-500' : ''}
                      />
                      {errors['captain.password'] && <p className="text-red-600 text-sm mt-1">{errors['captain.password']}</p>}
                      {data.captain.password && data.captain.password.length < 8 && (
                        <p className="text-red-600 text-sm mt-1">Password should be at least 8 characters long</p>
                      )}
                    </div>
                    
                    <div>
                      <Label htmlFor="captain_password_confirmation">Confirm Password *</Label>
                      <Input
                        id="captain_password_confirmation"
                        type="password"
                        value={data.captain.password_confirmation}
                        onChange={(e) => setData('captain', { ...data.captain, password_confirmation: e.target.value })}
                        placeholder="Confirm your password"
                        required
                        className={data.captain.password_confirmation && data.captain.password !== data.captain.password_confirmation ? 'border-red-500' : ''}
                      />
                      {errors['captain.password_confirmation'] && <p className="text-red-600 text-sm mt-1">{errors['captain.password_confirmation']}</p>}
                      {data.captain.password_confirmation && data.captain.password !== data.captain.password_confirmation && (
                        <p className="text-red-600 text-sm mt-1">Passwords do not match</p>
                      )}
                    </div>
                  </div>
                </TabsContent>

                {/* Players Section */}
                <TabsContent value="players" className="space-y-6">
                  <div className="flex items-center justify-between mb-6">
                    <div className="flex items-center space-x-3">
                      <Users className="h-6 w-6 text-green-600" />
                      <h2 className="text-2xl font-bold text-gray-900">Team Players</h2>
                    </div>
                    <div className="text-sm text-gray-600">
                      {data.players.length} / {maxPlayers} players
                    </div>
                  </div>

                  {/* Error Summary */}
                  {renderErrorSummary('players')}
                  
                  <div className="space-y-4 max-h-96 overflow-y-auto">
                    {data.players.map((player, idx) => (
                      <Card key={idx} className="p-4">
                        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                          <div>
                            <Label>Full Name *</Label>
                            <Input
                              value={player.name}
                              onChange={(e) => handlePlayerChange(idx, 'name', e.target.value)}
                              placeholder="Player name"
                              required
                              className={(player.name && !isValidName(player.name)) || errors[`players.${idx}.name`] ? 'border-red-500' : ''}
                            />
                            {errors[`players.${idx}.name`] && (
                              <p className="text-red-600 text-sm mt-1">{errors[`players.${idx}.name`]}</p>
                            )}
                            {!errors[`players.${idx}.name`] && player.name && !isValidName(player.name) && (
                              <p className="text-red-600 text-sm mt-1">Please enter a valid name (letters and spaces only, minimum 2 characters)</p>
                            )}
                          </div>
                          <div>
                            <Label>Email *</Label>
                            <Input
                              type="email"
                              value={player.email}
                              onChange={(e) => handlePlayerChange(idx, 'email', e.target.value)}
                              placeholder="player@example.com"
                              required
                              className={(player.email && !isValidEmail(player.email)) || errors[`players.${idx}.email`] ? 'border-red-500' : ''}
                            />
                            {errors[`players.${idx}.email`] && (
                              <p className="text-red-600 text-sm mt-1">{errors[`players.${idx}.email`]}</p>
                            )}
                            {!errors[`players.${idx}.email`] && player.email && !isValidEmail(player.email) && (
                              <p className="text-red-600 text-sm mt-1">Please enter a valid email address</p>
                            )}
                          </div>
                          <div>
                            <Label>Phone <span className="text-red-500">*</span></Label>
                            <Input
                              value={player.phone}
                              onChange={(e) => handlePlayerChange(idx, 'phone', e.target.value)}
                              placeholder="+254 700 000 000"
                              className={(player.phone && !isValidPhone(player.phone)) || errors[`players.${idx}.phone`] ? 'border-red-500' : ''}
                              required
                            />
                            {errors[`players.${idx}.phone`] && (
                              <p className="text-red-600 text-sm mt-1">{errors[`players.${idx}.phone`]}</p>
                            )}
                            {!errors[`players.${idx}.phone`] && player.phone && !isValidPhone(player.phone) && (
                              <p className="text-red-600 text-sm mt-1">Please enter a valid phone number</p>
                            )}
                          </div>
                          <div>
                            <Label>Position *</Label>
                            <select
                              value={player.position}
                              onChange={(e) => handlePlayerChange(idx, 'position', e.target.value)}
                              className={`w-full p-2 border rounded-lg ${errors[`players.${idx}.position`] ? 'border-red-500' : 'border-gray-300'}`}
                              required
                            >
                              <option value="Goalkeeper">Goalkeeper</option>
                              <option value="Defender">Defender</option>
                              <option value="Midfielder">Midfielder</option>
                              <option value="Forward">Forward</option>
                            </select>
                            {errors[`players.${idx}.position`] && (
                              <p className="text-red-600 text-sm mt-1">{errors[`players.${idx}.position`]}</p>
                            )}
                          </div>
                          <div>
                            <Label>ID No./Birth Cert No. *</Label>
                            <Input
                              value={player.id_number}
                              onChange={(e) => handlePlayerChange(idx, 'id_number', e.target.value)}
                              placeholder="Enter ID or Birth Certificate number"
                              className={errors[`players.${idx}.id_number`] ? 'border-red-500' : ''}
                              required
                            />
                            {errors[`players.${idx}.id_number`] && (
                              <p className="text-red-600 text-sm mt-1">{errors[`players.${idx}.id_number`]}</p>
                            )}
                          </div>
                          <div>
                            <Label>Country *</Label>
                            <select
                              value={player.country || countries[0]}
                              onChange={(e) => handlePlayerChange(idx, 'country', e.target.value)}
                              className={`w-full p-2 border rounded-lg ${errors[`players.${idx}.country`] ? 'border-red-500' : 'border-gray-300'}`}
                              required
                            >
                              {countries.map((country) => (
                                <option key={country} value={country}>
                                  {country}
                                </option>
                              ))}
                            </select>
                            {errors[`players.${idx}.country`] && (
                              <p className="text-red-600 text-sm mt-1">{errors[`players.${idx}.country`]}</p>
                            )}
                          </div>
                          <div>
                            <Label>County *</Label>
                            <select
                              
                              onChange={(e) => handlePlayerChange(idx, 'county', e.target.value)}
                              className={`w-full p-2 border rounded-lg ${errors[`players.${idx}.county`] ? 'border-red-500' : 'border-gray-300'}`}
                              required
                            >
                              {counties.map((county) => (
                                <option key={county} value={county}>
                                  {county}
                                </option>
                              ))}
                            </select>
                            {errors[`players.${idx}.county`] && (
                              <p className="text-red-600 text-sm mt-1">{errors[`players.${idx}.county`]}</p>
                            )}
                          </div>
                          <div>
                            <Label>Location *</Label>
                            {(() => {
                              const categoryLoc = category_locations[data.team.category];
                              const isRestricted = categoryLoc?.restrict_locations && categoryLoc?.locations?.length > 0;
                              
                              if (isRestricted) {
                                return (
                                  <>
                                    <select
                                      value={player.location || ''}
                                      onChange={(e) => handlePlayerChange(idx, 'location', e.target.value)}
                                      className={`w-full p-2 border rounded-lg ${errors[`players.${idx}.location`] ? 'border-red-500' : 'border-gray-300'}`}
                                      required
                                    >
                                      <option value="">Select a location</option>
                                      {categoryLoc.locations.map((location) => (
                                        <option key={location} value={location}>
                                          {location}
                                        </option>
                                      ))}
                                    </select>
                                    {errors[`players.${idx}.location`] && (
                                      <p className="text-red-600 text-sm mt-1">{errors[`players.${idx}.location`]}</p>
                                    )}
                                  </>
                                );
                              } else {
                                return (
                                  <>
                                    <Input
                                      value={player.location}
                                      onChange={(e) => handlePlayerChange(idx, 'location', e.target.value)}
                                      placeholder="Enter location"
                                      className={errors[`players.${idx}.location`] ? 'border-red-500' : ''}
                                      required
                                    />
                                    {errors[`players.${idx}.location`] && (
                                      <p className="text-red-600 text-sm mt-1">{errors[`players.${idx}.location`]}</p>
                                    )}
                                  </>
                                );
                              }
                            })()}
                          </div>
                          <div>
                            <Label>Date of Birth *</Label>
                            <Input
                              type="date"
                              value={player.date_of_birth}
                              onChange={(e) => handlePlayerChange(idx, 'date_of_birth', e.target.value)}
                              className={errors[`players.${idx}.date_of_birth`] ? 'border-red-500' : ''}
                              required
                              max={new Date().toISOString().split('T')[0]}
                            />
                            {errors[`players.${idx}.date_of_birth`] && (
                              <p className="text-red-600 text-sm mt-1">{errors[`players.${idx}.date_of_birth`]}</p>
                            )}
                          </div>
                          <div>
                            <Label>Gender *</Label>
                            <select
                              value={player.gender}
                              onChange={(e) => handlePlayerChange(idx, 'gender', e.target.value)}
                              className={`w-full p-2 border rounded-lg ${errors[`players.${idx}.gender`] ? 'border-red-500' : 'border-gray-300'}`}
                              required
                            >
                              <option value="male">Male</option>
                              <option value="female">Female</option>
                              <option value="other">Other</option>
                            </select>
                            {errors[`players.${idx}.gender`] && (
                              <p className="text-red-600 text-sm mt-1">{errors[`players.${idx}.gender`]}</p>
                            )}
                          </div>
                        </div>
                      </Card>
                    ))}
                  </div>
                  
                  {data.players.length < maxPlayers && (
                    <Button
                      type="button"
                      variant="outline"
                      onClick={addPlayer}
                      className="w-full border-green-300 text-green-600 hover:bg-green-50"
                    >
                      Add Player
                    </Button>
                  )}
                </TabsContent>

                {/* Officials Section */}
                <TabsContent value="officials" className="space-y-6">
                  <div className="flex items-center justify-between mb-6">
                    <div className="flex items-center space-x-3">
                      <Shield className="h-6 w-6 text-green-600" />
                      <h2 className="text-2xl font-bold text-gray-900">Team Officials</h2>
                    </div>
                    <div className="text-sm text-gray-600">
                      {data.officials.length} officials
                    </div>
                  </div>

                  {/* Error Summary */}
                  {renderErrorSummary('officials')}
                  
                  <div className="space-y-4 max-h-96 overflow-y-auto">
                    {data.officials.map((official, idx) => (
                      <Card key={idx} className="p-4">
                        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                          <div>
                            <Label>Name *</Label>
                            <Input
                              value={official.name}
                              onChange={(e) => handleOfficialChange(idx, 'name', e.target.value)}
                              placeholder="Official name"
                              required
                              className={(official.name && !isValidName(official.name)) || errors[`officials.${idx}.name`] ? 'border-red-500' : ''}
                            />
                            {errors[`officials.${idx}.name`] && (
                              <p className="text-red-600 text-sm mt-1">{errors[`officials.${idx}.name`]}</p>
                            )}
                            {!errors[`officials.${idx}.name`] && official.name && !isValidName(official.name) && (
                              <p className="text-red-600 text-sm mt-1">Please enter a valid name (letters and spaces only, minimum 2 characters)</p>
                            )}
                          </div>
                          <div>
                            <Label>Email *</Label>
                            <Input
                              type="email"
                              value={official.email}
                              onChange={(e) => handleOfficialChange(idx, 'email', e.target.value)}
                              placeholder="official@example.com"
                              required
                              className={(official.email && !isValidEmail(official.email)) || errors[`officials.${idx}.email`] ? 'border-red-500' : ''}
                            />
                            {errors[`officials.${idx}.email`] && (
                              <p className="text-red-600 text-sm mt-1">{errors[`officials.${idx}.email`]}</p>
                            )}
                            {!errors[`officials.${idx}.email`] && official.email && !isValidEmail(official.email) && (
                              <p className="text-red-600 text-sm mt-1">Please enter a valid email address</p>
                            )}
                          </div>
                          <div>
                            <Label>Phone *</Label>
                            <Input
                              value={official.phone}
                              onChange={(e) => handleOfficialChange(idx, 'phone', e.target.value)}
                              placeholder="+254 700 000 000"
                              required
                              className={(official.phone && !isValidPhone(official.phone)) || errors[`officials.${idx}.phone`] ? 'border-red-500' : ''}
                            />
                            {errors[`officials.${idx}.phone`] && (
                              <p className="text-red-600 text-sm mt-1">{errors[`officials.${idx}.phone`]}</p>
                            )}
                            {!errors[`officials.${idx}.phone`] && official.phone && !isValidPhone(official.phone) && (
                              <p className="text-red-600 text-sm mt-1">Please enter a valid phone number</p>
                            )}
                          </div>
                          <div>
                            <Label>Role *</Label>
                            <select
                              value={official.role}
                              onChange={(e) => handleOfficialChange(idx, 'role', e.target.value)}
                              className={`w-full p-2 border rounded-lg ${errors[`officials.${idx}.role`] ? 'border-red-500' : 'border-gray-300'}`}
                              required
                            >
                              <option value="Coach">Coach</option>
                              <option value="Assistant Coach">Assistant Coach</option>
                              <option value="Manager">Manager</option>
                              <option value="Physio">Physio</option>
                              <option value="Doctor">Doctor</option>
                              <option value="Other">Other</option>
                            </select>
                            {errors[`officials.${idx}.role`] && (
                              <p className="text-red-600 text-sm mt-1">{errors[`officials.${idx}.role`]}</p>
                            )}
                          </div>
                          <div className="md:col-span-2">
                            <Label>Qualifications (optional)</Label>
                            <Input
                              value={official.qualifications}
                              onChange={(e) => handleOfficialChange(idx, 'qualifications', e.target.value)}
                              placeholder="Relevant qualifications, certifications, or experience"
                            />
                          </div>
                        </div>
                      </Card>
                    ))}
                  </div>
                  
                  {data.officials.length < 10 && (
                    <Button
                      type="button"
                      variant="outline"
                      onClick={addOfficial}
                      className="w-full border-green-300 text-green-600 hover:bg-green-50"
                    >
                      Add Official
                    </Button>
                  )}
                </TabsContent>

                {/* Review Section */}
                <TabsContent value="review" className="space-y-6">
                  <div className="flex items-center space-x-3 mb-6">
                    <FileText className="h-6 w-6 text-green-600" />
                    <h2 className="text-2xl font-bold text-gray-900">Review & Submit</h2>
                  </div>

                  {/* Error Summary */}
                  {renderErrorSummary('review')}
                  
                  <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-4 mb-6">
                    <h4 className="font-medium text-yellow-900 mb-2">Take Note:</h4>
                    <ul className="text-sm text-yellow-800 space-y-1">
                      <li>• Players and officials must verify their emails to access their portal</li>
                      <li>• Payment of {selectedTournament?.currency || 'KES'} {selectedTournament?.entry_fee?.toLocaleString() || registration_fee.toLocaleString()} is required to complete registration</li>
                      <li>• Team manager will receive admin access after payment</li>
                      <li>• Team will be verified after payment of the registration fee</li>
                      <li>• Each Team player will be verified after their profile is completed</li>
                      <li>• If the team is verified, the team manager will receive a notification and the team will be listed <a href="/teams" target="_blank" className="text-green-600 hover:text-green-700 underline">on the teams page</a></li>
                      <li>• Players, Team and others deserving certificate will be sent a certificate after the tournament</li>
                    </ul>
                  </div>
                  
                  {/* Registration Summary */}
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <Card className="p-4">
                      <h3 className="font-medium text-gray-900 mb-3">Team Information</h3>
                      <div className="space-y-2 text-sm">
                        <div><strong>Name:</strong> {data.team.name}</div>
                        {data.team.category && (
                          <div>
                            <strong>Category:</strong>{' '}
                            <Badge className="bg-blue-100 text-blue-800">
                              {categories[data.team.category]?.name || data.team.category.replace('_', ' ').replace(/\b\w/g, (l: string) => l.toUpperCase())}
                            </Badge>
                          </div>
                        )}
                        <div><strong>Country:</strong> {data.team.country}</div>
                        <div><strong>County:</strong> {data.team.county}</div>
                        <div><strong>Location:</strong> {data.team.location}</div>
                        <div><strong>Founded:</strong> {data.team.year}</div>
                      </div>
                    </Card>
                    
                    <Card className="p-4 bg-green-50 border-green-200">
                      <h3 className="font-medium text-green-900 mb-3">Registration Fee</h3>
                      <div className="space-y-2 text-sm">
                        <div><strong>Tournament:</strong> {selectedTournament?.name || 'Selected Tournament'}</div>
                        {data.team.category && (
                          <div>
                            <strong>Category:</strong>{' '}
                            <span className="text-green-700">
                              {categories[data.team.category]?.name || data.team.category.replace('_', ' ').replace(/\b\w/g, (l: string) => l.toUpperCase())}
                            </span>
                          </div>
                        )}
                        <div>
                          <strong>{data.team.category && selectedTournament?.category_fees?.[data.team.category] ? 'Category Fee:' : 'Entry Fee:'}</strong>{' '}
                          <span className="text-lg font-bold text-green-600">
                            {selectedTournament?.currency || 'KES'}{' '}
                            {(
                              data.team.category && selectedTournament?.category_fees?.[data.team.category]
                                ? selectedTournament.category_fees[data.team.category]
                                : selectedTournament?.entry_fee || registration_fee || 0
                            ).toLocaleString()}
                          </span>
                        </div>
                        
                        <div className="text-xs text-green-600 mt-2 pt-2 border-t border-green-200">
                          💳 Payment required within 48 hours after registration
                        </div>
                      </div>
                    </Card>
                    
                    <Card className="p-4">
                      <h3 className="font-medium text-gray-900 mb-3">Team Manager</h3>
                      <div className="space-y-2 text-sm">
                        <div><strong>Name:</strong> {data.captain.name}</div>
                        <div><strong>Email:</strong> {data.captain.email}</div>
                        <div><strong>Phone:</strong> {data.captain.phone}</div>
                      </div>
                    </Card>
                    
                    <Card className="p-4">
                      <h3 className="font-medium text-gray-900 mb-3">Players</h3>
                      <div className="text-sm">
                        <div><strong>Total Players:</strong> {data.players.length}</div>
                        <div><strong>Goalkeepers:</strong> {data.players.filter(p => p.position === 'Goalkeeper').length}</div>
                        <div><strong>Defenders:</strong> {data.players.filter(p => p.position === 'Defender').length}</div>
                        <div><strong>Midfielders:</strong> {data.players.filter(p => p.position === 'Midfielder').length}</div>
                        <div><strong>Forwards:</strong> {data.players.filter(p => p.position === 'Forward').length}</div>
                      </div>
                    </Card>
                    
                    <Card className="p-4">
                      <h3 className="font-medium text-gray-900 mb-3">Officials</h3>
                      <div className="text-sm">
                        <div><strong>Total Officials:</strong> {data.officials.length}</div>
                        {['Coach', 'Assistant Coach', 'Manager', 'Physio', 'Doctor', 'Other'].map(role => {
                          const count = data.officials.filter(o => o.role === role).length;
                          return count > 0 ? <div key={role}><strong>{role}:</strong> {count}</div> : null;
                        })}
                      </div>
                    </Card>
                  </div>
                  
                  {/* Terms and Conditions */}
                  <Card className="p-6">
                    <h3 className="font-medium text-gray-900 mb-4">Terms and Conditions</h3>
                    <div className="flex items-start space-x-3">
                      <input
                        type="checkbox"
                        id="terms"
                        checked={termsAccepted}
                        onChange={(e) => setTermsAccepted(e.target.checked)}
                        className="h-4 w-4 text-green-600 focus:ring-green-500 border-gray-300 rounded mt-1"
                        required
                      />
                      <div className="text-sm">
                        <label htmlFor="terms" className="text-gray-700">
                          I have read and agree to the{' '}
                          <button
                            type="button"
                            onClick={() => setShowTermsModal(true)}
                            className="text-green-600 hover:text-green-700 underline"
                          >
                            Tournament Terms and Conditions
                          </button>
                          {' '}and confirm that all information provided is accurate.
                        </label>
                      </div>
                    </div>
                  </Card>
                </TabsContent>
              </Tabs>

              {/* Navigation Buttons */}
              <div className="flex justify-between mt-8 pt-6 border-t border-gray-200">
                <Button
                  type="button"
                  variant="outline"
                  onClick={prevStep}
                  disabled={currentStep === "tournament-selection"}
                >
                  Previous
                </Button>
                
                {currentStep !== "review" ? (
                  <Button
                    type="button"
                    onClick={nextStep}
                    className="bg-green-600 hover:bg-green-700 text-white"
                  >
                    {getStepCompletionStatus(currentStep) ? 'Next' : 'Skip to Next (Fill Later)'}
                  </Button>
                ) : (
                  <Button
                    type="submit"
                    disabled={!termsAccepted || processing}
                    className="bg-green-600 hover:bg-green-700 text-white"
                  >
                    {processing ? (
                      <>
                        <LoaderCircle className="animate-spin h-4 w-4 mr-2" />
                        Registering...
                      </>
                    ) : (
                      'Complete Registration'
                    )}
                  </Button>
                )}
              </div>
            </div>
          </form>

          {/* Already have account */}
          <div className="mt-8 text-center">
            <p className="text-gray-600">
              Already have an account?{' '}
              <button
                onClick={() => router.visit(route('login'))}
                className="text-green-600 hover:text-green-700 font-medium"
              >
                Sign in here
              </button>
            </p>
          </div>
        </div>
      </div>

      {/* Terms Modal */}
      {showTermsModal && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center p-4 z-50">
          <div className="bg-white rounded-lg max-w-2xl max-h-[80vh] overflow-y-auto p-6">
            <h2 className="text-2xl font-bold text-gray-900 mb-4">Tournament Terms and Conditions</h2>
            <div className="prose prose-sm text-gray-600 space-y-4">
              <h3>1. Registration Requirements</h3>
              <p>Teams must provide accurate information for all players and officials. All team members must verify their email addresses before the tournament begins.</p>
              
              <h3>2. Payment Terms</h3>
              <p>A registration fee of {selectedTournament?.currency || 'KES'} {selectedTournament?.entry_fee?.toLocaleString() || registration_fee.toLocaleString()} is required to complete registration. Payment must be made within 7 days of registration to secure your spot.</p>
              
              <h3>3. Team Composition</h3>
              <p>Teams must have a minimum of {minPlayers} players and maximum of {maxPlayers} players. At least one goalkeeper must be registered.</p>
              
              <h3>4. Code of Conduct</h3>
              <p>All participants must maintain good sportsmanship and follow tournament rules. Unsporting behavior may result in disqualification.</p>
              
              <h3>5. Tournament Rules</h3>
              <p>Standard FIFA football rules apply unless otherwise stated. Tournament organizers have final say in all disputes.</p>
              
              <h3>6. Liability</h3>
              <p>Participants play at their own risk. Tournament organizers are not liable for injuries sustained during the tournament.</p>
            </div>
            <div className="flex justify-end space-x-3 mt-6">
              <Button
                type="button"
                variant="outline"
                onClick={() => setShowTermsModal(false)}
              >
                Close
              </Button>
            </div>
          </div>
        </div>
      )}

      <Footer />
    </>
  );
};

export default RegisterTeam; 