import React, { useState } from "react";
import { router, useForm } from "@inertiajs/react";
import { Tabs, TabsList, TabsTrigger, TabsContent } from "@/components/ui/tabs";
import { Input } from "@/components/ui/input";
import { Button } from "@/components/ui/button";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import Header from "@/components/frontend/layout/Header";
import Footer from "@/components/frontend/sections/Footer";
import { Plus, Minus, LoaderCircle, Users, Shield, CreditCard, FileText } from "lucide-react";


const footballPositions = [
  "Goalkeeper",
  "Defender", 
  "Midfielder",
  "Forward"
];

const officialRoles = [
  "Coach",
  "Assistant Coach", 
  "Manager",
  "Physio",
  "Doctor",
  "Other"
];

interface Player {
  name: string;
  email: string;
  phone: string;
  position: string;
  jersey_number: string;
}

interface Official {
  name: string;
  email: string;
  phone: string;
  role: string;
  qualifications: string;
}

const defaultPlayer: Player = {
  name: "",
  email: "",
  phone: "",
  position: "Midfielder",
  jersey_number: ""
};

const defaultOfficial: Official = {
  name: "",
  email: "",
  phone: "",
  role: "Coach",
  qualifications: ""
};

interface RegisterTeamProps {
  registration_fee: number;
}

const RegisterTeam: React.FC<RegisterTeamProps> = ({ registration_fee }) => {
  const [currentStep, setCurrentStep] = useState("team");
  const [termsAccepted, setTermsAccepted] = useState(false);
  const [showTermsModal, setShowTermsModal] = useState(false);

  const { data, setData, post, processing, errors, reset } = useForm({
    team: {
      name: "",
      email: "",
      phone: "",
      location: "",
      coach: "",
      year: new Date().getFullYear()
    },
    captain: {
      name: "",
      email: "",
      phone: "",
      password: "",
      password_confirmation: ""
    },
    players: Array.from({ length: 11 }, () => ({ ...defaultPlayer })),
    officials: [{ ...defaultOfficial }],
    terms_accepted: false
  });

  const handlePlayerChange = (idx: number, field: keyof Player, value: string) => {
    setData('players', data.players.map((p, i) => i === idx ? { ...p, [field]: value } : p));
  };

  const addPlayer = () => {
    if (data.players.length < 23) {
      setData('players', [...data.players, { ...defaultPlayer }]);
    }
  };

  const removePlayer = (idx: number) => {
    if (data.players.length > 11) {
      setData('players', data.players.filter((_, i) => i !== idx));
    }
  };

  const handleOfficialChange = (idx: number, field: keyof Official, value: string) => {
    setData('officials', data.officials.map((o, i) => i === idx ? { ...o, [field]: value } : o));
  };

  const addOfficial = () => {
    if (data.officials.length < 10) {
      setData('officials', [...data.officials, { ...defaultOfficial }]);
    }
  };

  const removeOfficial = (idx: number) => {
    if (data.officials.length > 1) {
      setData('officials', data.officials.filter((_, i) => i !== idx));
    }
  };

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    if (!termsAccepted) {
      alert('Please accept the terms and conditions to continue.');
      return;
    }
    
    // Set terms_accepted and then submit
    setData('terms_accepted', true as any);
    
    // Submit after state update
    setTimeout(() => {
      post(route('register'), {
        onSuccess: () => {
          // Handle success - will redirect to profile completion
        },
        onError: (errors) => {
        }
      });
    }, 0);
  };

  const canProceedToNext = () => {
    switch (currentStep) {
      case "team":
        return data.team.name && data.team.email && data.team.phone && data.team.location && data.team.coach;
      case "captain":
        return data.captain.name && data.captain.email && data.captain.phone && 
               data.captain.password && data.captain.password_confirmation;
      case "players":
        return data.players.every(p => p.name && p.email && p.phone && p.position);
      case "officials":
        return data.officials.every(o => o.name && o.email && o.phone && o.role);
      default:
        return true;
    }
  };

  const nextStep = () => {
    const steps = ["team", "captain", "players", "officials", "review"];
    const currentIndex = steps.indexOf(currentStep);
    if (currentIndex < steps.length - 1 && canProceedToNext()) {
      setCurrentStep(steps[currentIndex + 1]);
    }
  };

  const prevStep = () => {
    const steps = ["team", "captain", "players", "officials", "review"];
    const currentIndex = steps.indexOf(currentStep);
    if (currentIndex > 0) {
      setCurrentStep(steps[currentIndex - 1]);
    }
  };

  return (
    <>
      <Header />
      <div className="min-h-screen bg-gradient-to-br from-green-50 to-gray-50 py-8">
        <div className="container mx-auto px-4 max-w-4xl">
          {/* Header */}
          <div className="text-center mb-8">
            
            <h1 className="text-3xl font-bold text-gray-900">Team Registration</h1>
            <p className="text-gray-600 mt-2">Register your team for the Milimus Tournament</p>
          </div>

          {/* Progress Indicator */}
          <div className="mb-8">
            <div className="flex justify-between items-center">
              {["team", "captain", "players", "officials", "review"].map((step, index) => (
                <div key={step} className="flex flex-col items-center">
                  <div className={`w-10 h-10 rounded-full flex items-center justify-center text-sm font-medium ${
                    currentStep === step ? 'bg-green-600 text-white' :
                    ["team", "captain", "players", "officials", "review"].indexOf(currentStep) > index 
                      ? 'bg-green-200 text-green-800' : 'bg-gray-200 text-gray-600'
                  }`}>
                    {index + 1}
                  </div>
                  <span className="text-xs mt-1 capitalize text-gray-600">{step}</span>
                </div>
              ))}
            </div>
          </div>

          <form onSubmit={handleSubmit}>
            <div className="bg-white rounded-lg shadow-xl border border-gray-200 p-8">
              <Tabs value={currentStep} onValueChange={setCurrentStep}>
                {/* Team Information */}
                <TabsContent value="team" className="space-y-6">
                  <div className="flex items-center space-x-3 mb-6">
                    <Users className="h-6 w-6 text-green-600" />
                    <h2 className="text-2xl font-bold text-gray-900">Team Information</h2>
                  </div>
                  
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <div>
                      <Label htmlFor="team_name">Team Name *</Label>
                      <Input
                        id="team_name"
                        value={data.team.name}
                        onChange={(e) => setData('team', { ...data.team, name: e.target.value })}
                        placeholder="Enter team name"
                        required
                      />
                      {errors['team.name'] && <p className="text-red-600 text-sm mt-1">{errors['team.name']}</p>}
                    </div>
                    
                    <div>
                      <Label htmlFor="team_email">Team Email *</Label>
                      <Input
                        id="team_email"
                        type="email"
                        value={data.team.email}
                        onChange={(e) => setData('team', { ...data.team, email: e.target.value })}
                        placeholder="team@example.com"
                        required
                      />
                      {errors['team.email'] && <p className="text-red-600 text-sm mt-1">{errors['team.email']}</p>}
                    </div>
                    
                    <div>
                      <Label htmlFor="team_phone">Team Phone *</Label>
                      <Input
                        id="team_phone"
                        value={data.team.phone}
                        onChange={(e) => setData('team', { ...data.team, phone: e.target.value })}
                        placeholder="+254 700 000 000"
                        required
                      />
                      {errors['team.phone'] && <p className="text-red-600 text-sm mt-1">{errors['team.phone']}</p>}
                    </div>
                    
                    <div>
                      <Label htmlFor="team_location">Location *</Label>
                      <Input
                        id="team_location"
                        value={data.team.location}
                        onChange={(e) => setData('team', { ...data.team, location: e.target.value })}
                        placeholder="City, Country"
                        required
                      />
                      {errors['team.location'] && <p className="text-red-600 text-sm mt-1">{errors['team.location']}</p>}
                    </div>
                    
                    <div>
                      <Label htmlFor="team_coach">Head Coach *</Label>
                      <Input
                        id="team_coach"
                        value={data.team.coach}
                        onChange={(e) => setData('team', { ...data.team, coach: e.target.value })}
                        placeholder="Coach name"
                        required
                      />
                      {errors['team.coach'] && <p className="text-red-600 text-sm mt-1">{errors['team.coach']}</p>}
                    </div>
                    
                    <div>
                      <Label htmlFor="team_year">Founded Year *</Label>
                      <Input
                        id="team_year"
                        type="number"
                        value={data.team.year}
                        onChange={(e) => setData('team', { ...data.team, year: parseInt(e.target.value) })}
                        placeholder="2020"
                        min="1900"
                        max={new Date().getFullYear()}
                        required
                      />
                      {errors['team.year'] && <p className="text-red-600 text-sm mt-1">{errors['team.year']}</p>}
                    </div>
                  </div>
                </TabsContent>

                {/* Captain Information */}
                <TabsContent value="captain" className="space-y-6">
                  <div className="flex items-center space-x-3 mb-6">
                    <Shield className="h-6 w-6 text-green-600" />
                    <h2 className="text-2xl font-bold text-gray-900">Team Captain</h2>
                  </div>
                  
                  <div className="bg-blue-50 border border-blue-200 rounded-lg p-4 mb-6">
                    <p className="text-blue-800 text-sm">
                      <strong>Note:</strong> The team captain will have administrative access to manage the team.
                      They will be able to view team status, payment information, and team member verification status.
                    </p>
                  </div>
                  
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <div>
                      <Label htmlFor="captain_name">Captain Name *</Label>
                      <Input
                        id="captain_name"
                        value={data.captain.name}
                        onChange={(e) => setData('captain', { ...data.captain, name: e.target.value })}
                        placeholder="Enter captain's full name"
                        required
                      />
                      {errors['captain.name'] && <p className="text-red-600 text-sm mt-1">{errors['captain.name']}</p>}
                    </div>
                    
                    <div>
                      <Label htmlFor="captain_email">Captain Email *</Label>
                      <Input
                        id="captain_email"
                        type="email"
                        value={data.captain.email}
                        onChange={(e) => setData('captain', { ...data.captain, email: e.target.value })}
                        placeholder="captain@example.com"
                        required
                      />
                      {errors['captain.email'] && <p className="text-red-600 text-sm mt-1">{errors['captain.email']}</p>}
                    </div>
                    
                    <div>
                      <Label htmlFor="captain_phone">Captain Phone *</Label>
                      <Input
                        id="captain_phone"
                        value={data.captain.phone}
                        onChange={(e) => setData('captain', { ...data.captain, phone: e.target.value })}
                        placeholder="+254 700 000 000"
                        required
                      />
                      {errors['captain.phone'] && <p className="text-red-600 text-sm mt-1">{errors['captain.phone']}</p>}
                    </div>
                    
                    <div className="md:col-span-2">
                      <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <div>
                          <Label htmlFor="captain_password">Password *</Label>
                          <Input
                            id="captain_password"
                            type="password"
                            value={data.captain.password}
                            onChange={(e) => setData('captain', { ...data.captain, password: e.target.value })}
                            placeholder="Create a strong password"
                            required
                          />
                          {errors['captain.password'] && <p className="text-red-600 text-sm mt-1">{errors['captain.password']}</p>}
                        </div>
                        
                        <div>
                          <Label htmlFor="captain_password_confirmation">Confirm Password *</Label>
                          <Input
                            id="captain_password_confirmation"
                            type="password"
                            value={data.captain.password_confirmation}
                            onChange={(e) => setData('captain', { ...data.captain, password_confirmation: e.target.value })}
                            placeholder="Confirm your password"
                            required
                          />
                          {errors['captain.password_confirmation'] && <p className="text-red-600 text-sm mt-1">{errors['captain.password_confirmation']}</p>}
                        </div>
                      </div>
                    </div>
                  </div>
                </TabsContent>

                {/* Players Section - I'll continue this in the next part due to length */}
                <TabsContent value="players" className="space-y-6">
                  <div className="flex items-center justify-between mb-6">
                    <div className="flex items-center space-x-3">
                      <Users className="h-6 w-6 text-green-600" />
                      <h2 className="text-2xl font-bold text-gray-900">Team Players</h2>
                    </div>
                    <div className="text-sm text-gray-600">
                      {data.players.length} / 23 players (minimum 11 required)
                    </div>
                  </div>
                  
                  <div className="space-y-4 max-h-96 overflow-y-auto">
                    {data.players.map((player, idx) => (
                      <div key={idx} className="border border-gray-200 rounded-lg p-4">
                        <div className="flex items-center justify-between mb-4">
                          <h3 className="font-medium text-gray-900">
                            Player {idx + 1} {idx === 0 && "(Captain)"}
                          </h3>
                          {idx > 10 && (
                            <Button
                              type="button"
                              variant="outline"
                              size="sm"
                              onClick={() => removePlayer(idx)}
                              className="text-red-600 border-red-300 hover:bg-red-50"
                            >
                              <Minus className="h-4 w-4" />
                            </Button>
                          )}
                        </div>
                        
                        <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
                          <div>
                            <Label>Name *</Label>
                            <Input
                              value={player.name}
                              onChange={(e) => handlePlayerChange(idx, 'name', e.target.value)}
                              placeholder="Player name"
                              required
                            />
                          </div>
                          <div>
                            <Label>Email *</Label>
                            <Input
                              type="email"
                              value={player.email}
                              onChange={(e) => handlePlayerChange(idx, 'email', e.target.value)}
                              placeholder="player@example.com"
                              required
                            />
                          </div>
                          <div>
                            <Label>Phone *</Label>
                            <Input
                              value={player.phone}
                              onChange={(e) => handlePlayerChange(idx, 'phone', e.target.value)}
                              placeholder="+254 700 000 000"
                              required
                            />
                          </div>
                          <div>
                            <Label>Position *</Label>
                            <select
                              value={player.position}
                              onChange={(e) => handlePlayerChange(idx, 'position', e.target.value)}
                              className="w-full p-2 border border-gray-300 rounded-lg"
                              required
                            >
                              {footballPositions.map((position) => (
                                <option key={position} value={position}>{position}</option>
                              ))}
                            </select>
                          </div>
                          <div>
                            <Label>Jersey Number (optional)</Label>
                            <Input
                              value={player.jersey_number}
                              onChange={(e) => handlePlayerChange(idx, 'jersey_number', e.target.value)}
                              placeholder="1-99"
                              maxLength={2}
                            />
                          </div>
                        </div>
                      </div>
                    ))}
                  </div>
                  
                  {data.players.length < 23 && (
                    <Button
                      type="button"
                      variant="outline"
                      onClick={addPlayer}
                      className="w-full border-green-300 text-green-600 hover:bg-green-50"
                    >
                      <Plus className="h-4 w-4 mr-2" />
                      Add Player
                    </Button>
                  )}
                </TabsContent>

                {/* Officials Section */}
                <TabsContent value="officials" className="space-y-6">
                  <div className="flex items-center justify-between mb-6">
                    <div className="flex items-center space-x-3">
                      <Shield className="h-6 w-6 text-green-600" />
                      <h2 className="text-2xl font-bold text-gray-900">Team Officials</h2>
                    </div>
                    <div className="text-sm text-gray-600">
                      {data.officials.length} / 10 officials (minimum 1 required)
                    </div>
                  </div>
                  
                  <div className="space-y-4 max-h-96 overflow-y-auto">
                    {data.officials.map((official, idx) => (
                      <div key={idx} className="border border-gray-200 rounded-lg p-4">
                        <div className="flex items-center justify-between mb-4">
                          <h3 className="font-medium text-gray-900">Official {idx + 1}</h3>
                          {idx > 0 && (
                            <Button
                              type="button"
                              variant="outline"
                              size="sm"
                              onClick={() => removeOfficial(idx)}
                              className="text-red-600 border-red-300 hover:bg-red-50"
                            >
                              <Minus className="h-4 w-4" />
                            </Button>
                          )}
                        </div>
                        
                        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                          <div>
                            <Label>Name *</Label>
                            <Input
                              value={official.name}
                              onChange={(e) => handleOfficialChange(idx, 'name', e.target.value)}
                              placeholder="Official name"
                              required
                            />
                          </div>
                          <div>
                            <Label>Email *</Label>
                            <Input
                              type="email"
                              value={official.email}
                              onChange={(e) => handleOfficialChange(idx, 'email', e.target.value)}
                              placeholder="official@example.com"
                              required
                            />
                          </div>
                          <div>
                            <Label>Phone *</Label>
                            <Input
                              value={official.phone}
                              onChange={(e) => handleOfficialChange(idx, 'phone', e.target.value)}
                              placeholder="+254 700 000 000"
                              required
                            />
                          </div>
                          <div>
                            <Label>Role *</Label>
                            <select
                              value={official.role}
                              onChange={(e) => handleOfficialChange(idx, 'role', e.target.value)}
                              className="w-full p-2 border border-gray-300 rounded-lg"
                              required
                            >
                              {officialRoles.map((role) => (
                                <option key={role} value={role}>{role}</option>
                              ))}
                            </select>
                          </div>
                          <div className="md:col-span-2">
                            <Label>Qualifications (optional)</Label>
                            <Textarea
                              value={official.qualifications}
                              onChange={(e) => handleOfficialChange(idx, 'qualifications', e.target.value)}
                              placeholder="Relevant qualifications, certifications, or experience"
                              rows={2}
                            />
                          </div>
                        </div>
                      </div>
                    ))}
                  </div>
                  
                  {data.officials.length < 10 && (
                    <Button
                      type="button"
                      variant="outline"
                      onClick={addOfficial}
                      className="w-full border-green-300 text-green-600 hover:bg-green-50"
                    >
                      <Plus className="h-4 w-4 mr-2" />
                      Add Official
                    </Button>
                  )}
                </TabsContent>

                {/* Review Section */}
                <TabsContent value="review" className="space-y-6">
                  <div className="flex items-center space-x-3 mb-6">
                    <FileText className="h-6 w-6 text-green-600" />
                    <h2 className="text-2xl font-bold text-gray-900">Review & Submit</h2>
                  </div>
                  
                  <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-4 mb-6">
                    <h4 className="font-medium text-yellow-900 mb-2">Before you submit:</h4>
                    <ul className="text-sm text-yellow-800 space-y-1">
                      <li>• OTP codes will be sent to all team members for email verification</li>
                      <li>• Players and officials must verify their emails to participate</li>
                      <li>• Payment of KES 1,000 is required to complete registration</li>
                      <li>• Team captain will receive admin access after payment</li>
                    </ul>
                  </div>
                  
                  {/* Registration Summary */}
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <div className="border border-gray-200 rounded-lg p-4">
                      <h3 className="font-medium text-gray-900 mb-3">Team Information</h3>
                      <div className="space-y-2 text-sm">
                        <div><strong>Name:</strong> {data.team.name}</div>
                        <div><strong>Location:</strong> {data.team.location}</div>
                        <div><strong>Coach:</strong> {data.team.coach}</div>
                        <div><strong>Founded:</strong> {data.team.year}</div>
                      </div>
                    </div>
                    
                    <div className="border border-gray-200 rounded-lg p-4">
                      <h3 className="font-medium text-gray-900 mb-3">Team Captain</h3>
                      <div className="space-y-2 text-sm">
                        <div><strong>Name:</strong> {data.captain.name}</div>
                        <div><strong>Email:</strong> {data.captain.email}</div>
                        <div><strong>Phone:</strong> {data.captain.phone}</div>
                      </div>
                    </div>
                    
                    <div className="border border-gray-200 rounded-lg p-4">
                      <h3 className="font-medium text-gray-900 mb-3">Players</h3>
                      <div className="text-sm">
                        <div><strong>Total Players:</strong> {data.players.length}</div>
                        <div><strong>Goalkeepers:</strong> {data.players.filter(p => p.position === 'Goalkeeper').length}</div>
                        <div><strong>Defenders:</strong> {data.players.filter(p => p.position === 'Defender').length}</div>
                        <div><strong>Midfielders:</strong> {data.players.filter(p => p.position === 'Midfielder').length}</div>
                        <div><strong>Forwards:</strong> {data.players.filter(p => p.position === 'Forward').length}</div>
                      </div>
                    </div>
                    
                    <div className="border border-gray-200 rounded-lg p-4">
                      <h3 className="font-medium text-gray-900 mb-3">Officials</h3>
                      <div className="text-sm">
                        <div><strong>Total Officials:</strong> {data.officials.length}</div>
                        {officialRoles.map(role => {
                          const count = data.officials.filter(o => o.role === role).length;
                          return count > 0 ? <div key={role}><strong>{role}:</strong> {count}</div> : null;
                        })}
                      </div>
                    </div>
                  </div>
                  
                  {/* Terms and Conditions */}
                  <div className="border border-gray-200 rounded-lg p-6">
                    <h3 className="font-medium text-gray-900 mb-4">Terms and Conditions</h3>
                    <div className="flex items-start space-x-3">
                      <input
                        type="checkbox"
                        id="terms"
                        checked={termsAccepted}
                        onChange={(e) => setTermsAccepted(e.target.checked)}
                        className="h-4 w-4 text-green-600 focus:ring-green-500 border-gray-300 rounded mt-1"
                        required
                      />
                      <div className="text-sm">
                        <label htmlFor="terms" className="text-gray-700">
                          I have read and agree to the{' '}
                          <button
                            type="button"
                            onClick={() => setShowTermsModal(true)}
                            className="text-green-600 hover:text-green-700 underline"
                          >
                            Tournament Terms and Conditions
                          </button>
                          {' '}and confirm that all information provided is accurate.
                        </label>
                      </div>
                    </div>
                  </div>
                </TabsContent>
              </Tabs>

              {/* Navigation Buttons */}
              <div className="flex justify-between mt-8 pt-6 border-t border-gray-200">
                <Button
                  type="button"
                  variant="outline"
                  onClick={prevStep}
                  disabled={currentStep === "team"}
                >
                  Previous
                </Button>
                
                {currentStep !== "review" ? (
                  <Button
                    type="button"
                    onClick={nextStep}
                    disabled={!canProceedToNext()}
                    className="bg-green-600 hover:bg-green-700 text-white"
                  >
                    Next
                  </Button>
                ) : (
                  <Button
                    type="submit"
                    disabled={!termsAccepted || processing}
                    className="bg-green-600 hover:bg-green-700 text-white"
                  >
                    {processing ? (
                      <>
                        <LoaderCircle className="animate-spin h-4 w-4 mr-2" />
                        Registering...
                      </>
                    ) : (
                      'Complete Registration'
                    )}
                  </Button>
                )}
              </div>
            </div>
          </form>

          {/* Already have account */}
          <div className="mt-8 text-center">
            <p className="text-gray-600">
              Already have an account?{' '}
              <button
                onClick={() => router.visit(route('login'))}
                className="text-green-600 hover:text-green-700 font-medium"
              >
                Sign in here
              </button>
            </p>
          </div>
        </div>
      </div>

      {/* Terms Modal */}
      {showTermsModal && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center p-4 z-50">
          <div className="bg-white rounded-lg max-w-2xl max-h-[80vh] overflow-y-auto p-6">
            <h2 className="text-2xl font-bold text-gray-900 mb-4">Tournament Terms and Conditions</h2>
            <div className="prose prose-sm text-gray-600 space-y-4">
              <h3>1. Registration Requirements</h3>
              <p>Teams must provide accurate information for all players and officials. All team members must verify their email addresses before the tournament begins.</p>
              
              <h3>2. Payment Terms</h3>
              <p>A registration fee of KES {registration_fee.toLocaleString()} is required to complete registration. Payment must be made within 7 days of registration to secure your spot.</p>
              
              <h3>3. Team Composition</h3>
              <p>Teams must have a minimum of 11 players and maximum of 23 players. At least one goalkeeper must be registered.</p>
              
              <h3>4. Code of Conduct</h3>
              <p>All participants must maintain good sportsmanship and follow tournament rules. Unsporting behavior may result in disqualification.</p>
              
              <h3>5. Tournament Rules</h3>
              <p>Standard FIFA football rules apply unless otherwise stated. Tournament organizers have final say in all disputes.</p>
              
              <h3>6. Liability</h3>
              <p>Participants play at their own risk. Tournament organizers are not liable for injuries sustained during the tournament.</p>
            </div>
            <div className="flex justify-end space-x-3 mt-6">
              <Button
                type="button"
                variant="outline"
                onClick={() => setShowTermsModal(false)}
              >
                Close
              </Button>
            </div>
          </div>
        </div>
      )}

      <Footer />
    </>
  );
};

export default RegisterTeam; 