import React, { useState } from "react";
import { router, useForm } from "@inertiajs/react";
import { Input } from "@/components/ui/input";
import { Button } from "@/components/ui/button";
import { Label } from "@/components/ui/label";
import Header from "@/components/frontend/layout/Header";
import Footer from "@/components/frontend/sections/Footer";
import { Eye, EyeOff, LoaderCircle } from "lucide-react";

interface LoginProps {
  canResetPassword: boolean;
  status?: string;
}

const Login: React.FC<LoginProps> = ({ canResetPassword, status }) => {
  const [showPassword, setShowPassword] = useState(false);
  
  const { data, setData, post, processing, errors, reset } = useForm<{
    email: string;
    password: string;
    remember: boolean;
  }>({
    email: '',
    password: '',
    remember: false,
  });

  const submit = (e: React.FormEvent) => {
    e.preventDefault();
    post(route('login'), {
      onFinish: () => reset('password'),
    });
  };

  return (
    <>
      <Header />
      <div className="flex flex-col items-center justify-center min-h-[80vh] bg-gradient-to-br from-green-50 to-gray-50 py-8">
        <div className="w-full max-w-md p-8 bg-white rounded-lg shadow-xl border border-gray-200">
          {/* Header */}
          <div className="text-center mb-8">
            
            <h2 className="text-2xl font-bold text-gray-900">Welcome Back</h2>
            <p className="text-gray-600 mt-2">Sign in to access the tournament system</p>
          </div>

          {/* Status Message */}
          {status && (
            <div className="mb-6 p-4 bg-green-50 border border-green-200 rounded-lg">
              <p className="text-green-800 text-sm text-center">{status}</p>
            </div>
          )}

          {/* Login Form */}
          <form onSubmit={submit} className="space-y-6">
            {/* Email Field */}
            <div>
              <Label htmlFor="email" className="text-gray-700 font-medium">
                Email Address
              </Label>
              <Input
                id="email"
                type="email"
                value={data.email}
                onChange={(e) => setData('email', e.target.value)}
                placeholder="Enter your email"
                className="mt-1"
                required
                disabled={processing}
              />
              {errors.email && (
                <p className="text-red-600 text-sm mt-1">{errors.email}</p>
              )}
            </div>

            {/* Password Field */}
            <div>
              <div className="flex items-center justify-between">
                <Label htmlFor="password" className="text-gray-700 font-medium">
                  Password
                </Label>
                {canResetPassword && (
                  <button
                    type="button"
                    onClick={() => router.visit(route('password.request'))}
                    className="text-sm text-green-600 hover:text-green-700 font-medium"
                  >
                    Forgot password?
                  </button>
                )}
              </div>
              <div className="relative mt-1">
                <Input
                  id="password"
                  type={showPassword ? "text" : "password"}
                  value={data.password}
                  onChange={(e) => setData('password', e.target.value)}
                  placeholder="Enter your password"
                  required
                  disabled={processing}
                />
                <button
                  type="button"
                  onClick={() => setShowPassword(!showPassword)}
                  className="absolute right-3 top-1/2 transform -translate-y-1/2 text-gray-500 hover:text-gray-700"
                >
                  {showPassword ? (
                    <EyeOff className="h-4 w-4" />
                  ) : (
                    <Eye className="h-4 w-4" />
                  )}
                </button>
              </div>
              {errors.password && (
                <p className="text-red-600 text-sm mt-1">{errors.password}</p>
              )}
            </div>

            {/* Remember Me */}
            <div className="flex items-center space-x-2">
              <input
                type="checkbox"
                id="remember"
                checked={data.remember}
                onChange={(e) => setData('remember', e.target.checked)}
                disabled={processing}
                className="h-4 w-4 text-green-600 focus:ring-green-500 border-gray-300 rounded"
              />
              <Label htmlFor="remember" className="text-sm text-gray-700">
                Remember me for 30 days
              </Label>
            </div>

            {/* Submit Button */}
            <Button
              type="submit"
              className="w-full bg-green-600 hover:bg-green-700 text-white font-medium py-2 px-4 rounded-lg transition-colors"
              disabled={processing}
            >
              {processing ? (
                <>
                  <LoaderCircle className="animate-spin h-4 w-4 mr-2" />
                  Signing in...
                </>
              ) : (
                'Sign In'
              )}
            </Button>
          </form>

          {/* Register Link */}
          <div className="mt-8 text-center">
            <p className="text-gray-600">
              Don't have an account?{' '}
              <button
                onClick={() => router.visit(route('register'))}
                className="text-green-600 hover:text-green-700 font-medium"
              >
                Register your team
              </button>
            </p>
          </div>

          {/* Help Text */}
          {/* <div className="mt-6 p-4 bg-blue-50 border border-blue-200 rounded-lg">
            <h4 className="text-sm font-medium text-blue-900 mb-2">Need help?</h4>
            <ul className="text-xs text-blue-800 space-y-1">
              <li>• Team captains use the email used during registration</li>
              <li>• Players and officials will receive login credentials after email verification</li>
              <li>• Contact support if you're unable to access your account</li>
            </ul>
          </div> */}
        </div>
      </div>
      <Footer />
    </>
  );
};

export default Login; 