import Header from "@/components/frontend/layout/Header";
import Footer from "@/components/frontend/sections/Footer";
import { Video, Play, Heart, Share2, ExternalLink, ChevronLeft, ChevronRight, Grid, List, Clock, Eye, ThumbsUp, X } from "lucide-react";
import React, { useState, useEffect } from "react";

interface VlogVideo {
  id: number;
  title: string;
  description?: string;
  platform: string;
  platform_icon: string;
  video_url: string;
  embed_url: string;
  thumbnail_url?: string;
  duration?: string;
  views: string;
  likes: number;
  category: string;
  tags?: string[];
  featured: boolean;
  published_at?: string;
}

interface VlogCategory {
  id: string;
  name: string;
}

interface VlogPlatform {
  id: string;
  name: string;
}

interface VlogProps {
  vlogVideos: VlogVideo[];
  vlogCategories: VlogCategory[];
  vlogPlatforms: VlogPlatform[];
}

const Vlog: React.FC<VlogProps> = ({ vlogVideos, vlogCategories, vlogPlatforms }) => {
  const [selectedCategory, setSelectedCategory] = useState("all");
  const [selectedPlatform, setSelectedPlatform] = useState("all");
  const [selectedVideo, setSelectedVideo] = useState<VlogVideo | null>(null);
  const [currentSlide, setCurrentSlide] = useState(0);
  const [layoutMode, setLayoutMode] = useState("grid"); // "grid" or "list"

  // Hero slides for vlog
  const vlogHeroSlides = [
    {
      id: 1,
      title: "Tournament Highlights",
      subtitle: "Watch the best moments from our tournaments",
      image: "/images/gallery-1.JPG",
    },
    {
      id: 2,
      title: "Behind the Scenes",
      subtitle: "Exclusive content from our events and training sessions",
      image: "/images/gallery-2.JPG",
    },
    {
      id: 3,
      title: "Player Interviews",
      subtitle: "Get to know our amazing players and their stories",
      image: "/images/gallery-3.JPG",
    },
  ];

  // Auto-play functionality
  useEffect(() => {
    const interval = setInterval(() => {
      setCurrentSlide((prev) => (prev + 1) % vlogHeroSlides.length);
    }, 5000); // Change slide every 5 seconds

    return () => clearInterval(interval);
  }, []);

  const nextSlide = () => {
    setCurrentSlide((prev) => (prev + 1) % vlogHeroSlides.length);
  };

  const prevSlide = () => {
    setCurrentSlide((prev) => (prev - 1 + vlogHeroSlides.length) % vlogHeroSlides.length);
  };

  const goToSlide = (index: number) => {
    setCurrentSlide(index);
  };

  const filteredVideos = vlogVideos.filter((video: VlogVideo) => {
    const matchesCategory = selectedCategory === "all" || video.category === selectedCategory;
    const matchesPlatform = selectedPlatform === "all" || video.platform === selectedPlatform;
    return matchesCategory && matchesPlatform;
  });

  const openVideoModal = (video: VlogVideo) => {
    setSelectedVideo(video);
  };

  const closeVideoModal = () => {
    setSelectedVideo(null);
  };

  const openExternalVideo = (video: VlogVideo) => {
    window.open(video.video_url, '_blank');
  };

  return (
    <div className="min-h-screen bg-background">
      <Header />
      
      {/* Hero Slider Section */}
      <section className="relative h-[380px] md:h-[480px] overflow-hidden">
        {/* Slides */}
        <div className="relative w-full h-full">
          {vlogHeroSlides.map((slide, index) => {
            const IconComponent = Video;
            return (
              <div
                key={slide.id}
                className={`absolute inset-0 transition-transform duration-1000 ease-in-out ${
                  index === currentSlide ? 'translate-x-0' : 
                  index < currentSlide ? '-translate-x-full' : 'translate-x-full'
                }`}
              >
                <div className="relative flex items-center justify-center h-full bg-gradient-to-br from-primary/90 to-secondary/90 overflow-hidden">
                  <img
                    src={slide.image}
                    alt={`Slide ${slide.id}`}
                    className="absolute inset-0 w-full h-full object-cover opacity-20 pointer-events-none select-none scale-110"
                  />
                  <div className="absolute inset-0 bg-gradient-to-b from-black/60 to-black/80" />
                  <div className="relative z-10 text-center text-white flex flex-col items-center justify-center w-full animate-fade-in-up">
                    <IconComponent className="mx-auto mb-4 h-16 w-16 text-green-700 animate-bounce-gentle" />
                    <h1 className="text-5xl md:text-6xl font-extrabold drop-shadow mb-4 tracking-tight">{slide.title}</h1>
                    <p className="text-xl md:text-2xl max-w-2xl mx-auto drop-shadow font-medium">
                      {slide.subtitle}
                    </p>
                  </div>
                </div>
              </div>
            );
          })}
        </div>

        {/* Navigation Arrows */}
        <button
          onClick={prevSlide}
          className="absolute left-4 top-1/2 -translate-y-1/2 z-20 p-3 rounded-full bg-white/20 hover:bg-white/30 transition-colors text-white backdrop-blur-sm"
          aria-label="Previous slide"
        >
          <ChevronLeft className="h-6 w-6" />
        </button>
        <button
          onClick={nextSlide}
          className="absolute right-4 top-1/2 -translate-y-1/2 z-20 p-3 rounded-full bg-white/20 hover:bg-white/30 transition-colors text-white backdrop-blur-sm"
          aria-label="Next slide"
        >
          <ChevronRight className="h-6 w-6" />
        </button>

        {/* Slide Indicators */}
        <div className="absolute bottom-6 left-1/2 -translate-x-1/2 z-20 flex gap-2">
          {vlogHeroSlides.map((_, index) => (
            <button
              key={index}
              onClick={() => goToSlide(index)}
              className={`w-3 h-3 rounded-full transition-all duration-300 ${
                index === currentSlide 
                  ? 'bg-white scale-125' 
                  : 'bg-white/50 hover:bg-white/75'
              }`}
              aria-label={`Go to slide ${index + 1}`}
            />
          ))}
        </div>
      </section>

      {/* Category Filter */}
      <section className="max-w-6xl mx-auto py-6 px-4">
        <div className="text-center mb-4">
          <h3 className="text-lg font-semibold text-gray-700 mb-2">Filter by Category</h3>
        </div>
        <div className="flex flex-wrap justify-center gap-2 md:gap-4">
          {vlogCategories.map((category: VlogCategory) => (
            <button
              key={category.id}
              onClick={() => setSelectedCategory(category.id)}
              className={`px-4 py-2 rounded-full text-sm font-medium transition-all duration-200 ${
                selectedCategory === category.id
                  ? "bg-secondary text-white shadow-lg"
                  : "bg-white/80 text-gray-700 hover:bg-secondary/10 border border-gray-200"
              }`}
            >
              {category.name}
            </button>
          ))}
        </div>
      </section>

      {/* Platform Filter */}
      <section className="max-w-6xl mx-auto py-6 px-4 mb-10">
        <div className="text-center mb-4">
          <h3 className="text-lg font-semibold text-gray-700 mb-2">Filter by Platform</h3>
        </div>
        <div className="flex flex-wrap justify-center gap-2 md:gap-4">
          {vlogPlatforms.map((platform: VlogPlatform) => (
            <button
              key={platform.id}
              onClick={() => setSelectedPlatform(platform.id)}
              className={`px-4 py-2 rounded-full text-sm font-medium transition-all duration-200 ${
                selectedPlatform === platform.id
                  ? "bg-primary text-white shadow-lg"
                  : "bg-white/80 text-gray-700 hover:bg-primary/10 border border-gray-200"
              }`}
            >
              {platform.name}
            </button>
          ))}
        </div>
      </section>

      {/* Layout Toggle */}
      <section className="max-w-6xl mx-auto py-4 px-4">
        <div className="text-center mb-4">
          <h3 className="text-lg font-semibold text-gray-700 mb-2">Layout View</h3>
        </div>
        <div className="flex justify-center gap-4">
          <button
            onClick={() => setLayoutMode("grid")}
            className={`flex items-center gap-2 px-4 py-2 rounded-full text-sm font-medium transition-all duration-200 ${
              layoutMode === "grid"
                ? "bg-primary text-white shadow-lg"
                : "bg-white/80 text-gray-700 hover:bg-primary/10 border border-gray-200"
            }`}
          >
            <Grid className="h-4 w-4" />
            Grid View
          </button>
          <button
            onClick={() => setLayoutMode("list")}
            className={`flex items-center gap-2 px-4 py-2 rounded-full text-sm font-medium transition-all duration-200 ${
              layoutMode === "list"
                ? "bg-primary text-white shadow-lg"
                : "bg-white/80 text-gray-700 hover:bg-primary/10 border border-gray-200"
            }`}
          >
            <List className="h-4 w-4" />
            List View
          </button>
        </div>
      </section>

      {/* Video Grid/List */}
      <section className="mx-auto px-4 pb-16">
        {layoutMode === "grid" ? (
          // Grid Layout
          <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-6">
            {filteredVideos.map((video: VlogVideo) => (
              <div
                key={video.id}
                className="group relative bg-white rounded-xl overflow-hidden shadow-lg hover:shadow-2xl transition-all duration-300"
              >
                {/* Video Thumbnail */}
                <div className="relative aspect-video overflow-hidden">
                  <img
                    src={video.thumbnail_url || "/images/video-placeholder.jpg"}
                    alt={video.title}
                    className="w-full h-full object-cover transition-transform duration-300 group-hover:scale-110"
                  />
                  <div className="absolute inset-0 bg-black/40 opacity-0 group-hover:opacity-100 transition-opacity duration-300 flex items-center justify-center">
                    <button
                      onClick={() => openVideoModal(video)}
                      className="p-4 rounded-full bg-white/20 hover:bg-white/30 transition-colors text-white backdrop-blur-sm"
                    >
                      <Play className="h-8 w-8 ml-1" />
                    </button>
                  </div>
                  
                  {/* Platform Badge */}
                  <div className="absolute top-2 left-2">
                    <span className="text-2xl">{video.platform_icon}</span>
                  </div>
                  
                  {/* Duration */}
                  {video.duration && (
                    <div className="absolute bottom-2 right-2 bg-black/70 text-white text-xs px-2 py-1 rounded">
                      {video.duration}
                    </div>
                  )}
                  
                  {/* Featured Badge */}
                  {video.featured && (
                    <div className="absolute top-2 right-2 bg-yellow-500 text-white text-xs px-2 py-1 rounded-full font-medium">
                      Featured
                    </div>
                  )}
                </div>

                {/* Video Info */}
                <div className="p-4">
                  <h3 className="font-semibold text-lg mb-2 line-clamp-2 group-hover:text-primary transition-colors">
                    {video.title}
                  </h3>
                  
                  {video.description && (
                    <p className="text-sm text-gray-600 mb-3 line-clamp-2">
                      {video.description}
                    </p>
                  )}
                  
                  {/* Stats */}
                  <div className="flex items-center gap-4 text-xs text-gray-500 mb-3">
                    <div className="flex items-center gap-1">
                      <Eye className="h-3 w-3" />
                      <span>{video.views}</span>
                    </div>
                    <div className="flex items-center gap-1">
                      <ThumbsUp className="h-3 w-3" />
                      <span>{video.likes}</span>
                    </div>
                    {video.duration && (
                      <div className="flex items-center gap-1">
                        <Clock className="h-3 w-3" />
                        <span>{video.duration}</span>
                      </div>
                    )}
                  </div>
                  
                  {/* Tags */}
                  {video.tags && video.tags.length > 0 && (
                    <div className="flex flex-wrap gap-1 mb-3">
                      {video.tags.slice(0, 3).map((tag, index) => (
                        <span
                          key={index}
                          className="text-xs bg-gray-100 text-gray-600 px-2 py-1 rounded-full"
                        >
                          {tag}
                        </span>
                      ))}
                    </div>
                  )}
                  
                  {/* Actions */}
                  <div className="flex items-center justify-between">
                    <button
                      onClick={() => openVideoModal(video)}
                      className="flex items-center gap-2 text-sm text-primary hover:text-primary/80 transition-colors"
                    >
                      <Play className="h-4 w-4" />
                      Watch
                    </button>
                    <button
                      onClick={() => openExternalVideo(video)}
                      className="flex items-center gap-2 text-sm text-gray-500 hover:text-gray-700 transition-colors"
                    >
                      <ExternalLink className="h-4 w-4" />
                      Open
                    </button>
                  </div>
                </div>
              </div>
            ))}
          </div>
        ) : (
          // List Layout
          <div className="space-y-4">
            {filteredVideos.map((video: VlogVideo) => (
              <div
                key={video.id}
                className="group bg-white rounded-xl overflow-hidden shadow-lg hover:shadow-xl transition-all duration-300"
              >
                <div className="flex flex-col md:flex-row">
                  {/* Video Thumbnail */}
                  <div className="relative md:w-80 aspect-video md:aspect-auto overflow-hidden">
                    <img
                      src={video.thumbnail_url || "/images/video-placeholder.jpg"}
                      alt={video.title}
                      className="w-full h-full object-cover transition-transform duration-300 group-hover:scale-105"
                    />
                    <div className="absolute inset-0 bg-black/40 opacity-0 group-hover:opacity-100 transition-opacity duration-300 flex items-center justify-center">
                      <button
                        onClick={() => openVideoModal(video)}
                        className="p-4 rounded-full bg-white/20 hover:bg-white/30 transition-colors text-white backdrop-blur-sm"
                      >
                        <Play className="h-8 w-8 ml-1" />
                      </button>
                    </div>
                    
                    {/* Platform Badge */}
                    <div className="absolute top-2 left-2">
                      <span className="text-2xl">{video.platform_icon}</span>
                    </div>
                    
                    {/* Duration */}
                    {video.duration && (
                      <div className="absolute bottom-2 right-2 bg-black/70 text-white text-xs px-2 py-1 rounded">
                        {video.duration}
                      </div>
                    )}
                    
                    {/* Featured Badge */}
                    {video.featured && (
                      <div className="absolute top-2 right-2 bg-yellow-500 text-white text-xs px-2 py-1 rounded-full font-medium">
                        Featured
                      </div>
                    )}
                  </div>

                  {/* Video Info */}
                  <div className="flex-1 p-6">
                    <div className="flex items-start justify-between mb-3">
                      <h3 className="font-semibold text-xl mb-2 group-hover:text-primary transition-colors">
                        {video.title}
                      </h3>
                      <button
                        onClick={() => openExternalVideo(video)}
                        className="text-gray-400 hover:text-gray-600 transition-colors"
                      >
                        <ExternalLink className="h-5 w-5" />
                      </button>
                    </div>
                    
                    {video.description && (
                      <p className="text-gray-600 mb-4 line-clamp-3">
                        {video.description}
                      </p>
                    )}
                    
                    {/* Stats */}
                    <div className="flex items-center gap-6 text-sm text-gray-500 mb-4">
                      <div className="flex items-center gap-1">
                        <Eye className="h-4 w-4" />
                        <span>{video.views} views</span>
                      </div>
                      <div className="flex items-center gap-1">
                        <ThumbsUp className="h-4 w-4" />
                        <span>{video.likes} likes</span>
                      </div>
                      {video.duration && (
                        <div className="flex items-center gap-1">
                          <Clock className="h-4 w-4" />
                          <span>{video.duration}</span>
                        </div>
                      )}
                      {video.published_at && (
                        <div className="text-gray-400">
                          {new Date(video.published_at).toLocaleDateString()}
                        </div>
                      )}
                    </div>
                    
                    {/* Tags */}
                    {video.tags && video.tags.length > 0 && (
                      <div className="flex flex-wrap gap-2 mb-4">
                        {video.tags.map((tag, index) => (
                          <span
                            key={index}
                            className="text-xs bg-gray-100 text-gray-600 px-3 py-1 rounded-full"
                          >
                            {tag}
                          </span>
                        ))}
                      </div>
                    )}
                    
                    {/* Actions */}
                    <div className="flex items-center gap-3">
                      <button
                        onClick={() => openVideoModal(video)}
                        className="flex items-center gap-2 px-4 py-2 bg-primary text-white rounded-lg hover:bg-primary/90 transition-colors"
                      >
                        <Play className="h-4 w-4" />
                        Watch Video
                      </button>
                      <button className="flex items-center gap-2 px-4 py-2 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition-colors">
                        <Heart className="h-4 w-4" />
                        Like
                      </button>
                      <button className="flex items-center gap-2 px-4 py-2 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition-colors">
                        <Share2 className="h-4 w-4" />
                        Share
                      </button>
                    </div>
                  </div>
                </div>
              </div>
            ))}
          </div>
        )}
        
        {filteredVideos.length === 0 && (
          <div className="text-center py-16">
            <Video className="mx-auto h-16 w-16 text-gray-400 mb-4" />
            <h3 className="text-xl font-semibold text-gray-600 mb-2">No videos found</h3>
            <p className="text-gray-500">Try selecting different filters.</p>
          </div>
        )}
      </section>

      {/* Video Modal */}
      {selectedVideo && (
        <div className="fixed inset-0 bg-black/90 z-50 flex items-center justify-center p-4">
          <div className="relative max-w-4xl w-full max-h-[90vh]">
            <div className="bg-white rounded-lg overflow-hidden">
              {/* Modal Header */}
              <div className="flex items-center justify-between p-4 border-b">
                <div className="flex items-center gap-3">
                  <span className="text-2xl">{selectedVideo.platform_icon}</span>
                  <h3 className="font-semibold text-lg">{selectedVideo.title}</h3>
                </div>
                <button
                  onClick={closeVideoModal}
                  className="p-2 rounded-full hover:bg-gray-100 transition-colors"
                >
                  <X className="h-5 w-5" />
                </button>
              </div>
              
              {/* Video Embed */}
              <div className="aspect-video">
                {selectedVideo.embed_url ? (
                  <iframe
                    src={selectedVideo.embed_url}
                    title={selectedVideo.title}
                    className="w-full h-full"
                    frameBorder="0"
                    allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture"
                    allowFullScreen
                  />
                ) : (
                  <div className="w-full h-full bg-gray-100 flex items-center justify-center">
                    <div className="text-center">
                      <Video className="h-16 w-16 text-gray-400 mx-auto mb-4" />
                      <p className="text-gray-500">Video not available for embedding</p>
                      <button
                        onClick={() => openExternalVideo(selectedVideo)}
                        className="mt-4 px-4 py-2 bg-primary text-white rounded-lg hover:bg-primary/90 transition-colors"
                      >
                        <ExternalLink className="h-4 w-4 inline mr-2" />
                        Watch on {selectedVideo.platform}
                      </button>
                    </div>
                  </div>
                )}
              </div>
              
              {/* Video Info */}
              <div className="p-4">
                {selectedVideo.description && (
                  <p className="text-gray-600 mb-4">{selectedVideo.description}</p>
                )}
                
                <div className="flex items-center justify-between">
                  <div className="flex items-center gap-4 text-sm text-gray-500">
                    <div className="flex items-center gap-1">
                      <Eye className="h-4 w-4" />
                      <span>{selectedVideo.views}</span>
                    </div>
                    <div className="flex items-center gap-1">
                      <ThumbsUp className="h-4 w-4" />
                      <span>{selectedVideo.likes}</span>
                    </div>
                    {selectedVideo.duration && (
                      <div className="flex items-center gap-1">
                        <Clock className="h-4 w-4" />
                        <span>{selectedVideo.duration}</span>
                      </div>
                    )}
                  </div>
                  
                  <button
                    onClick={() => openExternalVideo(selectedVideo)}
                    className="flex items-center gap-2 px-4 py-2 bg-primary text-white rounded-lg hover:bg-primary/90 transition-colors"
                  >
                    <ExternalLink className="h-4 w-4" />
                    Open on {selectedVideo.platform}
                  </button>
                </div>
              </div>
            </div>
          </div>
        </div>
      )}

      <Footer />
    </div>
  );
};

export default Vlog; 