import { useState, useEffect } from "react";
import { Head, Link } from "@inertiajs/react";
import Header from "@/components/frontend/layout/Header";
import Footer from "@/components/frontend/sections/Footer";
import { EyeIcon } from "lucide-react";

interface Player {
  id: number;
  display_name: string;
  user: {
    name: string;
  };
  jersey_number: number;
  football_position: string;
  status: string;
  avatar: string;
}

interface Formation {
  id: number;
  name: string;
  formation: {
    id: number;
    name: string;
    display_name: string;
  };
  player_positions: any;
  formation_data: any;
}

interface Team {
  id: number;
  name: string;
  description: string;
  logo: string;
  country: string;
  county: string;
  location: string;
  captain: {
    id: number;
    name: string;
  } | null;
  manager: {
    id: number;
    name: string;
  } | null;
  player_count: number;
  active_tournaments: Array<{
    id: number;
    name: string;
    slug: string;
  }>;
  has_formation: boolean;
  formation_name: string | null;
  stats: any;
  players?: Player[];
  default_formation?: Formation;
  team_stats?: {
    total_players: number;
    active_players: number;
    total_officials: number;
    tournaments_participated: number;
    matches_played: number;
    matches_won: number;
  };
}

interface Tournament {
  id: number;
  name: string;
  slug: string;
}

interface TeamsPageProps {
  teams: {
    data: Team[];
    links: any[];
    meta: any;
  };
  tournaments: Tournament[];
  filters: {
    tournament: string;
    status: string;
    search: string;
  };
}

const Teams = ({ teams, tournaments, filters }: TeamsPageProps) => {
  const [activeTab, setActiveTab] = useState<'teams' | 'players' | 'formations'>('teams');
  const [selectedTournament, setSelectedTournament] = useState(filters.tournament || '');
  const [searchTerm, setSearchTerm] = useState(filters.search || '');
  const [filteredTeams, setFilteredTeams] = useState(teams.data);

  useEffect(() => {
    let filtered = teams.data;

    if (selectedTournament) {
      filtered = filtered.filter(team => 
        Array.isArray(team.active_tournaments) && team.active_tournaments.some(tournament => tournament.id.toString() === selectedTournament)
      );
    }

    if (searchTerm) {
      filtered = filtered.filter(team => 
        team.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
        team.description?.toLowerCase().includes(searchTerm.toLowerCase())
      );
    }

    setFilteredTeams(filtered);
  }, [selectedTournament, searchTerm, teams.data]);

  const allPlayers = filteredTeams.flatMap(team => 
    team.players?.map(player => ({ ...player, team_name: team.name, team_logo: team.logo })) || []
  );

  const teamsWithFormations = filteredTeams.filter(team => team.has_formation);

  const getStatusBadge = (status: string) => {
    const statusColors = {
      active: 'bg-green-100 text-green-800',
      pending: 'bg-yellow-100 text-yellow-800',
      approved: 'bg-blue-100 text-blue-800',
      rejected: 'bg-red-100 text-red-800',
    };
    
    return (
      <span className={`px-2 py-1 text-xs font-medium rounded-full ${statusColors[status as keyof typeof statusColors] || 'bg-gray-100 text-gray-800'}`}>
        {status.charAt(0).toUpperCase() + status.slice(1)}
      </span>
    );
  };

  const renderTeamCard = (team: Team) => (
    <div key={team.id} className="bg-white rounded-lg shadow-md hover:shadow-lg transition-shadow duration-300 overflow-hidden">
      <div className="p-6">
        <div className="flex items-center space-x-4 mb-4">
          {team.logo && (
            <img 
              src={team.logo} 
              alt={`${team.name} logo`}
              className="w-16 h-16 object-cover rounded-full border-2 border-gray-200"
            />
          )}
          <div className="flex-1">
            <div className="flex items-center space-x-2 justify-between">
              <Link 
                href={route('teams.show', team.id)}
                className="text-xl font-bold text-gray-900 mb-1 hover:text-blue-600 transition-colors duration-200"
              >
                {team.name}
              </Link>

              <Link 
                href={route('teams.show', team.id)}
                className="text-xl font-bold text-gray-900 mb-1 hover:text-blue-600 transition-colors duration-200"
              >
                <EyeIcon className="w-4 h-4" />
              </Link>
            </div>
            <div className="flex items-center space-x-2 text-sm text-gray-600">
              {team.country && <span>{team.country}</span>}
              {team.county && <span>• {team.county}</span>}
              {team.location && <span>• {team.location}</span>}
            </div>
          </div>
        </div>

        {team.description && (
          <p className="text-gray-700 mb-4 line-clamp-2">{team.description}</p>
        )}

        <div className="grid grid-cols-2 gap-4 mb-4">
          <div className="text-center">
            <div className="text-2xl font-bold text-blue-600">{team.player_count}</div>
            <div className="text-sm text-gray-600">Players</div>
          </div>
          <div className="text-center">
            <div className="text-2xl font-bold text-green-600">{Array.isArray(team.active_tournaments) ? team.active_tournaments.length : 0}</div>
            <div className="text-sm text-gray-600">Tournaments</div>
          </div>
        </div>

        <div className="space-y-2 mb-4">
          {team.captain && (
            <div className="flex items-center justify-between text-sm">
              <span className="text-gray-600">Captain:</span>
              <span className="font-medium">{team.captain.name}</span>
            </div>
          )}
          {team.manager && (
            <div className="flex items-center justify-between text-sm">
              <span className="text-gray-600">Manager:</span>
              <span className="font-medium">{team.manager.name}</span>
            </div>
          )}
        </div>

        <div className="flex flex-wrap gap-2 mb-4">
          {Array.isArray(team.active_tournaments) && team.active_tournaments.map(tournament => (
            <span key={tournament.id} className="px-2 py-1 bg-blue-100 text-blue-800 text-xs rounded-full">
              {tournament.name}
            </span>
          ))}
        </div>

        {team.has_formation && (
          <div className="flex items-center justify-between text-sm text-green-600 mb-4">
            <span>✓ Formation Available</span>
            <span className="font-medium">{team.formation_name}</span>
          </div>
        )}
      </div>
    </div>
  );

  const renderPlayerCard = (player: Player & { team_name: string; team_logo: string }) => (
    <div key={player.id} className="bg-white rounded-lg shadow-md hover:shadow-lg transition-shadow duration-300 p-4">
      <div className="flex items-center space-x-4">
        {player.avatar && (
          <img 
            src={player.avatar} 
            alt={`${player.display_name} avatar`}
            className="w-12 h-12 object-cover rounded-full border-2 border-gray-200"
          />
        )}
        <div className="flex-1">
          <h4 className="font-semibold text-gray-900">{player.display_name}</h4>
          <p className="text-sm text-gray-600">{player.user.name}</p>
          <div className="flex items-center space-x-4 mt-1">
            {player.jersey_number && (
              <span className="text-xs bg-gray-100 px-2 py-1 rounded">#{player.jersey_number}</span>
            )}
            {player.football_position && (
              <span className="text-xs bg-blue-100 text-blue-800 px-2 py-1 rounded">
                {player.football_position}
              </span>
            )}
            {getStatusBadge(player.status)}
          </div>
        </div>
        <div className="text-right">
          <div className="text-sm font-medium text-gray-900">{player.team_name}</div>
          {player.team_logo && (
            <img 
              src={player.team_logo} 
              alt={`${player.team_name} logo`}
              className="w-6 h-6 object-cover rounded-full mt-1"
            />
          )}
        </div>
      </div>
    </div>
  );

  const renderFormationCard = (team: Team) => (
    <div key={team.id} className="bg-white rounded-lg shadow-md hover:shadow-lg transition-shadow duration-300 p-6">
      <div className="flex items-center space-x-4 mb-4">
        {team.logo && (
          <img 
            src={team.logo} 
            alt={`${team.name} logo`}
            className="w-12 h-12 object-cover rounded-full border-2 border-gray-200"
          />
        )}
        <div>
          <h3 className="text-lg font-bold text-gray-900">{team.name}</h3>
          <p className="text-sm text-gray-600">{team.formation_name}</p>
        </div>
      </div>

      {team.default_formation && (
        <div className="bg-gray-50 rounded-lg p-4">
          <div className="text-center mb-4">
            <h4 className="font-semibold text-gray-900">{team.default_formation.formation.display_name}</h4>
            <p className="text-sm text-gray-600">{team.default_formation.name}</p>
          </div>
          
          {/* Formation visualization placeholder */}
          <div className="bg-white rounded border-2 border-dashed border-gray-300 p-8 text-center">
            <div className="text-gray-500">
              <div className="text-4xl mb-2">⚽</div>
              <p className="text-sm">Formation visualization</p>
              <p className="text-xs text-gray-400">Interactive formation display coming soon</p>
            </div>
          </div>

          {team.default_formation.player_positions && Object.keys(team.default_formation.player_positions).length > 0 && (
            <div className="mt-4">
              <h5 className="font-medium text-gray-900 mb-2">Player Positions</h5>
              <div className="grid grid-cols-2 gap-2 text-sm">
                {Object.entries(team.default_formation.player_positions).map(([position, playerData]: [string, any]) => (
                  <div key={position} className="flex justify-between items-center bg-white p-2 rounded">
                    <span className="font-medium">{position}</span>
                    <span className="text-gray-600">{playerData?.player_name || 'TBD'}</span>
                  </div>
                ))}
              </div>
            </div>
          )}
        </div>
      )}
    </div>
  );

  return (
    <>
      <Head title="Teams - Tournament" />
    <Header />
      
      <div className="min-h-screen bg-gray-50">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
          {/* Header */}
          <div className="text-center mb-8">
            <h1 className="text-4xl font-bold text-gray-900 mb-4">Tournament Teams</h1>
            <p className="text-lg text-gray-600 max-w-2xl mx-auto">
              Discover all approved teams, their players, and formations participating in our tournaments.
            </p>
          </div>

          {/* Filters */}
          <div className="bg-white rounded-lg shadow-md p-6 mb-8">
            <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Tournament</label>
                <select
                  value={selectedTournament}
                  onChange={(e) => setSelectedTournament(e.target.value)}
                  className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                >
                  <option value="">All Tournaments</option>
                  {tournaments.map(tournament => (
                    <option key={tournament.id} value={tournament.id.toString()}>
                      {tournament.name}
                    </option>
                  ))}
                </select>
              </div>
              
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Search Teams</label>
                <input
                  type="text"
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                  placeholder="Search by team name or description..."
                  className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                />
              </div>

              {/* <div className="flex items-end">
                <div className="text-sm text-gray-600">
                  Showing {filteredTeams.length} teams
                </div>
              </div> */}
            </div>
          </div>

          {/* Tabs */}
          <div className="bg-white rounded-lg shadow-md mb-8">
            <div className="flex justify-between items-center border-b border-gray-200">
              <nav className="-mb-px flex space-x-8 px-6 items-center">
                <button
                  onClick={() => setActiveTab('teams')}
                  className={`py-4 px-1 border-b-2 font-medium text-sm ${
                    activeTab === 'teams'
                      ? 'border-blue-500 text-blue-600'
                      : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                  }`}
                >
                  Teams ({filteredTeams.length})
                </button>
                <button
                  onClick={() => setActiveTab('players')}
                  className={`py-4 px-1 border-b-2 font-medium text-sm ${
                    activeTab === 'players'
                      ? 'border-blue-500 text-blue-600'
                      : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                  }`}
                >
                  Players ({allPlayers.length})
                </button>                
              </nav>
              <div className="flex items-end justify-end items-center px-6">
                  Showing {filteredTeams.length} teams
              </div>
            </div>

            <div className="p-6">
              {activeTab === 'teams' && (
                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                  {filteredTeams.map(renderTeamCard)}
                </div>
              )}

              {activeTab === 'players' && (
                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                  {allPlayers.map(renderPlayerCard)}
                </div>
              )}

              {activeTab === 'teams' && filteredTeams.length === 0 && (
                <div className="text-center py-12">
                  <div className="text-gray-500 text-lg">No teams found matching your criteria</div>
                </div>
              )}

              {activeTab === 'players' && allPlayers.length === 0 && (
                <div className="text-center py-12">
                  <div className="text-gray-500 text-lg">No players found</div>
                </div>
              )}

              {activeTab === 'formations' && teamsWithFormations.length === 0 && (
                <div className="text-center py-12">
                  <div className="text-gray-500 text-lg">No formations available</div>
                </div>
              )}
            </div>
          </div>
      </div>
    </div>

    <Footer />
  </>
);
};

export default Teams; 