import { useState } from "react";
import { Head, Link } from "@inertiajs/react";
import Header from "@/components/frontend/layout/Header";
import Footer from "@/components/frontend/sections/Footer";
import FormationField from "@/components/TeamManager/FormationField";

interface Player {
  id: number;
  display_name: string;
  user: {
    name: string;
    email: string;
  };
  jersey_number: number;
  football_position: string;
  status: string;
  avatar: string;
  country: string;
  city: string;
  date_of_birth: string;
  gender: string;
  phone: string;
  is_team_captain: boolean;
  global_rating: number;
  tournament_rating: number;
  tournament_matches_played: number;
  tournament_matches_won: number;
  tournament_win_rate: number;
}

interface Official {
  id: number;
  user: {
    name: string;
    email: string;
  };
  role: string;
  status: string;
}

interface Formation {
  id: number;
  name: string;
  formation: {
    id: number;
    name: string;
    display_name: string;
    description: string;
    category: string;
  };
  player_positions: any;
  formation_data: any;
  description: string;
}

interface Tournament {
  id: number;
  name: string;
  slug: string;
  start_date: string;
  end_date: string;
}

interface TeamStats {
  total_players: number;
  active_players: number;
  total_officials: number;
  tournaments_participated: number;
  matches_played: number;
  matches_won: number;
}

interface Team {
  id: number;
  name: string;
  description: string;
  logo: string;
  country: string;
  county: string;
  location: string;
  captain: {
    id: number;
    name: string;
    email: string;
  } | null;
  manager: {
    id: number;
    name: string;
    email: string;
  } | null;
  players: Player[];
  officials: Official[];
  active_tournaments: Tournament[];
  default_formation: Formation | null;
  stats: any;
  team_stats: TeamStats;
}

interface TeamPageProps {
  team: Team;
}

const Team = ({ team }: TeamPageProps) => {
  const [activeTab, setActiveTab] = useState<'overview' | 'players' | 'formation' | 'stats'>('overview');

  const getStatusBadge = (status: string) => {
    const statusColors = {
      active: 'bg-green-100 text-green-800',
      pending: 'bg-yellow-100 text-yellow-800',
      approved: 'bg-blue-100 text-blue-800',
      rejected: 'bg-red-100 text-red-800',
    };
    
    return (
      <span className={`px-2 py-1 text-xs font-medium rounded-full ${statusColors[status as keyof typeof statusColors] || 'bg-gray-100 text-gray-800'}`}>
        {status.charAt(0).toUpperCase() + status.slice(1)}
      </span>
    );
  };

  const getPositionBadge = (position: string) => {
    const positionColors = {
      'Goalkeeper': 'bg-blue-100 text-blue-800',
      'Defender': 'bg-green-100 text-green-800',
      'Midfielder': 'bg-yellow-100 text-yellow-800',
      'Forward': 'bg-red-100 text-red-800',
      'Striker': 'bg-red-100 text-red-800',
    };
    
    return (
      <span className={`px-2 py-1 text-xs font-medium rounded-full ${positionColors[position as keyof typeof positionColors] || 'bg-gray-100 text-gray-800'}`}>
        {position}
      </span>
    );
  };

  const getAge = (dateOfBirth: string) => {
    if (!dateOfBirth) return '';
    const today = new Date();
    const birthDate = new Date(dateOfBirth);
    let age = today.getFullYear() - birthDate.getFullYear();
    const monthDiff = today.getMonth() - birthDate.getMonth();
    if (monthDiff < 0 || (monthDiff === 0 && today.getDate() < birthDate.getDate())) {
      age--;
    }
    return age;
  };

  const renderOverview = () => (
    <div className="space-y-8">
      {/* Team Header */}
      <div className="bg-gradient-to-r from-blue-600 to-blue-800 rounded-lg p-8 text-white">
        <div className="flex items-center space-x-6">
          {team.logo && (
            <img 
              src={team.logo} 
              alt={`${team.name} logo`}
              className="w-24 h-24 object-cover rounded-full border-4 border-white shadow-lg"
            />
          )}
          <div className="flex-1">
            <h1 className="text-4xl font-bold mb-2">{team.name}</h1>
            <div className="flex items-center space-x-4 text-blue-100">
              {team.country && <span>🌍 {team.country}</span>}
              {team.county && <span>📍 {team.county}</span>}
              {team.location && <span>🏠 {team.location}</span>}
            </div>
          </div>
        </div>
        
        {team.description && (
          <div className="mt-6">
            <p className="text-lg text-blue-100 leading-relaxed">{team.description}</p>
          </div>
        )}
      </div>

      {/* Quick Stats */}
      <div className="grid grid-cols-2 md:grid-cols-4 gap-6">
        <div className="bg-white rounded-lg shadow-md p-6 text-center">
          <div className="text-3xl font-bold text-blue-600 mb-2">{team.team_stats.total_players}</div>
          <div className="text-sm text-gray-600">Total Players</div>
        </div>
        <div className="bg-white rounded-lg shadow-md p-6 text-center">
          <div className="text-3xl font-bold text-green-600 mb-2">{team.team_stats.active_players}</div>
          <div className="text-sm text-gray-600">Active Players</div>
        </div>
        <div className="bg-white rounded-lg shadow-md p-6 text-center">
          <div className="text-3xl font-bold text-purple-600 mb-2">{team.team_stats.tournaments_participated}</div>
          <div className="text-sm text-gray-600">Tournaments</div>
        </div>
        <div className="bg-white rounded-lg shadow-md p-6 text-center">
          <div className="text-3xl font-bold text-orange-600 mb-2">{team.team_stats.matches_won}</div>
          <div className="text-sm text-gray-600">Matches Won</div>
        </div>
      </div>

      {/* Leadership */}
      <div className="bg-white rounded-lg shadow-md p-6">
        <h3 className="text-xl font-bold text-gray-900 mb-4">Leadership</h3>
        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
          {team.captain && (
            <div className="flex items-center space-x-4 p-4 bg-blue-50 rounded-lg">
              <div className="w-12 h-12 bg-blue-600 rounded-full flex items-center justify-center text-white font-bold text-lg">
                C
              </div>
              <div>
                <h4 className="font-semibold text-gray-900">Captain</h4>
                <p className="text-gray-600">{team.captain.name}</p>
                <p className="text-sm text-gray-500">{team.captain.email}</p>
              </div>
            </div>
          )}
          
          {team.manager && (
            <div className="flex items-center space-x-4 p-4 bg-green-50 rounded-lg">
              <div className="w-12 h-12 bg-green-600 rounded-full flex items-center justify-center text-white font-bold text-lg">
                M
              </div>
              <div>
                <h4 className="font-semibold text-gray-900">Manager</h4>
                <p className="text-gray-600">{team.manager.name}</p>
                <p className="text-sm text-gray-500">{team.manager.email}</p>
              </div>
            </div>
          )}
        </div>
      </div>

      {/* Active Tournaments */}
      <div className="bg-white rounded-lg shadow-md p-6">
        <h3 className="text-xl font-bold text-gray-900 mb-4">Active Tournaments</h3>
        {team.active_tournaments.length > 0 ? (
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            {team.active_tournaments.map(tournament => (
              <Link
                key={tournament.id}
                href={`/tournaments/${tournament.slug}`}
                className="p-4 border border-gray-200 rounded-lg hover:border-blue-500 hover:shadow-md transition-all duration-200"
              >
                <h4 className="font-semibold text-gray-900 mb-2">{tournament.name}</h4>
                <div className="text-sm text-gray-600">
                  <div>📅 {tournament.start_date} - {tournament.end_date}</div>
                </div>
              </Link>
            ))}
          </div>
        ) : (
          <p className="text-gray-500 text-center py-8">No active tournaments</p>
        )}
      </div>
    </div>
  );

  const renderPlayers = () => (
    <div className="space-y-6">
      <div className="flex justify-between items-center">
        <h3 className="text-xl font-bold text-gray-900">Players ({team.players.length})</h3>
        <div className="text-sm text-gray-600">
          {team.players.filter(p => p.status === 'active').length} active players
        </div>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
        {team.players.map(player => (
          <div key={player.id} className="bg-white rounded-lg shadow-md hover:shadow-lg transition-shadow duration-300 p-6">
            <div className="flex items-center space-x-4 mb-4">
              {player.avatar && (
                <img 
                  src={player.avatar} 
                  alt={`${player.display_name} avatar`}
                  className="w-16 h-16 object-cover rounded-full border-2 border-gray-200"
                />
              )}
              <div className="flex-1">
                <h4 className="font-bold text-gray-900 text-lg">{player.user.name}</h4>
                <p className="text-gray-600">{player.display_name}</p>
                {player.is_team_captain && (
                  <span className="inline-block px-2 py-1 bg-blue-100 text-blue-800 text-xs font-medium rounded-full mt-1">
                    Captain
                  </span>
                )}
              </div>
            </div>

            <div className="space-y-3">
              <div className="flex items-center justify-between">
                <span className="text-gray-600">Jersey No.</span>
                <span className="font-semibold">{player.jersey_number || ''}</span>
              </div>
              
              <div className="flex items-center justify-between">
                <span className="text-gray-600">Position</span>
                {getPositionBadge(player.football_position)}
              </div>
              
              <div className="flex items-center justify-between">
                <span className="text-gray-600">Age</span>
                <span className="font-semibold">{getAge(player.date_of_birth)}</span>
              </div>
              
              <div className="flex items-center justify-between">
                <span className="text-gray-600">Status</span>
                {getStatusBadge(player.status)}
              </div>
            </div>

            {/* Player Stats */}
            <div className="mt-4 pt-4 border-t border-gray-200">
              <h5 className="font-semibold text-gray-900 mb-2">Performance</h5>
              <div className="grid grid-cols-2 gap-2 text-sm">
                <div className="flex justify-between">
                  <span className="text-gray-600">Matches:</span>
                  <span className="font-medium">{player.tournament_matches_played}</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-gray-600">Wins:</span>
                  <span className="font-medium">{player.tournament_matches_won}</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-gray-600">Win Rate:</span>
                  <span className="font-medium">{player.tournament_win_rate}%</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-gray-600">Rating:</span>
                  <span className="font-medium">{player.tournament_rating || ''}</span>
                </div>
              </div>
            </div>
          </div>
        ))}
      </div>
    </div>
  );

  const renderFormation = () => (
    <div className="space-y-6">
      {team.default_formation ? (
        <>
          <div className="bg-white rounded-lg shadow-md p-6">
            <div className="flex items-center justify-between mb-6">
              <div>
                <h3 className="text-2xl font-bold text-gray-900">{team.default_formation.formation.display_name}</h3>
                <p className="text-gray-600">{team.default_formation.name}</p>
                <p className="text-sm text-gray-500 mt-1">{team.default_formation.formation.description}</p>
              </div>
              <div className="text-right">
                <span className="inline-block px-3 py-1 bg-blue-100 text-blue-800 text-sm font-medium rounded-full">
                  {team.default_formation.formation.category}
                </span>
              </div>
            </div>

            {/* Formation Visualization */}
            <div className="bg-gray-50 rounded-lg p-6">
              <FormationField
                formation={team.default_formation}
                players={team.players}
                showPlayerNames={true}
                className="mx-auto"
              />
            </div>

            {/* Formation Description */}
            {team.default_formation.description && (
              <div className="mt-6 p-4 bg-blue-50 rounded-lg">
                <h4 className="font-semibold text-gray-900 mb-2">Formation Strategy</h4>
                <p className="text-gray-700">{team.default_formation.description}</p>
              </div>
            )}
          </div>

          {/* Player Positions */}
          {team.default_formation.player_positions && Object.keys(team.default_formation.player_positions).length > 0 && (
            <div className="bg-white rounded-lg shadow-md p-6">
              <h4 className="text-xl font-bold text-gray-900 mb-4">Player Positions</h4>
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                {Object.entries(team.default_formation.player_positions).map(([position, playerData]: [string, any]) => {
                  const player = team.players.find(p => p.id === playerData?.playerId);
                  return (
                    <div key={position} className="flex items-center justify-between p-3 bg-gray-50 rounded-lg">
                      <div>
                        <span className="font-medium text-gray-900">{position.replace('_', ' ').toUpperCase()}</span>
                        {player && (
                          <div className="text-sm text-gray-600">
                            {player.display_name} (#{player.jersey_number})
                          </div>
                        )}
                      </div>
                      {player && (
                        <div className="text-right">
                          {getPositionBadge(player.football_position)}
                        </div>
                      )}
                    </div>
                  );
                })}
              </div>
            </div>
          )}
        </>
      ) : (
        <div className="bg-white rounded-lg shadow-md p-12 text-center">
          <div className="text-gray-500">
            <div className="text-6xl mb-4">⚽</div>
            <h3 className="text-xl font-semibold text-gray-900 mb-2">No Formation Available</h3>
            <p className="text-gray-600">This team hasn't set up their formation yet.</p>
          </div>
        </div>
      )}
    </div>
  );

  const renderStats = () => (
    <div className="space-y-6">
      {/* Team Performance */}
      <div className="bg-white rounded-lg shadow-md p-6">
        <h3 className="text-xl font-bold text-gray-900 mb-6">Team Performance</h3>
        <div className="grid grid-cols-2 md:grid-cols-4 gap-6">
          <div className="text-center">
            <div className="text-3xl font-bold text-blue-600 mb-2">{team.team_stats.matches_played}</div>
            <div className="text-sm text-gray-600">Matches Played</div>
          </div>
          <div className="text-center">
            <div className="text-3xl font-bold text-green-600 mb-2">{team.team_stats.matches_won}</div>
            <div className="text-sm text-gray-600">Matches Won</div>
          </div>
          <div className="text-center">
            <div className="text-3xl font-bold text-purple-600 mb-2">
              {team.team_stats.matches_played > 0 
                ? Math.round((team.team_stats.matches_won / team.team_stats.matches_played) * 100)
                : 0}%
            </div>
            <div className="text-sm text-gray-600">Win Rate</div>
          </div>
          <div className="text-center">
            <div className="text-3xl font-bold text-orange-600 mb-2">{team.team_stats.tournaments_participated}</div>
            <div className="text-sm text-gray-600">Tournaments</div>
          </div>
        </div>
      </div>

      {/* Player Statistics */}
      <div className="bg-white rounded-lg shadow-md p-6">
        <h3 className="text-xl font-bold text-gray-900 mb-6">Player Statistics</h3>
        <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
          <div className="text-center p-4 bg-blue-50 rounded-lg">
            <div className="text-2xl font-bold text-blue-600 mb-2">{team.team_stats.total_players}</div>
            <div className="text-sm text-gray-600">Total Players</div>
          </div>
          <div className="text-center p-4 bg-green-50 rounded-lg">
            <div className="text-2xl font-bold text-green-600 mb-2">{team.team_stats.active_players}</div>
            <div className="text-sm text-gray-600">Active Players</div>
          </div>
          <div className="text-center p-4 bg-purple-50 rounded-lg">
            <div className="text-2xl font-bold text-purple-600 mb-2">{team.team_stats.total_officials}</div>
            <div className="text-sm text-gray-600">Officials</div>
          </div>
        </div>
      </div>

      {/* Position Distribution */}
      <div className="bg-white rounded-lg shadow-md p-6">
        <h3 className="text-xl font-bold text-gray-900 mb-6">Position Distribution</h3>
        <div className="space-y-4">
          {['Goalkeeper', 'Defender', 'Midfielder', 'Forward', 'Striker'].map(position => {
            const count = team.players.filter(p => p.football_position === position).length;
            const percentage = team.players.length > 0 ? (count / team.players.length) * 100 : 0;
            
            return (
              <div key={position} className="flex items-center space-x-4">
                <div className="w-24 text-sm font-medium text-gray-700">{position}</div>
                <div className="flex-1 bg-gray-200 rounded-full h-4">
                  <div 
                    className="bg-blue-600 h-4 rounded-full transition-all duration-300"
                    style={{ width: `${percentage}%` }}
                  ></div>
                </div>
                <div className="w-16 text-sm text-gray-600 text-right">{count}</div>
              </div>
            );
          })}
        </div>
      </div>
    </div>
  );

  return (
    <>
      <Head title={`${team.name} - Team Details`} />
      <Header />
      
      <div className="min-h-screen bg-gray-50">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
          {/* Breadcrumb */}
          <nav className="flex items-center space-x-2 text-sm text-gray-600 mb-8">
            <Link href="/" className="hover:text-blue-600">Home</Link>
            <span>›</span>
            <Link href="/teams" className="hover:text-blue-600">Teams</Link>
            <span>›</span>
            <span className="text-gray-900 font-medium">{team.name}</span>
          </nav>

          {/* Tabs */}
          <div className="bg-white rounded-lg shadow-md mb-8">
            <div className="border-b border-gray-200">
              <nav className="-mb-px flex space-x-8 px-6">
                <button
                  onClick={() => setActiveTab('overview')}
                  className={`py-4 px-1 border-b-2 font-medium text-sm ${
                    activeTab === 'overview'
                      ? 'border-blue-500 text-blue-600'
                      : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                  }`}
                >
                  Overview
                </button>
                <button
                  onClick={() => setActiveTab('players')}
                  className={`py-4 px-1 border-b-2 font-medium text-sm ${
                    activeTab === 'players'
                      ? 'border-blue-500 text-blue-600'
                      : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                  }`}
                >
                  Players ({team.players.length})
                </button>
                <button
                  onClick={() => setActiveTab('formation')}
                  className={`py-4 px-1 border-b-2 font-medium text-sm ${
                    activeTab === 'formation'
                      ? 'border-blue-500 text-blue-600'
                      : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                  }`}
                >
                  Formation
                </button>
                <button
                  onClick={() => setActiveTab('stats')}
                  className={`py-4 px-1 border-b-2 font-medium text-sm ${
                    activeTab === 'stats'
                      ? 'border-blue-500 text-blue-600'
                      : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                  }`}
                >
                  Statistics
                </button>
              </nav>
            </div>

            <div className="p-6">
              {activeTab === 'overview' && renderOverview()}
              {activeTab === 'players' && renderPlayers()}
              {activeTab === 'formation' && renderFormation()}
              {activeTab === 'stats' && renderStats()}
            </div>
          </div>
        </div>
      </div>

      <Footer />
    </>
  );
};

export default Team;
