import React from "react";
import { Head } from "@inertiajs/react";
import Header from "@/components/frontend/layout/Header";
import Footer from "@/components/frontend/sections/Footer";
import { Trophy, Medal, Award, Star, Crown, Gift, DollarSign, Users, Target, Calendar } from "lucide-react";

interface PrizeAward {
  id: number;
  tournament_id: number;
  category: string;
  position: string;
  title: string;
  description: string;
  prize_value: number;
  prize_type: string;
  additional_benefits: string;
  is_active: boolean;
  sort_order: number;
  tournament: {
    id: number;
    name: string;
  };
}

interface PrizesPageProps {
  prizeAwards: Record<string, PrizeAward[]>;
  tournamentStats: Array<{
    icon: string;
    value: number;
    label: string;
  }>;
  totalPrizePool: number;
  prizeDistribution: Record<string, {
    total: number;
    percentage: number;
  }>;
  categories: Record<string, string>;
  prizeTypes: Record<string, string>;
  positions: Record<string, string>;
}

const Prizes = ({ 
  prizeAwards, 
  tournamentStats, 
  totalPrizePool, 
  prizeDistribution, 
  categories, 
  prizeTypes, 
  positions 
}: PrizesPageProps) => {
  
  const categoryIcons = {
    team: Trophy,
    individual: Medal,
    special: Award,
    performance: Star,
    fair_play: Crown,
  };

  const categoryColors = {
    team: 'from-blue-500 to-blue-600',
    individual: 'from-green-500 to-green-600',
    special: 'from-purple-500 to-purple-600',
    performance: 'from-yellow-500 to-yellow-600',
    fair_play: 'from-indigo-500 to-indigo-600',
  };

  const prizeTypeIcons = {
    cash: DollarSign,
    trophy: Trophy,
    medal: Medal,
    certificate: Award,
    merchandise: Gift,
    experience: Star,
    mixed: Crown,
  };

  const formatCurrency = (amount: number, currency: string) => {
    return new Intl.NumberFormat('en-KE', {
      style: 'currency',
      currency: currency || 'KES',
      minimumFractionDigits: 0,
      maximumFractionDigits: 0,
    }).format(amount);
  };

  const getIconComponent = (iconName: string) => {
    const icons: Record<string, any> = {
      Trophy,
      Medal,
      Award,
      Star,
    };
    return icons[iconName] || Trophy;
  };  

  const getPositionColor = (position: string) => {
    const colors = {
      "Champions": "bg-gradient-to-r from-yellow-400 to-orange-500",
      "Runners-up": "bg-gradient-to-r from-gray-300 to-gray-400",
      "Third Place": "bg-gradient-to-r from-orange-400 to-red-500",
      "Best Player": "bg-gradient-to-r from-blue-400 to-purple-500",
      "Top Scorer": "bg-gradient-to-r from-green-400 to-teal-500",
      "Best Goalkeeper": "bg-gradient-to-r from-purple-400 to-pink-500",
      "Most Valuable Player": "bg-gradient-to-r from-yellow-400 to-orange-500",
      "Fair Play Award": "bg-gradient-to-r from-green-400 to-emerald-500",
      "Best Young Player": "bg-gradient-to-r from-blue-400 to-cyan-500",
      "Best Coach": "bg-gradient-to-r from-indigo-400 to-purple-500"
    };
    return colors[position as keyof typeof colors] || "bg-gradient-to-r from-gray-400 to-gray-500";
  };

  return (
    <>
      <Head title="Prizes & Awards - Tournament" />
    <Header />
      
      <div className="min-h-screen bg-gray-50">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-12">
          {/* Header */}
          <div className="text-center mb-12">
            <div className="inline-flex items-center justify-center w-20 h-20 bg-gradient-to-r from-yellow-400 to-orange-500 rounded-full mb-6 shadow-lg">
              <Trophy className="w-10 h-10 text-white" />
            </div>
            <h1 className="text-4xl font-bold text-gray-900 mb-4">Prizes & Awards</h1>
            <p className="text-lg text-gray-600 max-w-3xl mx-auto">
              Compete for amazing prizes and recognition in our prestigious tournament. 
              From team championships to individual excellence awards.
            </p>
          </div>

          {/* Prize Pool Stats */}
          <div className="grid grid-cols-2 md:grid-cols-4 gap-6 mb-12">
            {tournamentStats.map((stat: any, index: number) => {
              const IconComponent = getIconComponent(stat.icon);
              return (
                <div key={index} className="bg-white rounded-lg shadow-md p-6 text-center hover:shadow-lg transition-shadow duration-300">
                  <div className="inline-flex items-center justify-center w-12 h-12 bg-blue-100 rounded-full mb-4">
                    <IconComponent className="w-6 h-6 text-blue-600" />
                  </div>
                  <div className="text-2xl font-bold text-gray-900 mb-2">{stat.value}</div>
                  <div className="text-sm text-gray-600">{stat.label}</div>
                </div>
              );
            })}
          </div>

          {/* Prize Categories */}
          <div className="space-y-12">
            {Object.entries(prizeAwards).map(([categoryKey, awards], categoryIndex) => {
              const categoryIcon = categoryIcons[categoryKey as keyof typeof categoryIcons] || Trophy;
              const categoryColor = categoryColors[categoryKey as keyof typeof categoryColors] || 'from-gray-500 to-gray-600';
              const categoryTitle = categories[categoryKey] || categoryKey;
              
              return (
                <div key={categoryKey} className="bg-white rounded-lg shadow-lg overflow-hidden">
                  <div className={`bg-gradient-to-r ${categoryColor} px-6 py-6 text-white`}>
                    <div className="flex items-center space-x-4">
                      <div className="p-3 bg-white/20 rounded-lg backdrop-blur-sm">
                        {React.createElement(categoryIcon, { className: "w-8 h-8" })}
                      </div>
                      <div>
                        <h2 className="text-3xl font-bold">{categoryTitle}</h2>
                        <p className="text-white/90 mt-1">{awards.length} award{awards.length !== 1 ? 's' : ''} available</p>
                      </div>
                    </div>
                  </div>
                  
                  <div className="p-6">
                    <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                      {awards.map((prize) => {
                        const PrizeTypeIcon = prizeTypeIcons[prize.prize_type as keyof typeof prizeTypeIcons] || Gift;
                        return (
                          <div key={prize.id} className="relative group">
                            <div className="bg-white border border-gray-200 rounded-lg p-6 hover:shadow-lg transition-all duration-300 group-hover:border-blue-300">
                              {/* Position Badge */}
                              <div className={`inline-flex items-center px-3 py-1 rounded-full text-sm font-medium mb-4 ${getPositionColor(positions[prize.position] || prize.position)}`}>
                                <PrizeTypeIcon className="w-4 h-4 mr-2" />
                                {positions[prize.position] || prize.position}
                              </div>
                              
                              {/* Prize Title */}
                              <h3 className="text-xl font-bold text-gray-900 mb-2">{prize.title}</h3>
                              
                              {/* Prize Amount */}
                              <div className="text-2xl font-bold text-green-600 mb-3">
                                {formatCurrency(prize.prize_value, 'KES')}
                              </div>
                              
                              {/* Description */}
                              <div className="text-gray-700 mb-4">
                                <p className="text-sm">{prize.description}</p>
                              </div>
                              
                              {/* Additional Benefits */}
                              {prize.additional_benefits && (
                                <div className="text-gray-600 text-sm">
                                  <p className="font-medium mb-1">Additional Benefits:</p>
                                  <p>{prize.additional_benefits}</p>
                                </div>
                              )}
                              
                              {/* Tournament Info */}
                              <div className="mt-4 pt-4 border-t border-gray-200">
                                <p className="text-xs text-gray-500">Tournament: {prize.tournament.name}</p>
                              </div>
                              
                              {/* Decorative Element */}
                              <div className="absolute top-4 right-4 opacity-10 group-hover:opacity-20 transition-opacity duration-300">
                                <PrizeTypeIcon className="w-8 h-8 text-gray-400" />
                              </div>
                            </div>
                          </div>
                        );
                      })}
                    </div>
                  </div>
                </div>
              );
            })}
          </div>

          {/* Prize Distribution Chart */}
          <div className="mt-12 bg-white rounded-lg shadow-lg p-8">
            <h2 className="text-2xl font-bold text-gray-900 mb-6 text-center">Prize Distribution</h2>
            <div className="text-center mb-6">
              <div className="text-4xl font-bold text-green-600 mb-2">
                {formatCurrency(totalPrizePool, 'KES')}
              </div>
              <div className="text-lg text-gray-600">Total Prize Pool</div>
            </div>
            <div className="grid grid-cols-1 md:grid-cols-3 gap-8">
              <div className="text-center">
                <div className="w-24 h-24 bg-gradient-to-r from-yellow-400 to-orange-500 rounded-full flex items-center justify-center mx-auto mb-4">
                  <Trophy className="w-12 h-12 text-white" />
                </div>
                <h3 className="text-lg font-semibold text-gray-900 mb-2">Team Prizes</h3>
                <div className="text-2xl font-bold text-gray-900 mb-1">
                  {formatCurrency(prizeDistribution.team.total, 'KES')}
                </div>
                <div className="text-sm text-gray-600">{prizeDistribution.team.percentage}% of total pool</div>
              </div>
              
              <div className="text-center">
                <div className="w-24 h-24 bg-gradient-to-r from-blue-400 to-purple-500 rounded-full flex items-center justify-center mx-auto mb-4">
                  <Star className="w-12 h-12 text-white" />
                </div>
                <h3 className="text-lg font-semibold text-gray-900 mb-2">Individual Awards</h3>
                <div className="text-2xl font-bold text-gray-900 mb-1">
                  {formatCurrency(prizeDistribution.individual.total, 'KES')}
                </div>
                <div className="text-sm text-gray-600">{prizeDistribution.individual.percentage}% of total pool</div>
              </div>
              
              <div className="text-center">
                <div className="w-24 h-24 bg-gradient-to-r from-green-400 to-teal-500 rounded-full flex items-center justify-center mx-auto mb-4">
                  <Gift className="w-12 h-12 text-white" />
                </div>
                <h3 className="text-lg font-semibold text-gray-900 mb-2">Special Recognition</h3>
                <div className="text-2xl font-bold text-gray-900 mb-1">
                  {formatCurrency(prizeDistribution.special.total, 'KES')}
                </div>
                <div className="text-sm text-gray-600">{prizeDistribution.special.percentage}% of total pool</div>
              </div>
            </div>
          </div>

          {/* Terms & Conditions */}
          <div className="mt-8 bg-gradient-to-r from-blue-50 to-purple-50 border border-blue-200 rounded-lg p-6">
            <h3 className="text-lg font-semibold text-blue-900 mb-4">Terms & Conditions</h3>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6 text-blue-800">
              <div>
                <h4 className="font-medium mb-2">Prize Eligibility:</h4>
                <ul className="text-sm space-y-1">
                  <li>• All prizes subject to verification</li>
                  <li>• Winners must be present at award ceremony</li>
                  <li>• Cash prizes paid within 30 days</li>
                  <li>• Trophies and medals awarded immediately</li>
                </ul>
              </div>
              <div>
                <h4 className="font-medium mb-2">Additional Notes:</h4>
                <ul className="text-sm space-y-1">
                  <li>• Photo sessions included with all prizes</li>
                  <li>• Certificates provided for all winners</li>
                  <li>• Prize values subject to change</li>
                  <li>• Tournament organizers' decisions are final</li>
                </ul>
              </div>
            </div>
          </div>          
      </div>
    </div>

    <Footer />
  </>
);
};

export default Prizes; 