import Header from "@/components/frontend/layout/Header";
import Footer from "@/components/frontend/sections/Footer";
import { Calendar, User, Clock, Share2, Facebook, Twitter, Linkedin, ArrowLeft, Bookmark, Heart } from "lucide-react";
import { Card, CardContent } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { useState } from "react";
import { Link, usePage } from "@inertiajs/react";
import { newsArticles } from "@/lib/data";

const NewsArticle = () => {
  const { id } = usePage().props as unknown as { id: string };
  const [isLiked, setIsLiked] = useState(false);
  const [isBookmarked, setIsBookmarked] = useState(false);

  // Get article data based on ID
  const article = newsArticles.find(article => article.id === Number(id)) || newsArticles[0];

  // Get related articles (excluding current article)
  const relatedArticles = newsArticles.filter(article => article.id !== Number(id)).slice(0, 3);

  const formatDate = (dateString: string) => {
    const date = new Date(dateString);
    return date.toLocaleDateString('en-US', { 
      year: 'numeric', 
      month: 'long', 
      day: 'numeric' 
    });
  };

  const handleShare = (platform: string) => {
    const url = window.location.href;
    const text = article.title;
    
    switch (platform) {
      case 'facebook':
        window.open(`https://www.facebook.com/sharer/sharer.php?u=${encodeURIComponent(url)}`);
        break;
      case 'twitter':
        window.open(`https://twitter.com/intent/tweet?url=${encodeURIComponent(url)}&text=${encodeURIComponent(text)}`);
        break;
      case 'linkedin':
        window.open(`https://www.linkedin.com/sharing/share-offsite/?url=${encodeURIComponent(url)}`);
        break;
    }
  };

  return (
    <div className="min-h-screen bg-background">
      <Header />
      
      {/* Article Header */}
      <section className="relative py-20 bg-gradient-to-br from-primary/90 to-secondary/90 overflow-hidden">
        <div className="absolute inset-0 bg-gradient-to-b from-black/60 to-black/80" />
        <div className="relative z-10 max-w-4xl mx-auto px-4 sm:px-6 lg:px-8 text-center text-white">
          <Link href="/news" className="inline-flex items-center gap-2 text-white/80 hover:text-white mb-6">
            <ArrowLeft className="h-4 w-4" />
            Back to News
          </Link>
          
          <div className="mb-6">
            <span className="px-3 py-1 bg-primary text-white text-sm font-semibold rounded-full">
              Registration
            </span>
          </div>
          
          <h1 className="text-3xl md:text-5xl font-bold mb-6">
            {article.title}
          </h1>
          
          <p className="text-lg md:text-xl text-white/90 max-w-3xl mx-auto mb-8">
            {article.excerpt}
          </p>
          
          <div className="flex flex-col sm:flex-row items-center justify-center gap-6 text-sm text-white/80">
            <div className="flex items-center gap-2">
              <User className="h-4 w-4" />
              {article.author}
            </div>
            <div className="flex items-center gap-2">
              <Calendar className="h-4 w-4" />
              {formatDate(article.date)}
            </div>
            <div className="flex items-center gap-2">
              <Clock className="h-4 w-4" />
              {article.readTime}
            </div>
          </div>
        </div>
      </section>

      {/* Article Content */}
      <section className="py-16">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="grid lg:grid-cols-4 gap-8">
            {/* Main Content */}
            <div className="lg:col-span-3">
              <div className="bg-white rounded-2xl shadow-xl overflow-hidden">
                <img
                  src={article.image}
                  alt={article.title}
                  className="w-full h-64 md:h-80 object-cover"
                />
                
                <div className="p-8">
                  {/* Article Actions */}
                  <div className="flex items-center justify-between mb-8 pb-6 border-b">
                    <div className="flex items-center gap-4">
                      <button
                        onClick={() => setIsLiked(!isLiked)}
                        className={`flex items-center gap-2 px-4 py-2 rounded-full transition-all duration-200 ${
                          isLiked 
                            ? "bg-red-500 text-white" 
                            : "bg-gray-100 text-gray-700 hover:bg-red-50"
                        }`}
                      >
                        <Heart className={`h-4 w-4 ${isLiked ? "fill-current" : ""}`} />
                        {article.likes ? article.likes + (isLiked ? 1 : 0) : 0}
                      </button>
                      <button
                        onClick={() => setIsBookmarked(!isBookmarked)}
                        className={`flex items-center gap-2 px-4 py-2 rounded-full transition-all duration-200 ${
                          isBookmarked 
                            ? "bg-yellow-500 text-white" 
                            : "bg-gray-100 text-gray-700 hover:bg-yellow-50"
                        }`}
                      >
                        <Bookmark className={`h-4 w-4 ${isBookmarked ? "fill-current" : ""}`} />
                        Bookmark
                      </button>
                    </div>
                    
                    <div className="flex items-center gap-2">
                      <span className="text-sm text-gray-500">{article.views} views</span>
                      <div className="flex gap-2">
                        <button
                          onClick={() => handleShare('facebook')}
                          className="p-2 rounded-full bg-blue-600 text-white hover:bg-blue-700 transition-colors"
                        >
                          <Facebook className="h-4 w-4" />
                        </button>
                        <button
                          onClick={() => handleShare('twitter')}
                          className="p-2 rounded-full bg-blue-400 text-white hover:bg-blue-500 transition-colors"
                        >
                          <Twitter className="h-4 w-4" />
                        </button>
                        <button
                          onClick={() => handleShare('linkedin')}
                          className="p-2 rounded-full bg-blue-700 text-white hover:bg-blue-800 transition-colors"
                        >
                          <Linkedin className="h-4 w-4" />
                        </button>
                      </div>
                    </div>
                  </div>

                  {/* Article Content */}
                  <div 
                    className="prose prose-lg max-w-none"
                    dangerouslySetInnerHTML={{ __html: article.content || "" }}
                  />

                  {/* Tags */}
                  <div className="mt-8 pt-6 border-t">
                    <h3 className="text-lg font-semibold text-foreground mb-4">Tags</h3>
                    <div className="flex flex-wrap gap-2">
                      {article.tags?.map((tag: string, index: number) => (
                        <span
                          key={index}
                          className="px-3 py-1 bg-gray-100 text-gray-700 text-sm rounded-full"
                        >
                          {tag}
                        </span>
                      ))}
                    </div>
                  </div>
                </div>
              </div>
            </div>

            {/* Sidebar */}
            <div className="lg:col-span-1">
              {/* Author Info */}
              <Card className="mb-8">
                <CardContent className="p-6">
                  <h3 className="text-lg font-semibold text-foreground mb-4">About the Author</h3>
                  <div className="flex items-center gap-4">
                    <img
                      src={article.authorImage}
                      alt={article.author}
                      className="w-12 h-12 rounded-full object-cover"
                    />
                    <div>
                      <h4 className="font-semibold text-foreground">{article.author}</h4>
                      <p className="text-sm text-muted-foreground">Tournament Committee</p>
                    </div>
                  </div>
                </CardContent>
              </Card>

              {/* Related Articles */}
              <Card>
                <CardContent className="p-6">
                  <h3 className="text-lg font-semibold text-foreground mb-4">Related Articles</h3>
                  <div className="space-y-4">
                    {relatedArticles.map((relatedArticle) => (
                                             <Link
                         key={relatedArticle.id}
                         href={`/news/${relatedArticle.id}`}
                         className="block group"
                       >
                        <div className="flex gap-3">
                          <img
                            src={relatedArticle.image}
                            alt={relatedArticle.title}
                            className="w-16 h-16 rounded object-cover"
                          />
                          <div>
                            <h4 className="font-medium text-foreground group-hover:text-primary transition-colors line-clamp-2">
                              {relatedArticle.title}
                            </h4>
                            <p className="text-xs text-muted-foreground mt-1">
                              {formatDate(relatedArticle.date)}
                            </p>
                          </div>
                        </div>
                      </Link>
                    ))}
                  </div>
                </CardContent>
              </Card>
            </div>
          </div>
        </div>
      </section>

      <Footer />
    </div>
  );
};

export default NewsArticle;