import Header from "@/components/frontend/layout/Header";
import Footer from "@/components/frontend/sections/Footer";
import { Calendar, Search, Trophy, CheckCircle, Clock, LayoutGrid, List, ChevronDown } from "lucide-react";
import React, { useState, useMemo, useEffect, useRef } from "react";


interface FixturesProps {
  fixtures: FixtureData[];
}

interface FixtureData {
  id: number;
  date: string;
  time: string;
  teamA: FixtureTeam;
  teamB: FixtureTeam;
  venue: string;
  status: string;
  result: string;
}

interface FixtureTeam {
  name: string;
  logo: string;
}




const statusBadge = (status: string) => {
  if (status === "live") return <span className="px-3 py-1 rounded-full bg-red-500 text-white text-xs font-bold animate-pulse">Live</span>;
  if (status === "completed") return <span className="px-3 py-1 rounded-full bg-green-600 text-white text-xs font-bold flex items-center gap-1"><CheckCircle className="h-4 w-4" />Completed</span>;
  return <span className="px-3 py-1 rounded-full bg-gray-300 text-gray-700 text-xs font-bold flex items-center gap-1"><Clock className="h-4 w-4" />Upcoming</span>;
};

const Fixtures = ({ fixtures }: FixturesProps) => {
  const [search, setSearch] = useState("");
  const [view, setView] = useState("list"); // 'grid' or 'list'
  const [selectedYear, setSelectedYear] = useState(new Date().getFullYear());
  const [showYearDropdown, setShowYearDropdown] = useState(false);
  const yearDropdownRef = useRef<HTMLDivElement>(null);

  // Extract unique years from fixtures
  const availableYears = useMemo(() => {
    const years = fixtures
      .map(f => {
        const date = new Date(f.date);
        return date.getFullYear();
      })
      .filter((year, index, arr) => arr.indexOf(year) === index)
      .sort((a, b) => b - a); // Sort in descending order (newest first)
    
    return years.length > 0 ? years : [new Date().getFullYear()];
  }, [fixtures]);

  // Filter fixtures by year and search
  const filteredFixtures = useMemo(() => {
    return fixtures.filter(f => {
      const fixtureDate = new Date(f.date);
      const fixtureYear = fixtureDate.getFullYear();
      const q = search.toLowerCase();
      
      return (
        fixtureYear === selectedYear &&
        (f.teamA.name.toLowerCase().includes(q) ||
         f.teamB.name.toLowerCase().includes(q) ||
         f.date.includes(q))
      );
    });
  }, [fixtures, selectedYear, search]);

  // Close dropdown when clicking outside
  useEffect(() => {
    const handleClickOutside = (event: MouseEvent) => {
      if (yearDropdownRef.current && !yearDropdownRef.current.contains(event.target as Node)) {
        setShowYearDropdown(false);
      }
    };

    document.addEventListener('mousedown', handleClickOutside);
    return () => {
      document.removeEventListener('mousedown', handleClickOutside);
    };
  }, []);

  return (
    <div className="min-h-screen bg-background">
      <Header />
      {/* Hero Section */}
      <section className="relative flex items-center justify-center h-[260px] md:h-[320px] bg-gradient-to-br from-primary/80 to-secondary/80 overflow-hidden">
        <img
          src="/lovable-uploads/18075074-5322-458c-b0cf-a75db311c3c0.png"
          alt="Fixtures Hero"
          className="absolute inset-0 w-full h-full object-cover opacity-10 pointer-events-none select-none"
        />
        <div className="absolute inset-0 bg-gradient-to-b from-black/60 to-black/80" />
        <div className="relative z-10 text-center text-white flex flex-col items-center justify-center w-full">
          <Calendar className="mx-auto mb-4 h-12 w-12 text-green-700 animate-bounce-gentle" />
          <h1 className="text-4xl md:text-5xl font-extrabold drop-shadow mb-2">Fixtures & Results</h1>
          <p className="text-lg md:text-xl max-w-2xl mx-auto drop-shadow">
            Stay up to date with the latest fixtures, results, and live action.
          </p>
        </div>
      </section>
      {/* Filter/Search Bar and View Toggle */}
      <section className="max-w-7xl mx-auto py-8 px-4">
        <form className="flex flex-col md:flex-row gap-4 items-center justify-between bg-white rounded-xl shadow p-4 mb-8">
          <div className="flex items-center w-full md:w-auto gap-2">
            <Search className="h-5 w-5 text-gray-400" />
            <input
              type="text"
              placeholder="Search team or date..."
              className="w-full md:w-64 px-3 py-2 rounded border border-gray-200 focus:outline-none focus:ring-2 focus:ring-primary"
              value={search}
              onChange={e => setSearch(e.target.value)}
            />
          </div>
          
          {/* Year Filter */}
          <div className="relative" ref={yearDropdownRef}>
            <button
              type="button"
              onClick={() => setShowYearDropdown(!showYearDropdown)}
              className="flex items-center gap-2 px-4 py-2 border border-gray-200 rounded-lg hover:border-primary focus:outline-none focus:ring-2 focus:ring-primary"
            >
              <Calendar className="h-4 w-4 text-gray-500" />
              <span className="text-gray-700 font-medium">{selectedYear}</span>
              <ChevronDown className={`h-4 w-4 text-gray-500 transition-transform ${showYearDropdown ? 'rotate-180' : ''}`} />
            </button>
            
            {showYearDropdown && (
              <div className="absolute top-full left-0 mt-1 w-32 bg-white border border-gray-200 rounded-lg shadow-lg z-10 max-h-48 overflow-y-auto">
                {availableYears.map((year) => (
                  <button
                    key={year}
                    type="button"
                    onClick={() => {
                      setSelectedYear(year);
                      setShowYearDropdown(false);
                    }}
                    className={`w-full px-4 py-2 text-left hover:bg-gray-50 ${
                      selectedYear === year ? 'bg-primary text-white' : 'text-gray-700'
                    }`}
                  >
                    {year}
                  </button>
                ))}
              </div>
            )}
          </div>
          
          <div className="flex gap-2 items-center">
            <button
              type="button"
              className={`p-2 rounded-full border ${view === "grid" ? "bg-primary text-white border-primary" : "bg-gray-100 text-gray-500 border-gray-200"}`}
              onClick={() => setView("grid")}
              aria-label="Grid view"
            >
              <LayoutGrid className="h-5 w-5" />
            </button>
            <button
              type="button"
              className={`p-2 rounded-full border ${view === "list" ? "bg-primary text-white border-primary" : "bg-gray-100 text-gray-500 border-gray-200"}`}
              onClick={() => setView("list")}
              aria-label="List view"
            >
              <List className="h-5 w-5" />
            </button>
          </div>
        </form>
        
        {/* Filter Summary */}
        <div className="mb-6 text-center">
          <p className="text-gray-600">
            Showing {filteredFixtures.length} fixture{filteredFixtures.length !== 1 ? 's' : ''} for {selectedYear}
            {search && (
              <span className="ml-2">
                matching "<span className="font-semibold">{search}</span>"
              </span>
            )}
          </p>
        </div>
        
        {/* Fixtures List */}
        {view === "grid" ? (
          <div className="grid gap-6 sm:grid-cols-1 md:grid-cols-2 justify-center w-full overflow-x-hidden max-w-7xl mx-auto">
            {filteredFixtures.map((f, i) => (
              <div key={i} className="bg-white rounded-xl shadow-lg p-6 flex flex-col gap-4 border border-gray-100 hover:shadow-2xl transition-shadow duration-300 w-full">
                {/* Teams Row */}
                  <div className="font-bold text-lg text-primary flex justify-center items-center gap-2">
                    <span className="text-gray-500 font-bold text-xl flex items-center gap-2"><img src={f.teamA.logo} alt={f.teamA.name} className="w-12 h-12 rounded-full object-cover border-2 border-primary" /> {f.teamA.name} </span> 
                    <span className="text-red-500 font-bold text-4xl mb-8">vs</span> 
                    <span className="text-gray-500 font-bold text-xl flex items-center gap-2">{f.teamB.name} <img src={f.teamB.logo} alt={f.teamB.name} className="w-12 h-12 rounded-full object-cover border-2 border-secondary" /></span>
                  </div> 
                {/* Date/Venue and Result/Status Row */}
                <div className="flex flex-col mt-4 sm:flex-row items-center justify-between gap-4 w-full">
                  <div className="flex flex-col items-center sm:items-start gap-1 min-w-[120px]">
                    <div className="text-gray-700 font-medium">{f.date} <span className="text-gray-400">|</span> {f.time}</div>
                    <div className="text-gray-500 text-sm">{f.venue}</div>
                  </div>
                  <div className="flex flex-col items-center gap-1 min-w-[100px]">
                    <div className="font-bold text-lg">{f.result}</div>
                    {statusBadge(f.status)}
                  </div>
                </div>
              </div>
            ))}
            {filteredFixtures.length === 0 && (
              <div className="col-span-full text-center text-gray-500 py-8">No fixtures found.</div>
            )}
          </div>
        ) : (
          <div className="overflow-x-auto">
            <table className="w-full text-left border rounded-xl bg-white shadow max-w-7xl mx-auto border-gray-100 shadow-lg">
              <thead>
                <tr className="bg-gray-100">
                  <th className="py-2 px-4">Date</th>
                  <th className="py-2 px-4">Time</th>
                  <th className="py-2 px-4">Team A</th>
                  <th className="py-2 px-4">vs</th>
                  <th className="py-2 px-4">Team B</th>
                  <th className="py-2 px-4">Result</th>
                </tr>
              </thead>
              <tbody>
                {filteredFixtures.map((f, i) => (
                  <tr key={i} className="border-b hover:bg-gray-50">
                    <td className="py-2 px-4">{f.date}</td>
                    <td className="py-2 px-4">{f.time}</td>
                    <td className="py-2 px-4">
                      <div className="flex items-center gap-2">
                        <img src={f.teamA.logo} alt={f.teamA.name} className="w-8 h-8 rounded-full object-cover border-2 border-primary" />
                        <span className="text-primary">{f.teamA.name}</span>
                      </div>
                    </td>
                    <td className="py-2 px-4 m-auto text-red-500 font-bold text-4xl mb-4">vs</td>
                    <td className="py-2 px-4">
                      <div className="flex items-center gap-2">
                        <img src={f.teamB.logo} alt={f.teamB.name} className="w-8 h-8 rounded-full object-cover border-2 border-secondary" />
                        <span className="text-secondary">{f.teamB.name}</span>
                      </div>
                    </td>
                    <td className="py-2 px-4 font-bold">{f.result}</td>
                  </tr>
                ))}
                {filteredFixtures.length === 0 && (
                  <tr><td colSpan={7} className="text-center text-gray-500 py-8">No fixtures found.</td></tr>
                )}
              </tbody>
            </table>
          </div>
        )}
      </section>
      {/* Call to Action */}
      <section className="py-12 bg-gradient-to-r from-primary/10 to-secondary/10">
        <div className="max-w-2xl mx-auto text-center">
          <h2 className="text-2xl md:text-3xl font-bold mb-4 text-primary">Want to see your team here?</h2>
          <p className="mb-6 text-lg text-gray-700">Register your team now or check out the full results!</p>
          <a href="/register" className="inline-block px-8 py-3 rounded-full bg-gradient-to-r from-primary to-secondary text-white font-bold text-lg shadow hover:from-secondary hover:to-primary transition-colors mr-4">Register Team</a>
          <a href="/certificates" className="inline-block px-8 py-3 rounded-full bg-white text-primary font-bold text-lg shadow border border-primary hover:bg-primary hover:text-white transition-colors">View Results</a>
        </div>
      </section>
      <Footer />
    </div>
  );
};

export default Fixtures; 