import React, { useState } from "react";
import { Head } from "@inertiajs/react";
import Header from "@/components/frontend/layout/Header";
import Footer from "@/components/frontend/sections/Footer";
import { ChevronDown, ChevronUp, HelpCircle, Users, Trophy, Calendar, Shield, CreditCard, Mail, Phone, MapPin } from "lucide-react";

interface FaqData {
  id: number;
  category: string;
  question: string;
  answer: string;
  sort_order: number;
  is_active: boolean;
}

interface FaqsPageProps {
  faqs: Record<string, FaqData[]>;
  categories: Record<string, string>;
}

const FAQ = ({ faqs, categories }: FaqsPageProps) => {
  const [openItems, setOpenItems] = useState<number[]>([]);
  const [activeCategory, setActiveCategory] = useState<string>('all');

  const toggleItem = (faqId: number) => {
    setOpenItems(prev => 
      prev.includes(faqId) 
        ? prev.filter(item => item !== faqId)
        : [...prev, faqId]
    );
  };

  const faqCategories = [
    {
      id: 'general',
      title: 'General Information',
      icon: HelpCircle,
      color: 'from-blue-500 to-purple-600'
    },
    {
      id: 'registration',
      title: 'Team Registration',
      icon: Users,
      color: 'from-green-500 to-teal-600'
    },
    {
      id: 'tournament',
      title: 'Tournament Rules',
      icon: Trophy,
      color: 'from-yellow-500 to-orange-600'
    },
    {
      id: 'schedule',
      title: 'Schedule & Fixtures',
      icon: Calendar,
      color: 'from-purple-500 to-pink-600'
    },
    {
      id: 'prizes',
      title: 'Prizes & Awards',
      icon: Trophy,
      color: 'from-red-500 to-rose-600'
    },
    {
      id: 'support',
      title: 'Support & Contact',
      icon: Shield,
      color: 'from-indigo-500 to-blue-600'
    }
  ];

  // Convert grouped FAQs to flat array for easier handling
  const allFaqs = Object.values(faqs).flat();

  const filteredFAQs = activeCategory === 'all' 
    ? allFaqs 
    : allFaqs.filter(faq => faq.category === activeCategory);

  const contactInfo = [
    { icon: Mail, label: "Email", value: "info@milimustournament.com" },
    { icon: Phone, label: "Phone", value: "+254 700 000 000" },
    { icon: MapPin, label: "Office", value: "Nairobi, Kenya" }
  ];

  return (
    <>
      <Head title="FAQ - Tournament" />
    <Header />
      
      <div className="min-h-screen bg-gray-50">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-12">
          {/* Header */}
          <div className="text-center mb-12">
            <div className="inline-flex items-center justify-center w-20 h-20 bg-gradient-to-r from-blue-500 to-purple-600 rounded-full mb-6 shadow-lg">
              <HelpCircle className="w-10 h-10 text-white" />
            </div>
            <h1 className="text-4xl font-bold text-gray-900 mb-4">Frequently Asked Questions</h1>
            <p className="text-lg text-gray-600 max-w-3xl mx-auto">
              Find answers to common questions about our tournament, registration process, 
              rules, and more. Can't find what you're looking for? Contact us directly.
            </p>
          </div>

          {/* Category Filter */}
          <div className="mb-8">
            <div className="flex flex-wrap justify-center gap-3">
              <button
                onClick={() => setActiveCategory('all')}
                className={`px-6 py-3 rounded-lg font-medium transition-all duration-300 ${
                  activeCategory === 'all'
                    ? 'bg-gradient-to-r from-blue-500 to-purple-600 text-white shadow-lg'
                    : 'bg-white text-gray-700 hover:bg-blue-50 hover:text-blue-600 shadow-md'
                }`}
              >
                All Questions
              </button>
              {faqCategories.map((category) => (
                <button
                  key={category.id}
                  onClick={() => setActiveCategory(category.id)}
                  className={`px-6 py-3 rounded-lg font-medium transition-all duration-300 flex items-center space-x-2 ${
                    activeCategory === category.id
                      ? `bg-gradient-to-r ${category.color} text-white shadow-lg`
                      : 'bg-white text-gray-700 hover:bg-blue-50 hover:text-blue-600 shadow-md'
                  }`}
                >
                  <category.icon className="w-4 h-4" />
                  <span>{category.title}</span>
                </button>
              ))}
            </div>
          </div>

          {/* FAQ Items */}
          <div className="max-w-4xl mx-auto">
            <div className="space-y-4">
              {filteredFAQs.map((faq, index) => (
                <div key={faq.id} className="bg-white rounded-lg shadow-md overflow-hidden">
                  <button
                    onClick={() => toggleItem(faq.id)}
                    className="w-full px-6 py-4 text-left flex items-center justify-between hover:bg-gray-50 transition-colors duration-200"
                  >
                    <div className="flex items-center space-x-4">
                      <div className={`p-2 rounded-lg bg-gradient-to-r ${
                        faqCategories.find(cat => cat.id === faq.category)?.color || 'from-gray-400 to-gray-500'
                      }`}>
                        {React.createElement(
                          faqCategories.find(cat => cat.id === faq.category)?.icon || HelpCircle,
                          { className: "w-5 h-5 text-white" }
                        )}
                      </div>
                      <h3 className="text-lg font-semibold text-gray-900">{faq.question}</h3>
                    </div>
                    {openItems.includes(faq.id) ? (
                      <ChevronUp className="w-5 h-5 text-gray-500" />
                    ) : (
                      <ChevronDown className="w-5 h-5 text-gray-500" />
                    )}
                  </button>
                  
                  {openItems.includes(faq.id) && (
                    <div className="px-6 pb-4">
                      <div className="pl-16">
                        <p className="text-gray-700 leading-relaxed whitespace-pre-wrap">{faq.answer}</p>
                      </div>
                    </div>
                  )}
                </div>
              ))}
            </div>
          </div>

          {/* Contact Section */}
          <div className="mt-16 bg-white rounded-lg shadow-lg p-8">
            <div className="text-center mb-8">
              <h2 className="text-2xl font-bold text-gray-900 mb-4">Still Have Questions?</h2>
              <p className="text-gray-600">
                Can't find the answer you're looking for? Our support team is here to help.
              </p>
            </div>
            
            <div className="grid grid-cols-1 md:grid-cols-3 gap-6 mb-8">
              {contactInfo.map((contact, index) => (
                <div key={index} className="text-center p-6 bg-gray-50 rounded-lg">
                  <div className="inline-flex items-center justify-center w-12 h-12 bg-blue-100 rounded-full mb-4">
                    <contact.icon className="w-6 h-6 text-blue-600" />
                  </div>
                  <h3 className="font-semibold text-gray-900 mb-2">{contact.label}</h3>
                  <p className="text-gray-600">{contact.value}</p>
                </div>
              ))}
            </div>

            <div className="text-center">
              <div className="bg-gradient-to-r from-blue-50 to-purple-50 border border-blue-200 rounded-lg p-6">
                <h3 className="text-lg font-semibold text-blue-900 mb-2">Quick Response Guarantee</h3>
                <p className="text-blue-800 text-sm">
                  We aim to respond to all inquiries within 24 hours during business days. 
                  For urgent matters, please call our hotline.
                </p>
              </div>
            </div>
          </div>

          {/* Additional Resources */}
          <div className="mt-12 grid grid-cols-1 md:grid-cols-3 gap-6">
            <div className="bg-white rounded-lg shadow-md p-6 text-center">
              <div className="inline-flex items-center justify-center w-12 h-12 bg-green-100 rounded-full mb-4">
                <Users className="w-6 h-6 text-green-600" />
              </div>
              <h3 className="text-lg font-semibold text-gray-900 mb-2">Team Registration</h3>
              <p className="text-gray-600 text-sm mb-4">Ready to join the tournament?</p>
              <a 
                href="/register-team" 
                className="inline-block bg-green-600 text-white px-4 py-2 rounded-lg hover:bg-green-700 transition-colors duration-200"
              >
                Register Now
              </a>
            </div>

            <div className="bg-white rounded-lg shadow-md p-6 text-center">
              <div className="inline-flex items-center justify-center w-12 h-12 bg-blue-100 rounded-full mb-4">
                <Trophy className="w-6 h-6 text-blue-600" />
              </div>
              <h3 className="text-lg font-semibold text-gray-900 mb-2">Tournament Rules</h3>
              <p className="text-gray-600 text-sm mb-4">Understand the competition rules</p>
              <a 
                href="/rules" 
                className="inline-block bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700 transition-colors duration-200"
              >
                View Rules
              </a>
            </div>

            <div className="bg-white rounded-lg shadow-md p-6 text-center">
              <div className="inline-flex items-center justify-center w-12 h-12 bg-purple-100 rounded-full mb-4">
                <Calendar className="w-6 h-6 text-purple-600" />
              </div>
              <h3 className="text-lg font-semibold text-gray-900 mb-2">Tournament Schedule</h3>
              <p className="text-gray-600 text-sm mb-4">Check upcoming matches</p>
              <a 
                href="/fixtures" 
                className="inline-block bg-purple-600 text-white px-4 py-2 rounded-lg hover:bg-purple-700 transition-colors duration-200"
              >
                View Fixtures
              </a>
            </div>
          </div>
      </div>
    </div>

    <Footer />
  </>
);
};

export default FAQ; 