import React, { useState } from 'react';
import { Head } from '@inertiajs/react';
import Header from "@/components/frontend/layout/Header";
import Footer from "@/components/frontend/sections/Footer";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Badge } from "@/components/ui/badge";
import { 
  QrCode, 
  Search, 
  CheckCircle, 
  XCircle, 
  AlertCircle,
  Calendar,
  User,
  Award,
  MapPin,
  FileText
} from 'lucide-react';

interface CertificateData {
  id: string;
  type: string;
  recipient: {
    name: string;
    team?: string;
    position?: string;
  };
  tournament: {
    name: string;
    startDate: string;
    endDate: string;
    location: string;
  };
  generatedAt: string;
  validUntil: string;
  status: 'valid' | 'expired' | 'invalid';
}

const CertificateVerification = () => {
  const [certificateId, setCertificateId] = useState('');
  const [isLoading, setIsLoading] = useState(false);
  const [verificationResult, setVerificationResult] = useState<CertificateData | null>(null);
  const [error, setError] = useState<string | null>(null);

  const handleVerify = async () => {
    if (!certificateId.trim()) {
      setError('Please enter a certificate ID');
      return;
    }

    setIsLoading(true);
    setError(null);
    setVerificationResult(null);

    try {
      // In real implementation, this would make an API call to verify the certificate
      // For demo purposes, we'll simulate the verification
      await new Promise(resolve => setTimeout(resolve, 1000));
      
      // Mock verification result
      const mockResult: CertificateData = {
        id: certificateId,
        type: 'participation',
        recipient: {
          name: 'Victor Kipkoech',
          team: 'Lwanda K FC',
          position: 'Striker'
        },
        tournament: {
          name: 'Milimus Tournament Championship 2024',
          startDate: '2024-08-03',
          endDate: '2024-08-31',
          location: 'Nairobi, Kenya'
        },
        generatedAt: 'September 20, 2025',
        validUntil: 'September 20, 2026',
        status: 'valid'
      };

      setVerificationResult(mockResult);
    } catch (err) {
      setError('Failed to verify certificate. Please try again.');
    } finally {
      setIsLoading(false);
    }
  };

  const getStatusIcon = (status: string) => {
    switch (status) {
      case 'valid':
        return <CheckCircle className="h-6 w-6 text-green-500" />;
      case 'expired':
        return <AlertCircle className="h-6 w-6 text-yellow-500" />;
      case 'invalid':
        return <XCircle className="h-6 w-6 text-red-500" />;
      default:
        return <AlertCircle className="h-6 w-6 text-gray-500" />;
    }
  };

  const getStatusBadge = (status: string) => {
    switch (status) {
      case 'valid':
        return <Badge className="bg-green-100 text-green-800">Valid</Badge>;
      case 'expired':
        return <Badge className="bg-yellow-100 text-yellow-800">Expired</Badge>;
      case 'invalid':
        return <Badge className="bg-red-100 text-red-800">Invalid</Badge>;
      default:
        return <Badge className="bg-gray-100 text-gray-800">Unknown</Badge>;
    }
  };

  return (
    <>
      <Head title="Certificate Verification" />
      <Header />
      <div className="min-h-screen bg-gray-50 py-12">
        <div className="container mx-auto px-4">
          {/* Header Section */}
          <div className="text-center mb-12">
            <QrCode className="h-16 w-16 text-blue-500 mx-auto mb-4" />
            <h1 className="text-4xl font-bold text-gray-900 mb-4">
              Certificate Verification
            </h1>
            <p className="text-xl text-gray-600 max-w-3xl mx-auto">
              Verify the authenticity of tournament certificates using certificate IDs or QR codes. 
              All certificates are digitally signed and can be verified for authenticity.
            </p>
          </div>

          {/* Verification Form */}
          <div className="max-w-2xl mx-auto mb-12">
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center">
                  <Search className="h-5 w-5 mr-2" />
                  Verify Certificate
                </CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div>
                  <label htmlFor="certificateId" className="block text-sm font-medium text-gray-700 mb-2">
                    Certificate ID or QR Code Data
                  </label>
                  <Input
                    id="certificateId"
                    type="text"
                    placeholder="Enter certificate ID (e.g., CERT-000075-770DE884)"
                    value={certificateId}
                    onChange={(e) => setCertificateId(e.target.value)}
                    className="w-full"
                  />
                </div>
                <Button 
                  onClick={handleVerify}
                  disabled={isLoading}
                  className="w-full"
                >
                  {isLoading ? (
                    <>
                      <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-white mr-2"></div>
                      Verifying...
                    </>
                  ) : (
                    <>
                      <Search className="h-4 w-4 mr-2" />
                      Verify Certificate
                    </>
                  )}
                </Button>
                
                {error && (
                  <div className="bg-red-50 border border-red-200 rounded-md p-3">
                    <div className="flex">
                      <XCircle className="h-5 w-5 text-red-400 mr-2" />
                      <p className="text-sm text-red-700">{error}</p>
                    </div>
                  </div>
                )}
              </CardContent>
            </Card>
          </div>

          {/* Verification Result */}
          {verificationResult && (
            <div className="max-w-4xl mx-auto">
              <Card className="overflow-hidden">
                <CardHeader className={`${
                  verificationResult.status === 'valid' ? 'bg-green-50' : 
                  verificationResult.status === 'expired' ? 'bg-yellow-50' : 'bg-red-50'
                }`}>
                  <div className="flex items-center justify-between">
                    <CardTitle className="flex items-center">
                      {getStatusIcon(verificationResult.status)}
                      <span className="ml-2">Verification Result</span>
                    </CardTitle>
                    {getStatusBadge(verificationResult.status)}
                  </div>
                </CardHeader>
                
                <CardContent className="p-6">
                  <div className="grid md:grid-cols-2 gap-6">
                    {/* Certificate Details */}
                    <div className="space-y-4">
                      <h3 className="text-lg font-semibold text-gray-900">Certificate Details</h3>
                      
                      <div className="space-y-3">
                        <div className="flex items-center">
                          <FileText className="h-4 w-4 text-gray-400 mr-3" />
                          <div>
                            <p className="text-sm text-gray-600">Certificate ID</p>
                            <p className="font-medium">{verificationResult.id}</p>
                          </div>
                        </div>
                        
                        <div className="flex items-center">
                          <Award className="h-4 w-4 text-gray-400 mr-3" />
                          <div>
                            <p className="text-sm text-gray-600">Type</p>
                            <p className="font-medium capitalize">{verificationResult.type} Certificate</p>
                          </div>
                        </div>
                        
                        <div className="flex items-center">
                          <Calendar className="h-4 w-4 text-gray-400 mr-3" />
                          <div>
                            <p className="text-sm text-gray-600">Generated</p>
                            <p className="font-medium">{verificationResult.generatedAt}</p>
                          </div>
                        </div>
                        
                        <div className="flex items-center">
                          <Calendar className="h-4 w-4 text-gray-400 mr-3" />
                          <div>
                            <p className="text-sm text-gray-600">Valid Until</p>
                            <p className="font-medium">{verificationResult.validUntil}</p>
                          </div>
                        </div>
                      </div>
                    </div>

                    {/* Recipient Details */}
                    <div className="space-y-4">
                      <h3 className="text-lg font-semibold text-gray-900">Recipient Details</h3>
                      
                      <div className="space-y-3">
                        <div className="flex items-center">
                          <User className="h-4 w-4 text-gray-400 mr-3" />
                          <div>
                            <p className="text-sm text-gray-600">Name</p>
                            <p className="font-medium">{verificationResult.recipient.name}</p>
                          </div>
                        </div>
                        
                        {verificationResult.recipient.team && (
                          <div className="flex items-center">
                            <User className="h-4 w-4 text-gray-400 mr-3" />
                            <div>
                              <p className="text-sm text-gray-600">Team</p>
                              <p className="font-medium">{verificationResult.recipient.team}</p>
                            </div>
                          </div>
                        )}
                        
                        {verificationResult.recipient.position && (
                          <div className="flex items-center">
                            <Award className="h-4 w-4 text-gray-400 mr-3" />
                            <div>
                              <p className="text-sm text-gray-600">Position</p>
                              <p className="font-medium">{verificationResult.recipient.position}</p>
                            </div>
                          </div>
                        )}
                      </div>
                    </div>
                  </div>

                  {/* Tournament Details */}
                  <div className="mt-6 pt-6 border-t border-gray-200">
                    <h3 className="text-lg font-semibold text-gray-900 mb-4">Tournament Details</h3>
                    <div className="grid md:grid-cols-3 gap-4">
                      <div>
                        <p className="text-sm text-gray-600">Tournament</p>
                        <p className="font-medium">{verificationResult.tournament.name}</p>
                      </div>
                      <div className="flex items-center">
                        <Calendar className="h-4 w-4 text-gray-400 mr-2" />
                        <div>
                          <p className="text-sm text-gray-600">Duration</p>
                          <p className="font-medium">
                            {new Date(verificationResult.tournament.startDate).toLocaleDateString()} - 
                            {new Date(verificationResult.tournament.endDate).toLocaleDateString()}
                          </p>
                        </div>
                      </div>
                      <div className="flex items-center">
                        <MapPin className="h-4 w-4 text-gray-400 mr-2" />
                        <div>
                          <p className="text-sm text-gray-600">Location</p>
                          <p className="font-medium">{verificationResult.tournament.location}</p>
                        </div>
                      </div>
                    </div>
                  </div>

                  {/* Status Message */}
                  <div className={`mt-6 p-4 rounded-lg ${
                    verificationResult.status === 'valid' ? 'bg-green-50 border border-green-200' :
                    verificationResult.status === 'expired' ? 'bg-yellow-50 border border-yellow-200' :
                    'bg-red-50 border border-red-200'
                  }`}>
                    <div className="flex items-start">
                      {getStatusIcon(verificationResult.status)}
                      <div className="ml-3">
                        <p className={`font-medium ${
                          verificationResult.status === 'valid' ? 'text-green-800' :
                          verificationResult.status === 'expired' ? 'text-yellow-800' :
                          'text-red-800'
                        }`}>
                          {verificationResult.status === 'valid' ? 'Certificate is Valid' :
                           verificationResult.status === 'expired' ? 'Certificate has Expired' :
                           'Certificate is Invalid'}
                        </p>
                        <p className={`text-sm mt-1 ${
                          verificationResult.status === 'valid' ? 'text-green-700' :
                          verificationResult.status === 'expired' ? 'text-yellow-700' :
                          'text-red-700'
                        }`}>
                          {verificationResult.status === 'valid' ? 
                            'This certificate has been verified and is authentic. The recipient has successfully participated in the tournament.' :
                           verificationResult.status === 'expired' ? 
                            'This certificate was valid but has now expired. Please contact the tournament organizers for a new certificate.' :
                            'This certificate could not be verified. Please check the certificate ID or contact support if you believe this is an error.'}
                        </p>
                      </div>
                    </div>
                  </div>
                </CardContent>
              </Card>
            </div>
          )}

          {/* Information Section */}
          <div className="max-w-4xl mx-auto mt-12">
            <Card>
              <CardContent className="p-6">
                <div className="text-center">
                  <h3 className="text-lg font-semibold mb-4">About Certificate Verification</h3>
                  <div className="grid md:grid-cols-3 gap-6 text-sm text-gray-600">
                    <div>
                      <QrCode className="h-8 w-8 text-blue-500 mx-auto mb-2" />
                      <h4 className="font-medium text-gray-900 mb-2">QR Code Scanning</h4>
                      <p>Scan the QR code on any certificate to instantly verify its authenticity and view details.</p>
                    </div>
                    <div>
                      <Search className="h-8 w-8 text-blue-500 mx-auto mb-2" />
                      <h4 className="font-medium text-gray-900 mb-2">Manual Verification</h4>
                      <p>Enter the certificate ID manually to verify certificates when QR scanning is not available.</p>
                    </div>
                    <div>
                      <CheckCircle className="h-8 w-8 text-blue-500 mx-auto mb-2" />
                      <h4 className="font-medium text-gray-900 mb-2">Secure & Reliable</h4>
                      <p>All certificates are digitally signed and verified against our secure database.</p>
                    </div>
                  </div>
                </div>
              </CardContent>
            </Card>
          </div>
        </div>
      </div>
      <Footer />
    </>
  );
};

export default CertificateVerification;
