import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Alert, AlertDescription } from '@/components/ui/alert';
import { Badge } from '@/components/ui/badge';
import { Link } from '@inertiajs/react';
import { 
    ShieldX, 
    ArrowLeft, 
    Mail, 
    Home, 
    Info,
    AlertCircle,
    Users,
    Key
} from 'lucide-react';

interface PermissionDeniedProps {
    error: {
        message: string;
        required_permission: string;
        user_roles: string[];
        user_permissions: string[];
        suggestions: string[];
    };
}

export default function PermissionDenied({ error }: PermissionDeniedProps) {
    const {
        message,
        required_permission,
        user_roles,
        user_permissions,
        suggestions
    } = error;

    // Parse permission for better display
    const [resource, action] = required_permission.split('.');
    const resourceName = resource?.charAt(0).toUpperCase() + resource?.slice(1) || 'Unknown';
    const actionName = action?.charAt(0).toUpperCase() + action?.slice(1) || 'Unknown';

    return (
        <div className="min-h-screen flex items-center justify-center bg-gray-50 px-4">
            <div className="max-w-2xl w-full space-y-6">
                {/* Main Error Card */}
                <Card className="border-red-200">
                    <CardHeader className="text-center">
                        <div className="mx-auto w-16 h-16 bg-red-100 rounded-full flex items-center justify-center mb-4">
                            <ShieldX className="w-8 h-8 text-red-600" />
                        </div>
                        <CardTitle className="text-2xl text-red-800">Access Denied</CardTitle>
                        <CardDescription className="text-lg">
                            {message}
                        </CardDescription>
                    </CardHeader>
                    <CardContent className="space-y-6">
                        {/* Permission Details */}
                        <div className="bg-gray-50 rounded-lg p-4 space-y-3">
                            <div className="flex items-center gap-2 text-sm font-medium text-gray-700">
                                <Key className="w-4 h-4" />
                                Required Permission
                            </div>
                            <div className="ml-6">
                                <Badge variant="destructive" className="text-sm">
                                    {required_permission}
                                </Badge>
                                <p className="text-sm text-gray-600 mt-1">
                                    {actionName} access to {resourceName}
                                </p>
                            </div>
                        </div>

                        {/* User Information */}
                        <div className="bg-blue-50 rounded-lg p-4 space-y-3">
                            <div className="flex items-center gap-2 text-sm font-medium text-gray-700">
                                <Users className="w-4 h-4" />
                                Your Current Access
                            </div>
                            <div className="ml-6 space-y-2">
                                <div>
                                    <span className="text-sm text-gray-600">Roles:</span>
                                    <div className="flex flex-wrap gap-1 mt-1">
                                        {user_roles.length > 0 ? (
                                            user_roles.map((role) => (
                                                <Badge key={role} variant="secondary" className="text-xs">
                                                    {role}
                                                </Badge>
                                            ))
                                        ) : (
                                            <span className="text-xs text-gray-500">No roles assigned</span>
                                        )}
                                    </div>
                                </div>
                                <div>
                                    <span className="text-sm text-gray-600">Permissions ({user_permissions.length}):</span>
                                    {user_permissions.length > 0 ? (
                                        <details className="mt-1">
                                            <summary className="text-xs text-blue-600 cursor-pointer hover:text-blue-800">
                                                View all permissions
                                            </summary>
                                            <div className="flex flex-wrap gap-1 mt-2 max-h-32 overflow-y-auto">
                                                {user_permissions.map((permission) => (
                                                    <Badge key={permission} variant="outline" className="text-xs">
                                                        {permission}
                                                    </Badge>
                                                ))}
                                            </div>
                                        </details>
                                    ) : (
                                        <p className="text-xs text-gray-500 mt-1">No permissions assigned</p>
                                    )}
                                </div>
                            </div>
                        </div>

                        {/* Suggestions */}
                        {suggestions.length > 0 && (
                            <Alert>
                                <Info className="h-4 w-4" />
                                <AlertDescription>
                                    <div className="space-y-2">
                                        <p className="font-medium">What you can do:</p>
                                        <ul className="list-disc list-inside space-y-1 text-sm">
                                            {suggestions.map((suggestion, index) => (
                                                <li key={index}>{suggestion}</li>
                                            ))}
                                        </ul>
                                    </div>
                                </AlertDescription>
                            </Alert>
                        )}

                        {/* Action Buttons */}
                        <div className="flex flex-col sm:flex-row gap-3 pt-4">
                            <Button variant="outline" asChild className="flex-1">
                                <Link href="/admin/dashboard">
                                    <ArrowLeft className="w-4 h-4 mr-2" />
                                    Back to Dashboard
                                </Link>
                            </Button>
                            
                            <Button asChild className="flex-1">
                                <Link href="/">
                                    <Home className="w-4 h-4 mr-2" />
                                    Go Home
                                </Link>
                            </Button>
                        </div>
                    </CardContent>
                </Card>

                {/* Contact Support */}
                <Card>
                    <CardHeader>
                        <CardTitle className="text-lg flex items-center gap-2">
                            <AlertCircle className="w-5 h-5 text-blue-600" />
                            Need Help?
                        </CardTitle>
                        <CardDescription>
                            If you believe you should have access to this resource, please contact support.
                        </CardDescription>
                    </CardHeader>
                    <CardContent>
                        <div className="flex flex-col sm:flex-row gap-3">
                            <Button variant="outline" asChild>
                                <a href="mailto:admin@example.com">
                                    <Mail className="w-4 h-4 mr-2" />
                                    Contact Administrator
                                </a>
                            </Button>
                            
                            <Button variant="outline" asChild>
                                <Link href="/help">
                                    <Info className="w-4 h-4 mr-2" />
                                    View Help Documentation
                                </Link>
                            </Button>
                        </div>
                        
                        <div className="mt-4 p-3 bg-gray-50 rounded text-xs text-gray-600">
                            <p><strong>Error Details:</strong></p>
                            <p>Permission: {required_permission}</p>
                            <p>Timestamp: {new Date().toISOString()}</p>
                            <p>Include this information when contacting support.</p>
                        </div>
                    </CardContent>
                </Card>
            </div>
        </div>
    );
}