import React from 'react';
import { Head } from '@inertiajs/react';
import { router } from '@inertiajs/react';
import AppLayout from '@/layouts/app-layout';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Button } from '@/components/ui/button';
import { 
  Trophy, 
  Users, 
  Calendar, 
  Target, 
  TrendingUp, 
  Clock,
  Gamepad2,
  Medal,
  Play,
  CheckCircle,
  AlertCircle,
  User
} from 'lucide-react';
import { type BreadcrumbItem } from '@/types';

const breadcrumbs: BreadcrumbItem[] = [
  {
    title: 'Dashboard',
    href: '/dashboard',
  },
];

interface DashboardProps {
  stats: {
    totalTournaments: number;
    activeTournaments: number;
    totalPlayers: number;
    totalTeams: number;
    totalMatches: number;
    activeMatches: number;
    totalUsers: number;
  };
  recentTournaments: Array<{
    id: number;
    name: string;
    status: string;
    start_date: string;
    end_date: string;
    matches_count: number;
    game_type: string;
  }>;
  upcomingMatches: Array<{
    id: number;
    name: string;
    tournament: string;
    scheduled_time: string;
    match_type: string;
    participant1: string;
    participant2: string;
    status: string;
  }>;
  recentMatches: Array<{
    id: number;
    name: string;
    tournament: string;
    completed_at: string;
    match_type: string;
    participant1: string;
    participant2: string;
    score: string;
  }>;
  tournamentStatusStats: Record<string, number>;
  matchStatusStats: Record<string, number>;
  monthlyTournamentStats: Array<{
    month: string;
    count: number;
  }>;
}

const StatCard: React.FC<{
  title: string;
  value: number;
  icon: React.ReactNode;
  color: string;
  href?: string;
}> = ({ title, value, icon, color, href }) => {
  const handleClick = () => {
    if (href) {
      router.get(href);
    }
  };

  return (
    <Card className={`cursor-pointer hover:shadow-md transition-shadow ${href ? 'hover:bg-gray-50' : ''}`} onClick={handleClick}>
      <CardContent className="p-6">
        <div className="flex items-center justify-between">
          <div>
            <p className="text-sm font-medium text-muted-foreground">{title}</p>
            <p className="text-3xl font-bold">{value.toLocaleString()}</p>
          </div>
          <div className={`p-3 rounded-full ${color}`}>
            {icon}
          </div>
        </div>
      </CardContent>
    </Card>
  );
};

const StatusBadge: React.FC<{ status: string }> = ({ status }) => {
  const getStatusColor = (status: string) => {
    switch (status.toLowerCase()) {
      case 'active':
      case 'in_progress':
      case 'scheduled':
        return 'bg-green-100 text-green-800';
      case 'completed':
        return 'bg-blue-100 text-blue-800';
      case 'cancelled':
        return 'bg-red-100 text-red-800';
      case 'draft':
        return 'bg-gray-100 text-gray-800';
      default:
        return 'bg-gray-100 text-gray-800';
    }
  };

  return (
    <Badge className={getStatusColor(status)}>
      {status.replace('_', ' ').toUpperCase()}
    </Badge>
  );
};

export default function Dashboard({ 
  stats, 
  recentTournaments, 
  upcomingMatches, 
  recentMatches,
  tournamentStatusStats,
  matchStatusStats,
  monthlyTournamentStats 
}: DashboardProps) {
  return (
    <AppLayout breadcrumbs={breadcrumbs}>
      <Head title="Dashboard" />
      
      <div className="space-y-6 p-6">
        {/* Welcome Section */}
        <div className="mb-8">
          <h1 className="text-3xl font-bold text-gray-900">Tournament Management Dashboard</h1>
          <p className="text-gray-600 mt-2">Overview of your tournament system performance and activities</p>
        </div>

        {/* Stats Cards */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
          <StatCard
            title="Total Tournaments"
            value={stats.totalTournaments}
            icon={<Trophy className="w-6 h-6 text-white" />}
            color="bg-blue-500"
            href="/admin/tournaments"
          />
          <StatCard
            title="Active Tournaments"
            value={stats.activeTournaments}
            icon={<Play className="w-6 h-6 text-white" />}
            color="bg-green-500"
            href="/admin/tournaments"
          />
          <StatCard
            title="Total Players"
            value={stats.totalPlayers}
            icon={<Users className="w-6 h-6 text-white" />}
            color="bg-purple-500"
            href="/admin/players"
          />
          <StatCard
            title="Total Teams"
            value={stats.totalTeams}
            icon={<Target className="w-6 h-6 text-white" />}
            color="bg-orange-500"
            href="/admin/teams"
          />
        </div>

        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
          <StatCard
            title="Total Matches"
            value={stats.totalMatches}
            icon={<Gamepad2 className="w-6 h-6 text-white" />}
            color="bg-indigo-500"
            href="/admin/matches"
          />
          <StatCard
            title="Active Matches"
            value={stats.activeMatches}
            icon={<Clock className="w-6 h-6 text-white" />}
            color="bg-red-500"
            href="/admin/matches"
          />
          <StatCard
            title="System Users"
            value={stats.totalUsers}
            icon={<User className="w-6 h-6 text-white" />}
            color="bg-teal-500"
            href="/admin/users"
          />
        </div>

        {/* Main Content Grid */}
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
          {/* Recent Tournaments */}
          <Card>
            <CardHeader className="flex flex-row items-center justify-between">
              <CardTitle className="flex items-center gap-2">
                <Trophy className="w-5 h-5" />
                Recent Tournaments
              </CardTitle>
              <Button 
                variant="outline" 
                size="sm"
                onClick={() => router.get('/admin/tournaments')}
              >
                View All
              </Button>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                {recentTournaments.length > 0 ? (
                  recentTournaments.map((tournament) => (
                    <div key={tournament.id} className="flex items-center justify-between p-3 bg-gray-50 rounded-lg">
                      <div className="flex-1">
                        <h4 className="font-medium">{tournament.name}</h4>
                        <p className="text-sm text-gray-600">
                          {tournament.game_type} • {tournament.matches_count} matches
                        </p>
                        <p className="text-xs text-gray-500">
                          {new Date(tournament.start_date).toLocaleDateString()} - {new Date(tournament.end_date).toLocaleDateString()}
                        </p>
                      </div>
                      <StatusBadge status={tournament.status} />
                    </div>
                  ))
                ) : (
                  <p className="text-gray-500 text-center py-4">No tournaments found</p>
                )}
              </div>
            </CardContent>
          </Card>

          {/* Upcoming Matches */}
          <Card>
            <CardHeader className="flex flex-row items-center justify-between">
              <CardTitle className="flex items-center gap-2">
                <Calendar className="w-5 h-5" />
                Upcoming Matches
              </CardTitle>
              <Button 
                variant="outline" 
                size="sm"
                onClick={() => router.get('/admin/matches')}
              >
                View All
              </Button>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                {upcomingMatches.length > 0 ? (
                  upcomingMatches.map((match) => (
                    <div key={match.id} className="flex items-center justify-between p-3 bg-gray-50 rounded-lg">
                      <div className="flex-1">
                        <h4 className="font-medium">{match.name}</h4>
                        <p className="text-sm text-gray-600">
                          {match.participant1} vs {match.participant2}
                        </p>
                        <p className="text-xs text-gray-500">
                          {match.tournament} • {new Date(match.scheduled_time).toLocaleString()}
                        </p>
                      </div>
                      <StatusBadge status={match.status} />
                    </div>
                  ))
                ) : (
                  <p className="text-gray-500 text-center py-4">No upcoming matches</p>
                )}
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Recent Matches */}
        <Card>
          <CardHeader className="flex flex-row items-center justify-between">
            <CardTitle className="flex items-center gap-2">
              <CheckCircle className="w-5 h-5" />
              Recent Completed Matches
            </CardTitle>
            <Button 
              variant="outline" 
              size="sm"
              onClick={() => router.get('/admin/matches')}
            >
              View All
            </Button>
          </CardHeader>
          <CardContent>
            <div className="space-y-4">
              {recentMatches.length > 0 ? (
                recentMatches.map((match) => (
                  <div key={match.id} className="flex items-center justify-between p-3 bg-gray-50 rounded-lg">
                    <div className="flex-1">
                      <h4 className="font-medium">{match.name}</h4>
                      <p className="text-sm text-gray-600">
                        {match.participant1} vs {match.participant2}
                      </p>
                      <p className="text-xs text-gray-500">
                        {match.tournament} • {new Date(match.completed_at).toLocaleString()}
                      </p>
                    </div>
                    <div className="text-right">
                      <p className="font-bold text-lg">{match.score}</p>
                      <StatusBadge status="completed" />
                    </div>
                  </div>
                ))
              ) : (
                <p className="text-gray-500 text-center py-4">No completed matches</p>
              )}
            </div>
          </CardContent>
        </Card>

        {/* Statistics Overview */}
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
          {/* Tournament Status Distribution */}
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <TrendingUp className="w-5 h-5" />
                Tournament Status Distribution
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-3">
                {Object.entries(tournamentStatusStats).map(([status, count]) => (
                  <div key={status} className="flex items-center justify-between">
                    <div className="flex items-center gap-2">
                      <StatusBadge status={status} />
                    </div>
                    <span className="font-semibold">{count}</span>
                  </div>
                ))}
              </div>
            </CardContent>
          </Card>

          {/* Match Status Distribution */}
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Medal className="w-5 h-5" />
                Match Status Distribution
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-3">
                {Object.entries(matchStatusStats).map(([status, count]) => (
                  <div key={status} className="flex items-center justify-between">
                    <div className="flex items-center gap-2">
                      <StatusBadge status={status} />
                    </div>
                    <span className="font-semibold">{count}</span>
                  </div>
                ))}
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Monthly Tournament Creation Trend */}
        {monthlyTournamentStats.length > 0 && (
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <TrendingUp className="w-5 h-5" />
                Tournament Creation Trend (Last 6 Months)
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="flex items-end space-x-2 h-40">
                {monthlyTournamentStats.map((stat, index) => (
                  <div key={index} className="flex flex-col items-center flex-1">
                    <div 
                      className="bg-blue-500 w-full rounded-t"
                      style={{ 
                        height: `${Math.max((stat.count / Math.max(...monthlyTournamentStats.map(s => s.count))) * 120, 10)}px` 
                      }}
                    />
                    <p className="text-xs mt-2 text-center">{stat.month}</p>
                    <p className="text-xs font-semibold">{stat.count}</p>
                  </div>
                ))}
              </div>
            </CardContent>
          </Card>
        )}

        {/* Quick Actions */}
        <Card>
          <CardHeader>
            <CardTitle>Quick Actions</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
              <Button 
                onClick={() => router.get('/admin/tournaments/create')}
                className="flex flex-col items-center gap-2 h-auto py-4"
              >
                <Trophy className="w-6 h-6" />
                Create Tournament
              </Button>
              <Button 
                onClick={() => router.get('/admin/matches/create')}
                variant="outline"
                className="flex flex-col items-center gap-2 h-auto py-4"
              >
                <Gamepad2 className="w-6 h-6" />
                Create Match
              </Button>
              <Button 
                onClick={() => router.get('/admin/players/create')}
                variant="outline"
                className="flex flex-col items-center gap-2 h-auto py-4"
              >
                <Users className="w-6 h-6" />
                Add Player
              </Button>
              <Button 
                onClick={() => router.get('/admin/teams/create')}
                variant="outline"
                className="flex flex-col items-center gap-2 h-auto py-4"
              >
                <Target className="w-6 h-6" />
                Create Team
              </Button>
            </div>
          </CardContent>
        </Card>
      </div>
    </AppLayout>
  );
}
