import React from 'react';
import { Head } from '@inertiajs/react';
import { router } from '@inertiajs/react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { 
  ArrowLeft, 
  Edit, 
  Shield, 
  Mail, 
  Calendar, 
  CheckCircle, 
  XCircle,
  User,
  UserCheck,
  Trophy,
  Users,
  Activity
} from 'lucide-react';
import AppLayout from '@/layouts/app-layout';

interface Props {
  user: {
    id: number;
    name: string;
    email: string;
    email_verified_at: string | null;
    created_at: string;
    updated_at: string;
    phone?: string;
    id_number?: string;
    passport_number?: string;
    date_of_birth?: string;
    gender?: string;
    nationality?: string;
    county?: string;
    sub_county?: string;
    ward?: string;
    address?: string;
    postal_code?: string;
    occupation?: string;
    employer?: string;
    emergency_contact_name?: string;
    emergency_contact_phone?: string;
    emergency_contact_relationship?: string;
    bio?: string;
    kyc_status?: string;
    kyc_verified_at?: string;
    kyc_notes?: string;
    roles: Array<{
      id: number;
      name: string;
      display_name: string;
      description: string;
      color: string;
    }>;
    players?: Array<{
      id: number;
      display_name: string;
      game_tag: string;
    }>;
    teams?: Array<{
      id: number;
      name: string;
      tag: string;
    }>;
  };
}

const ShowUser: React.FC<Props> = ({ user }) => {
  const getRoleColor = (role: any) => {
    const colors: Record<string, string> = {
      'super_admin': 'bg-red-100 text-red-800',
      'admin': 'bg-purple-100 text-purple-800',
      'organizer': 'bg-blue-100 text-blue-800',
      'referee': 'bg-green-100 text-green-800',
      'player': 'bg-yellow-100 text-yellow-800',
      'spectator': 'bg-gray-100 text-gray-800',
    };
    return colors[role.name] || 'bg-gray-100 text-gray-800';
  };

  const handleImpersonate = () => {
    if (confirm(`Are you sure you want to impersonate ${user.name}?`)) {
      router.post(route('admin.users.impersonate', user.id));
    }
  };

  const handleDelete = () => {
    if (confirm(`Are you sure you want to delete ${user.name}? This action cannot be undone.`)) {
      router.delete(route('admin.users.destroy', user.id));
    }
  };

  return (
    <AppLayout breadcrumbs={[{ title: 'Users', href: '/admin/users' }]}> 
      <Head title={`User - ${user.name}`} />
      
      <div className="space-y-6 p-4 flex flex-col gap-4">
        {/* Header */}
        <div className="flex items-center justify-between">
          <div className="flex items-center gap-4">
            <Button
              variant="outline"
              size="sm"
              onClick={() => router.get(route('admin.users.index'))}
            >
              <ArrowLeft className="w-4 h-4 mr-2" />
              Back to Users
            </Button>
            
          </div>
          
          <div className="flex items-center gap-2">
            <Button
              variant="outline"
              onClick={handleImpersonate}
            >
              <Shield className="w-4 h-4 mr-2" />
              Login as User
            </Button>
            <Button
              onClick={() => router.get(route('admin.users.edit', user.id))}
            >
              <Edit className="w-4 h-4 mr-2" />
              Edit User
            </Button>
          </div>
        </div>

        <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
          {/* User Information */}
          <div className="lg:col-span-2 space-y-6">
            {/* Basic Information */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <User className="w-5 h-5" />
                  Basic Information
                </CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <div>
                    <label className="text-sm font-medium text-muted-foreground">Full Name</label>
                    <p className="text-lg font-medium">{user.name}</p>
                  </div>
                  <div>
                    <label className="text-sm font-medium text-muted-foreground">Email Address</label>
                    <div className="flex items-center gap-2">
                      <p className="text-lg">{user.email}</p>
                      {user.email_verified_at ? (
                        <CheckCircle className="w-5 h-5 text-green-600" />
                      ) : (
                        <XCircle className="w-5 h-5 text-red-600" />
                      )}
                    </div>
                  </div>
                  <div>
                    <label className="text-sm font-medium text-muted-foreground">User ID</label>
                    <p className="text-lg font-mono">{user.id}</p>
                  </div>
                  <div>
                    <label className="text-sm font-medium text-muted-foreground">Email Status</label>
                    <Badge 
                      className={user.email_verified_at 
                        ? 'bg-green-100 text-green-800' 
                        : 'bg-red-100 text-red-800'
                      }
                    >
                      {user.email_verified_at ? 'Verified' : 'Unverified'}
                    </Badge>
                  </div>
                </div>

                <div className="grid grid-cols-1 md:grid-cols-2 gap-4 pt-4 border-t">
                  <div>
                    <label className="text-sm font-medium text-muted-foreground">Member Since</label>
                    <div className="flex items-center gap-2">
                      <Calendar className="w-4 h-4" />
                      <p>{new Date(user.created_at).toLocaleDateString('en-US', {
                        year: 'numeric',
                        month: 'long',
                        day: 'numeric'
                      })}</p>
                    </div>
                  </div>
                  <div>
                    <label className="text-sm font-medium text-muted-foreground">Last Updated</label>
                    <div className="flex items-center gap-2">
                      <Activity className="w-4 h-4" />
                      <p>{new Date(user.updated_at).toLocaleDateString('en-US', {
                        year: 'numeric',
                        month: 'long',
                        day: 'numeric'
                      })}</p>
                    </div>
                  </div>
                </div>
              </CardContent>
            </Card>

            {/* KYC Information */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <UserCheck className="w-5 h-5" />
                  KYC Information
                </CardTitle>
              </CardHeader>
              <CardContent className="space-y-6">
                {/* KYC Status */}
                <div className="flex items-center justify-between p-4 bg-gray-50 rounded-lg">
                  <div>
                    <h4 className="font-medium">KYC Status</h4>
                    <p className="text-sm text-muted-foreground">
                      {user.kyc_status === 'verified' ? 'Verified' : 
                       user.kyc_status === 'pending' ? 'Pending Review' :
                       user.kyc_status === 'rejected' ? 'Rejected' : 'Not Submitted'}
                    </p>
                  </div>
                  <Badge 
                    variant={user.kyc_status === 'verified' ? 'default' : 
                            user.kyc_status === 'pending' ? 'secondary' :
                            user.kyc_status === 'rejected' ? 'destructive' : 'outline'}
                  >
                    {user.kyc_status === 'verified' ? 'Verified' : 
                     user.kyc_status === 'pending' ? 'Pending' :
                     user.kyc_status === 'rejected' ? 'Rejected' : 'Not Submitted'}
                  </Badge>
                </div>

                {/* Personal Details */}
                {(user.phone || user.id_number || user.passport_number || user.date_of_birth || user.gender || user.nationality) && (
                  <div className="space-y-4">
                    <h4 className="font-medium text-sm text-gray-700 border-b pb-2">Personal Details</h4>
                    <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                      {user.phone && (
                        <div>
                          <label className="text-sm font-medium text-muted-foreground">Phone Number</label>
                          <p className="text-sm">{user.phone}</p>
                        </div>
                      )}
                      {user.id_number && (
                        <div>
                          <label className="text-sm font-medium text-muted-foreground">National ID</label>
                          <p className="text-sm">{user.id_number}</p>
                        </div>
                      )}
                      {user.passport_number && (
                        <div>
                          <label className="text-sm font-medium text-muted-foreground">Passport Number</label>
                          <p className="text-sm">{user.passport_number}</p>
                        </div>
                      )}
                      {user.date_of_birth && (
                        <div>
                          <label className="text-sm font-medium text-muted-foreground">Date of Birth</label>
                          <p className="text-sm">{new Date(user.date_of_birth).toLocaleDateString()}</p>
                        </div>
                      )}
                      {user.gender && (
                        <div>
                          <label className="text-sm font-medium text-muted-foreground">Gender</label>
                          <p className="text-sm capitalize">{user.gender}</p>
                        </div>
                      )}
                      {user.nationality && (
                        <div>
                          <label className="text-sm font-medium text-muted-foreground">Nationality</label>
                          <p className="text-sm">{user.nationality}</p>
                        </div>
                      )}
                    </div>
                  </div>
                )}

                {/* Location Details */}
                {(user.county || user.sub_county || user.ward || user.address || user.postal_code) && (
                  <div className="space-y-4">
                    <h4 className="font-medium text-sm text-gray-700 border-b pb-2">Location Details</h4>
                    <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                      {user.county && (
                        <div>
                          <label className="text-sm font-medium text-muted-foreground">County</label>
                          <p className="text-sm">{user.county}</p>
                        </div>
                      )}
                      {user.sub_county && (
                        <div>
                          <label className="text-sm font-medium text-muted-foreground">Sub County</label>
                          <p className="text-sm">{user.sub_county}</p>
                        </div>
                      )}
                      {user.ward && (
                        <div>
                          <label className="text-sm font-medium text-muted-foreground">Ward</label>
                          <p className="text-sm">{user.ward}</p>
                        </div>
                      )}
                      {user.address && (
                        <div className="md:col-span-2">
                          <label className="text-sm font-medium text-muted-foreground">Address</label>
                          <p className="text-sm">{user.address}</p>
                        </div>
                      )}
                      {user.postal_code && (
                        <div>
                          <label className="text-sm font-medium text-muted-foreground">Postal Code</label>
                          <p className="text-sm">{user.postal_code}</p>
                        </div>
                      )}
                    </div>
                  </div>
                )}

                {/* Employment Details */}
                {(user.occupation || user.employer) && (
                  <div className="space-y-4">
                    <h4 className="font-medium text-sm text-gray-700 border-b pb-2">Employment Details</h4>
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                      {user.occupation && (
                        <div>
                          <label className="text-sm font-medium text-muted-foreground">Occupation</label>
                          <p className="text-sm">{user.occupation}</p>
                        </div>
                      )}
                      {user.employer && (
                        <div>
                          <label className="text-sm font-medium text-muted-foreground">Employer</label>
                          <p className="text-sm">{user.employer}</p>
                        </div>
                      )}
                    </div>
                  </div>
                )}

                {/* Emergency Contact */}
                {(user.emergency_contact_name || user.emergency_contact_phone || user.emergency_contact_relationship) && (
                  <div className="space-y-4">
                    <h4 className="font-medium text-sm text-gray-700 border-b pb-2">Emergency Contact</h4>
                    <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                      {user.emergency_contact_name && (
                        <div>
                          <label className="text-sm font-medium text-muted-foreground">Contact Name</label>
                          <p className="text-sm">{user.emergency_contact_name}</p>
                        </div>
                      )}
                      {user.emergency_contact_phone && (
                        <div>
                          <label className="text-sm font-medium text-muted-foreground">Contact Phone</label>
                          <p className="text-sm">{user.emergency_contact_phone}</p>
                        </div>
                      )}
                      {user.emergency_contact_relationship && (
                        <div>
                          <label className="text-sm font-medium text-muted-foreground">Relationship</label>
                          <p className="text-sm">{user.emergency_contact_relationship}</p>
                        </div>
                      )}
                    </div>
                  </div>
                )}

                {/* Additional Information */}
                {(user.bio || user.kyc_notes) && (
                  <div className="space-y-4">
                    <h4 className="font-medium text-sm text-gray-700 border-b pb-2">Additional Information</h4>
                    {user.bio && (
                      <div>
                        <label className="text-sm font-medium text-muted-foreground">Biography</label>
                        <p className="text-sm mt-1">{user.bio}</p>
                      </div>
                    )}
                    {user.kyc_notes && (
                      <div>
                        <label className="text-sm font-medium text-muted-foreground">KYC Notes</label>
                        <p className="text-sm mt-1">{user.kyc_notes}</p>
                      </div>
                    )}
                    {user.kyc_verified_at && (
                      <div>
                        <label className="text-sm font-medium text-muted-foreground">KYC Verified At</label>
                        <p className="text-sm mt-1">{new Date(user.kyc_verified_at).toLocaleDateString()}</p>
                      </div>
                    )}
                  </div>
                )}

                {/* Empty State */}
                {!user.phone && !user.id_number && !user.passport_number && !user.date_of_birth && 
                 !user.gender && !user.nationality && !user.county && !user.sub_county && 
                 !user.ward && !user.address && !user.postal_code && !user.occupation && 
                 !user.employer && !user.emergency_contact_name && !user.emergency_contact_phone && 
                 !user.emergency_contact_relationship && !user.bio && !user.kyc_notes && (
                  <div className="text-center py-8 text-muted-foreground">
                    <UserCheck className="w-12 h-12 mx-auto mb-2 opacity-50" />
                    <p>No KYC information provided yet.</p>
                  </div>
                )}
              </CardContent>
            </Card>

            {/* Players & Teams */}
            {(user.players && user.players.length > 0) || (user.teams && user.teams.length > 0) ? (
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <Trophy className="w-5 h-5" />
                    Gaming Profiles
                  </CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                  {user.players && user.players.length > 0 && (
                    <div>
                      <label className="text-sm font-medium text-muted-foreground">Player Profiles</label>
                      <div className="flex flex-wrap gap-2 mt-2">
                        {user.players.map((player) => (
                          <Badge key={player.id} variant="outline" className="text-sm">
                            <UserCheck className="w-3 h-3 mr-1" />
                            {player.display_name} ({player.game_tag})
                          </Badge>
                        ))}
                      </div>
                    </div>
                  )}

                  {user.teams && user.teams.length > 0 && (
                    <div>
                      <label className="text-sm font-medium text-muted-foreground">Team Memberships</label>
                      <div className="flex flex-wrap gap-2 mt-2">
                        {user.teams.map((team) => (
                          <Badge key={team.id} variant="outline" className="text-sm">
                            <Users className="w-3 h-3 mr-1" />
                            {team.name} ({team.tag})
                          </Badge>
                        ))}
                      </div>
                    </div>
                  )}
                </CardContent>
              </Card>
            ) : null}
          </div>

          {/* Sidebar */}
          <div className="space-y-6">
            {/* Roles & Permissions */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Shield className="w-5 h-5" />
                  Roles & Permissions
                </CardTitle>
              </CardHeader>
              <CardContent className="space-y-3">
                {user.roles.length > 0 ? (
                  user.roles.map((role) => (
                    <div key={role.id} className="space-y-2">
                      <Badge className={getRoleColor(role)}>
                        {role.display_name}
                      </Badge>
                      <p className="text-sm text-muted-foreground">
                        {role.description}
                      </p>
                    </div>
                  ))
                ) : (
                  <div className="text-center py-6 text-muted-foreground">
                    <Shield className="w-8 h-8 mx-auto mb-2 opacity-50" />
                    <p>No roles assigned</p>
                    <p className="text-sm">User has minimal permissions</p>
                  </div>
                )}
              </CardContent>
            </Card>

            {/* Quick Actions */}
            <Card>
              <CardHeader>
                <CardTitle>Quick Actions</CardTitle>
              </CardHeader>
              <CardContent className="space-y-2">
                <Button
                  variant="outline"
                  className="w-full justify-start"
                  onClick={() => router.get(route('admin.users.edit', user.id))}
                >
                  <Edit className="w-4 h-4 mr-2" />
                  Edit User
                </Button>
                <Button
                  variant="outline"
                  className="w-full justify-start"
                  onClick={handleImpersonate}
                >
                  <Shield className="w-4 h-4 mr-2" />
                  Login as User
                </Button>
                <Button
                  variant="outline"
                  className="w-full justify-start text-red-600 hover:text-red-700"
                  onClick={handleDelete}
                >
                  <XCircle className="w-4 h-4 mr-2" />
                  Delete User
                </Button>
              </CardContent>
            </Card>
          </div>
        </div>
      </div>
    </AppLayout>
  );
};

export default ShowUser; 