import React, { useState } from 'react';
import { Head } from '@inertiajs/react';
import { router } from '@inertiajs/react';
import { Button } from '@/components/ui/button';
import { ArrowLeft, Save, UserCog } from 'lucide-react';
import AppLayout from '@/layouts/app-layout';
import UserForm from '@/components/users/user-form';

interface Props {
  user: {
    id: number;
    name: string;
    email: string;
    email_verified_at: string | null;
    phone?: string;
    id_number?: string;
    passport_number?: string;
    date_of_birth?: string;
    gender?: string;
    nationality?: string;
    county?: string;
    sub_county?: string;
    ward?: string;
    address?: string;
    postal_code?: string;
    occupation?: string;
    employer?: string;
    emergency_contact_name?: string;
    emergency_contact_phone?: string;
    emergency_contact_relationship?: string;
    bio?: string;
    kyc_status?: string;
    kyc_notes?: string;
    roles: Array<{
      id: number;
      name: string;
      display_name: string;
    }>;
  };
  roles: any[];
}

const EditUser: React.FC<Props> = ({ user, roles }) => {
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [errors, setErrors] = useState<any>({});
  const [form, setForm] = useState({
    name: user.name,
    email: user.email,
    password: '',
    password_confirmation: '',
    roles: user.roles.map(role => role.id),
    email_verified_at: !!user.email_verified_at,
    send_welcome_email: false,
    // KYC fields
    phone: user.phone || '',
    id_number: user.id_number || '',
    passport_number: user.passport_number || '',
    date_of_birth: user.date_of_birth || '',
    gender: user.gender || '',
    nationality: user.nationality || '',
    county: user.county || '',
    sub_county: user.sub_county || '',
    ward: user.ward || '',
    address: user.address || '',
    postal_code: user.postal_code || '',
    occupation: user.occupation || '',
    employer: user.employer || '',
    emergency_contact_name: user.emergency_contact_name || '',
    emergency_contact_phone: user.emergency_contact_phone || '',
    emergency_contact_relationship: user.emergency_contact_relationship || '',
    bio: user.bio || '',
    kyc_status: user.kyc_status || 'not_submitted',
    kyc_notes: user.kyc_notes || '',
  });

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    setIsSubmitting(true);
    setErrors({});

    router.put(route('admin.users.update', user.id), form, {
      onSuccess: () => {
        setIsSubmitting(false);
      },
      onError: (errors) => {
        setErrors(errors);
        setIsSubmitting(false);
      },
    });
  };

  return (
    <AppLayout breadcrumbs={[{ title: 'Users', href: '/admin/users' }]}> 
      <Head title={`Edit User - ${user.name}`} />
      
      <div className="space-y-6 p-4 flex flex-col gap-4">
        {/* Header */}
        <div className="flex items-center gap-4">
          <Button
            variant="outline"
            size="sm"
            onClick={() => router.get(route('admin.users.index'))}
          >
            <ArrowLeft className="w-4 h-4 mr-2" />
            Back to Users
          </Button>
         
        </div>

        {/* Form */}
        <form onSubmit={handleSubmit} className="space-y-6">
          <UserForm
            form={form}
            setForm={setForm}
            errors={errors}
            roles={roles}
            isEditing={true}
          />

          {/* Submit Actions */}
          <div className="flex items-center justify-end gap-4 p-6 bg-muted/50 rounded-lg">
            <Button
              type="button"
              variant="outline"
              onClick={() => router.get(route('admin.users.index'))}
              disabled={isSubmitting}
            >
              Cancel
            </Button>
            <Button 
              type="submit" 
              disabled={isSubmitting}
              className="bg-green-600 hover:bg-green-700"
            >
              <Save className="w-4 h-4 mr-2" />
              {isSubmitting ? 'Updating...' : 'Update User'}
            </Button>
          </div>
        </form>
      </div>
    </AppLayout>
  );
};

export default EditUser; 