import React, { useState } from 'react';
import { Head, router } from '@inertiajs/react';
import AppLayout from '@/layouts/app-layout';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { 
  Trophy, 
  ArrowLeft,
  TrendingUp,
  TrendingDown,
  Users,
  Clock,
  Calendar,
  Award,
  Target,
  BarChart3,
  PieChart,
  Activity,
  Download,
  RefreshCw,
  Info,
  Star,
  Medal,
  Timer,
  Zap
} from 'lucide-react';
import { Alert, AlertDescription } from '@/components/ui/alert';
import { Progress } from '@/components/ui/progress';
import { Label } from '@/components/ui/label';

interface Tournament {
  id: number;
  name: string;
  description: string;
  status: string;
  type: string;
  start_date: string;
  end_date: string;
}

interface Stats {
  overview: {
    total_participants: number;
    total_matches: number;
    completed_matches: number;
    average_match_duration: number;
    total_prize_pool: number;
    registration_rate: number;
  };
  participants: {
    by_status: Record<string, number>;
    by_team_size: Record<string, number>;
    registration_timeline: Array<{ date: string; count: number }>;
  };
  matches: {
    by_status: Record<string, number>;
    by_type: Record<string, number>;
    average_scores: Record<string, number>;
    duration_distribution: Record<string, number>;
  };
  performance: {
    top_teams: Array<{ team: string; wins: number; losses: number; win_rate: number }>;
    most_active_teams: Array<{ team: string; matches_played: number }>;
    longest_matches: Array<{ match: string; duration: number; teams: string }>;
  };
  trends: {
    daily_matches: Array<{ date: string; count: number }>;
    weekly_participation: Array<{ week: string; participants: number }>;
    monthly_growth: Array<{ month: string; growth: number }>;
  };
}

interface Props {
  tournament: Tournament;
  stats: Stats;
  filters: {
    period: string;
    view: string;
  };
}

const TournamentStatistics: React.FC<Props> = ({ 
  tournament, 
  stats, 
  filters 
}) => {
  const [localFilters, setLocalFilters] = useState({
    period: filters?.period || 'all',
    view: filters?.view || 'overview',
  });
  const [activeTab, setActiveTab] = useState('overview');

  const handleFilterChange = (key: string, value: string) => {
    const newFilters = { ...localFilters, [key]: value };
    setLocalFilters(newFilters);
    router.get(route('admin.tournaments.statistics', tournament.id), newFilters, {
      preserveState: true,
      preserveScroll: true,
    });
  };

  const getStatusColor = (status: string) => {
    const colors = {
      'draft': 'bg-gray-100 text-gray-800',
      'active': 'bg-green-100 text-green-800',
      'completed': 'bg-blue-100 text-blue-800',
      'cancelled': 'bg-red-100 text-red-800',
    };
    return colors[status as keyof typeof colors] || 'bg-gray-100 text-gray-800';
  };

  const formatDuration = (minutes: number) => {
    if (!minutes) return 'N/A';
    const hours = Math.floor(minutes / 60);
    const mins = minutes % 60;
    return hours > 0 ? `${hours}h ${mins}m` : `${mins}m`;
  };

  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat('en-KE', {
      style: 'currency',
      currency: 'KES'
    }).format(amount);
  };

  const progressPercentage = stats.overview.total_matches > 0 
    ? (stats.overview.completed_matches / stats.overview.total_matches) * 100 
    : 0;

  return (
    <AppLayout>
      <Head title={`Statistics - ${tournament.name}`} />
      
      <div className="p-4">
        {/* Header */}
        <div className="flex items-center justify-between mb-6">
          <div className="flex items-center space-x-4">
            <Button variant="outline" size="sm" asChild>
              <a href={route('admin.tournaments.show', tournament.id)}>
                <ArrowLeft className="h-4 w-4 mr-2" />
                Back to Tournament
              </a>
            </Button>
            
          </div>
          <div className="flex space-x-2">
            <Button variant="outline" size="sm">
              <Download className="h-4 w-4 mr-2" />
              Export Report
            </Button>
            <Button variant="outline" size="sm">
              <RefreshCw className="h-4 w-4 mr-2" />
              Refresh
            </Button>
          </div>
        </div>

        {/* Filters */}
        <Card className="mb-6">
          <CardContent className="p-4">
            <div className="flex flex-col sm:flex-row gap-4">
              <div>
                <Label className="text-sm font-medium text-gray-500 mb-2 block">Time Period</Label>
                <Select value={localFilters.period} onValueChange={(value) => handleFilterChange('period', value)}>
                  <SelectTrigger className="w-48">
                    <SelectValue placeholder="Select period" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">All Time</SelectItem>
                    <SelectItem value="week">Last Week</SelectItem>
                    <SelectItem value="month">Last Month</SelectItem>
                    <SelectItem value="quarter">Last Quarter</SelectItem>
                  </SelectContent>
                </Select>
              </div>
              <div>
                <Label className="text-sm font-medium text-gray-500 mb-2 block">View</Label>
                <Select value={localFilters.view} onValueChange={(value) => handleFilterChange('view', value)}>
                  <SelectTrigger className="w-48">
                    <SelectValue placeholder="Select view" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="overview">Overview</SelectItem>
                    <SelectItem value="detailed">Detailed</SelectItem>
                    <SelectItem value="comparative">Comparative</SelectItem>
                  </SelectContent>
                </Select>
              </div>
            </div>
          </CardContent>
        </Card>

        {/* Overview Stats */}
        <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-4 mb-6">
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center">
                <div className="p-2 bg-blue-100 rounded-lg">
                  <Users className="w-5 h-5 text-blue-600" />
                </div>
                <div className="ml-3">
                  <p className="text-sm font-medium text-gray-500">Participants</p>
                  <p className="text-xl font-bold">{stats.overview.total_participants}</p>
                  <p className="text-xs text-green-600 flex items-center">
                    <TrendingUp className="w-3 h-3 mr-1" />
                    +{stats.overview.registration_rate}% this week
                  </p>
                </div>
              </div>
            </CardContent>
          </Card>
          
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center">
                <div className="p-2 bg-green-100 rounded-lg">
                  <Trophy className="w-5 h-5 text-green-600" />
                </div>
                <div className="ml-3">
                  <p className="text-sm font-medium text-gray-500">Matches</p>
                  <p className="text-xl font-bold">{stats.overview.completed_matches}/{stats.overview.total_matches}</p>
                  <p className="text-xs text-blue-600">
                    {Math.round(progressPercentage)}% completed
                  </p>
                </div>
              </div>
            </CardContent>
          </Card>
          
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center">
                <div className="p-2 bg-yellow-100 rounded-lg">
                  <Timer className="w-5 h-5 text-yellow-600" />
                </div>
                <div className="ml-3">
                  <p className="text-sm font-medium text-gray-500">Avg Duration</p>
                  <p className="text-xl font-bold">{formatDuration(stats.overview.average_match_duration)}</p>
                  <p className="text-xs text-gray-500">per match</p>
                </div>
              </div>
            </CardContent>
          </Card>
          
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center">
                <div className="p-2 bg-purple-100 rounded-lg">
                  <Award className="w-5 h-5 text-purple-600" />
                </div>
                <div className="ml-3">
                  <p className="text-sm font-medium text-gray-500">Prize Pool</p>
                  <p className="text-xl font-bold">{formatCurrency(stats.overview.total_prize_pool)}</p>
                  <p className="text-xs text-gray-500">total distributed</p>
                </div>
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Progress Bar */}
        <Card className="mb-6">
          <CardContent className="p-4">
            <div className="flex items-center justify-between mb-2">
              <span className="text-sm font-medium">Tournament Progress</span>
              <span className="text-sm text-gray-500">{Math.round(progressPercentage)}%</span>
            </div>
            <Progress value={progressPercentage} className="h-2" />
          </CardContent>
        </Card>

        {/* Statistics Tabs */}
        <Tabs value={activeTab} onValueChange={setActiveTab} className="space-y-4">
          <TabsList className="grid w-full grid-cols-4">
            <TabsTrigger value="overview">Overview</TabsTrigger>
            <TabsTrigger value="participants">Participants</TabsTrigger>
            <TabsTrigger value="matches">Matches</TabsTrigger>
            <TabsTrigger value="performance">Performance</TabsTrigger>
          </TabsList>

          <TabsContent value="overview">
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
              {/* Key Metrics */}
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center space-x-2">
                    <BarChart3 className="h-5 w-5 text-blue-600" />
                    <span>Key Metrics</span>
                  </CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div className="grid grid-cols-2 gap-4">
                    <div>
                      <Label className="text-sm font-medium text-gray-500">Registration Rate</Label>
                      <p className="text-2xl font-bold text-green-600">{stats.overview.registration_rate}%</p>
                    </div>
                    <div>
                      <Label className="text-sm font-medium text-gray-500">Completion Rate</Label>
                      <p className="text-2xl font-bold text-blue-600">{Math.round(progressPercentage)}%</p>
                    </div>
                    <div>
                      <Label className="text-sm font-medium text-gray-500">Avg Match Duration</Label>
                      <p className="text-lg font-semibold">{formatDuration(stats.overview.average_match_duration)}</p>
                    </div>
                    <div>
                      <Label className="text-sm font-medium text-gray-500">Total Prize Pool</Label>
                      <p className="text-lg font-semibold">{formatCurrency(stats.overview.total_prize_pool)}</p>
                    </div>
                  </div>
                </CardContent>
              </Card>

              {/* Quick Trends */}
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center space-x-2">
                    <TrendingUp className="h-5 w-5 text-green-600" />
                    <span>Quick Trends</span>
                  </CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div className="space-y-3">
                    <div className="flex items-center justify-between">
                      <span className="text-sm">Participant Growth</span>
                      <div className="flex items-center text-green-600">
                        <TrendingUp className="w-4 h-4 mr-1" />
                        <span className="text-sm font-medium">+12%</span>
                      </div>
                    </div>
                    <div className="flex items-center justify-between">
                      <span className="text-sm">Match Completion</span>
                      <div className="flex items-center text-blue-600">
                        <Activity className="w-4 h-4 mr-1" />
                        <span className="text-sm font-medium">+8%</span>
                      </div>
                    </div>
                    <div className="flex items-center justify-between">
                      <span className="text-sm">Average Score</span>
                      <div className="flex items-center text-purple-600">
                        <Target className="w-4 h-4 mr-1" />
                        <span className="text-sm font-medium">+5%</span>
                      </div>
                    </div>
                  </div>
                </CardContent>
              </Card>
            </div>
          </TabsContent>

          <TabsContent value="participants">
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
              {/* Participant Status */}
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center space-x-2">
                    <Users className="h-5 w-5 text-blue-600" />
                    <span>Participant Status</span>
                  </CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="space-y-3">
                    {Object.entries(stats.participants.by_status).map(([status, count]) => (
                      <div key={status} className="flex items-center justify-between">
                        <span className="text-sm capitalize">{status.replace('_', ' ')}</span>
                        <div className="flex items-center space-x-2">
                          <div className="w-20 bg-gray-200 rounded-full h-2">
                            <div 
                              className="bg-blue-600 h-2 rounded-full" 
                              style={{ width: `${(count / stats.overview.total_participants) * 100}%` }}
                            ></div>
                          </div>
                          <span className="text-sm font-medium">{count}</span>
                        </div>
                      </div>
                    ))}
                  </div>
                </CardContent>
              </Card>

              {/* Registration Timeline */}
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center space-x-2">
                    <Calendar className="h-5 w-5 text-green-600" />
                    <span>Registration Timeline</span>
                  </CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="space-y-3">
                    {stats.participants.registration_timeline.slice(-7).map((item, index) => (
                      <div key={index} className="flex items-center justify-between">
                        <span className="text-sm">{new Date(item.date).toLocaleDateString()}</span>
                        <div className="flex items-center space-x-2">
                          <div className="w-20 bg-gray-200 rounded-full h-2">
                            <div 
                              className="bg-green-600 h-2 rounded-full" 
                              style={{ width: `${(item.count / Math.max(...stats.participants.registration_timeline.map(i => i.count))) * 100}%` }}
                            ></div>
                          </div>
                          <span className="text-sm font-medium">{item.count}</span>
                        </div>
                      </div>
                    ))}
                  </div>
                </CardContent>
              </Card>
            </div>
          </TabsContent>

          <TabsContent value="matches">
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
              {/* Match Status */}
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center space-x-2">
                    <Trophy className="h-5 w-5 text-yellow-600" />
                    <span>Match Status</span>
                  </CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="space-y-3">
                    {Object.entries(stats.matches.by_status).map(([status, count]) => (
                      <div key={status} className="flex items-center justify-between">
                        <span className="text-sm capitalize">{status.replace('_', ' ')}</span>
                        <div className="flex items-center space-x-2">
                          <div className="w-20 bg-gray-200 rounded-full h-2">
                            <div 
                              className="bg-yellow-600 h-2 rounded-full" 
                              style={{ width: `${(count / stats.overview.total_matches) * 100}%` }}
                            ></div>
                          </div>
                          <span className="text-sm font-medium">{count}</span>
                        </div>
                      </div>
                    ))}
                  </div>
                </CardContent>
              </Card>

              {/* Match Types */}
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center space-x-2">
                    <Target className="h-5 w-5 text-purple-600" />
                    <span>Match Types</span>
                  </CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="space-y-3">
                    {Object.entries(stats.matches.by_type).map(([type, count]) => (
                      <div key={type} className="flex items-center justify-between">
                        <span className="text-sm capitalize">{type.replace('_', ' ')}</span>
                        <div className="flex items-center space-x-2">
                          <div className="w-20 bg-gray-200 rounded-full h-2">
                            <div 
                              className="bg-purple-600 h-2 rounded-full" 
                              style={{ width: `${(count / stats.overview.total_matches) * 100}%` }}
                            ></div>
                          </div>
                          <span className="text-sm font-medium">{count}</span>
                        </div>
                      </div>
                    ))}
                  </div>
                </CardContent>
              </Card>
            </div>
          </TabsContent>

          <TabsContent value="performance">
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
              {/* Top Teams */}
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center space-x-2">
                    <Medal className="h-5 w-5 text-yellow-600" />
                    <span>Top Teams</span>
                  </CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="space-y-3">
                    {stats.performance.top_teams.slice(0, 5).map((team, index) => (
                      <div key={index} className="flex items-center justify-between p-2 bg-gray-50 rounded">
                        <div className="flex items-center space-x-2">
                          <span className="text-sm font-medium">#{index + 1}</span>
                          <span className="text-sm">{team.team}</span>
                        </div>
                        <div className="flex items-center space-x-2">
                          <span className="text-xs text-gray-500">{team.wins}W/{team.losses}L</span>
                          <Badge className="text-xs">{Math.round(team.win_rate)}%</Badge>
                        </div>
                      </div>
                    ))}
                  </div>
                </CardContent>
              </Card>

              {/* Most Active Teams */}
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center space-x-2">
                    <Activity className="h-5 w-5 text-green-600" />
                    <span>Most Active Teams</span>
                  </CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="space-y-3">
                    {stats.performance.most_active_teams.slice(0, 5).map((team, index) => (
                      <div key={index} className="flex items-center justify-between p-2 bg-gray-50 rounded">
                        <div className="flex items-center space-x-2">
                          <span className="text-sm font-medium">#{index + 1}</span>
                          <span className="text-sm">{team.team}</span>
                        </div>
                        <div className="flex items-center space-x-2">
                          <span className="text-xs text-gray-500">{team.matches_played} matches</span>
                          <Badge className="text-xs bg-green-100 text-green-800">Active</Badge>
                        </div>
                      </div>
                    ))}
                  </div>
                </CardContent>
              </Card>
            </div>
          </TabsContent>
        </Tabs>

        {/* Info Alert */}
        <Alert className="mt-6">
          <Info className="h-4 w-4" />
          <AlertDescription>
            <strong>Tournament Analytics:</strong> View comprehensive statistics about your tournament including 
            participant engagement, match performance, and team rankings. Use filters to analyze different time periods 
            and export reports for further analysis.
          </AlertDescription>
        </Alert>
      </div>
    </AppLayout>
  );
};

export default TournamentStatistics; 