import AppLayout from '@/layouts/app-layout';
import { Head, router } from '@inertiajs/react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Progress } from '@/components/ui/progress';
import { 
  Trophy, 
  Calendar, 
  Users, 
  MapPin, 
  DollarSign, 
  Edit, 
  ArrowLeft,
  Play,
  Pause,
  Award,
  Clock,
  Globe,
  Mail,
  Phone,
  ExternalLink,
  Share2,
  Download,
  Settings,
  Eye,
  TrendingUp,
  Target,
  Star
} from 'lucide-react';
import { usePermissions } from '@/hooks/use-permissions';

interface Tournament {
  id: number;
  name: string;
  slug: string;
  description: string;
  type: string;
  status: string;
  format: string;
  start_date: string;
  end_date: string;
  registration_start: string;
  registration_end: string;
  location: string;
  venue: string;
  organizer: string;
  contact_email: string;
  contact_phone: string;
  website: string;
  total_participants: number;
  max_participants: number;
  min_participants: number;
  entry_fee: number;
  prize_pool: number;
  prize_distribution: any;
  visibility: string;
  banner_image: string;
  logo_image: string;
  social_links: any;
  announcement: string;
  rules: any;
  settings: any;
  created_at: string;
  updated_at: string;
  completed_matches: number;
  total_matches: number;
  average_match_duration: number;
}

interface TournamentShowProps {
  tournament: Tournament;
  participants?: any[];
  matches?: any[];
  brackets?: any[];
}

export default function AdminTournamentShow({ tournament, participants = [], matches = [], brackets = [] }: TournamentShowProps) {
  const { hasPermission } = usePermissions();
  const getStatusColor = (status: string) => {
    const colors = {
      'draft': 'bg-gray-100 text-gray-800',
      'active': 'bg-green-100 text-green-800',
      'completed': 'bg-blue-100 text-blue-800',
      'cancelled': 'bg-red-100 text-red-800',
    };
    return colors[status as keyof typeof colors] || 'bg-gray-100 text-gray-800';
  };

  const getTypeColor = (type: string) => {
    const colors = {
      'single_elimination': 'bg-orange-100 text-orange-800',
      'double_elimination': 'bg-purple-100 text-purple-800',
      'round_robin': 'bg-teal-100 text-teal-800',
    };
    return colors[type as keyof typeof colors] || 'bg-gray-100 text-gray-800';
  };

  const formatCurrency = (amount: number, currency: string) => {
    return new Intl.NumberFormat('en-US', {
      style: 'currency',
      currency: currency || 'KES',
      minimumFractionDigits: 0,
      maximumFractionDigits: 0,
    }).format(amount);
  };

  const formatDate = (dateString: string) => {
    if (!dateString) return 'Not set';
    return new Date(dateString).toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'long',
      day: 'numeric',
      hour: '2-digit',
      minute: '2-digit'
    });
  };



  const progressPercentage = tournament.total_matches > 0 
    ? (tournament.completed_matches / tournament.total_matches) * 100 
    : 0;

  const registrationProgress = tournament.max_participants > 0
    ? (tournament.total_participants / tournament.max_participants) * 100
    : 0;

  return (
    <AppLayout breadcrumbs={[
      { title: 'Tournaments', href: '/admin/tournaments' },
      { title: tournament.name, href: `/admin/tournaments/${tournament.id}` }
    ]}> 
      <Head title={`Tournament: ${tournament.name}`} />
      
      <div className="space-y-4 sm:space-y-6 p-4">
        {/* Header */}
        <div className="flex flex-col lg:flex-row lg:items-center lg:justify-between gap-4">
          <div className="flex flex-col sm:flex-row sm:items-center gap-3 sm:gap-4">
            <Button 
              variant="outline" 
              size="sm"
              onClick={() => router.get(route('admin.tournaments.index'))}
              className="w-full sm:w-auto"
            >
              <ArrowLeft className="w-4 h-4 mr-2" />
              Back
            </Button>
            
          </div>
          
          <div className="flex flex-wrap gap-2">
            <Button variant="outline" size="sm" className="flex-1 sm:flex-none">
              <Share2 className="w-4 h-4 mr-2" />
              Share
            </Button>
            <Button variant="outline" size="sm" className="flex-1 sm:flex-none">
              <Download className="w-4 h-4 mr-2" />
              Export
            </Button>
            {hasPermission('tournaments.edit') && (
            <Button 
              variant="outline" 
              size="sm"
              onClick={() => router.get(route('admin.tournaments.edit', tournament.id))}
              className="flex-1 sm:flex-none"
            >
              <Edit className="w-4 h-4 mr-2" />
              Edit
            </Button>
            )}
          </div>
        </div>

        {/* Quick Stats */}
        <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-4 sm:gap-6">
          <Card>
            <CardContent className="p-4 sm:p-6">
              <div className="flex items-center">
                <div className="p-2 bg-blue-100 rounded-lg">
                  <Users className="w-5 h-5 sm:w-6 sm:h-6 text-blue-600" />
                </div>
                <div className="ml-3 sm:ml-4">
                  <p className="text-xs sm:text-sm font-medium text-muted-foreground">Participants</p>
                  <p className="text-xl sm:text-2xl font-bold text-foreground">
                    {tournament.total_participants || 0}
                    {tournament.max_participants && (
                      <span className="text-xs sm:text-sm text-muted-foreground">
                        /{tournament.max_participants}
                      </span>
                    )}
                  </p>
                </div>
              </div>
            </CardContent>
          </Card>
          
          <Card>
            <CardContent className="p-4 sm:p-6">
              <div className="flex items-center">
                <div className="p-2 bg-green-100 rounded-lg">
                  <Trophy className="w-5 h-5 sm:w-6 sm:h-6 text-green-600" />
                </div>
                <div className="ml-3 sm:ml-4">
                  <p className="text-xs sm:text-sm font-medium text-muted-foreground">Matches</p>
                  <p className="text-xl sm:text-2xl font-bold text-foreground">
                    {tournament.completed_matches || 0}
                    {tournament.total_matches && (
                      <span className="text-xs sm:text-sm text-muted-foreground">
                        /{tournament.total_matches}
                      </span>
                    )}
                  </p>
                </div>
              </div>
            </CardContent>
          </Card>
          
          <Card>
            <CardContent className="p-4 sm:p-6">
              <div className="flex items-center">
                <div className="p-2 bg-yellow-100 rounded-lg">
                  <Award className="w-5 h-5 sm:w-6 sm:h-6 text-yellow-600" />
                </div>
                <div className="ml-3 sm:ml-4">
                  <p className="text-xs sm:text-sm font-medium text-muted-foreground">Prize Pool</p>
                  <p className="text-xl sm:text-2xl font-bold text-foreground">
                    {tournament.prize_pool ? formatCurrency(tournament.prize_pool, 'KES') : 'TBD'}
                  </p>
                </div>
              </div>
            </CardContent>
          </Card>
          
          <Card>
            <CardContent className="p-4 sm:p-6">
              <div className="flex items-center">
                <div className="p-2 bg-purple-100 rounded-lg">
                  <TrendingUp className="w-5 h-5 sm:w-6 sm:h-6 text-purple-600" />
                </div>
                <div className="ml-3 sm:ml-4">
                  <p className="text-xs sm:text-sm font-medium text-muted-foreground">Progress</p>
                  <p className="text-xl sm:text-2xl font-bold text-foreground">
                    {Math.round(progressPercentage)}%
                  </p>
                </div>
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Progress Bars */}
        {(tournament.total_matches > 0 || tournament.max_participants > 0) && (
          <div className="grid grid-cols-1 sm:grid-cols-2 gap-4 sm:gap-6">
            {tournament.total_matches > 0 && (
              <Card>
                <CardContent className="p-4 sm:p-6">
                  <div className="flex items-center justify-between mb-2">
                    <h3 className="text-sm sm:text-base font-medium">Tournament Progress</h3>
                    <span className="text-xs sm:text-sm text-muted-foreground">
                      {tournament.completed_matches}/{tournament.total_matches} matches
                    </span>
                  </div>
                  <Progress value={progressPercentage} className="h-2" />
                </CardContent>
              </Card>
            )}
            
            {tournament.max_participants > 0 && (
              <Card>
                <CardContent className="p-4 sm:p-6">
                  <div className="flex items-center justify-between mb-2">
                    <h3 className="text-sm sm:text-base font-medium">Registration Progress</h3>
                    <span className="text-xs sm:text-sm text-muted-foreground">
                      {tournament.total_participants}/{tournament.max_participants} slots
                    </span>
                  </div>
                  <Progress value={registrationProgress} className="h-2" />
                </CardContent>
              </Card>
            )}
          </div>
        )}

        {/* Main Content Tabs */}
        <Tabs defaultValue="overview" className="space-y-4 sm:space-y-6">
          <TabsList className="grid w-full grid-cols-2 sm:grid-cols-3 lg:grid-cols-5">
            <TabsTrigger value="overview" className="text-xs sm:text-sm">Overview</TabsTrigger>
            <TabsTrigger value="participants" className="text-xs sm:text-sm">Participants</TabsTrigger>
            <TabsTrigger value="matches" className="text-xs sm:text-sm">Matches</TabsTrigger>
            <TabsTrigger value="settings" className="text-xs sm:text-sm">Settings</TabsTrigger>
            <TabsTrigger value="media" className="text-xs sm:text-sm">Media</TabsTrigger>
          </TabsList>

          <TabsContent value="overview" className="space-y-4 sm:space-y-6">
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-4 sm:gap-6">
              {/* Tournament Details */}
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2 text-sm sm:text-base">
                    <Trophy className="w-4 h-4 sm:w-5 sm:h-5" />
                    Tournament Details
                  </CardTitle>
                </CardHeader>
                <CardContent className="space-y-3 sm:space-y-4">
                  <div className="grid grid-cols-1 sm:grid-cols-2 gap-3 sm:gap-4">
                    <div>
                      <label className="text-sm font-medium text-muted-foreground">Format</label>
                      <p className="text-sm font-medium capitalize">{tournament.format || 'Not specified'}</p>
                    </div>
                    <div>
                      <label className="text-sm font-medium text-muted-foreground">Visibility</label>
                      <p className="text-sm font-medium capitalize">{tournament.visibility}</p>
                    </div>
                    <div>
                      <label className="text-sm font-medium text-muted-foreground">Entry Fee</label>
                      <p className="text-sm font-medium">
                        {tournament.entry_fee ? formatCurrency(tournament.entry_fee, 'KES') : 'Free'}
                      </p>
                    </div>
                    <div>
                      <label className="text-sm font-medium text-muted-foreground">Min Participants</label>
                      <p className="text-sm font-medium">{tournament.min_participants || 'Not set'}</p>
                    </div>
                  </div>
                </CardContent>
              </Card>

              {/* Schedule & Location */}
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <Calendar className="w-5 h-5" />
                    Schedule & Location
                  </CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div>
                    <label className="text-sm font-medium text-muted-foreground">Tournament Period</label>
                    <p className="text-sm font-medium">
                      {formatDate(tournament.start_date)} - {formatDate(tournament.end_date)}
                    </p>
                  </div>
                  {(tournament.registration_start || tournament.registration_end) && (
                    <div>
                      <label className="text-sm font-medium text-muted-foreground">Registration Period</label>
                      <p className="text-sm font-medium">
                        {formatDate(tournament.registration_start)} - {formatDate(tournament.registration_end)}
                      </p>
                    </div>
                  )}
                  {tournament.location && (
                    <div>
                      <label className="text-sm font-medium text-muted-foreground">Location</label>
                      <p className="text-sm font-medium flex items-center gap-1">
                        <MapPin className="w-4 h-4" />
                        {tournament.location}
                      </p>
                    </div>
                  )}
                  {tournament.venue && (
                    <div>
                      <label className="text-sm font-medium text-muted-foreground">Venue</label>
                      <p className="text-sm font-medium">{tournament.venue}</p>
                    </div>
                  )}
                </CardContent>
              </Card>

              {/* Contact Information */}
              {(tournament.organizer || tournament.contact_email || tournament.contact_phone || tournament.website) && (
                <Card>
                  <CardHeader>
                    <CardTitle className="flex items-center gap-2">
                      <Mail className="w-5 h-5" />
                      Contact Information
                    </CardTitle>
                  </CardHeader>
                  <CardContent className="space-y-4">
                    {tournament.organizer && (
                      <div>
                        <label className="text-sm font-medium text-muted-foreground">Organizer</label>
                        <p className="text-sm font-medium">{tournament.organizer}</p>
                      </div>
                    )}
                    {tournament.contact_email && (
                      <div>
                        <label className="text-sm font-medium text-muted-foreground">Email</label>
                        <p className="text-sm font-medium flex items-center gap-1">
                          <Mail className="w-4 h-4" />
                          <a href={`mailto:${tournament.contact_email}`} className="text-blue-600 hover:underline">
                            {tournament.contact_email}
                          </a>
                        </p>
                      </div>
                    )}
                    {tournament.contact_phone && (
                      <div>
                        <label className="text-sm font-medium text-muted-foreground">Phone</label>
                        <p className="text-sm font-medium flex items-center gap-1">
                          <Phone className="w-4 h-4" />
                          <a href={`tel:${tournament.contact_phone}`} className="text-blue-600 hover:underline">
                            {tournament.contact_phone}
                          </a>
                        </p>
                      </div>
                    )}
                    {tournament.website && (
                      <div>
                        <label className="text-sm font-medium text-muted-foreground">Website</label>
                        <p className="text-sm font-medium flex items-center gap-1">
                          <Globe className="w-4 h-4" />
                          <a href={tournament.website} target="_blank" rel="noopener noreferrer" className="text-blue-600 hover:underline">
                            {tournament.website}
                            <ExternalLink className="w-3 h-3 ml-1 inline" />
                          </a>
                        </p>
                      </div>
                    )}
                  </CardContent>
                </Card>
              )}

              {/* Quick Actions */}
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <Settings className="w-5 h-5" />
                    Quick Actions
                  </CardTitle>
                </CardHeader>
                  <CardContent className="space-y-3">
                    <Button
                    className="w-full justify-start"
                    variant="outline"
                    onClick={() => router.get(route('admin.tournaments.register-teams', tournament.id))}
                    >
                      <Users className="w-4 h-4 mr-2" />
                      Register Teams
                    </Button>
                   <Button 
                     className="w-full justify-start" 
                     variant="outline"
                     onClick={() => router.get(route('admin.tournaments.participants', tournament.id))}
                   >
                     <Users className="w-4 h-4 mr-2" />
                     Manage Participants
                   </Button>
                  <Button className="w-full justify-start" variant="outline"
                  onClick={() => router.get(route('admin.tournaments.matches', tournament.id))}
                  >
                    <Trophy className="w-4 h-4 mr-2" />
                    View Matches
                  </Button>
                  <Button className="w-full justify-start" variant="outline"
                  onClick={() => router.get(route('admin.tournaments.brackets', tournament.id))}
                  >
                    <Target className="w-4 h-4 mr-2" />
                    View Brackets
                  </Button>
                  <Button className="w-full justify-start" variant="outline"
                  onClick={() => router.get(route('admin.tournaments.statistics', tournament.id))}
                  >
                    <TrendingUp className="w-4 h-4 mr-2" />
                    Statistics
                  </Button>
                </CardContent>
              </Card>
            </div>

            {/* Announcement */}
            {tournament.announcement && (
              <Card>
                <CardHeader>
                  <CardTitle>Announcement</CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
                    <p className="text-blue-800">{tournament.announcement}</p>
                  </div>
                </CardContent>
              </Card>
            )}
          </TabsContent>

          <TabsContent value="participants">
            <Card>
              <CardHeader>
                <CardTitle>Participants ({tournament.total_participants || 0})</CardTitle>
              </CardHeader>
              <CardContent>
                <div className="text-center py-8 text-muted-foreground">
                  <Users className="w-12 h-12 mx-auto mb-4 opacity-50" />
                  <p>Participant management coming soon</p>
                </div>
              </CardContent>
            </Card>
          </TabsContent>

          <TabsContent value="matches">
            <Card>
              <CardHeader>
                <CardTitle>Matches ({tournament.total_matches || 0})</CardTitle>
              </CardHeader>
              <CardContent>
                <div className="text-center py-8 text-muted-foreground">
                  <Trophy className="w-12 h-12 mx-auto mb-4 opacity-50" />
                  <p>Match management coming soon</p>
                </div>
              </CardContent>
            </Card>
          </TabsContent>

          <TabsContent value="settings">
            <Card>
              <CardHeader>
                <CardTitle>Tournament Settings</CardTitle>
              </CardHeader>
              <CardContent>
                <div className="space-y-6">
                  <div>
                    <h3 className="text-lg font-medium mb-3">Rules</h3>
                    {tournament.rules && Object.keys(tournament.rules).length > 0 ? (
                      <div className="bg-gray-50 rounded-lg p-4">
                        <pre className="text-sm">{JSON.stringify(tournament.rules, null, 2)}</pre>
                      </div>
                    ) : (
                      <p className="text-muted-foreground">No rules configured</p>
                    )}
                  </div>
                  
                  <div>
                    <h3 className="text-lg font-medium mb-3">Additional Settings</h3>
                    {tournament.settings && Object.keys(tournament.settings).length > 0 ? (
                      <div className="bg-gray-50 rounded-lg p-4">
                        <pre className="text-sm">{JSON.stringify(tournament.settings, null, 2)}</pre>
                      </div>
                    ) : (
                      <p className="text-muted-foreground">No additional settings configured</p>
                    )}
                  </div>
                </div>
              </CardContent>
            </Card>
          </TabsContent>

          <TabsContent value="media">
            <Card>
              <CardHeader>
                <CardTitle>Tournament Media</CardTitle>
              </CardHeader>
              <CardContent>
                <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                  <div>
                    <h3 className="font-medium mb-3">Banner Image</h3>
                    {tournament.banner_image ? (
                      <img
                        src={tournament.banner_image}
                        alt="Tournament banner"
                        className="w-full h-48 object-cover rounded-lg border"
                      />
                    ) : (
                      <div className="w-full h-48 bg-gray-100 rounded-lg border flex items-center justify-center">
                        <p className="text-muted-foreground">No banner image</p>
                      </div>
                    )}
                  </div>
                  
                  <div>
                    <h3 className="font-medium mb-3">Logo</h3>
                    {tournament.logo_image ? (
                      <img
                        src={tournament.logo_image}
                        alt="Tournament logo"
                        className="w-32 h-32 object-contain rounded-lg border"
                      />
                    ) : (
                      <div className="w-32 h-32 bg-gray-100 rounded-lg border flex items-center justify-center">
                        <p className="text-muted-foreground text-sm">No logo</p>
                      </div>
                    )}
                  </div>
                </div>
              </CardContent>
            </Card>
          </TabsContent>
        </Tabs>
      </div>
    </AppLayout>
  );
}