import React, { useState, useEffect } from 'react';
import { Head, router } from '@inertiajs/react';
import AppLayout from '@/layouts/app-layout';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Label } from '@/components/ui/label';
import { Badge } from '@/components/ui/badge';
import { Checkbox } from '@/components/ui/checkbox';
import { 
  Trophy, 
  Users, 
  Search, 
  ArrowLeft,
  CheckCircle,
  AlertCircle,
  Info,
  Plus,
  UserCheck,
  UserX
} from 'lucide-react';
import { Alert, AlertDescription } from '@/components/ui/alert';

interface Team {
  id: number;
  name: string;
  description: string;
  status: string;
  captain_id: number;
  captain?: {
    name: string;
    email: string;
  };
  tournaments?: Array<{
    id: number;
    name: string;
    pivot: {
      status: string;
      registered_at: string;
    };
  }>;
}

interface Tournament {
  id: number;
  name: string;
  description: string;
  status: string;
  entry_fee: number;
  max_participants: number;
  total_participants: number;
}

interface Props {
  tournament: Tournament;
  available_teams: Team[];
  registered_teams: Team[];
  errors: Record<string, string>;
}

const TournamentRegisterTeams: React.FC<Props> = ({ 
  tournament, 
  available_teams, 
  registered_teams, 
  errors 
}) => {
  const [selectedTeams, setSelectedTeams] = useState<number[]>([]);
  const [searchTerm, setSearchTerm] = useState('');
  const [statusFilter, setStatusFilter] = useState('all');
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [showRegistered, setShowRegistered] = useState(false);

  // Filter teams based on search and status
  const filteredTeams = available_teams.filter(team => {
    const matchesSearch = team.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
                         team.description?.toLowerCase().includes(searchTerm.toLowerCase());
    const matchesStatus = statusFilter === 'all' || team.status === statusFilter;
    return matchesSearch && matchesStatus;
  });

  const handleTeamSelection = (teamId: number, checked: boolean) => {
    if (checked) {
      setSelectedTeams([...selectedTeams, teamId]);
    } else {
      setSelectedTeams(selectedTeams.filter(id => id !== teamId));
    }
  };

  const handleSelectAll = (checked: boolean) => {
    if (checked) {
      setSelectedTeams(filteredTeams.map(team => team.id));
    } else {
      setSelectedTeams([]);
    }
  };

  const handleRegisterTeams = () => {
    if (selectedTeams.length === 0) {
      return;
    }

    setIsSubmitting(true);
    router.post(route('admin.tournaments.register-teams', { tournament: tournament.id }), {
      team_ids: selectedTeams,
      registration_notes: `Bulk registration by admin on ${new Date().toLocaleDateString()}`
    }, {
      onSuccess: () => {
        setSelectedTeams([]);
        setIsSubmitting(false);
      },
      onError: () => {
        setIsSubmitting(false);
      }
    });
  };

  const handleUnregisterTeam = (teamId: number) => {
    router.delete(route('admin.tournaments.unregister-team', { 
      tournament: tournament.id, 
      team: teamId 
    }));
  };

  const getRegistrationStatus = (team: Team) => {
    if (!team.tournaments) return 'Not registered';
    
    const tournamentRegistration = team.tournaments.find(t => t.id === tournament.id);
    if (!tournamentRegistration) return 'Not registered';
    
    return tournamentRegistration.pivot.status;
  };

  const isTeamRegistered = (team: Team) => {
    if (!team.tournaments) return false;
    return team.tournaments.some(t => t.id === tournament.id);
  };

  const getStatusColor = (status: string) => {
    const colors = {
      'registered': 'bg-blue-100 text-blue-800',
      'active': 'bg-green-100 text-green-800',
      'eliminated': 'bg-red-100 text-red-800',
      'withdrawn': 'bg-gray-100 text-gray-800',
    };
    return colors[status as keyof typeof colors] || 'bg-gray-100 text-gray-800';
  };

  return (
    <AppLayout>
      <Head title={`Register Teams - ${tournament.name}`} />
      
      <div className="p-4">
        {/* Header */}
        <div className="flex items-center justify-between mb-6">
          <div className="flex items-center space-x-4">
            <Button variant="outline" size="sm" asChild>
              <a href={route('admin.tournaments.index')}>
                <ArrowLeft className="h-4 w-4 mr-2" />
                Back to Tournaments
              </a>
            </Button>
            
          </div>
        </div>

        {/* Tournament Info Card */}
        <Card className="mb-6">
          <CardHeader>
            <CardTitle className="flex items-center space-x-2">
              <Trophy className="h-5 w-5 text-yellow-600" />
              <span>{tournament.name}</span>
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
              <div>
                <Label className="text-sm font-medium text-gray-500">Status</Label>
                <Badge className={getStatusColor(tournament.status)}>
                  {tournament.status}
                </Badge>
              </div>
              <div>
                <Label className="text-sm font-medium text-gray-500">Entry Fee</Label>
                <p className="text-lg font-semibold">KES {tournament?.entry_fee?.toLocaleString()}</p>
              </div>
              <div>
                <Label className="text-sm font-medium text-gray-500">Participants</Label>
                <p className="text-lg font-semibold">
                  {tournament?.total_participants} / {tournament?.max_participants}
                </p>
              </div>
              <div>
                <Label className="text-sm font-medium text-gray-500">Available Slots</Label>
                <p className="text-lg font-semibold text-green-600">
                  {tournament?.max_participants - tournament?.total_participants}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>

        {/* Tabs for Available vs Registered Teams */}
        <div className="mb-6">
          <div className="flex space-x-4 border-b">
            <button
              onClick={() => setShowRegistered(false)}
              className={`pb-2 px-1 border-b-2 font-medium ${
                !showRegistered 
                  ? 'border-blue-500 text-blue-600' 
                  : 'border-transparent text-gray-500 hover:text-gray-700'
              }`}
            >
              Available Teams ({available_teams.length})
            </button>
            <button
              onClick={() => setShowRegistered(true)}
              className={`pb-2 px-1 border-b-2 font-medium ${
                showRegistered 
                  ? 'border-blue-500 text-blue-600' 
                  : 'border-transparent text-gray-500 hover:text-gray-700'
              }`}
            >
              Registered Teams ({registered_teams.length})
            </button>
          </div>
        </div>

        {!showRegistered ? (
          /* Available Teams Section */
          <div>
            {/* Filters and Actions */}
            <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center mb-6 space-y-4 sm:space-y-0">
              <div className="flex flex-col sm:flex-row space-y-2 sm:space-y-0 sm:space-x-4">
                <div className="relative">
                  <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 h-4 w-4 text-gray-400" />
                  <Input
                    placeholder="Search teams..."
                    value={searchTerm}
                    onChange={(e) => setSearchTerm(e.target.value)}
                    className="pl-10 w-64"
                  />
                </div>
                <Select value={statusFilter} onValueChange={setStatusFilter}>
                  <SelectTrigger className="w-48">
                    <SelectValue placeholder="Filter by status" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">All Status</SelectItem>
                    <SelectItem value="active">Active</SelectItem>
                    <SelectItem value="inactive">Inactive</SelectItem>
                  </SelectContent>
                </Select>
              </div>
              
              {selectedTeams.length > 0 && (
                <div className="flex items-center space-x-4">
                  <span className="text-sm text-gray-600">
                    {selectedTeams.length} team(s) selected
                  </span>
                  <Button 
                    onClick={handleRegisterTeams}
                    disabled={isSubmitting}
                    className="bg-green-600 hover:bg-green-700"
                  >
                    {isSubmitting ? 'Registering...' : `Register ${selectedTeams.length} Team(s)`}
                  </Button>
                </div>
              )}
            </div>

            {/* Teams List */}
            <Card>
              <CardContent className="p-0">
                {filteredTeams.length === 0 ? (
                  <div className="p-8 text-center text-gray-500">
                    <Users className="h-12 w-12 mx-auto mb-4 text-gray-300" />
                    <p>No teams found matching your criteria.</p>
                  </div>
                ) : (
                  <div className="divide-y">
                    {/* Select All Header */}
                    <div className="p-4 bg-gray-50 border-b">
                      <div className="flex items-center space-x-3">
                        <Checkbox
                          checked={selectedTeams.length === filteredTeams.length && filteredTeams.length > 0}
                          onCheckedChange={handleSelectAll}
                        />
                        <span className="font-medium">Select All</span>
                      </div>
                    </div>

                    {/* Teams */}
                    {filteredTeams.map((team) => (
                      <div key={team.id} className="p-4 hover:bg-gray-50">
                        <div className="flex items-center space-x-4">
                          <Checkbox
                            checked={selectedTeams.includes(team.id)}
                            onCheckedChange={(checked) => handleTeamSelection(team.id, checked as boolean)}
                          />
                          <div className="flex-1">
                            <div className="flex items-center justify-between">
                              <div>
                                <h3 className="font-semibold">{team.name}</h3>
                                <p className="text-sm text-gray-600">{team.description}</p>
                                {team.captain && (
                                  <p className="text-sm text-gray-500">
                                    Captain: {team.captain.name} ({team.captain.email})
                                  </p>
                                )}
                              </div>
                              <div className="flex items-center space-x-2">
                                <Badge className={getStatusColor(team.status)}>
                                  {team.status}
                                </Badge>
                                {isTeamRegistered(team) && (
                                  <Badge className="bg-green-100 text-green-800">
                                    <CheckCircle className="h-3 w-3 mr-1" />
                                    Registered
                                  </Badge>
                                )}
                              </div>
                            </div>
                          </div>
                        </div>
                      </div>
                    ))}
                  </div>
                )}
              </CardContent>
            </Card>
          </div>
        ) : (
          /* Registered Teams Section */
          <div>
            <Card>
              <CardContent className="p-0">
                {registered_teams.length === 0 ? (
                  <div className="p-8 text-center text-gray-500">
                    <Users className="h-12 w-12 mx-auto mb-4 text-gray-300" />
                    <p>No teams registered for this tournament yet.</p>
                  </div>
                ) : (
                  <div className="divide-y">
                    {registered_teams.map((team) => (
                      <div key={team.id} className="p-4 hover:bg-gray-50">
                        <div className="flex items-center justify-between">
                          <div>
                            <h3 className="font-semibold">{team.name}</h3>
                            <p className="text-sm text-gray-600">{team.description}</p>
                            {team.captain && (
                              <p className="text-sm text-gray-500">
                                Captain: {team.captain.name} ({team.captain.email})
                              </p>
                            )}
                            <div className="flex items-center space-x-2 mt-2">
                              <Badge className={getStatusColor(team.status)}>
                                {team.status}
                              </Badge>
                              <Badge className={getStatusColor(getRegistrationStatus(team))}>
                                {getRegistrationStatus(team)}
                              </Badge>
                            </div>
                          </div>
                          <div className="flex items-center space-x-2">
                            <Button
                              variant="outline"
                              size="sm"
                              onClick={() => handleUnregisterTeam(team.id)}
                              className="text-red-600 hover:text-red-700"
                            >
                              <UserX className="h-4 w-4 mr-1" />
                              Unregister
                            </Button>
                          </div>
                        </div>
                      </div>
                    ))}
                  </div>
                )}
              </CardContent>
            </Card>
          </div>
        )}

        {/* Info Alert */}
        <Alert className="mt-6">
          <Info className="h-4 w-4" />
          <AlertDescription>
            <strong>Registration Notes:</strong> Teams can be registered for multiple tournaments. 
            Each team will receive an email notification about their registration. 
            The tournament entry fee will be handled separately through the payment system.
          </AlertDescription>
        </Alert>
      </div>
    </AppLayout>
  );
};

export default TournamentRegisterTeams; 