import React, { useState } from 'react';
import { Head, router } from '@inertiajs/react';
import AppLayout from '@/layouts/app-layout';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Badge } from '@/components/ui/badge';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { usePermissions } from '@/hooks/use-permissions';
import { CategoryBadge } from '@/components/ui/category-badge';
import { 
  Users, 
  Search, 
  Filter, 
  ArrowLeft,
  CheckCircle,
  XCircle,
  Clock,
  UserCheck,
  UserX,
  Mail,
  Phone,
  MapPin,
  Calendar,
  Trophy,
  Award,
  Star,
  Eye,
  Edit,
  Trash2,
  MoreHorizontal,
  Download,
  Upload,
  Plus,
  Info
} from 'lucide-react';
import { 
  DropdownMenu, 
  DropdownMenuContent, 
  DropdownMenuItem, 
  DropdownMenuTrigger,
  DropdownMenuSeparator
} from '@/components/ui/dropdown-menu';
import { Alert, AlertDescription } from '@/components/ui/alert';
import { Progress } from '@/components/ui/progress';
import { Label } from '@/components/ui/label';

interface Participant {
  id: number;
  team_id: number;
  team: {
    id: number;
    name: string;
    description: string;
    status: string;
    category?: 'men' | 'women' | 'youth_male' | 'youth_female' | 'children';
    captain: {
      id: number;
      name: string;
      email: string;
      phone?: string;
    };
    players: Array<{
      id: number;
      name: string;
      email: string;
      phone?: string;
      position?: string;
      gaming_handle?: string;
    }>;
    officials: Array<{
      id: number;
      name: string;
      email: string;
      phone?: string;
      role: string;
    }>;
  };
  pivot: {
    status: string;
    registered_at: string;
    approved_at?: string;
    registration_notes?: string;
    tournament_specific_data?: any;
    approval_status?: string;
    first_approved_by?: number;
    first_approved_at?: string;
    second_approved_by?: number;
    second_approved_at?: string;
    first_approval_notes?: string;
    second_approval_notes?: string;
    rejection_reason?: string;
    rejected_by?: number;
    rejected_at?: string;
  };
}

interface Tournament {
  id: number;
  name: string;
  description: string;
  enable_categories?: boolean;
  category_fees?: Record<string, number>;
  available_categories?: string[];
  status: string;
  entry_fee: number | null;
  max_participants: number;
  total_participants: number;
  start_date: string;
  end_date: string;
  enable_two_tier_approval?: boolean;
  first_approver_id?: number;
  second_approver_id?: number;
  first_approver?: {
    id: number;
    name: string;
    email: string;
  };
  second_approver?: {
    id: number;
    name: string;
    email: string;
  };
}

interface Props {
  tournament: Tournament;
  participants: Participant[];
  filters: {
    search: string;
    status: string;
    team_status: string;
    category?: string;
  };
  stats: {
    total: number;
    registered: number;
    active: number;
    eliminated: number;
    withdrawn: number;
    pending_approval: number;
    inactive: number;
    pending_first_approval: number;
    first_approved: number;
    second_approved: number;
    rejected: number;
  };
  userPermissions: string[];
  userRoles: string[];
}

const TournamentParticipants: React.FC<Props> = ({ 
  tournament, 
  participants, 
  filters,
  stats,
  userPermissions,
  userRoles
}) => {
  const { hasPermission } = usePermissions();
  const [localFilters, setLocalFilters] = useState({
    search: filters?.search || '',
    status: filters?.status || 'all',
    team_status: filters?.team_status || 'all',
    category: filters?.category || 'all',
  });

  const [selectedParticipants, setSelectedParticipants] = useState<number[]>([]);
  const [activeTab, setActiveTab] = useState('all');
  const [showApprovalModal, setShowApprovalModal] = useState(false);
  const [selectedParticipant, setSelectedParticipant] = useState<Participant | null>(null);
  const [approvalAction, setApprovalAction] = useState<'first-approve' | 'second-approve' | 'reject' | null>(null);
  const [approvalNotes, setApprovalNotes] = useState('');

  const handleFilterChange = (key: string, value: string) => {
    const newFilters = { ...localFilters, [key]: value };
    setLocalFilters(newFilters);
    router.get(route('admin.tournaments.participants', tournament.id), newFilters, {
      preserveState: true,
      preserveScroll: true,
    });
  };

  const handleParticipantSelection = (participantId: number, checked: boolean) => {
    if (checked) {
      setSelectedParticipants([...selectedParticipants, participantId]);
    } else {
      setSelectedParticipants(selectedParticipants.filter(id => id !== participantId));
    }
  };

  const handleSelectAll = (checked: boolean) => {
    if (checked) {
      setSelectedParticipants(displayParticipants.map(p => p.id));
    } else {
      setSelectedParticipants([]);
    }
  };

  const handleBulkAction = (action: string) => {
    if (selectedParticipants.length === 0) return;

    const data = {
      participant_ids: selectedParticipants,
      action: action,
    };

    router.post(route('admin.tournaments.participants.bulk-action', tournament.id), data, {
      onSuccess: () => {
        setSelectedParticipants([]);
      },
    });
  };

  const handleStatusChange = (participantId: number, newStatus: string) => {
    router.put(route('admin.tournaments.participants.update-status', { 
      tournament: tournament.id, 
      team: participantId
    }), {
      status: newStatus,
    });
  };

  // Approval helper functions
  const canApprove = (participant: Participant): boolean => {
    if (userRoles.includes('admin') || userRoles.includes('super_admin')) {
      return true;
    }
    
    if (tournament.enable_two_tier_approval) {
      return userRoles.includes('first_approver') || userRoles.includes('second_approver');
    }
    
    return userPermissions.includes('tournaments.approve_participants');
  };

  const canFirstApprove = (participant: Participant): boolean => {
    if (userRoles.includes('admin') || userRoles.includes('super_admin')) {
      return true;
    }
    
    if (tournament.enable_two_tier_approval) {
      return userRoles.includes('first_approver') && participant.pivot.approval_status === 'pending';
    }
    
    return false;
  };

  const canSecondApprove = (participant: Participant): boolean => {
    if (userRoles.includes('admin') || userRoles.includes('super_admin')) {
      return true;
    }
    
    if (tournament.enable_two_tier_approval) {
      return userRoles.includes('second_approver') && participant.pivot.approval_status === 'first_approved';
    }
    
    return false;
  };

  const canReject = (participant: Participant): boolean => {
    if (userRoles.includes('admin') || userRoles.includes('super_admin')) {
      return true;
    }
    
    if (tournament.enable_two_tier_approval) {
      return (userRoles.includes('first_approver') || userRoles.includes('second_approver')) && 
             participant.pivot.approval_status !== 'rejected';
    }
    
    return userPermissions.includes('tournaments.approve_participants');
  };

  const showFirstApprovalNotMade = (participant: Participant): boolean => {
    return Boolean(tournament.enable_two_tier_approval) && 
           userRoles.includes('second_approver') && 
           !userRoles.includes('first_approver') && 
           participant.pivot.approval_status === 'pending';
  };

  const handleApprovalAction = (participant: Participant, action: 'first-approve' | 'second-approve' | 'reject') => {
    setSelectedParticipant(participant);
    setApprovalAction(action);
    setApprovalNotes('');
    setShowApprovalModal(true);
  };

  const submitApprovalAction = () => {
    if (!selectedParticipant || !approvalAction) return;

    const data = approvalAction === 'reject' 
      ? { reason: approvalNotes }
      : { notes: approvalNotes };

    let routeName = '';
    switch (approvalAction) {
      case 'first-approve':
        routeName = 'admin.tournaments.participants.first-approve';
        break;
      case 'second-approve':
        routeName = 'admin.tournaments.participants.second-approve';
        break;
      case 'reject':
        routeName = 'admin.tournaments.participants.reject';
        break;
    }

    router.post(route(routeName, { 
      tournament: tournament.id, 
      team: selectedParticipant.id 
    }), data, {
      onSuccess: () => {
        setShowApprovalModal(false);
        setSelectedParticipant(null);
        setApprovalAction(null);
        setApprovalNotes('');
      }
    });
  };

  const getStatusColor = (status: string) => {
    const colors = {
      'registered': 'bg-blue-100 text-blue-800',
      'active': 'bg-green-100 text-green-800',
      'eliminated': 'bg-red-100 text-red-800',
      'withdrawn': 'bg-gray-100 text-gray-800',
    };
    return colors[status as keyof typeof colors] || 'bg-gray-100 text-gray-800';
  };

  const getApprovalStatusColor = (approvalStatus: string) => {
    const colors = {
      'pending': 'bg-yellow-100 text-yellow-800',
      'first_approved': 'bg-blue-100 text-blue-800',
      'second_approved': 'bg-green-100 text-green-800',
      'rejected': 'bg-red-100 text-red-800',
    };
    return colors[approvalStatus as keyof typeof colors] || 'bg-gray-100 text-gray-800';
  };

  const getApprovalStatusText = (approvalStatus: string) => {
    const texts = {
      'pending': 'Pending Approval',
      'first_approved': 'First Approved',
      'second_approved': 'Fully Approved',
      'rejected': 'Rejected',
    };
    return texts[approvalStatus as keyof typeof texts] || 'Unknown';
  };

  const getTeamStatusColor = (status: string) => {
    const colors = {
      'active': 'bg-green-100 text-green-800',
      'inactive': 'bg-gray-100 text-gray-800',
    };
    return colors[status as keyof typeof colors] || 'bg-gray-100 text-gray-800';
  };

  const formatDate = (dateString: string) => {
    if (!dateString) return 'Not set';
    return new Date(dateString).toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric',
      hour: '2-digit',
      minute: '2-digit'
    });
  };



  const filteredParticipants = participants.filter(participant => {
    // Search filter - only apply if search term is not empty
    const matchesSearch = 
      !localFilters.search || 
      localFilters.search.trim() === '' ||
      participant.team.name?.toLowerCase().includes(localFilters.search.toLowerCase()) ||
      participant.team.captain.name?.toLowerCase().includes(localFilters.search.toLowerCase()) ||
      participant.team.captain.email?.toLowerCase().includes(localFilters.search.toLowerCase());
    
    // Status filter
    const matchesStatus = localFilters.status === 'all' || participant.pivot.status === localFilters.status;
    
    // Team status filter
    const matchesTeamStatus = localFilters.team_status === 'all' || participant.team.status === localFilters.team_status;
    
    // Category filter
    const matchesCategory = localFilters.category === 'all' || !localFilters.category || participant.team.category === localFilters.category;
    
    return matchesSearch && matchesStatus && matchesTeamStatus && matchesCategory;
  });

  // Fallback: If no filtered results but we have participants, show all
  const displayParticipants = filteredParticipants.length === 0 && participants.length > 0 
    ? participants 
    : filteredParticipants;

  const getParticipantsByStatus = (status: string) => {
    if (status === 'all') return displayParticipants;
    return displayParticipants.filter(p => p.pivot.status === status);
  };



  const registrationProgress = tournament.max_participants > 0
    ? (tournament.total_participants / tournament.max_participants) * 100
    : 0;

  return (
    <AppLayout>
      <Head title={`Participants - ${tournament.name}`} />
      
      <div className="p-4">
        {/* Header */}
        <div className="flex items-center justify-between mb-6">
          <div className="flex items-center space-x-4">
            <Button variant="outline" size="sm" asChild>
              <a href={route('admin.tournaments.show', tournament.id)}>
                <ArrowLeft className="h-4 w-4 mr-2" />
                Back to Tournament
              </a>
            </Button>
           
          </div>
          <div className="flex space-x-2">
            <Button variant="outline" size="sm">
              <Download className="h-4 w-4 mr-2" />
              Export
            </Button>
            <Button 
              variant="outline" 
              size="sm"
              onClick={() => {
                const form = document.createElement('form');
                form.method = 'POST';
                form.action = route('admin.tournaments.certificates.bulk', tournament.id);
                form.style.display = 'none';
                
                const csrfToken = document.createElement('input');
                csrfToken.type = 'hidden';
                csrfToken.name = '_token';
                csrfToken.value = document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || '';
                form.appendChild(csrfToken);
                
                const statusFilter = document.createElement('input');
                statusFilter.type = 'hidden';
                statusFilter.name = 'status_filter';
                statusFilter.value = 'active';
                form.appendChild(statusFilter);
                
                document.body.appendChild(form);
                form.submit();
                document.body.removeChild(form);
              }}
            >
              <Award className="h-4 w-4 mr-2" />
              Generate Certificates
            </Button>
            <Button 
              variant="outline" 
              size="sm"
              onClick={() => {
                // Get all registered participants
                const registeredParticipants = participants.filter(p => p.pivot.status === 'registered');
                if (registeredParticipants.length === 0) {
                  alert('No registered participants to approve.');
                  return;
                }
                
                // Set selected participants to all registered ones
                setSelectedParticipants(registeredParticipants.map(p => p.id));
                
                // Trigger bulk approve action
                handleBulkAction('approve');
              }}
              className="bg-green-50 text-green-700 border-green-200 hover:bg-green-100"
            >
              <UserCheck className="h-4 w-4 mr-2" />
              Approve All Registered
            </Button>
            <Button variant="outline" size="sm">
              <Upload className="h-4 w-4 mr-2" />
              Import
            </Button>
            <Button size="sm">
              <Plus className="h-4 w-4 mr-2" />
              Add Participant
            </Button>
          </div>
        </div>

        {/* Tournament Info Card */}
        <Card className="mb-6">
          <CardHeader>
            <CardTitle className="flex items-center space-x-2">
              <Trophy className="h-5 w-5 text-yellow-600" />
              <span>{tournament.name}</span>
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
              <div>
                <Label className="text-sm font-medium text-gray-500">Status</Label>
                <Badge className={getStatusColor(tournament.status)}>
                  {tournament.status}
                </Badge>
              </div>
              <div>
                <Label className="text-sm font-medium text-gray-500">Entry Fee</Label>
                <p className="text-lg font-semibold">KES {tournament.entry_fee?.toLocaleString() ?? '0'}</p>
              </div>
              <div>
                <Label className="text-sm font-medium text-gray-500">Participants</Label>
                <p className="text-lg font-semibold">
                  {tournament.total_participants} / {tournament.max_participants}
                </p>
              </div>
              <div>
                <Label className="text-sm font-medium text-gray-500">Available Slots</Label>
                <p className="text-lg font-semibold text-green-600">
                  {tournament.max_participants - tournament.total_participants}
                </p>
              </div>
            </div>
            
            {/* Registration Progress */}
            <div className="mt-4">
              <div className="flex items-center justify-between mb-2">
                <span className="text-sm font-medium">Registration Progress</span>
                <span className="text-sm text-gray-500">{Math.round(registrationProgress)}%</span>
              </div>
              <Progress value={registrationProgress} className="h-2" />
            </div>
          </CardContent>
        </Card>

        {/* Stats Cards */}
        <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-6 gap-4 mb-6">
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center">
                <div className="p-2 bg-blue-100 rounded-lg">
                  <Users className="w-5 h-5 text-blue-600" />
                </div>
                <div className="ml-3">
                  <p className="text-sm font-medium text-gray-500">Total</p>
                  <p className="text-xl font-bold">{stats.total}</p>
                </div>
              </div>
            </CardContent>
          </Card>
          
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center">
                <div className="p-2 bg-blue-100 rounded-lg">
                  <Clock className="w-5 h-5 text-blue-600" />
                </div>
                <div className="ml-3">
                  <p className="text-sm font-medium text-gray-500">Registered</p>
                  <p className="text-xl font-bold">{stats.registered}</p>
                </div>
              </div>
            </CardContent>
          </Card>
          
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center">
                <div className="p-2 bg-green-100 rounded-lg">
                  <UserCheck className="w-5 h-5 text-green-600" />
                </div>
                <div className="ml-3">
                  <p className="text-sm font-medium text-gray-500">Active</p>
                  <p className="text-xl font-bold">{stats.active}</p>
                </div>
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardContent className="p-4">
              <div className="flex items-center">
                <div className="p-2 bg-gray-100 rounded-lg">
                  <XCircle className="w-5 h-5 text-gray-600" />
                </div>
                <div className="ml-3">
                  <p className="text-sm font-medium text-gray-500">Inactive</p>
                  <p className="text-xl font-bold">{stats.inactive}</p>
                </div>
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardContent className="p-4">
              <div className="flex items-center">
                <div className="p-2 bg-yellow-100 rounded-lg">
                  <Clock className="w-5 h-5 text-yellow-600" />
                </div>
                <div className="ml-3">
                  <p className="text-sm font-medium text-gray-500">Pending Approval</p>
                  <p className="text-xl font-bold">{stats.pending_approval}</p>
                </div>
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardContent className="p-4">
              <div className="flex items-center">
                <div className="p-2 bg-red-100 rounded-lg">
                  <XCircle className="w-5 h-5 text-red-600" />
                </div>
                <div className="ml-3">
                  <p className="text-sm font-medium text-gray-500">Eliminated</p>
                  <p className="text-xl font-bold">{stats.eliminated}</p>
                </div>
              </div>
            </CardContent>
          </Card>
          
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center">
                <div className="p-2 bg-gray-100 rounded-lg">
                  <UserX className="w-5 h-5 text-gray-600" />
                </div>
                <div className="ml-3">
                  <p className="text-sm font-medium text-gray-500">Withdrawn</p>
                  <p className="text-xl font-bold">{stats.withdrawn}</p>
                </div>
              </div>
            </CardContent>
          </Card>

          {/* Two-tier approval stats */}
          {tournament.enable_two_tier_approval && (
            <>
              <Card>
                <CardContent className="p-4">
                  <div className="flex items-center">
                    <div className="p-2 bg-yellow-100 rounded-lg">
                      <Clock className="w-5 h-5 text-yellow-600" />
                    </div>
                    <div className="ml-3">
                      <p className="text-sm font-medium text-gray-500">Pending First</p>
                      <p className="text-xl font-bold">{stats.pending_first_approval}</p>
                    </div>
                  </div>
                </CardContent>
              </Card>
              
              <Card>
                <CardContent className="p-4">
                  <div className="flex items-center">
                    <div className="p-2 bg-blue-100 rounded-lg">
                      <UserCheck className="w-5 h-5 text-blue-600" />
                    </div>
                    <div className="ml-3">
                      <p className="text-sm font-medium text-gray-500">First Approved</p>
                      <p className="text-xl font-bold">{stats.first_approved}</p>
                    </div>
                  </div>
                </CardContent>
              </Card>
              
              <Card>
                <CardContent className="p-4">
                  <div className="flex items-center">
                    <div className="p-2 bg-green-100 rounded-lg">
                      <CheckCircle className="w-5 h-5 text-green-600" />
                    </div>
                    <div className="ml-3">
                      <p className="text-sm font-medium text-gray-500">Fully Approved</p>
                      <p className="text-xl font-bold">{stats.second_approved}</p>
                    </div>
                  </div>
                </CardContent>
              </Card>
              
              <Card>
                <CardContent className="p-4">
                  <div className="flex items-center">
                    <div className="p-2 bg-red-100 rounded-lg">
                      <XCircle className="w-5 h-5 text-red-600" />
                    </div>
                    <div className="ml-3">
                      <p className="text-sm font-medium text-gray-500">Rejected</p>
                      <p className="text-xl font-bold">{stats.rejected}</p>
                    </div>
                  </div>
                </CardContent>
              </Card>
            </>
          )}
        </div>

        {/* Filters */}
        <Card className="mb-6">
          <CardContent className="p-4">
            <div className="flex flex-col sm:flex-row gap-4">
              <div className="flex-1">
                <div className="relative">
                  <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 h-4 w-4 text-gray-400" />
                  <Input
                    placeholder="Search teams, captains, emails..."
                    value={localFilters.search}
                    onChange={(e) => handleFilterChange('search', e.target.value)}
                    className="pl-10"
                  />
                </div>
              </div>
              <Select value={localFilters.status} onValueChange={(value) => handleFilterChange('status', value)}>
                <SelectTrigger className="w-48">
                  <SelectValue placeholder="All Status" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Status</SelectItem>
                  <SelectItem value="registered">Registered</SelectItem>
                  <SelectItem value="active">Active</SelectItem>
                  <SelectItem value="eliminated">Eliminated</SelectItem>
                  <SelectItem value="withdrawn">Withdrawn</SelectItem>
                </SelectContent>
              </Select>
              <Select value={localFilters.team_status} onValueChange={(value) => handleFilterChange('team_status', value)}>
                <SelectTrigger className="w-48">
                  <SelectValue placeholder="All Team Status" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Team Status</SelectItem>
                  <SelectItem value="active">Active Teams</SelectItem>
                  <SelectItem value="inactive">Inactive Teams</SelectItem>
                </SelectContent>
              </Select>

              {/* Category Filter */}
              {tournament.enable_categories && (
                <Select value={localFilters.category} onValueChange={(value) => handleFilterChange('category', value)}>
                  <SelectTrigger className="w-48">
                    <SelectValue placeholder="All Categories" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">All Categories</SelectItem>
                    <SelectItem value="men"><span className="mr-2">👨</span>Men</SelectItem>
                    <SelectItem value="women"><span className="mr-2">👩</span>Women</SelectItem>
                    <SelectItem value="youth_male"><span className="mr-2">🧑</span>Youth Male</SelectItem>
                    <SelectItem value="youth_female"><span className="mr-2">👧</span>Youth Female</SelectItem>
                    <SelectItem value="children"><span className="mr-2">🧒</span>Children</SelectItem>
                  </SelectContent>
                </Select>
              )}
            </div>
          </CardContent>
        </Card>

        {/* Bulk Actions */}
        {selectedParticipants.length > 0 && (
          <div className="mb-4 p-4 bg-blue-50 border border-blue-200 rounded-lg">
            <div className="flex items-center justify-between">
              <span className="text-blue-800">
                {selectedParticipants.length} participant(s) selected
              </span>
              <div className="flex space-x-2">
                <Button 
                  variant="outline" 
                  size="sm" 
                  onClick={() => handleBulkAction('approve')}
                >
                  <UserCheck className="h-4 w-4 mr-1" />
                  Approve
                </Button>
                <Button 
                  variant="outline" 
                  size="sm" 
                  onClick={() => handleBulkAction('eliminate')}
                >
                  <XCircle className="h-4 w-4 mr-1" />
                  Eliminate
                </Button>
                <Button 
                  variant="outline" 
                  size="sm" 
                  onClick={() => handleBulkAction('withdraw')}
                >
                  <UserX className="h-4 w-4 mr-1" />
                  Withdraw
                </Button>
              </div>
            </div>
          </div>
        )}

        {/* Participants Tabs */}
        <Tabs value={activeTab} onValueChange={setActiveTab} className="space-y-4">
          <TabsList className={`grid w-full ${tournament.enable_two_tier_approval ? 'grid-cols-10' : 'grid-cols-6'}`}>
            <TabsTrigger value="all">All ({displayParticipants.length})</TabsTrigger>
            <TabsTrigger value="registered" className="bg-yellow-50 text-yellow-700">Registered ({getParticipantsByStatus('registered').length})</TabsTrigger>
            <TabsTrigger value="active">Active ({getParticipantsByStatus('active').length})</TabsTrigger>
            <TabsTrigger value="eliminated">Eliminated ({getParticipantsByStatus('eliminated').length})</TabsTrigger>
            <TabsTrigger value="withdrawn">Withdrawn ({getParticipantsByStatus('withdrawn').length})</TabsTrigger>
            <TabsTrigger value="pending_approval">Pending ({getParticipantsByStatus('pending_approval').length})</TabsTrigger>
            
            {/* Two-tier approval tabs */}
            {tournament.enable_two_tier_approval && (
              <>
                <TabsTrigger value="pending_first" className="bg-yellow-50 text-yellow-700">Pending First ({displayParticipants.filter(p => p.pivot.approval_status === 'pending').length})</TabsTrigger>
                <TabsTrigger value="first_approved" className="bg-blue-50 text-blue-700">First Approved ({displayParticipants.filter(p => p.pivot.approval_status === 'first_approved').length})</TabsTrigger>
                <TabsTrigger value="second_approved" className="bg-green-50 text-green-700">Fully Approved ({displayParticipants.filter(p => p.pivot.approval_status === 'second_approved').length})</TabsTrigger>
                <TabsTrigger value="rejected" className="bg-red-50 text-red-700">Rejected ({displayParticipants.filter(p => p.pivot.approval_status === 'rejected').length})</TabsTrigger>
              </>
            )}
          </TabsList>

          <TabsContent value={activeTab}>
            <Card>
              <CardContent className="p-0">
                {displayParticipants.length === 0 ? (
                  <div className="p-8 text-center text-gray-500">
                    <Users className="h-12 w-12 mx-auto mb-4 text-gray-300" />
                    <p>No participants found matching your criteria.</p>
                  </div>
                ) : (
                  <div className="divide-y">
                    {/* Select All Header */}
                    <div className="p-4 bg-gray-50 border-b">
                      <div className="flex items-center space-x-3">
                        <input
                          type="checkbox"
                          checked={selectedParticipants.length === displayParticipants.length && displayParticipants.length > 0}
                          onChange={(e) => handleSelectAll(e.target.checked)}
                          className="rounded border-gray-300"
                        />
                        <span className="font-medium">Select All</span>
                      </div>
                    </div>

                    {/* Participants List */}
                    {displayParticipants.map((participant) => (
                      <div key={participant.id} className="p-4 hover:bg-gray-50">
                        <div className="flex items-center space-x-4">
                          <input
                            type="checkbox"
                            checked={selectedParticipants.includes(participant.id)}
                            onChange={(e) => handleParticipantSelection(participant.id, e.target.checked)}
                            className="rounded border-gray-300"
                          />
                          <div className="flex-1">
                            <div className="flex items-center justify-between">
                              <div>
                                <div className="flex items-center gap-2">
                                  <h3 className="font-semibold">{participant?.team?.name}</h3>
                                  {participant?.team?.category && (
                                    <CategoryBadge category={participant.team.category} size="sm" />
                                  )}
                                </div>
                                <p className="text-sm text-gray-600">{participant?.team?.description}</p>
                                <div className="flex items-center space-x-2 mt-1">
                                  <span className="text-sm text-gray-500">
                                    Captain: {participant?.team?.captain?.name}
                                  </span>
                                  <span className="text-sm text-gray-500">
                                    • {participant?.team?.players?.length} players
                                  </span>
                                  <span className="text-sm text-gray-500">
                                    • {participant?.team?.officials?.length} officials
                                  </span>
                                </div>
                              </div>
                              <div className="flex items-center space-x-2">
                                <Badge className={getStatusColor(participant?.pivot?.status)}>
                                  {participant?.pivot?.status}
                                </Badge>
                                <Badge className={getTeamStatusColor(participant?.team?.status)}>
                                  {participant?.team?.status}
                                </Badge>
                                {tournament.enable_two_tier_approval && participant?.pivot?.approval_status && (
                                  <Badge className={getApprovalStatusColor(participant?.pivot?.approval_status)}>
                                    {getApprovalStatusText(participant?.pivot?.approval_status)}
                                  </Badge>
                                )}
                                <DropdownMenu>
                                  <DropdownMenuTrigger asChild>
                                    <Button variant="ghost" size="sm">
                                      <MoreHorizontal className="h-4 w-4" />
                                    </Button>
                                  </DropdownMenuTrigger>
                                                                     <DropdownMenuContent align="end">
                                     <DropdownMenuItem onClick={() => router.get(route('admin.tournaments.participant-detail', { 
                                       tournament: tournament.id, 
                                       team: participant?.id 
                                     }))}>
                                       <Eye className="w-4 h-4 mr-2" />
                                       View Details
                                     </DropdownMenuItem>
                                     <DropdownMenuItem onClick={() => router.get(route('admin.teams.show', participant.id))}>
                                       <Eye className="w-4 h-4 mr-2" />
                                       View Team
                                     </DropdownMenuItem>
                                    <DropdownMenuItem onClick={() => router.get(route('admin.teams.edit', participant.id))}>
                                      <Edit className="w-4 h-4 mr-2" />
                                      Edit Team
                                    </DropdownMenuItem>
                                    <DropdownMenuSeparator />
                                    <DropdownMenuItem 
                                      onClick={() => {
                                        if (['active', 'eliminated', 'withdrawn'].includes(participant.pivot.status)) {
                                          window.open(route('admin.tournaments.certificate', { 
                                            tournament: tournament.id, 
                                            team: participant.id 
                                          }), '_blank');
                                        }
                                      }}
                                      disabled={!['active', 'eliminated', 'withdrawn'].includes(participant.pivot.status)}
                                    >
                                      <Award className="w-4 h-4 mr-2" />
                                      Download Certificate
                                    </DropdownMenuItem>
                                    <DropdownMenuSeparator />
                                    
                                    {/* Approval Actions */}
                                    {tournament.enable_two_tier_approval ? (
                                      <>
                                        {canFirstApprove(participant) && (
                                          <DropdownMenuItem onClick={() => handleApprovalAction(participant, 'first-approve')}>
                                            <UserCheck className="w-4 h-4 mr-2" />
                                            First Approve
                                          </DropdownMenuItem>
                                        )}
                                        {canSecondApprove(participant) && (
                                          <DropdownMenuItem onClick={() => handleApprovalAction(participant, 'second-approve')}>
                                            <CheckCircle className="w-4 h-4 mr-2" />
                                            Second Approve
                                          </DropdownMenuItem>
                                        )}
                                        {showFirstApprovalNotMade(participant) && (
                                          <DropdownMenuItem disabled>
                                            <Clock className="w-4 h-4 mr-2" />
                                            First Approval Not Made
                                          </DropdownMenuItem>
                                        )}
                                        {canReject(participant) && (
                                          <DropdownMenuItem onClick={() => handleApprovalAction(participant, 'reject')}>
                                            <XCircle className="w-4 h-4 mr-2" />
                                            Reject
                                          </DropdownMenuItem>
                                        )}
                                      </>
                                    ) : (
                                      <>
                                        {canApprove(participant) && (
                                          <DropdownMenuItem onClick={() => handleStatusChange(participant?.id, 'active')}>
                                            <UserCheck className="w-4 h-4 mr-2" />
                                            Approve
                                          </DropdownMenuItem>
                                        )}
                                      </>
                                    )}                                    
                                    {/* Standard Actions */}
                                    {hasPermission('tournaments.participants.eliminate') && (
                                    <DropdownMenuItem onClick={() => handleStatusChange(participant?.id, 'eliminated')}>
                                      <XCircle className="w-4 h-4 mr-2" />
                                      Eliminate
                                    </DropdownMenuItem>
                                    )}
                                    {hasPermission('tournaments.participants.withdraw') && (
                                    <DropdownMenuItem onClick={() => handleStatusChange(participant?.id, 'withdrawn')}>
                                      <UserX className="w-4 h-4 mr-2" />
                                      Withdraw
                                    </DropdownMenuItem>
                                    )}
                                    {hasPermission('tournaments.participants.remove') && (
                                    <>
                                    <DropdownMenuSeparator />
                                    <DropdownMenuItem 
                                      onClick={() => router.delete(route('admin.tournaments.unregister-team', { 
                                        tournament: tournament.id, 
                                        team: participant?.id 
                                      }))}
                                      className="text-red-600"
                                    >
                                      <Trash2 className="w-4 h-4 mr-2" />
                                      Remove
                                    </DropdownMenuItem>
                                    </>
                                    )}

                                  </DropdownMenuContent>
                                </DropdownMenu>
                              </div>
                            </div>
                            
                            {/* Registration Details */}
                            <div className="mt-3 text-sm text-gray-500">
                              <div className="flex items-center space-x-4">
                                <span>Registered: {formatDate(participant?.pivot?.registered_at)}</span>
                                {participant?.pivot?.approved_at && (
                                  <span>Approved: {formatDate(participant?.pivot?.approved_at)}</span>
                                )}
                                {participant?.pivot?.registration_notes && (
                                  <span>Notes: {participant?.pivot?.registration_notes}</span>
                                )}
                              </div>
                            </div>
                          </div>
                        </div>
                      </div>
                    ))}
                  </div>
                )}
              </CardContent>
            </Card>
          </TabsContent>
        </Tabs>



        {/* Info Alert */}
        <Alert className="mt-6">
          <Info className="h-4 w-4" />
          <AlertDescription>
            <strong>Participant Management:</strong> You can approve, eliminate, or withdraw participants. 
            Changes will be reflected immediately and participants will be notified via email. 
            Use bulk actions to manage multiple participants at once.
            {tournament.enable_two_tier_approval && (
              <><br /><strong>Two-Tier Approval:</strong> This tournament uses a two-tier approval system. 
              First approvers can approve participants, and second approvers can give final approval. 
              Only fully approved participants become active.</>
            )}
          </AlertDescription>
        </Alert>

        {/* Approval Modal */}
        {showApprovalModal && selectedParticipant && approvalAction && (
          <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
            <div className="bg-white rounded-lg p-6 w-full max-w-md mx-4">
              <h3 className="text-lg font-semibold mb-4">
                {approvalAction === 'first-approve' && 'First Approve Participant'}
                {approvalAction === 'second-approve' && 'Second Approve Participant'}
                {approvalAction === 'reject' && 'Reject Participant'}
              </h3>
              
              <div className="mb-4">
                <p className="text-sm text-gray-600 mb-2">
                  <strong>Team:</strong> {selectedParticipant.team.name}
                </p>
                <p className="text-sm text-gray-600">
                  <strong>Captain:</strong> {selectedParticipant.team.captain.name}
                </p>
              </div>

              <div className="mb-4">
                <Label htmlFor="notes">
                  {approvalAction === 'reject' ? 'Rejection Reason' : 'Notes (Optional)'}
                </Label>
                <textarea
                  id="notes"
                  value={approvalNotes}
                  onChange={(e) => setApprovalNotes(e.target.value)}
                  className="w-full mt-1 p-2 border border-gray-300 rounded-md"
                  rows={3}
                  placeholder={approvalAction === 'reject' ? 'Enter reason for rejection...' : 'Enter any notes...'}
                  required={approvalAction === 'reject'}
                />
              </div>

              <div className="flex justify-end space-x-2">
                <Button
                  variant="outline"
                  onClick={() => {
                    setShowApprovalModal(false);
                    setSelectedParticipant(null);
                    setApprovalAction(null);
                    setApprovalNotes('');
                  }}
                >
                  Cancel
                </Button>
                <Button
                  onClick={submitApprovalAction}
                  className={approvalAction === 'reject' ? 'bg-red-600 hover:bg-red-700' : 'bg-green-600 hover:bg-green-700'}
                >
                  {approvalAction === 'first-approve' && 'First Approve'}
                  {approvalAction === 'second-approve' && 'Second Approve'}
                  {approvalAction === 'reject' && 'Reject'}
                </Button>
              </div>
            </div>
          </div>
        )}
      </div>
    </AppLayout>
  );
};

export default TournamentParticipants; 