import React, { useState } from 'react';
import { Head, router } from '@inertiajs/react';
import AppLayout from '@/layouts/app-layout';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { CategoryBadge } from '@/components/ui/category-badge';
import { 
  Users, 
  ArrowLeft,
  UserCheck,
  XCircle,
  UserX,
  Mail,
  Phone,
  MapPin,
  Calendar,
  Trophy,
  Award,
  Star,
  Eye,
  Edit,
  Trash2,
  MoreHorizontal,
  Download,
  Info,
  Shield,
  Crown,
  User
} from 'lucide-react';
import { 
  DropdownMenu, 
  DropdownMenuContent, 
  DropdownMenuItem, 
  DropdownMenuTrigger,
  DropdownMenuSeparator
} from '@/components/ui/dropdown-menu';
import { Alert, AlertDescription } from '@/components/ui/alert';
import { Label } from '@/components/ui/label';

interface Participant {
  id: number;
  team_id: number;
  team: {
    id: number;
    name: string;
    description: string;
    status: string;
    category?: 'men' | 'women' | 'youth_male' | 'youth_female' | 'children';
    captain: {
      id: number;
      name: string;
      email: string;
      phone?: string;
    };
    players: Array<{
      id: number;
      name: string;
      email: string;
      phone?: string;
      position?: string;
      gaming_handle?: string;
      user_id: number;
      user?: {
        id: number;
        name: string;
        email: string;
      };
    }>;
    officials: Array<{
      id: number;
      name: string;
      email: string;
      phone?: string;
      role: string;
      user_id: number;
      user?: {
        id: number;
        name: string;
        email: string;
      };
    }>;
  };
  pivot: {
    status: string;
    registered_at: string;
    approved_at?: string;
    registration_notes?: string;
    tournament_specific_data?: any;
  };
}

interface Tournament {
  id: number;
  name: string;
  description: string;
  status: string;
  entry_fee: number;
  max_participants: number;
  total_participants: number;
  start_date: string;
  end_date: string;
  enable_categories?: boolean;
  category_fees?: Record<string, number>;
  available_categories?: string[];
}

interface Props {
  tournament: Tournament;
  participant: Participant;
}

const ParticipantDetail: React.FC<Props> = ({ tournament, participant }) => {
  const [isUpdating, setIsUpdating] = useState(false);

  const handleStatusChange = (newStatus: string) => {
    setIsUpdating(true);
    router.put(route('admin.tournaments.participants.update-status', { 
      tournament: tournament.id, 
      team: participant.team.id 
    }), {
      status: newStatus,
    }, {
      onSuccess: () => setIsUpdating(false),
      onError: () => setIsUpdating(false),
    });
  };

  const getStatusColor = (status: string) => {
    const colors = {
      'registered': 'bg-blue-100 text-blue-800',
      'active': 'bg-green-100 text-green-800',
      'eliminated': 'bg-red-100 text-red-800',
      'withdrawn': 'bg-gray-100 text-gray-800',
    };
    return colors[status as keyof typeof colors] || 'bg-gray-100 text-gray-800';
  };

  const getTeamStatusColor = (status: string) => {
    const colors = {
      'active': 'bg-green-100 text-green-800',
      'inactive': 'bg-gray-100 text-gray-800',
    };
    return colors[status as keyof typeof colors] || 'bg-gray-100 text-gray-800';
  };

  const formatDate = (dateString: string) => {
    if (!dateString) return 'Not set';
    return new Date(dateString).toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'long',
      day: 'numeric',
      hour: '2-digit',
      minute: '2-digit'
    });
  };

  return (
    <AppLayout>
      <Head title={`${participant.team.name} - ${tournament.name}`} />
      
      <div className="p-4">
        {/* Header */}
        <div className="flex items-center justify-between mb-6">
          <div className="flex items-center space-x-4">
            <Button variant="outline" size="sm" asChild>
              <a href={route('admin.tournaments.participants', tournament.id)}>
                <ArrowLeft className="h-4 w-4 mr-2" />
                Back to Participants
              </a>
            </Button>
            
          </div>
          <div className="flex space-x-2">
            <Button 
              variant="outline" 
              size="sm"
              onClick={() => router.get(route('admin.teams.edit', participant.team.id))}
            >
              <Edit className="h-4 w-4 mr-2" />
              Edit Team
            </Button>
            <DropdownMenu>
              <DropdownMenuTrigger asChild>
                <Button variant="outline" size="sm">
                  <MoreHorizontal className="h-4 w-4" />
                </Button>
              </DropdownMenuTrigger>
              <DropdownMenuContent align="end">
                <DropdownMenuItem onClick={() => router.get(route('admin.teams.show', participant.team.id))}>
                  <Eye className="w-4 h-4 mr-2" />
                  View Team Details
                </DropdownMenuItem>
                <DropdownMenuItem>
                  <Download className="w-4 h-4 mr-2" />
                  Export Data
                </DropdownMenuItem>
                <DropdownMenuSeparator />
                <DropdownMenuItem 
                  onClick={() => router.delete(route('admin.tournaments.unregister-team', { 
                    tournament: tournament.id, 
                    team: participant.team.id 
                  }))}
                  className="text-red-600"
                >
                  <Trash2 className="w-4 h-4 mr-2" />
                  Remove from Tournament
                </DropdownMenuItem>
              </DropdownMenuContent>
            </DropdownMenu>
          </div>
        </div>

        {/* Status Management */}
        <Card className="mb-6">
          <CardHeader>
            <CardTitle className="flex items-center space-x-2">
              <Trophy className="h-5 w-5 text-yellow-600" />
              <span>Tournament Status</span>
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="flex items-center justify-between">
              <div className="flex items-center space-x-4">
                <Badge className={getStatusColor(participant.pivot.status)}>
                  {participant.pivot.status}
                </Badge>
                <Badge className={getTeamStatusColor(participant.team.status)}>
                  {participant.team.status}
                </Badge>
              </div>
              <div className="flex space-x-2">
                <Button 
                  variant="outline" 
                  size="sm"
                  onClick={() => handleStatusChange('active')}
                  disabled={isUpdating || participant.pivot.status === 'active'}
                  className="bg-green-50 text-green-700 border-green-200 hover:bg-green-100"
                >
                  <UserCheck className="h-4 w-4 mr-1" />
                  Approve
                </Button>
                <Button 
                  variant="outline" 
                  size="sm"
                  onClick={() => handleStatusChange('eliminated')}
                  disabled={isUpdating || participant.pivot.status === 'eliminated'}
                  className="bg-red-50 text-red-700 border-red-200 hover:bg-red-100"
                >
                  <XCircle className="h-4 w-4 mr-1" />
                  Eliminate
                </Button>
                <Button 
                  variant="outline" 
                  size="sm"
                  onClick={() => handleStatusChange('withdrawn')}
                  disabled={isUpdating || participant.pivot.status === 'withdrawn'}
                  className="bg-gray-50 text-gray-700 border-gray-200 hover:bg-gray-100"
                >
                  <UserX className="h-4 w-4 mr-1" />
                  Withdraw
                </Button>
                {['active', 'eliminated', 'withdrawn'].includes(participant.pivot.status) && (
                  <Button 
                    variant="outline" 
                    size="sm"
                    onClick={() => {
                      window.open(route('admin.tournaments.certificate', { 
                        tournament: tournament.id, 
                        team: participant.team.id 
                      }), '_blank');
                    }}
                    className="bg-blue-50 text-blue-700 border-blue-200 hover:bg-blue-100"
                  >
                    <Award className="h-4 w-4 mr-1" />
                    Download Certificate
                  </Button>
                )}
              </div>
            </div>
          </CardContent>
        </Card>

        {/* Registration Details */}
        <Card className="mb-6">
          <CardHeader>
            <CardTitle className="flex items-center space-x-2">
              <Calendar className="h-5 w-5 text-blue-600" />
              <span>Registration Details</span>
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div>
                <Label className="text-sm font-medium text-gray-500">Registered At</Label>
                <p className="text-sm font-medium">{formatDate(participant.pivot.registered_at)}</p>
              </div>
              {participant.pivot.approved_at && (
                <div>
                  <Label className="text-sm font-medium text-gray-500">Approved At</Label>
                  <p className="text-sm font-medium">{formatDate(participant.pivot.approved_at)}</p>
                </div>
              )}
              {participant.pivot.registration_notes && (
                <div className="md:col-span-2">
                  <Label className="text-sm font-medium text-gray-500">Registration Notes</Label>
                  <p className="text-sm">{participant.pivot.registration_notes}</p>
                </div>
              )}
              
              {/* Category Information */}
              {tournament.enable_categories && participant.team.category && (
                <>
                  <div>
                    <Label className="text-sm font-medium text-gray-500">Team Category</Label>
                    <div className="mt-1">
                      <CategoryBadge category={participant.team.category} size="md" />
                    </div>
                  </div>
                  <div>
                    <Label className="text-sm font-medium text-gray-500">Category Requirements</Label>
                    <p className="text-sm">
                      {participant.team.category === 'men' && '18+ years, Male players'}
                      {participant.team.category === 'women' && '18+ years, Female players'}
                      {participant.team.category === 'youth_male' && '13-17 years, Male players'}
                      {participant.team.category === 'youth_female' && '13-17 years, Female players'}
                      {participant.team.category === 'children' && 'Under 13 years, Any gender'}
                    </p>
                  </div>
                </>
              )}
            </div>
          </CardContent>
        </Card>

        {/* Team Information */}
        <Tabs defaultValue="overview" className="space-y-4">
          <TabsList className="grid w-full grid-cols-4">
            <TabsTrigger value="overview">Overview</TabsTrigger>
            <TabsTrigger value="players">Players ({participant.team.players.length})</TabsTrigger>
            <TabsTrigger value="officials">Officials ({participant.team.officials.length})</TabsTrigger>
            <TabsTrigger value="tournament">Tournament Data</TabsTrigger>
          </TabsList>

          <TabsContent value="overview">
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
              {/* Team Details */}
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center space-x-2">
                    <Shield className="h-5 w-5 text-blue-600" />
                    <span>Team Information</span>
                  </CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div>
                    <Label className="text-sm font-medium text-gray-500">Team Name</Label>
                    <div className="flex items-center gap-2 mt-1">
                      <p className="text-lg font-semibold">{participant.team.name}</p>
                      {participant.team.category && (
                        <CategoryBadge category={participant.team.category} size="md" />
                      )}
                    </div>
                  </div>
                  {participant.team.category && (
                    <div>
                      <Label className="text-sm font-medium text-gray-500">Team Category</Label>
                      <div className="mt-1">
                        <CategoryBadge category={participant.team.category} size="md" showTooltip={true} />
                      </div>
                      <p className="text-xs text-gray-500 mt-1">
                        {participant.team.category === 'men' && '18+ years, Male players'}
                        {participant.team.category === 'women' && '18+ years, Female players'}
                        {participant.team.category === 'youth_male' && '13-17 years, Male players'}
                        {participant.team.category === 'youth_female' && '13-17 years, Female players'}
                        {participant.team.category === 'children' && 'Under 13 years, Any gender'}
                      </p>
                    </div>
                  )}
                  <div>
                    <Label className="text-sm font-medium text-gray-500">Description</Label>
                    <p className="text-sm">{participant.team.description || 'No description provided'}</p>
                  </div>
                  <div>
                    <Label className="text-sm font-medium text-gray-500">Team Status</Label>
                    <Badge className={getTeamStatusColor(participant.team.status)}>
                      {participant.team.status}
                    </Badge>
                  </div>
                </CardContent>
              </Card>

              {/* Captain Information */}
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center space-x-2">
                    <Crown className="h-5 w-5 text-yellow-600" />
                    <span>Captain</span>
                  </CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div>
                    <Label className="text-sm font-medium text-gray-500">Name</Label>
                    <p className="text-lg font-semibold">{participant.team.captain.name}</p>
                  </div>
                  <div>
                    <Label className="text-sm font-medium text-gray-500">Email</Label>
                    <p className="text-sm flex items-center">
                      <Mail className="h-4 w-4 mr-1" />
                      <a href={`mailto:${participant.team.captain.email}`} className="text-blue-600 hover:underline">
                        {participant.team.captain.email}
                      </a>
                    </p>
                  </div>
                  {participant.team.captain.phone && (
                    <div>
                      <Label className="text-sm font-medium text-gray-500">Phone</Label>
                      <p className="text-sm flex items-center">
                        <Phone className="h-4 w-4 mr-1" />
                        <a href={`tel:${participant.team.captain.phone}`} className="text-blue-600 hover:underline">
                          {participant.team.captain.phone}
                        </a>
                      </p>
                    </div>
                  )}
                </CardContent>
              </Card>
            </div>
          </TabsContent>

          <TabsContent value="players">
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center space-x-2">
                  <Users className="h-5 w-5 text-green-600" />
                  <span>Team Players</span>
                </CardTitle>
              </CardHeader>
              <CardContent>
                {participant.team.players.length === 0 ? (
                  <div className="text-center py-8 text-gray-500">
                    <Users className="h-12 w-12 mx-auto mb-4 text-gray-300" />
                    <p>No players found</p>
                  </div>
                ) : (
                  <div className="space-y-4">
                    {participant.team.players.map((player) => (
                      <div key={player.id} className="flex items-center justify-between p-4 border rounded-lg">
                        <div className="flex items-center space-x-4">
                          <div className="p-2 bg-green-100 rounded-lg">
                            <User className="h-5 w-5 text-green-600" />
                          </div>
                          <div>
                            <h3 className="font-semibold">{player.name}</h3>
                            <p className="text-sm text-gray-600">{player.email}</p>
                            {player.gaming_handle && (
                              <p className="text-sm text-gray-500">Handle: {player.gaming_handle}</p>
                            )}
                            {player.position && (
                              <p className="text-sm text-gray-500">Position: {player.position}</p>
                            )}
                          </div>
                        </div>
                        <div className="flex items-center space-x-2">
                          {player.phone && (
                            <a href={`tel:${player.phone}`} className="text-blue-600 hover:text-blue-800">
                              <Phone className="h-4 w-4" />
                            </a>
                          )}
                          <a href={`mailto:${player.email}`} className="text-blue-600 hover:text-blue-800">
                            <Mail className="h-4 w-4" />
                          </a>
                        </div>
                      </div>
                    ))}
                  </div>
                )}
              </CardContent>
            </Card>
          </TabsContent>

          <TabsContent value="officials">
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center space-x-2">
                  <Shield className="h-5 w-5 text-purple-600" />
                  <span>Team Officials</span>
                </CardTitle>
              </CardHeader>
              <CardContent>
                {participant.team.officials.length === 0 ? (
                  <div className="text-center py-8 text-gray-500">
                    <Shield className="h-12 w-12 mx-auto mb-4 text-gray-300" />
                    <p>No officials found</p>
                  </div>
                ) : (
                  <div className="space-y-4">
                    {participant.team.officials.map((official) => (
                      <div key={official.id} className="flex items-center justify-between p-4 border rounded-lg">
                        <div className="flex items-center space-x-4">
                          <div className="p-2 bg-purple-100 rounded-lg">
                            <Shield className="h-5 w-5 text-purple-600" />
                          </div>
                          <div>
                            <h3 className="font-semibold">{official.name}</h3>
                            <p className="text-sm text-gray-600">{official.email}</p>
                            <p className="text-sm text-gray-500">Role: {official.role}</p>
                          </div>
                        </div>
                        <div className="flex items-center space-x-2">
                          {official.phone && (
                            <a href={`tel:${official.phone}`} className="text-blue-600 hover:text-blue-800">
                              <Phone className="h-4 w-4" />
                            </a>
                          )}
                          <a href={`mailto:${official.email}`} className="text-blue-600 hover:text-blue-800">
                            <Mail className="h-4 w-4" />
                          </a>
                        </div>
                      </div>
                    ))}
                  </div>
                )}
              </CardContent>
            </Card>
          </TabsContent>

          <TabsContent value="tournament">
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center space-x-2">
                  <Trophy className="h-5 w-5 text-yellow-600" />
                  <span>Tournament Specific Data</span>
                </CardTitle>
              </CardHeader>
              <CardContent>
                {participant.pivot.tournament_specific_data ? (
                  <div className="bg-gray-50 rounded-lg p-4">
                    <pre className="text-sm">{JSON.stringify(participant.pivot.tournament_specific_data, null, 2)}</pre>
                  </div>
                ) : (
                  <div className="text-center py-8 text-gray-500">
                    <Trophy className="h-12 w-12 mx-auto mb-4 text-gray-300" />
                    <p>No tournament-specific data available</p>
                  </div>
                )}
              </CardContent>
            </Card>
          </TabsContent>
        </Tabs>

        {/* Info Alert */}
        <Alert className="mt-6">
          <Info className="h-4 w-4" />
          <AlertDescription>
            <strong>Participant Management:</strong> You can change the participant's status, view team details, 
            and manage their tournament participation. All changes will be logged and participants will be notified.
          </AlertDescription>
        </Alert>
      </div>
    </AppLayout>
  );
};

export default ParticipantDetail; 