import React, { useState } from 'react';
import { Head, router } from '@inertiajs/react';
import AppLayout from '@/layouts/app-layout';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Badge } from '@/components/ui/badge';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { 
  Trophy, 
  Search, 
  ArrowLeft,
  Play,
  Pause,
  CheckCircle,
  XCircle,
  Clock,
  Calendar,
  Users,
  Eye,
  Edit,
  Trash2,
  MoreHorizontal,
  Download,
  Upload,
  Plus,
  Info,
  Target,
  Award,
  Timer
} from 'lucide-react';
import { 
  DropdownMenu, 
  DropdownMenuContent, 
  DropdownMenuItem, 
  DropdownMenuTrigger,
  DropdownMenuSeparator
} from '@/components/ui/dropdown-menu';
import { Alert, AlertDescription } from '@/components/ui/alert';
import { Progress } from '@/components/ui/progress';
import { Label } from '@/components/ui/label';

interface Match {
  id: number;
  tournament_id: number;
  match_type: string;
  status: string;
  scheduled_at: string;
  started_at?: string;
  completed_at?: string;
  duration?: number;
  venue?: string;
  team1_id: number;
  team2_id: number;
  team1_score?: number;
  team2_score?: number;
  winner_id?: number;
  notes?: string;
  team1: {
    id: number;
    name: string;
  };
  team2: {
    id: number;
    name: string;
  };
  winner?: {
    id: number;
    name: string;
  };
}

interface Tournament {
  id: number;
  name: string;
  description: string;
  status: string;
  start_date: string;
  end_date: string;
}

interface Props {
  tournament: Tournament;
  matches: Match[];
  filters: {
    search: string;
    status: string;
    match_type: string;
  };
  stats: {
    total: number;
    scheduled: number;
    in_progress: number;
    completed: number;
    cancelled: number;
  };
}

const TournamentMatches: React.FC<Props> = ({ 
  tournament, 
  matches, 
  filters,
  stats 
}) => {
  const [localFilters, setLocalFilters] = useState({
    search: filters?.search || '',
    status: filters?.status || 'all',
    match_type: filters?.match_type || 'all',
  });
  const [selectedMatches, setSelectedMatches] = useState<number[]>([]);
  const [activeTab, setActiveTab] = useState('all');

  const handleFilterChange = (key: string, value: string) => {
    const newFilters = { ...localFilters, [key]: value };
    setLocalFilters(newFilters);
    router.get(route('admin.tournaments.matches', tournament.id), newFilters, {
      preserveState: true,
      preserveScroll: true,
    });
  };

  const handleMatchSelection = (matchId: number, checked: boolean) => {
    if (checked) {
      setSelectedMatches([...selectedMatches, matchId]);
    } else {
      setSelectedMatches(selectedMatches.filter(id => id !== matchId));
    }
  };

  const handleSelectAll = (checked: boolean) => {
    if (checked) {
      setSelectedMatches(matches.map(m => m.id));
    } else {
      setSelectedMatches([]);
    }
  };

  const handleBulkAction = (action: string) => {
    if (selectedMatches.length === 0) return;

    const data = {
      match_ids: selectedMatches,
      action: action,
    };

    router.post(route('admin.tournaments.matches.bulk-action', tournament.id), data, {
      onSuccess: () => {
        setSelectedMatches([]);
      },
    });
  };

  const handleStatusChange = (matchId: number, newStatus: string) => {
    router.put(route('admin.tournaments.matches.update-status', { 
      tournament: tournament.id, 
      match: matchId 
    }), {
      status: newStatus,
    });
  };

  const getStatusColor = (status: string) => {
    const colors = {
      'scheduled': 'bg-blue-100 text-blue-800',
      'in_progress': 'bg-yellow-100 text-yellow-800',
      'completed': 'bg-green-100 text-green-800',
      'cancelled': 'bg-red-100 text-red-800',
    };
    return colors[status as keyof typeof colors] || 'bg-gray-100 text-gray-800';
  };

  const getMatchTypeColor = (type: string) => {
    const colors = {
      'group': 'bg-purple-100 text-purple-800',
      'knockout': 'bg-orange-100 text-orange-800',
      'semi': 'bg-pink-100 text-pink-800',
      'final': 'bg-yellow-100 text-yellow-800',
      'consolation': 'bg-gray-100 text-gray-800',
    };
    return colors[type as keyof typeof colors] || 'bg-gray-100 text-gray-800';
  };

  const formatDate = (dateString: string) => {
    if (!dateString) return 'Not set';
    return new Date(dateString).toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric',
      hour: '2-digit',
      minute: '2-digit'
    });
  };

  const formatDuration = (minutes: number) => {
    if (!minutes) return 'N/A';
    const hours = Math.floor(minutes / 60);
    const mins = minutes % 60;
    return hours > 0 ? `${hours}h ${mins}m` : `${mins}m`;
  };

  const filteredMatches = matches.filter(match => {
    const matchesSearch = 
      match.team1.name.toLowerCase().includes(localFilters.search.toLowerCase()) ||
      match.team2.name.toLowerCase().includes(localFilters.search.toLowerCase());
    
    const matchesStatus = localFilters.status === 'all' || match.status === localFilters.status;
    const matchesType = localFilters.match_type === 'all' || match.match_type === localFilters.match_type;
    
    return matchesSearch && matchesStatus && matchesType;
  });

  const getMatchesByStatus = (status: string) => {
    if (status === 'all') return filteredMatches;
    return filteredMatches.filter(m => m.status === status);
  };

  const progressPercentage = stats.total > 0 
    ? (stats.completed / stats.total) * 100 
    : 0;

  return (
    <AppLayout>
      <Head title={`Matches - ${tournament.name}`} />
      
      <div className="p-4">
        {/* Header */}
        <div className="flex items-center justify-between mb-6">
          <div className="flex items-center space-x-4">
            <Button variant="outline" size="sm" asChild>
              <a href={route('admin.tournaments.show', tournament.id)}>
                <ArrowLeft className="h-4 w-4 mr-2" />
                Back to Tournament
              </a>
            </Button>
            
          </div>
          <div className="flex space-x-2">
            <Button variant="outline" size="sm">
              <Download className="h-4 w-4 mr-2" />
              Export
            </Button>
            <Button variant="outline" size="sm">
              <Upload className="h-4 w-4 mr-2" />
              Import
            </Button>
            <Button size="sm">
              <Plus className="h-4 w-4 mr-2" />
              Create Match
            </Button>
          </div>
        </div>

        {/* Tournament Info Card */}
        <Card className="mb-6">
          <CardHeader>
            <CardTitle className="flex items-center space-x-2">
              <Trophy className="h-5 w-5 text-yellow-600" />
              <span>{tournament.name}</span>
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
              <div>
                <Label className="text-sm font-medium text-gray-500">Status</Label>
                <Badge className={getStatusColor(tournament.status)}>
                  {tournament.status}
                </Badge>
              </div>
              <div>
                <Label className="text-sm font-medium text-gray-500">Period</Label>
                <p className="text-sm font-medium">
                  {formatDate(tournament.start_date)} - {formatDate(tournament.end_date)}
                </p>
              </div>
              <div>
                <Label className="text-sm font-medium text-gray-500">Total Matches</Label>
                <p className="text-lg font-semibold">{stats.total}</p>
              </div>
              <div>
                <Label className="text-sm font-medium text-gray-500">Completed</Label>
                <p className="text-lg font-semibold text-green-600">
                  {stats.completed} / {stats.total}
                </p>
              </div>
            </div>
            
            {/* Progress Bar */}
            <div className="mt-4">
              <div className="flex items-center justify-between mb-2">
                <span className="text-sm font-medium">Tournament Progress</span>
                <span className="text-sm text-gray-500">{Math.round(progressPercentage)}%</span>
              </div>
              <Progress value={progressPercentage} className="h-2" />
            </div>
          </CardContent>
        </Card>

        {/* Stats Cards */}
        <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-5 gap-4 mb-6">
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center">
                <div className="p-2 bg-blue-100 rounded-lg">
                  <Trophy className="w-5 h-5 text-blue-600" />
                </div>
                <div className="ml-3">
                  <p className="text-sm font-medium text-gray-500">Total</p>
                  <p className="text-xl font-bold">{stats.total}</p>
                </div>
              </div>
            </CardContent>
          </Card>
          
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center">
                <div className="p-2 bg-blue-100 rounded-lg">
                  <Clock className="w-5 h-5 text-blue-600" />
                </div>
                <div className="ml-3">
                  <p className="text-sm font-medium text-gray-500">Scheduled</p>
                  <p className="text-xl font-bold">{stats.scheduled}</p>
                </div>
              </div>
            </CardContent>
          </Card>
          
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center">
                <div className="p-2 bg-yellow-100 rounded-lg">
                  <Play className="w-5 h-5 text-yellow-600" />
                </div>
                <div className="ml-3">
                  <p className="text-sm font-medium text-gray-500">In Progress</p>
                  <p className="text-xl font-bold">{stats.in_progress}</p>
                </div>
              </div>
            </CardContent>
          </Card>
          
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center">
                <div className="p-2 bg-green-100 rounded-lg">
                  <CheckCircle className="w-5 h-5 text-green-600" />
                </div>
                <div className="ml-3">
                  <p className="text-sm font-medium text-gray-500">Completed</p>
                  <p className="text-xl font-bold">{stats.completed}</p>
                </div>
              </div>
            </CardContent>
          </Card>
          
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center">
                <div className="p-2 bg-red-100 rounded-lg">
                  <XCircle className="w-5 h-5 text-red-600" />
                </div>
                <div className="ml-3">
                  <p className="text-sm font-medium text-gray-500">Cancelled</p>
                  <p className="text-xl font-bold">{stats.cancelled}</p>
                </div>
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Filters */}
        <Card className="mb-6">
          <CardContent className="p-4">
            <div className="flex flex-col sm:flex-row gap-4">
              <div className="flex-1">
                <div className="relative">
                  <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 h-4 w-4 text-gray-400" />
                  <Input
                    placeholder="Search teams..."
                    value={localFilters.search}
                    onChange={(e) => handleFilterChange('search', e.target.value)}
                    className="pl-10"
                  />
                </div>
              </div>
              <Select value={localFilters.status} onValueChange={(value) => handleFilterChange('status', value)}>
                <SelectTrigger className="w-48">
                  <SelectValue placeholder="All Status" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Status</SelectItem>
                  <SelectItem value="scheduled">Scheduled</SelectItem>
                  <SelectItem value="in_progress">In Progress</SelectItem>
                  <SelectItem value="completed">Completed</SelectItem>
                  <SelectItem value="cancelled">Cancelled</SelectItem>
                </SelectContent>
              </Select>
              <Select value={localFilters.match_type} onValueChange={(value) => handleFilterChange('match_type', value)}>
                <SelectTrigger className="w-48">
                  <SelectValue placeholder="All Types" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Types</SelectItem>
                  <SelectItem value="group">Group Stage</SelectItem>
                  <SelectItem value="knockout">Knockout</SelectItem>
                  <SelectItem value="semi">Semi Final</SelectItem>
                  <SelectItem value="final">Final</SelectItem>
                  <SelectItem value="consolation">Consolation</SelectItem>
                </SelectContent>
              </Select>
            </div>
          </CardContent>
        </Card>

        {/* Bulk Actions */}
        {selectedMatches.length > 0 && (
          <div className="mb-4 p-4 bg-blue-50 border border-blue-200 rounded-lg">
            <div className="flex items-center justify-between">
              <span className="text-blue-800">
                {selectedMatches.length} match(es) selected
              </span>
              <div className="flex space-x-2">
                <Button 
                  variant="outline" 
                  size="sm" 
                  onClick={() => handleBulkAction('start')}
                >
                  <Play className="h-4 w-4 mr-1" />
                  Start
                </Button>
                <Button 
                  variant="outline" 
                  size="sm" 
                  onClick={() => handleBulkAction('complete')}
                >
                  <CheckCircle className="h-4 w-4 mr-1" />
                  Complete
                </Button>
                <Button 
                  variant="outline" 
                  size="sm" 
                  onClick={() => handleBulkAction('cancel')}
                >
                  <XCircle className="h-4 w-4 mr-1" />
                  Cancel
                </Button>
              </div>
            </div>
          </div>
        )}

        {/* Matches Tabs */}
        <Tabs value={activeTab} onValueChange={setActiveTab} className="space-y-4">
          <TabsList className="grid w-full grid-cols-5">
            <TabsTrigger value="all">All ({filteredMatches.length})</TabsTrigger>
            <TabsTrigger value="scheduled">Scheduled ({getMatchesByStatus('scheduled').length})</TabsTrigger>
            <TabsTrigger value="in_progress">In Progress ({getMatchesByStatus('in_progress').length})</TabsTrigger>
            <TabsTrigger value="completed">Completed ({getMatchesByStatus('completed').length})</TabsTrigger>
            <TabsTrigger value="cancelled">Cancelled ({getMatchesByStatus('cancelled').length})</TabsTrigger>
          </TabsList>

          <TabsContent value={activeTab}>
            <Card>
              <CardContent className="p-0">
                {filteredMatches.length === 0 ? (
                  <div className="p-8 text-center text-gray-500">
                    <Trophy className="h-12 w-12 mx-auto mb-4 text-gray-300" />
                    <p>No matches found matching your criteria.</p>
                  </div>
                ) : (
                  <div className="divide-y">
                    {/* Select All Header */}
                    <div className="p-4 bg-gray-50 border-b">
                      <div className="flex items-center space-x-3">
                        <input
                          type="checkbox"
                          checked={selectedMatches.length === filteredMatches.length && filteredMatches.length > 0}
                          onChange={(e) => handleSelectAll(e.target.checked)}
                          className="rounded border-gray-300"
                        />
                        <span className="font-medium">Select All</span>
                      </div>
                    </div>

                    {/* Matches List */}
                    {filteredMatches.map((match) => (
                      <div key={match.id} className="p-4 hover:bg-gray-50">
                        <div className="flex items-center space-x-4">
                          <input
                            type="checkbox"
                            checked={selectedMatches.includes(match.id)}
                            onChange={(e) => handleMatchSelection(match.id, e.target.checked)}
                            className="rounded border-gray-300"
                          />
                          <div className="flex-1">
                            <div className="flex items-center justify-between">
                              <div>
                                <h3 className="font-semibold">
                                  {match.team1.name} vs {match.team2.name}
                                </h3>
                                <p className="text-sm text-gray-600">
                                  {match.match_type.replace('_', ' ').toUpperCase()} • {match.venue || 'TBD'}
                                </p>
                                <div className="flex items-center space-x-2 mt-1">
                                  <span className="text-sm text-gray-500">
                                    {formatDate(match.scheduled_at)}
                                  </span>
                                  {match.duration && (
                                    <span className="text-sm text-gray-500">
                                      • Duration: {formatDuration(match.duration)}
                                    </span>
                                  )}
                                  {match.team1_score !== null && match.team2_score !== null && (
                                    <span className="text-sm font-medium">
                                      • Score: {match.team1_score} - {match.team2_score}
                                    </span>
                                  )}
                                </div>
                              </div>
                              <div className="flex items-center space-x-2">
                                <Badge className={getStatusColor(match.status)}>
                                  {match.status.replace('_', ' ')}
                                </Badge>
                                <Badge className={getMatchTypeColor(match.match_type)}>
                                  {match.match_type}
                                </Badge>
                                <DropdownMenu>
                                  <DropdownMenuTrigger asChild>
                                    <Button variant="ghost" size="sm">
                                      <MoreHorizontal className="h-4 w-4" />
                                    </Button>
                                  </DropdownMenuTrigger>
                                  <DropdownMenuContent align="end">
                                    <DropdownMenuItem onClick={() => router.get(route('admin.tournaments.matches.show', { 
                                      tournament: tournament.id, 
                                      match: match.id 
                                    }))}>
                                      <Eye className="w-4 h-4 mr-2" />
                                      View Details
                                    </DropdownMenuItem>
                                    <DropdownMenuItem onClick={() => router.get(route('admin.matches.edit', { 
                                      tournament: tournament.id, 
                                      match: match.id 
                                    }))}>
                                      <Edit className="w-4 h-4 mr-2" />
                                      Edit Match
                                    </DropdownMenuItem>
                                    <DropdownMenuSeparator />
                                    <DropdownMenuItem onClick={() => handleStatusChange(match.id, 'in_progress')}>
                                      <Play className="w-4 h-4 mr-2" />
                                      Start Match
                                    </DropdownMenuItem>
                                    <DropdownMenuItem onClick={() => handleStatusChange(match.id, 'completed')}>
                                      <CheckCircle className="w-4 h-4 mr-2" />
                                      Complete
                                    </DropdownMenuItem>
                                    <DropdownMenuItem onClick={() => handleStatusChange(match.id, 'cancelled')}>
                                      <XCircle className="w-4 h-4 mr-2" />
                                      Cancel
                                    </DropdownMenuItem>
                                    <DropdownMenuSeparator />
                                    <DropdownMenuItem 
                                      onClick={() => router.delete(route('admin.tournaments.matches.destroy', { 
                                        tournament: tournament.id, 
                                        match: match.id 
                                      }))}
                                      className="text-red-600"
                                    >
                                      <Trash2 className="w-4 h-4 mr-2" />
                                      Delete
                                    </DropdownMenuItem>
                                  </DropdownMenuContent>
                                </DropdownMenu>
                              </div>
                            </div>
                            
                            {/* Match Details */}
                            <div className="mt-3 text-sm text-gray-500">
                              <div className="flex items-center space-x-4">
                                {match.started_at && (
                                  <span>Started: {formatDate(match.started_at)}</span>
                                )}
                                {match.completed_at && (
                                  <span>Completed: {formatDate(match.completed_at)}</span>
                                )}
                                {match.winner && (
                                  <span>Winner: {match.winner.name}</span>
                                )}
                                {match.notes && (
                                  <span>Notes: {match.notes}</span>
                                )}
                              </div>
                            </div>
                          </div>
                        </div>
                      </div>
                    ))}
                  </div>
                )}
              </CardContent>
            </Card>
          </TabsContent>
        </Tabs>

        {/* Info Alert */}
        <Alert className="mt-6">
          <Info className="h-4 w-4" />
          <AlertDescription>
            <strong>Match Management:</strong> You can start, complete, or cancel matches. 
            Match results will be automatically updated in the tournament brackets. 
            Use bulk actions to manage multiple matches at once.
          </AlertDescription>
        </Alert>
      </div>
    </AppLayout>
  );
};

export default TournamentMatches; 