import AppLayout from '@/layouts/app-layout';
import { Head, router } from '@inertiajs/react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Badge } from '@/components/ui/badge';
import { CategoryBadge } from '@/components/ui/category-badge';
import { useState } from 'react';
import { 
  Plus, 
  Search, 
  Filter, 
  Trophy, 
  Calendar, 
  Users, 
  Eye, 
  Edit, 
  Trash2, 
  MoreHorizontal,
  TrendingUp,
  Award,
  Clock,
  MapPin,
  Play,
  Target
} from 'lucide-react';
import { 
  DropdownMenu, 
  DropdownMenuContent, 
  DropdownMenuItem, 
  DropdownMenuTrigger,
  DropdownMenuSeparator
} from '@/components/ui/dropdown-menu';
import { usePermissions } from '@/hooks/use-permissions';

interface Tournament {
  id: number;
  name: string;
  slug: string;
  description: string;
  type: string;
  status: string;
  format: string;
  start_date: string;
  end_date: string;
  total_participants: number;
  max_participants: number;
  entry_fee: number;
  prize_pool: number;
  location: string;
  visibility: string;
  created_at: string;
  enable_categories?: boolean;
  category_fees?: Record<string, number>;
  available_categories?: string[];
}

interface TournamentsPageProps {
  tournaments: {
    data: Tournament[];
    current_page: number;
    last_page: number;
    per_page: number;
    total: number;
  };
  filters: {
    search: string;
    status: string;
    type: string;
    format: string;
  };
}

export default function AdminTournaments({ tournaments, filters }: TournamentsPageProps) {
  const { hasPermission } = usePermissions();
  const [localFilters, setLocalFilters] = useState({
    search: filters?.search || '',
    status: filters?.status || 'all',
    type: filters?.type || 'all',
    format: filters?.format || 'all',
  });

  const handleFilterChange = (key: string, value: string) => {
    const newFilters = { ...localFilters, [key]: value };
    setLocalFilters(newFilters);
    router.get(route('admin.tournaments.index'), newFilters, {
      preserveState: true,
      preserveScroll: true,
    });
  };

  const formatCurrency = (amount: number, currency: string) => {
    return new Intl.NumberFormat('en-US', {
      style: 'currency',
      currency: currency || 'KES',
      minimumFractionDigits: 0,
      maximumFractionDigits: 0,
    }).format(amount);
  };

  const getStatusColor = (status: string) => {
    const colors = {
      'draft': 'bg-gray-100 text-gray-800',
      'active': 'bg-green-100 text-green-800',
      'completed': 'bg-blue-100 text-blue-800',
      'cancelled': 'bg-red-100 text-red-800',
    };
    return colors[status as keyof typeof colors] || 'bg-gray-100 text-gray-800';
  };

  const getTypeColor = (type: string) => {
    const colors = {
      'single_elimination': 'bg-orange-100 text-orange-800',
      'double_elimination': 'bg-purple-100 text-purple-800',
      'round_robin': 'bg-teal-100 text-teal-800',
    };
    return colors[type as keyof typeof colors] || 'bg-gray-100 text-gray-800';
  };

  const formatDate = (dateString: string) => {
    if (!dateString) return 'Not set';
    return new Date(dateString).toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric',
    });
  };

  const handleDelete = (tournament: Tournament) => {
    if (confirm(`Are you sure you want to delete "${tournament.name}"?`)) {
      router.delete(route('admin.tournaments.destroy', tournament.id));
    }
  };

  return (
    <AppLayout>
      <Head title="Tournaments" />
      
      <div className="space-y-4 sm:space-y-6 p-4">
        {/* Header */}
        <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-3 sm:gap-4">
          {hasPermission('tournaments.create') && (
          <Button 
            onClick={() => router.visit(route('admin.tournaments.create'))}
            className="bg-primary hover:bg-primary/90 w-full sm:w-auto"
          >
            <Plus className="w-4 h-4 mr-2" />
            Create Tournament
          </Button>
          )}
        </div>

        {/* Stats Cards */}
        <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-4 sm:gap-6">
          <Card>
            <CardContent className="p-4 sm:p-6">
              <div className="flex items-center">
                <div className="p-2 bg-blue-100 rounded-lg">
                  <Trophy className="w-5 h-5 sm:w-6 sm:h-6 text-blue-600" />
                </div>
                <div className="ml-3 sm:ml-4">
                  <p className="text-xs sm:text-sm font-medium text-muted-foreground">Total Tournaments</p>
                  <p className="text-xl sm:text-2xl font-bold text-foreground">{tournaments.total || 0}</p>
                </div>
              </div>
            </CardContent>
          </Card>
          
          <Card>
            <CardContent className="p-4 sm:p-6">
              <div className="flex items-center">
                <div className="p-2 bg-green-100 rounded-lg">
                  <TrendingUp className="w-5 h-5 sm:w-6 sm:h-6 text-green-600" />
                </div>
                <div className="ml-3 sm:ml-4">
                  <p className="text-xs sm:text-sm font-medium text-muted-foreground">Active</p>
                  <p className="text-xl sm:text-2xl font-bold text-foreground">
                    {tournaments.data.filter(t => t.status === 'active').length || 0}
                  </p>
                </div>
              </div>
            </CardContent>
          </Card>
          
          <Card>
            <CardContent className="p-4 sm:p-6">
              <div className="flex items-center">
                <div className="p-2 bg-purple-100 rounded-lg">
                  <Users className="w-5 h-5 sm:w-6 sm:h-6 text-purple-600" />
                </div>
                <div className="ml-3 sm:ml-4">
                  <p className="text-xs sm:text-sm font-medium text-muted-foreground">Total Participants</p>
                  <p className="text-xl sm:text-2xl font-bold text-foreground">
                    {tournaments.data.reduce((sum, t) => sum + (t.total_participants) || 0, 0)}
                  </p>
                </div>
              </div>
            </CardContent>
          </Card>
          
          <Card>
            <CardContent className="p-4 sm:p-6">
              <div className="flex items-center">
                <div className="p-2 bg-yellow-100 rounded-lg">
                  <Award className="w-5 h-5 sm:w-6 sm:h-6 text-yellow-600" />
                </div>
                <div className="ml-3 sm:ml-4">
                  <p className="text-xs sm:text-sm font-medium text-muted-foreground">Prize Pool</p>
                  <p className="text-xl sm:text-2xl font-bold text-foreground">
                    {formatCurrency(tournaments.data.reduce((sum, t) => sum + (t.prize_pool) || 0, 0), 'KES')}
                  </p>
                </div>
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Filters */}
        <Card>
          <CardContent className="p-4 sm:p-6">
            <div className="flex flex-col gap-3 sm:flex-row sm:gap-4">
              <div className="flex-1">
                <div className="relative">
                  <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-muted-foreground w-4 h-4" />
                  <Input
                    placeholder="Search tournaments..."
                    value={localFilters.search}
                    onChange={(e) => handleFilterChange('search', e.target.value)}
                    className="pl-9"
                  />
                </div>
              </div>
              
              <Select value={localFilters.status} onValueChange={(value) => handleFilterChange('status', value)}>
                <SelectTrigger className="w-full sm:w-[180px]">
                  <SelectValue placeholder="All Status" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Status</SelectItem>
                  <SelectItem value="draft">Draft</SelectItem>
                  <SelectItem value="active">Active</SelectItem>
                  <SelectItem value="completed">Completed</SelectItem>
                  <SelectItem value="cancelled">Cancelled</SelectItem>
                </SelectContent>
              </Select>
              
              <Select value={localFilters.type} onValueChange={(value) => handleFilterChange('type', value)}>
                <SelectTrigger className="w-full sm:w-[180px]">
                  <SelectValue placeholder="All Types" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Types</SelectItem>
                  <SelectItem value="single_elimination">Single Elimination</SelectItem>
                  <SelectItem value="double_elimination">Double Elimination</SelectItem>
                  <SelectItem value="round_robin">Round Robin</SelectItem>
                </SelectContent>
              </Select>
              
              <Select value={localFilters.format} onValueChange={(value) => handleFilterChange('format', value)}>
                <SelectTrigger className="w-full sm:w-[180px]">
                  <SelectValue placeholder="All Formats" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Formats</SelectItem>
                  <SelectItem value="individual">Individual</SelectItem>
                  <SelectItem value="team">Team</SelectItem>
                  <SelectItem value="mixed">Mixed</SelectItem>
                </SelectContent>
              </Select>
            </div>
          </CardContent>
        </Card>

        {/* Tournaments Grid */}
        <div className="grid grid-cols-1 lg:grid-cols-2 xl:grid-cols-3 gap-4 sm:gap-6">
          {tournaments.data.map((tournament) => (
            <Card key={tournament.id} className="hover:shadow-lg transition-shadow">
              <CardHeader className="pb-3">
                <div className="flex items-start justify-between">
                  <div className="flex-1 min-w-0">
                    <CardTitle className="text-base sm:text-lg font-semibold mb-2 truncate">
                      {tournament.name}
                    </CardTitle>
                    <div className="flex flex-wrap gap-1 sm:gap-2 mb-2">
                      <Badge className={`${getStatusColor(tournament.status)} text-xs`}>
                        {tournament.status}
                      </Badge>
                      <Badge className={`${getTypeColor(tournament.type)} text-xs`}>
                        {tournament.type.replace('_', ' ')}
                      </Badge>
                    </div>
                  </div>
                  <DropdownMenu>
                    <DropdownMenuTrigger asChild>
                      <Button variant="ghost" size="sm" className="flex-shrink-0">
                        <MoreHorizontal className="w-4 h-4" />
                      </Button>
                    </DropdownMenuTrigger>
                    <DropdownMenuContent align="end">
                      <DropdownMenuItem onClick={() => router.get(route('admin.tournaments.show', tournament.id))}>
                        <Eye className="w-4 h-4 mr-2" />
                        View
                      </DropdownMenuItem>
                      {hasPermission('tournaments.edit') && (
                      <DropdownMenuItem onClick={() => router.get(route('admin.tournaments.edit', tournament.id))}>
                        <Edit className="w-4 h-4 mr-2" />
                        Edit
                      </DropdownMenuItem>
                      )}
                      {hasPermission('tournaments.register-teams') && (
                      <DropdownMenuItem onClick={() => router.get(route('admin.tournaments.register-teams', tournament.id))}>
                        <Users className="w-4 h-4 mr-2" />
                        Register Teams
                      </DropdownMenuItem>
                      )}
                      {hasPermission('tournaments.participants') && (
                      <DropdownMenuItem onClick={() => router.get(route('admin.tournaments.participants', tournament.id))}>
                        <Users className="w-4 h-4 mr-2" />
                        Manage Participants
                      </DropdownMenuItem>
                      )}
                      {hasPermission('tournaments.matches') && (
                      <DropdownMenuItem onClick={() => router.get(route('admin.tournaments.matches', tournament.id))}>
                        <Play className="w-4 h-4 mr-2" />
                        View Matches
                      </DropdownMenuItem>
                      )}
                      {hasPermission('tournaments.brackets') && (
                      <DropdownMenuItem onClick={() => router.get(route('admin.tournaments.brackets', tournament.id))}>
                        <Target className="w-4 h-4 mr-2" />
                        View Brackets
                      </DropdownMenuItem>
                      )}
                      {hasPermission('tournaments.statistics') && (
                      <DropdownMenuItem onClick={() => router.get(route('admin.tournaments.statistics', tournament.id))}>
                        <TrendingUp className="w-4 h-4 mr-2" />
                        Statistics
                      </DropdownMenuItem>
                      )}
                      {hasPermission('tournaments.delete') && (
                      <DropdownMenuSeparator />
                      )}
                      {hasPermission('tournaments.delete') && (
                      <DropdownMenuItem 
                        onClick={() => handleDelete(tournament)}
                        className="text-red-600"
                      >
                        <Trash2 className="w-4 h-4 mr-2" />
                        Delete
                      </DropdownMenuItem>
                      )}
                    </DropdownMenuContent>
                  </DropdownMenu>
                </div>
              </CardHeader>
              
              <CardContent className="pt-0">
                <div className="space-y-3">
                  {tournament.description && (
                    <p className="text-xs sm:text-sm text-muted-foreground line-clamp-2">
                      {tournament.description}
                    </p>
                  )}
                  
                  <div className="flex flex-col sm:flex-row sm:items-center gap-2 sm:gap-4 text-xs sm:text-sm text-muted-foreground">
                    <div className="flex items-center gap-1">
                      <Calendar className="w-3 h-3 sm:w-4 sm:h-4" />
                      {formatDate(tournament.start_date)}
                    </div>
                    {tournament.location && (
                      <div className="flex items-center gap-1">
                        <MapPin className="w-3 h-3 sm:w-4 sm:h-4" />
                        <span className="truncate">{tournament.location}</span>
                      </div>
                    )}
                  </div>
                  
                  <div className="flex items-center justify-between text-xs sm:text-sm">
                    <div className="flex items-center gap-1 text-muted-foreground">
                      <Users className="w-3 h-3 sm:w-4 sm:h-4" />
                      {tournament.total_participants || 0} / {tournament.max_participants || 'Unlimited'}
                    </div>
                    {tournament.prize_pool > 0 && (
                      <div className="font-medium text-green-600 text-xs sm:text-sm">
                        {formatCurrency(tournament.prize_pool, 'KES')}
                      </div>
                    )}
                  </div>

                  {/* Category Fees Section */}
                  {tournament.enable_categories && (
                    <div className="border-t pt-3 mt-1">
                      <div className="flex items-center gap-2 mb-2">
                        <Trophy className="w-3 h-3 text-blue-600" />
                        <span className="text-xs font-semibold text-gray-700">Category Fees:</span>
                      </div>
                      <div className="grid grid-cols-2 gap-2">
                        {tournament.category_fees?.men !== undefined && (
                          <div className="flex items-center justify-between text-xs bg-blue-50 px-2 py-1 rounded">
                            <span className="flex items-center gap-1">
                              <span>👨</span>
                              <span className="font-medium">Men</span>
                            </span>
                            <span className="font-bold">KES {tournament.category_fees.men.toLocaleString()}</span>
                          </div>
                        )}
                        {tournament.category_fees?.women !== undefined && (
                          <div className="flex items-center justify-between text-xs bg-pink-50 px-2 py-1 rounded">
                            <span className="flex items-center gap-1">
                              <span>👩</span>
                              <span className="font-medium">Women</span>
                            </span>
                            <span className="font-bold">KES {tournament.category_fees.women.toLocaleString()}</span>
                          </div>
                        )}
                        {tournament.category_fees?.youth_male !== undefined && tournament.category_fees.youth_male !== null && (
                          <div className="flex items-center justify-between text-xs bg-purple-50 px-2 py-1 rounded">
                            <span className="flex items-center gap-1">
                              <span>🧑</span>
                              <span className="font-medium">Youth M</span>
                            </span>
                            <span className="font-bold">{tournament.category_fees.youth_male > 0 ? `KES ${tournament.category_fees.youth_male.toLocaleString()}` : 'FREE'}</span>
                          </div>
                        )}
                        {tournament.category_fees?.youth_female !== undefined && tournament.category_fees.youth_female !== null && (
                          <div className="flex items-center justify-between text-xs bg-purple-100 px-2 py-1 rounded">
                            <span className="flex items-center gap-1">
                              <span>👧</span>
                              <span className="font-medium">Youth F</span>
                            </span>
                            <span className="font-bold">{tournament.category_fees.youth_female > 0 ? `KES ${tournament.category_fees.youth_female.toLocaleString()}` : 'FREE'}</span>
                          </div>
                        )}
                        {tournament.category_fees?.children !== undefined && tournament.category_fees.children !== null && (
                          <div className="flex items-center justify-between text-xs bg-green-50 px-2 py-1 rounded">
                            <span className="flex items-center gap-1">
                              <span>🧒</span>
                              <span className="font-medium">Children</span>
                            </span>
                            <span className="font-bold">{tournament.category_fees.children > 0 ? `KES ${tournament.category_fees.children.toLocaleString()}` : 'FREE'}</span>
                          </div>
                        )}
                      </div>
                    </div>
                  )}
                  
                  <div className="flex gap-2 pt-2">
                    {hasPermission('tournaments.view') && (
                    <Button 
                      variant="outline" 
                      size="sm" 
                      className="flex-1 text-xs"
                      onClick={() => router.get(route('admin.tournaments.show', tournament.id))}
                    >
                      <Eye className="w-3 h-3 sm:w-4 sm:h-4 mr-1" />
                      View
                    </Button>
                    )}
                    {hasPermission('tournaments.edit') && (
                    <Button 
                      variant="outline" 
                      size="sm" 
                      className="flex-1 text-xs"
                      onClick={() => router.get(route('admin.tournaments.edit', tournament.id))}
                    >
                      <Edit className="w-3 h-3 sm:w-4 sm:h-4 mr-1" />
                      Edit
                    </Button>
                    )}
                    {hasPermission('tournaments.delete') && (
                    <Button 
                      variant="outline" 
                      size="sm" 
                      className="flex-1 text-xs"
                      onClick={() => handleDelete(tournament)}
                    >
                      <Trash2 className="w-3 h-3 sm:w-4 sm:h-4 mr-1" />
                      Delete
                    </Button>
                    )}
                  </div>
                </div>
              </CardContent>
            </Card>
          ))}
        </div>

        {/* Empty State */}
        {tournaments.data.length === 0 && (
          <Card>
            <CardContent className="p-8 sm:p-12 text-center">
              <Trophy className="w-8 h-8 sm:w-12 sm:h-12 text-muted-foreground mx-auto mb-4" />
              <h3 className="text-base sm:text-lg font-semibold text-foreground mb-2">No tournaments found</h3>
              <p className="text-sm sm:text-base text-muted-foreground mb-4">
                {localFilters.search || localFilters.status !== 'all' || localFilters.type !== 'all' 
                  ? 'Try adjusting your filters to see more results.'
                  : 'Get started by creating your first tournament.'
                }
              </p>
              {!localFilters.search && localFilters.status === 'all' && localFilters.type === 'all' && (
                <Button onClick={() => router.get(route('admin.tournaments.create'))} className="w-full sm:w-auto">
                  <Plus className="w-4 h-4 mr-2" />
                  Create Tournament
                </Button>
              )}
            </CardContent>
          </Card>
        )}

        {/* Pagination */}
        {tournaments.last_page > 1 && (
          <Card>
            <CardContent className="p-4">
              <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
                <div className="text-xs sm:text-sm text-muted-foreground text-center sm:text-left">
                  Showing {((tournaments.current_page - 1) * tournaments.per_page) + 1} to{' '}
                  {Math.min(tournaments.current_page * tournaments.per_page, tournaments.total)} of{' '}
                  {tournaments.total} results
                </div>
                <div className="flex items-center justify-center sm:justify-end gap-2">
                  <Button
                    variant="outline"
                    size="sm"
                    disabled={tournaments.current_page === 1}
                    onClick={() => handleFilterChange('page', String(tournaments.current_page - 1))}
                    className="text-xs"
                  >
                    Previous
                  </Button>
                  <span className="text-xs sm:text-sm">
                    Page {tournaments.current_page} of {tournaments.last_page}
                  </span>
                  <Button
                    variant="outline"
                    size="sm"
                    disabled={tournaments.current_page === tournaments.last_page}
                    onClick={() => handleFilterChange('page', String(tournaments.current_page + 1))}
                    className="text-xs"
                  >
                    Next
                  </Button>
                </div>
              </div>
            </CardContent>
          </Card>
        )}
      </div>
    </AppLayout>
  );
}