import React, { useState } from 'react';
import { Head, router, useForm } from '@inertiajs/react';
import AppLayout from '@/layouts/app-layout';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Switch } from '@/components/ui/switch';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Badge } from '@/components/ui/badge';
import { 
  ArrowLeft, 
  Save, 
  Calendar, 
  MapPin, 
  DollarSign, 
  Settings, 
  Globe, 
  Mail, 
  Phone,
  Trophy,
  Users,
  Image,
  AlertCircle,
  Info
} from 'lucide-react';
import { Alert, AlertDescription } from '@/components/ui/alert';

interface Tournament {
  id: number;
  name: string;
  slug: string;
  description: string;
  type: string;
  status: string;
  format: string;
  start_date: string;
  end_date: string;
  registration_start: string;
  registration_end: string;
  location: string;
  venue: string;
  organizer: string;
  contact_email: string;
  contact_phone: string;
  website: string;
  max_participants: number;
  min_participants: number;
  entry_fee: number;
  prize_pool: number;
  visibility: string;
  auto_accept_registrations: boolean;
  registration_status: string;
  allow_spectators: boolean;
  require_approval: boolean;
  enable_two_tier_approval: boolean;
  first_approver_id: number;
  second_approver_id: number;
  first_approver: {
    id: number;
    name: string;
    email: string;
  };
  second_approver: {
    id: number;
    name: string;
    email: string;
  };
  is_featured: boolean;
  banner_image: string;
  logo_image: string;
  announcement: string;
  enable_categories?: boolean;
  category_fees?: Record<string, number>;
  available_categories?: string[];
  teams_count?: number;
}

interface EditTournamentProps {
  tournament: Tournament;
  firstApprovers: Array<{
    id: number;
    name: string;
    email: string;
  }>;
  secondApprovers: Array<{
    id: number;
    name: string;
    email: string;
  }>;
}

export default function EditTournament({ tournament, firstApprovers, secondApprovers }: EditTournamentProps) {
  const [currentTab, setCurrentTab] = useState('basic');
  const [isSubmitting, setIsSubmitting] = useState(false);



  const { data, setData, post, processing, errors, reset } = useForm({
    name: tournament.name || '',
    slug: tournament.slug || '',
    description: tournament.description || '',
    type: tournament.type || 'single_elimination',
    status: tournament.status || 'draft',
    format: tournament.format || 'team',
    start_date: tournament.start_date ? tournament.start_date.split('T')[0] : '',
    end_date: tournament.end_date ? tournament.end_date.split('T')[0] : '',
    registration_start: tournament.registration_start ? tournament.registration_start.split('T')[0] : '',
    registration_end: tournament.registration_end ? tournament.registration_end.split('T')[0] : '',
    location: tournament.location || '',
    venue: tournament.venue || '',
    organizer: tournament.organizer || '',
    contact_email: tournament.contact_email || '',
    contact_phone: tournament.contact_phone || '',
    website: tournament.website || '',
    max_participants: tournament.max_participants || '',
    min_participants: tournament.min_participants || '',
    min_players_per_team: tournament.min_players_per_team || '',
    max_players_per_team: tournament.max_players_per_team || '',
    entry_fee: tournament.entry_fee || '',
    prize_pool: tournament.prize_pool || '',
    visibility: tournament.visibility || 'public',
    auto_accept_registrations: tournament.auto_accept_registrations ?? true,
    registration_status: tournament.registration_status || 'open',
    allow_spectators: tournament.allow_spectators ?? true,
    require_approval: tournament.require_approval ?? false,
    enable_two_tier_approval: tournament.enable_two_tier_approval ?? false,
    first_approver_id: tournament.first_approver_id || null,
    second_approver_id: tournament.second_approver_id || null,
    is_featured: tournament.is_featured ?? false,
    banner_image: tournament.banner_image || '',
    logo_image: tournament.logo_image || '',
    announcement: tournament.announcement || '',
    enable_categories: tournament.enable_categories ?? false,
    category_fees: tournament.category_fees ?? null,
    available_categories: tournament.available_categories ?? null,
  });

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    setIsSubmitting(true);

    router.put(route('admin.tournaments.update', tournament.id), data, {
      onSuccess: () => {
        setIsSubmitting(false);
      },
      onError: (errors) => {
        setIsSubmitting(false);
      }
    });
  };

  const formatDateForInput = (dateString: string) => {
    if (!dateString) return '';
    return dateString.split('T')[0];
  };

  return (
    <AppLayout breadcrumbs={[
      { title: 'Tournaments', href: '/admin/tournaments' },
      { title: tournament.name, href: `/admin/tournaments/${tournament.id}` },
      { title: 'Edit', href: `/admin/tournaments/${tournament.id}/edit` }
    ]}>
      <Head title={`Edit Tournament - ${tournament.name}`} />
      
      <div className="space-y-4 sm:space-y-6 p-4">
        {/* Header */}
        <div className="flex flex-col lg:flex-row lg:items-center lg:justify-between gap-4">
          <div className="flex flex-col sm:flex-row sm:items-center gap-3 sm:gap-4">
            <Button 
              variant="outline" 
              size="sm"
              onClick={() => router.get(route('admin.tournaments.show', tournament.id))}
              className="w-full sm:w-auto"
            >
              <ArrowLeft className="w-4 h-4 mr-2" />
              Back to Tournament
            </Button>
            
          </div>
        </div>

        <form onSubmit={handleSubmit} className="space-y-4 sm:space-y-6">
          <Tabs value={currentTab} onValueChange={setCurrentTab} className="space-y-4 sm:space-y-6">
            <TabsList className="grid w-full grid-cols-2 sm:grid-cols-3 lg:grid-cols-6">
              <TabsTrigger value="basic" className="text-xs sm:text-sm">Basic Info</TabsTrigger>
              <TabsTrigger value="schedule" className="text-xs sm:text-sm">Schedule</TabsTrigger>
              <TabsTrigger value="participants" className="text-xs sm:text-sm">Participants</TabsTrigger>
              <TabsTrigger value="categories" className="text-xs sm:text-sm">
                <Trophy className="w-3 h-3 mr-1 hidden sm:inline" />
                Categories
              </TabsTrigger>
              <TabsTrigger value="settings" className="text-xs sm:text-sm">Settings</TabsTrigger>
              <TabsTrigger value="media" className="text-xs sm:text-sm">Media</TabsTrigger>
            </TabsList>

            {/* Basic Information Tab */}
            <TabsContent value="basic" className="space-y-4 sm:space-y-6">
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2 text-sm sm:text-base">
                    <Trophy className="w-4 h-4 sm:w-5 sm:h-5" />
                    Tournament Details
                  </CardTitle>
                </CardHeader>
                <CardContent className="space-y-3 sm:space-y-4">
                  <div className="grid grid-cols-1 lg:grid-cols-2 gap-3 sm:gap-4">
                    <div className="space-y-2">
                      <Label htmlFor="name">Tournament Name *</Label>
                      <Input
                        id="name"
                        value={data.name}
                        onChange={(e) => setData('name', e.target.value)}
                        placeholder="Enter tournament name"
                        className={errors.name ? 'border-red-500' : ''}
                      />
                      {errors.name && (
                        <p className="text-sm text-red-600 flex items-center gap-1">
                          <AlertCircle className="w-4 h-4" />
                          {errors.name}
                        </p>
                      )}
                    </div>

                    <div className="space-y-2">
                      <Label htmlFor="slug">URL Slug</Label>
                      <Input
                        id="slug"
                        value={data.slug}
                        onChange={(e) => setData('slug', e.target.value)}
                        placeholder="tournament-slug"
                        className={errors.slug ? 'border-red-500' : ''}
                      />
                      {errors.slug && (
                        <p className="text-sm text-red-600 flex items-center gap-1">
                          <AlertCircle className="w-4 h-4" />
                          {errors.slug}
                        </p>
                      )}
                    </div>
                  </div>

                  <div className="space-y-2">
                    <Label htmlFor="description" className="text-sm">Description</Label>
                    <Textarea
                      id="description"
                      value={data.description}
                      onChange={(e) => setData('description', e.target.value)}
                      placeholder="Tournament description"
                      rows={3}
                      className="text-sm"
                    />
                  </div>

                  <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-3 sm:gap-4">
                    <div className="space-y-2">
                      <Label htmlFor="type">Tournament Type *</Label>
                      <Select value={data.type} onValueChange={(value) => setData('type', value)}>
                        <SelectTrigger className={errors.type ? 'border-red-500' : ''}>
                          <SelectValue placeholder="Select type" />
                        </SelectTrigger>
                        <SelectContent>
                          <SelectItem value="single_elimination">Single Elimination</SelectItem>
                          <SelectItem value="double_elimination">Double Elimination</SelectItem>
                          <SelectItem value="round_robin">Round Robin</SelectItem>
                        </SelectContent>
                      </Select>
                      {errors.type && (
                        <p className="text-sm text-red-600">{errors.type}</p>
                      )}
                    </div>

                    <div className="space-y-2">
                      <Label htmlFor="format">Format *</Label>
                      <Select value={data.format} onValueChange={(value) => setData('format', value)}>
                        <SelectTrigger className={errors.format ? 'border-red-500' : ''}>
                          <SelectValue placeholder="Select format" />
                        </SelectTrigger>
                        <SelectContent>
                          <SelectItem value="individual">Individual</SelectItem>
                          <SelectItem value="team">Team</SelectItem>
                          <SelectItem value="mixed">Mixed</SelectItem>
                        </SelectContent>
                      </Select>
                      {errors.format && (
                        <p className="text-sm text-red-600">{errors.format}</p>
                      )}
                    </div>

                    <div className="space-y-2">
                      <Label htmlFor="status">Status *</Label>
                      <Select value={data.status} onValueChange={(value) => setData('status', value)}>
                        <SelectTrigger className={errors.status ? 'border-red-500' : ''}>
                          <SelectValue placeholder="Select status" />
                        </SelectTrigger>
                        <SelectContent>
                          <SelectItem value="draft">Draft</SelectItem>
                          <SelectItem value="active">Active</SelectItem>
                          <SelectItem value="completed">Completed</SelectItem>
                          <SelectItem value="cancelled">Cancelled</SelectItem>
                        </SelectContent>
                      </Select>
                      {errors.status && (
                        <p className="text-sm text-red-600">{errors.status}</p>
                      )}
                    </div>
                  </div>

                  <div className="grid grid-cols-1 lg:grid-cols-2 gap-4">
                    <div className="space-y-2">
                      <Label htmlFor="location">Location</Label>
                      <div className="relative">
                        <MapPin className="absolute left-3 top-1/2 transform -translate-y-1/2 text-muted-foreground w-4 h-4" />
                        <Input
                          id="location"
                          value={data.location}
                          onChange={(e) => setData('location', e.target.value)}
                          placeholder="City, Country"
                          className="pl-9"
                        />
                      </div>
                    </div>

                    <div className="space-y-2">
                      <Label htmlFor="venue">Venue</Label>
                      <Input
                        id="venue"
                        value={data.venue}
                        onChange={(e) => setData('venue', e.target.value)}
                        placeholder="Specific venue name"
                      />
                    </div>
                  </div>
                </CardContent>
              </Card>

              {/* Contact Information */}
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <Mail className="w-5 h-5" />
                    Contact Information
                  </CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div className="grid grid-cols-1 lg:grid-cols-2 gap-4">
                    <div className="space-y-2">
                      <Label htmlFor="organizer">Organizer</Label>
                      <Input
                        id="organizer"
                        value={data.organizer}
                        onChange={(e) => setData('organizer', e.target.value)}
                        placeholder="Organization or person name"
                      />
                    </div>

                    <div className="space-y-2">
                      <Label htmlFor="contact_email">Contact Email</Label>
                      <div className="relative">
                        <Mail className="absolute left-3 top-1/2 transform -translate-y-1/2 text-muted-foreground w-4 h-4" />
                        <Input
                          id="contact_email"
                          type="email"
                          value={data.contact_email}
                          onChange={(e) => setData('contact_email', e.target.value)}
                          placeholder="contact@example.com"
                          className="pl-9"
                        />
                      </div>
                    </div>

                    <div className="space-y-2">
                      <Label htmlFor="contact_phone">Contact Phone</Label>
                      <div className="relative">
                        <Phone className="absolute left-3 top-1/2 transform -translate-y-1/2 text-muted-foreground w-4 h-4" />
                        <Input
                          id="contact_phone"
                          value={data.contact_phone}
                          onChange={(e) => setData('contact_phone', e.target.value)}
                          placeholder="07XX XXX XXX"
                          className="pl-9"
                        />
                      </div>
                    </div>

                    <div className="space-y-2">
                      <Label htmlFor="website">Website</Label>
                      <div className="relative">
                        <Globe className="absolute left-3 top-1/2 transform -translate-y-1/2 text-muted-foreground w-4 h-4" />
                        <Input
                          id="website"
                          type="url"
                          value={data.website}
                          onChange={(e) => setData('website', e.target.value)}
                          placeholder="https://example.com"
                          className="pl-9"
                        />
                      </div>
                    </div>
                  </div>
                </CardContent>
              </Card>
            </TabsContent>

            {/* Schedule Tab */}
            <TabsContent value="schedule" className="space-y-6">
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <Calendar className="w-5 h-5" />
                    Tournament Schedule
                  </CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div className="grid grid-cols-1 lg:grid-cols-2 gap-4">
                    <div className="space-y-2">
                      <Label htmlFor="start_date">Start Date *</Label>
                      <Input
                        id="start_date"
                        type="date"
                        value={data.start_date}
                        onChange={(e) => setData('start_date', e.target.value)}
                        className={errors.start_date ? 'border-red-500' : ''}
                      />
                      {errors.start_date && (
                        <p className="text-sm text-red-600">{errors.start_date}</p>
                      )}
                    </div>

                    <div className="space-y-2">
                      <Label htmlFor="end_date">End Date *</Label>
                      <Input
                        id="end_date"
                        type="date"
                        value={data.end_date}
                        onChange={(e) => setData('end_date', e.target.value)}
                        className={errors.end_date ? 'border-red-500' : ''}
                      />
                      {errors.end_date && (
                        <p className="text-sm text-red-600">{errors.end_date}</p>
                      )}
                    </div>

                    <div className="space-y-2">
                      <Label htmlFor="registration_start">Registration Start</Label>
                      <Input
                        id="registration_start"
                        type="date"
                        value={data.registration_start}
                        onChange={(e) => setData('registration_start', e.target.value)}
                      />
                    </div>

                    <div className="space-y-2">
                      <Label htmlFor="registration_end">Registration End</Label>
                      <Input
                        id="registration_end"
                        type="date"
                        value={data.registration_end}
                        onChange={(e) => setData('registration_end', e.target.value)}
                      />
                    </div>
                  </div>
                </CardContent>
              </Card>
            </TabsContent>

            {/* Participants Tab */}
            <TabsContent value="participants" className="space-y-6">
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <Users className="w-5 h-5" />
                    Participant Settings
                  </CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                  {/* Number of Teams Section */}
                  <div className="mb-6">
                    <h3 className="text-base font-semibold mb-3 flex items-center gap-2">
                      <Trophy className="w-4 h-4" />
                      Tournament Capacity (Number of Teams)
                    </h3>
                    <div className="grid grid-cols-1 lg:grid-cols-2 gap-4">
                      <div className="space-y-2">
                        <Label htmlFor="min_participants">
                          Minimum Teams *
                          <span className="text-xs text-gray-500 font-normal ml-2">
                            (Required to start)
                          </span>
                        </Label>
                        <Input
                          id="min_participants"
                          type="number"
                          value={data.min_participants}
                          onChange={(e) => setData('min_participants', e.target.value)}
                          placeholder="8"
                          min="2"
                          max="1000"
                          className={errors.min_participants ? 'border-red-500' : ''}
                        />
                        {errors.min_participants && (
                          <p className="text-red-500 text-sm">{errors.min_participants}</p>
                        )}
                        <p className="text-xs text-gray-500">
                          Minimum number of teams to start tournament
                        </p>
                      </div>

                      <div className="space-y-2">
                        <Label htmlFor="max_participants">
                          Maximum Teams *
                          <span className="text-xs text-gray-500 font-normal ml-2">
                            (Tournament capacity)
                          </span>
                        </Label>
                        <Input
                          id="max_participants"
                          type="number"
                          value={data.max_participants}
                          onChange={(e) => setData('max_participants', e.target.value)}
                          placeholder="64"
                          min={data.min_participants || "2"}
                          max="1000"
                          className={errors.max_participants ? 'border-red-500' : ''}
                        />
                        {errors.max_participants && (
                          <p className="text-red-500 text-sm">{errors.max_participants}</p>
                        )}
                        <p className="text-xs text-gray-500">
                          Maximum teams allowed in tournament
                        </p>
                      </div>
                    </div>
                  </div>

                  {/* Players per Team Section */}
                  <div className="grid grid-cols-1 lg:grid-cols-2 gap-4">
                    <div className="space-y-2">
                      <Label htmlFor="min_players_per_team">
                        Minimum Players per Team *
                        <span className="text-xs text-gray-500 font-normal ml-2">
                          (Each team must have)
                        </span>
                      </Label>
                      <Input
                        id="min_players_per_team"
                        type="number"
                        value={data.min_players_per_team}
                        onChange={(e) => setData('min_players_per_team', e.target.value)}
                        placeholder="11"
                        min="1"
                        max="100"
                        className={errors.min_players_per_team ? 'border-red-500' : ''}
                      />
                      {errors.min_players_per_team && (
                        <p className="text-red-500 text-sm">{errors.min_players_per_team}</p>
                      )}
                      <p className="text-xs text-gray-500">
                        💡 Football: 11 players minimum (1 GK + 10 field)
                      </p>
                    </div>

                    <div className="space-y-2">
                      <Label htmlFor="max_players_per_team">
                        Maximum Players per Team *
                        <span className="text-xs text-gray-500 font-normal ml-2">
                          (Squad limit)
                        </span>
                      </Label>
                      <Input
                        id="max_players_per_team"
                        type="number"
                        value={data.max_players_per_team}
                        onChange={(e) => setData('max_players_per_team', e.target.value)}
                        placeholder="23"
                        min={data.min_players_per_team || "1"}
                        max="100"
                        className={errors.max_players_per_team ? 'border-red-500' : ''}
                      />
                      {errors.max_players_per_team && (
                        <p className="text-red-500 text-sm">{errors.max_players_per_team}</p>
                      )}
                      <p className="text-xs text-gray-500">
                        💡 Football: 23 players max (11 starting + 12 subs)
                      </p>
                    </div>

                    <div className="space-y-2">
                      <Label htmlFor="entry_fee">Entry Fee (KES)</Label>
                      <div className="relative">
                        <DollarSign className="absolute left-3 top-1/2 transform -translate-y-1/2 text-muted-foreground w-4 h-4" />
                        <Input
                          id="entry_fee"
                          type="number"
                          value={data.entry_fee}
                          onChange={(e) => setData('entry_fee', e.target.value)}
                          placeholder="0"
                          min="0"
                          step="0.01"
                          className="pl-9"
                        />
                      </div>
                    </div>

                    <div className="space-y-2">
                      <Label htmlFor="prize_pool">Prize Pool (KES)</Label>
                      <div className="relative">
                        <DollarSign className="absolute left-3 top-1/2 transform -translate-y-1/2 text-muted-foreground w-4 h-4" />
                        <Input
                          id="prize_pool"
                          type="number"
                          value={data.prize_pool}
                          onChange={(e) => setData('prize_pool', e.target.value)}
                          placeholder="0"
                          min="0"
                          step="0.01"
                          className="pl-9"
                        />
                      </div>
                    </div>
                  </div>

                  <div className="space-y-2">
                    <Label htmlFor="registration_status">Registration Status</Label>
                    <Select value={data.registration_status} onValueChange={(value) => setData('registration_status', value)}>
                      <SelectTrigger>
                        <SelectValue placeholder="Select registration status" />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="open">Open</SelectItem>
                        <SelectItem value="closed">Closed</SelectItem>
                        <SelectItem value="full">Full</SelectItem>
                      </SelectContent>
                    </Select>
                  </div>
                </CardContent>
              </Card>
            </TabsContent>

            {/* Settings Tab */}
            <TabsContent value="settings" className="space-y-6">
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <Settings className="w-5 h-5" />
                    Tournament Settings
                  </CardTitle>
                </CardHeader>
                <CardContent className="space-y-6">
                  <div className="space-y-2">
                    <Label htmlFor="visibility">Visibility</Label>
                    <Select value={data.visibility} onValueChange={(value) => setData('visibility', value)}>
                      <SelectTrigger>
                        <SelectValue placeholder="Select visibility" />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="public">Public</SelectItem>
                        <SelectItem value="private">Private</SelectItem>
                        <SelectItem value="unlisted">Unlisted</SelectItem>
                      </SelectContent>
                    </Select>
                  </div>

                  <div className="space-y-4">
                    <div className="flex items-center justify-between">
                      <div>
                        <Label htmlFor="auto_accept_registrations">Auto Accept Registrations</Label>
                        <p className="text-sm text-muted-foreground">
                          Automatically accept participant registrations
                        </p>
                      </div>
                      <Switch
                        id="auto_accept_registrations"
                        checked={data.auto_accept_registrations}
                        onCheckedChange={(checked) => setData('auto_accept_registrations', checked)}
                      />
                    </div>

                    <div className="flex items-center justify-between">
                      <div>
                        <Label htmlFor="allow_spectators">Allow Spectators</Label>
                        <p className="text-sm text-muted-foreground">
                          Allow spectators to view the tournament
                        </p>
                      </div>
                      <Switch
                        id="allow_spectators"
                        checked={data.allow_spectators}
                        onCheckedChange={(checked) => setData('allow_spectators', checked)}
                      />
                    </div>

                    <div className="flex items-center justify-between">
                      <div>
                        <Label htmlFor="require_approval">Require Approval</Label>
                        <p className="text-sm text-muted-foreground">
                          Require manual approval for participants
                        </p>
                      </div>
                      <Switch
                        id="require_approval"
                        checked={data.require_approval}
                        onCheckedChange={(checked) => setData('require_approval', checked)}
                      />
                    </div>

                    <div className="flex items-center justify-between">
                      <div>
                        <Label htmlFor="enable_two_tier_approval">Two-Tier Approval</Label>
                        <p className="text-sm text-muted-foreground">
                          Enable two-tier approval system with first and second approvers
                        </p>
                      </div>
                      <Switch
                        id="enable_two_tier_approval"
                        checked={data.enable_two_tier_approval}
                        onCheckedChange={(checked) => setData('enable_two_tier_approval', checked)}
                      />
                    </div>

                    {/* Two-tier approval settings */}
                    {data.enable_two_tier_approval && (
                      <div className="space-y-4 p-4 bg-gray-50 rounded-lg">                       
                        <div className="space-y-2">
                          <Label htmlFor="first_approver_id">First Approver</Label>
                          <Input
                            id="first_approver_id"
                            value={
                              firstApprovers.length > 0 
                                ? `${firstApprovers[0].name} (${firstApprovers[0].email})`
                                : 'Not configured'
                            }
                            disabled
                            className="bg-gray-100 cursor-not-allowed"
                          />
                        </div>

                        <div className="space-y-2">
                          <Label htmlFor="second_approver_id">Second Approver</Label>
                          <Input
                            id="second_approver_id"
                            value={
                              secondApprovers.length > 0 
                                ? `${secondApprovers[0].name} (${secondApprovers[0].email})`
                                : 'Not configured'
                            }
                            disabled
                            className="bg-gray-100 cursor-not-allowed"
                          />
                          
                        </div>
                      </div>
                    )}

                    <div className="flex items-center justify-between">
                      <div>
                        <Label htmlFor="is_featured">Featured Tournament</Label>
                        <p className="text-sm text-muted-foreground">
                          Display this tournament prominently
                        </p>
                      </div>
                      <Switch
                        id="is_featured"
                        checked={data.is_featured}
                        onCheckedChange={(checked) => setData('is_featured', checked)}
                      />
                    </div>
                  </div>
                </CardContent>
              </Card>

              <Card>
                <CardHeader>
                  <CardTitle>Announcement</CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="space-y-2">
                    <Label htmlFor="announcement">Tournament Announcement</Label>
                    <Textarea
                      id="announcement"
                      value={data.announcement}
                      onChange={(e) => setData('announcement', e.target.value)}
                      placeholder="Important announcements for participants..."
                      rows={4}
                    />
                  </div>
                </CardContent>
              </Card>
            </TabsContent>

            {/* Categories Tab */}
            <TabsContent value="categories" className="space-y-6">
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <Trophy className="w-5 h-5" />
                    Team Categories & Fees
                  </CardTitle>
                </CardHeader>
                <CardContent className="space-y-6">
                  <div className="flex items-center justify-between mb-4">
                    <div>
                      <h3 className="text-lg font-semibold">Team Categories</h3>
                      <p className="text-sm text-muted-foreground">Configure different categories with separate registration fees</p>
                    </div>
                    <Switch
                      checked={data.enable_categories || false}
                      onCheckedChange={(checked) => {
                        setData('enable_categories', checked);
                        if (!checked) {
                          setData('category_fees', null);
                          setData('available_categories', null);
                        } else {
                          // Initialize with default fees
                          setData('category_fees', {
                            men: 1000,
                            women: 800,
                            youth_male: 500,
                            youth_female: 500,
                            children: 300,
                          });
                          setData('available_categories', ['men', 'women', 'youth_male', 'youth_female', 'children']);
                        }
                      }}
                    />
                  </div>

                  {data.enable_categories && (
                    <div className="space-y-4">
                      <Alert>
                        <Info className="h-4 w-4" />
                        <AlertDescription>
                          Enable multiple team categories with different fees. Teams will select their category during registration.
                        </AlertDescription>
                      </Alert>

                      {/* Category Selection & Fees Combined */}
                      <div>
                        <Label className="text-base font-medium mb-3 block">Configure Categories & Fees</Label>
                        <div className="space-y-3">
                          {[
                            { id: 'men', name: 'Men', icon: '👨', desc: '18+ years, Male', defaultFee: 1000 },
                            { id: 'women', name: 'Women', icon: '👩', desc: '18+ years, Female', defaultFee: 800 },
                            { id: 'youth_male', name: 'Youth Male', icon: '🧑', desc: '13-17 years, Male', defaultFee: 500 },
                            { id: 'youth_female', name: 'Youth Female', icon: '👧', desc: '13-17 years, Female', defaultFee: 500 },
                            { id: 'children', name: 'Children', icon: '🧒', desc: 'Under 13, Any gender', defaultFee: 300 },
                          ].map((category) => {
                            const isEnabled = data.available_categories?.includes(category.id) || false;
                            return (
                              <div
                                key={category.id}
                                className={`p-4 border rounded-md transition-colors ${
                                  isEnabled ? 'bg-accent border-primary' : 'bg-gray-50'
                                }`}
                              >
                                <div className="flex items-center justify-between mb-3">
                                  <div className="flex items-center gap-2">
                                    <span className="text-xl">{category.icon}</span>
                                    <div>
                                      <div className="font-medium">{category.name}</div>
                                      <p className="text-xs text-muted-foreground">{category.desc}</p>
                                    </div>
                                  </div>
                                  <Switch
                                    checked={isEnabled}
                                    onCheckedChange={(checked) => {
                                      const currentCategories = data.available_categories || [];
                                      const newCategories = checked
                                        ? [...currentCategories, category.id]
                                        : currentCategories.filter((c: string) => c !== category.id);
                                      setData('available_categories', newCategories);
                                    }}
                                  />
                                </div>
                                
                                {isEnabled && (
                                  <div>
                                    <Label htmlFor={`fee_${category.id}`} className="text-sm">
                                      Registration Fee (KES)
                                    </Label>
                                    <div className="relative mt-1">
                                      <span className="absolute left-3 top-1/2 transform -translate-y-1/2 text-muted-foreground text-sm">
                                        KES
                                      </span>
                                      <Input
                                        id={`fee_${category.id}`}
                                        type="number"
                                        min="0"
                                        step="0.01"
                                        value={data.category_fees?.[category.id] || category.defaultFee}
                                        onChange={(e) => {
                                          const newFees = { ...(data.category_fees || {}) };
                                          newFees[category.id] = parseFloat(e.target.value) || 0;
                                          setData('category_fees', newFees);
                                        }}
                                        className="pl-16"
                                        placeholder={category.defaultFee.toString()}
                                      />
                                    </div>
                                    {errors?.[`category_fees.${category.id}`] && (
                                      <p className="text-red-500 text-sm mt-1">{errors[`category_fees.${category.id}`]}</p>
                                    )}
                                  </div>
                                )}
                              </div>
                            );
                          })}
                        </div>
                        <p className="text-xs text-muted-foreground mt-2">
                          Enable categories and set their registration fees. Only enabled categories will be available during team registration.
                        </p>
                      </div>
                    </div>
                  )}

                  {!data.enable_categories && (
                    <p className="text-sm text-muted-foreground">
                      Categories are disabled. All teams will use the default entry fee. Enable categories to configure different fees for men, women, youth, and children divisions.
                    </p>
                  )}
                </CardContent>
              </Card>
            </TabsContent>

            {/* Media Tab */}
            <TabsContent value="media" className="space-y-6">
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <Image className="w-5 h-5" />
                    Tournament Media
                  </CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div className="grid grid-cols-1 lg:grid-cols-2 gap-4">
                    <div className="space-y-2">
                      <Label htmlFor="banner_image">Banner Image URL</Label>
                      <Input
                        id="banner_image"
                        type="url"
                        value={data.banner_image}
                        onChange={(e) => setData('banner_image', e.target.value)}
                        placeholder="https://example.com/banner.jpg"
                      />
                    </div>

                    <div className="space-y-2">
                      <Label htmlFor="logo_image">Logo Image URL</Label>
                      <Input
                        id="logo_image"
                        type="url"
                        value={data.logo_image}
                        onChange={(e) => setData('logo_image', e.target.value)}
                        placeholder="https://example.com/logo.jpg"
                      />
                    </div>
                  </div>

                  {/* Image Previews */}
                  <div className="grid grid-cols-1 lg:grid-cols-2 gap-4">
                    {data.banner_image && (
                      <div>
                        <Label>Banner Preview</Label>
                        <img
                          src={data.banner_image}
                          alt="Banner preview"
                          className="w-full h-32 object-cover rounded-lg border mt-2"
                          onError={(e) => {
                            e.currentTarget.style.display = 'none';
                          }}
                        />
                      </div>
                    )}

                    {data.logo_image && (
                      <div>
                        <Label>Logo Preview</Label>
                        <img
                          src={data.logo_image}
                          alt="Logo preview"
                          className="w-24 h-24 object-contain rounded-lg border mt-2"
                          onError={(e) => {
                            e.currentTarget.style.display = 'none';
                          }}
                        />
                      </div>
                    )}
                  </div>
                </CardContent>
              </Card>
            </TabsContent>
          </Tabs>

          {/* Submit Actions */}
          <Card>
            <CardContent className="p-4 sm:p-6">
              <div className="flex flex-col sm:flex-row items-center justify-end gap-3 sm:gap-4">
                <Button
                  type="button"
                  variant="outline"
                  onClick={() => router.get(route('admin.tournaments.show', tournament.id))}
                  disabled={isSubmitting}
                  className="w-full sm:w-auto"
                >
                  Cancel
                </Button>
                <Button 
                  type="submit" 
                  disabled={isSubmitting}
                  className="bg-green-600 hover:bg-green-700 w-full sm:w-auto"
                >
                  <Save className="w-4 h-4 mr-2" />
                  {isSubmitting ? 'Updating...' : 'Update Tournament'}
                </Button>
              </div>
            </CardContent>
          </Card>
        </form>
      </div>
    </AppLayout>
  );
}