import React, { useState } from 'react';
import AppLayout from '@/layouts/app-layout';
import { Head, router } from '@inertiajs/react';
import TournamentFormSteps from '@/components/tournaments/tournament-form-steps';
import BasicInfoStep from '@/components/tournaments/form-steps/basic-info-step';
import TournamentSetupStep from '@/components/tournaments/form-steps/tournament-setup-step';
import ScheduleParticipantsStep from '@/components/tournaments/form-steps/schedule-participants-step';
import MediaSettingsStep from '@/components/tournaments/form-steps/media-settings-step';
import { Trophy, Settings, Calendar, Video } from 'lucide-react';

export default function AdminTournamentCreate() {
  const [currentStep, setCurrentStep] = useState(0);
  const [isSubmitting, setIsSubmitting] = useState(false);
  
  const [form, setForm] = useState({
    name: '',
    description: '',
    location: '',
    venue: '',
    organizer: '',
    contact_email: '',
    contact_phone: '',
    website: '',
    type: '',
    format: '',
    status: 'draft',
    registration_status: 'open',
    game_title: '',
    game_version: '',
    team_size: '',
    rounds: '',
    matches_per_round: '',
    auto_accept_registrations: true,
    allow_spectators: true,
    require_approval: false,
    is_featured: false,
    start_date: '',
    end_date: '',
    registration_start: '',
    registration_end: '',
    min_participants: '',
    max_participants: '',
    min_players_per_team: '',
    max_players_per_team: '',
    entry_fee: '',
    currency: 'KES',
    prize_pool: '',
    announcement: '',
    is_streamed: false,
    stream_url: '',
    twitch_url: '',
    youtube_url: '',
    banner_image: '',
    logo_image: '',
    visibility: 'public',
    enable_categories: false,
    category_fees: null,
    available_categories: null,
  });

  const [errors, setErrors] = useState<any>({});

  const steps = [
    {
      id: 'basic',
      title: 'Basic Information',
      description: 'Tournament details and contact information',
      icon: <Trophy className="w-5 h-5" />,
    },
    {
      id: 'setup',
      title: 'Tournament Setup',
      description: 'Type, format, and game configuration',
      icon: <Settings className="w-5 h-5" />,
    },
    {
      id: 'schedule',
      title: 'Schedule & Participants',
      description: 'Dates, participants, and entry details',
      icon: <Calendar className="w-5 h-5" />,
    },
    {
      id: 'media',
      title: 'Media & Settings',
      description: 'Streaming, branding, and final review',
      icon: <Video className="w-5 h-5" />,
    },
  ];

  const handleInputChange = (field: string, value: any) => {
    setForm((prev: any) => ({ ...prev, [field]: value }));
    if (errors[field]) {
      setErrors((prev: any) => ({ ...prev, [field]: null }));
    }
  };

  const validateCurrentStep = () => {
    const stepErrors: any = {};
    
    switch (currentStep) {
      case 0: // Basic Information
        if (!form.name) stepErrors.name = 'Tournament name is required';
        if (!form.visibility) stepErrors.visibility = 'Visibility is required';
        if (form.contact_email && !/\S+@\S+\.\S+/.test(form.contact_email)) {
          stepErrors.contact_email = 'Please enter a valid email address';
        }
        if (form.website && !form.website.startsWith('http')) {
          stepErrors.website = 'Please enter a valid URL starting with http:// or https://';
        }
        break;
      case 1: // Tournament Setup
        if (!form.type) stepErrors.type = 'Tournament type is required';
        if (!form.format) stepErrors.format = 'Tournament format is required';
        if (!form.status) stepErrors.status = 'Tournament status is required';
        if (!form.registration_status) stepErrors.registration_status = 'Registration status is required';
        break;
      case 2: // Schedule & Participants
        if (!form.start_date) stepErrors.start_date = 'Start date is required';
        if (!form.end_date) stepErrors.end_date = 'End date is required';
        if (form.start_date && form.end_date && new Date(form.start_date) >= new Date(form.end_date)) {
          stepErrors.end_date = 'End date must be after start date';
        }
        if (form.registration_start && form.registration_end && new Date(form.registration_start) >= new Date(form.registration_end)) {
          stepErrors.registration_end = 'Registration end must be after registration start';
        }
        if (form.min_participants && form.max_participants && parseInt(form.min_participants) > parseInt(form.max_participants)) {
          stepErrors.max_participants = 'Maximum participants must be greater than minimum';
        }
        break;
      case 3: // Media & Settings - No required fields
        if (form.is_streamed) {
          if (form.stream_url && !form.stream_url.startsWith('http')) {
            stepErrors.stream_url = 'Please enter a valid URL starting with http:// or https://';
          }
          if (form.twitch_url && !form.twitch_url.startsWith('http')) {
            stepErrors.twitch_url = 'Please enter a valid URL starting with http:// or https://';
          }
          if (form.youtube_url && !form.youtube_url.startsWith('http')) {
            stepErrors.youtube_url = 'Please enter a valid URL starting with http:// or https://';
          }
        }
        break;
    }

    setErrors(stepErrors);
    return Object.keys(stepErrors).length === 0;
  };

  const handleNext = () => {
    if (validateCurrentStep()) {
      setCurrentStep(prev => Math.min(prev + 1, steps.length - 1));
    }
  };

  const handlePrevious = () => {
    setCurrentStep(prev => Math.max(prev - 1, 0));
  };

  const handleStepChange = (step: number) => {
    // Allow going back to completed steps or current step
    if (step <= currentStep) {
      setCurrentStep(step);
    }
  };

  const handleSubmit = () => {
    if (!validateCurrentStep()) return;

    setIsSubmitting(true);
    router.post(route('admin.tournaments.store'), form, {
      onError: (errors) => {
        setErrors(errors);
        setIsSubmitting(false);
        // Go back to the step with errors
        if (errors.name || errors.visibility || errors.contact_email || errors.website) setCurrentStep(0);
        else if (errors.type || errors.format || errors.status || errors.registration_status) setCurrentStep(1);
        else if (errors.start_date || errors.end_date || errors.registration_start || errors.registration_end) setCurrentStep(2);
      },
      onSuccess: () => {
        router.get(route('admin.tournaments.index'));
      },
      onFinish: () => {
        setIsSubmitting(false);
      }
    });
  };

  const renderCurrentStep = () => {
    switch (currentStep) {
      case 0:
        return (
          <BasicInfoStep
            form={form}
            errors={errors}
            onInputChange={handleInputChange}
          />
        );
      case 1:
        return (
          <TournamentSetupStep
            form={form}
            errors={errors}
            onInputChange={handleInputChange}
          />
        );
      case 2:
        return (
          <ScheduleParticipantsStep
            form={form}
            errors={errors}
            onInputChange={handleInputChange}
          />
        );
      case 3:
        return (
          <MediaSettingsStep
            form={form}
            errors={errors}
            onInputChange={handleInputChange}
          />
        );
      default:
        return null;
    }
  };

  return (
    <AppLayout breadcrumbs={[
      { title: 'Tournaments', href: '/admin/tournaments' },
      { title: 'Create Tournament', href: '/admin/tournaments/create' }
    ]}>
      <Head title="Create Tournament" />
      
      <div className="flex flex-col gap-6 p-4">
        <div className="flex justify-between items-center">
          
          <div className="flex gap-2">
            <button
              onClick={() => router.get(route('admin.tournaments.index'))}
              className="px-4 py-2 text-sm border rounded-md hover:bg-muted transition-colors"
            >
              Cancel
            </button>
          </div>
        </div>

        <TournamentFormSteps
          steps={steps}
          currentStep={currentStep}
          onStepChange={handleStepChange}
          onNext={handleNext}
          onPrevious={handlePrevious}
          onSubmit={handleSubmit}
          isSubmitting={isSubmitting}
          canProceed={true}
          isLastStep={currentStep === steps.length - 1}
        >
          {renderCurrentStep()}
        </TournamentFormSteps>
      </div>
    </AppLayout>
  );
} 