import React, { useState } from 'react';
import { Head, router } from '@inertiajs/react';
import AppLayout from '@/layouts/app-layout';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { 
  Trophy, 
  ArrowLeft,
  Play,
  CheckCircle,
  XCircle,
  Clock,
  Calendar,
  Users,
  Eye,
  Edit,
  Trash2,
  MoreHorizontal,
  Download,
  Upload,
  Plus,
  Info,
  Target,
  Award,
  Timer,
  Zap,
  RefreshCw
} from 'lucide-react';
import { 
  DropdownMenu, 
  DropdownMenuContent, 
  DropdownMenuItem, 
  DropdownMenuTrigger,
  DropdownMenuSeparator
} from '@/components/ui/dropdown-menu';
import { Alert, AlertDescription } from '@/components/ui/alert';
import { Progress } from '@/components/ui/progress';
import { Label } from '@/components/ui/label';

interface Bracket {
  id: number;
  tournament_id: number;
  name: string;
  type: string;
  status: string;
  created_at: string;
  updated_at: string;
  rounds: BracketRound[];
}

interface BracketRound {
  id: number;
  bracket_id: number;
  name: string;
  order: number;
  status: string;
  matches: BracketMatch[];
}

interface BracketMatch {
  id: number;
  round_id: number;
  match_number: number;
  team1_id?: number;
  team2_id?: number;
  winner_id?: number;
  status: string;
  scheduled_at?: string;
  team1: {
    id: number;
    name: string;
  } | null;
  team2: {
    id: number;
    name: string;
  } | null;
  winner: {
    id: number;
    name: string;
  } | null;
}

interface Tournament {
  id: number;
  name: string;
  description: string;
  status: string;
  type: string;
  start_date: string;
  end_date: string;
}

interface Props {
  tournament: Tournament;
  brackets: Bracket[];
  stats: {
    total_brackets: number;
    active_brackets: number;
    completed_brackets: number;
    total_matches: number;
    completed_matches: number;
  };
}

const TournamentBrackets: React.FC<Props> = ({ 
  tournament, 
  brackets, 
  stats 
}) => {
  const [selectedBracket, setSelectedBracket] = useState<Bracket | null>(null);
  const [activeTab, setActiveTab] = useState('overview');

  const getStatusColor = (status: string) => {
    const colors = {
      'draft': 'bg-gray-100 text-gray-800',
      'active': 'bg-green-100 text-green-800',
      'completed': 'bg-blue-100 text-blue-800',
      'cancelled': 'bg-red-100 text-red-800',
    };
    return colors[status as keyof typeof colors] || 'bg-gray-100 text-gray-800';
  };

  const getBracketTypeColor = (type: string) => {
    const colors = {
      'single_elimination': 'bg-orange-100 text-orange-800',
      'double_elimination': 'bg-purple-100 text-purple-800',
      'round_robin': 'bg-teal-100 text-teal-800',
      'group_stage': 'bg-pink-100 text-pink-800',
    };
    return colors[type as keyof typeof colors] || 'bg-gray-100 text-gray-800';
  };

  const getMatchStatusColor = (status: string) => {
    const colors = {
      'scheduled': 'bg-blue-100 text-blue-800',
      'in_progress': 'bg-yellow-100 text-yellow-800',
      'completed': 'bg-green-100 text-green-800',
      'cancelled': 'bg-red-100 text-red-800',
    };
    return colors[status as keyof typeof colors] || 'bg-gray-100 text-gray-800';
  };

  const formatDate = (dateString: string) => {
    if (!dateString) return 'Not set';
    return new Date(dateString).toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric',
      hour: '2-digit',
      minute: '2-digit'
    });
  };

  const progressPercentage = stats.total_matches > 0 
    ? (stats.completed_matches / stats.total_matches) * 100 
    : 0;

  const handleGenerateBracket = () => {
    router.post(route('admin.tournaments.brackets.generate', tournament.id));
  };

  const handleAutoAdvance = () => {
    router.post(route('admin.tournaments.brackets.auto-advance', tournament.id));
  };

  return (
    <AppLayout>
      <Head title={`Brackets - ${tournament.name}`} />
      
      <div className="p-4">
        {/* Header */}
        <div className="flex items-center justify-between mb-6">
          <div className="flex items-center space-x-4">
            <Button variant="outline" size="sm" asChild>
              <a href={route('admin.tournaments.show', tournament.id)}>
                <ArrowLeft className="h-4 w-4 mr-2" />
                Back to Tournament
              </a>
            </Button>
            
          </div>
          <div className="flex space-x-2">
            <Button variant="outline" size="sm">
              <Download className="h-4 w-4 mr-2" />
              Export
            </Button>
            <Button variant="outline" size="sm" onClick={handleAutoAdvance}>
              <Zap className="h-4 w-4 mr-2" />
              Auto Advance
            </Button>
            <Button size="sm" onClick={handleGenerateBracket}>
              <Plus className="h-4 w-4 mr-2" />
              Generate Bracket
            </Button>
          </div>
        </div>

        {/* Tournament Info Card */}
        <Card className="mb-6">
          <CardHeader>
            <CardTitle className="flex items-center space-x-2">
              <Trophy className="h-5 w-5 text-yellow-600" />
              <span>{tournament.name}</span>
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
              <div>
                <Label className="text-sm font-medium text-gray-500">Status</Label>
                <Badge className={getStatusColor(tournament.status)}>
                  {tournament.status}
                </Badge>
              </div>
              <div>
                <Label className="text-sm font-medium text-gray-500">Type</Label>
                <Badge className={getBracketTypeColor(tournament.type)}>
                  {tournament.type.replace('_', ' ')}
                </Badge>
              </div>
              <div>
                <Label className="text-sm font-medium text-gray-500">Total Brackets</Label>
                <p className="text-lg font-semibold">{stats.total_brackets}</p>
              </div>
              <div>
                <Label className="text-sm font-medium text-gray-500">Active Brackets</Label>
                <p className="text-lg font-semibold text-green-600">{stats.active_brackets}</p>
              </div>
            </div>
            
            {/* Progress Bar */}
            <div className="mt-4">
              <div className="flex items-center justify-between mb-2">
                <span className="text-sm font-medium">Bracket Progress</span>
                <span className="text-sm text-gray-500">{Math.round(progressPercentage)}%</span>
              </div>
              <Progress value={progressPercentage} className="h-2" />
            </div>
          </CardContent>
        </Card>

        {/* Stats Cards */}
        <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-5 gap-4 mb-6">
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center">
                <div className="p-2 bg-blue-100 rounded-lg">
                  <Target className="w-5 h-5 text-blue-600" />
                </div>
                <div className="ml-3">
                  <p className="text-sm font-medium text-gray-500">Total Brackets</p>
                  <p className="text-xl font-bold">{stats.total_brackets}</p>
                </div>
              </div>
            </CardContent>
          </Card>
          
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center">
                <div className="p-2 bg-green-100 rounded-lg">
                  <Play className="w-5 h-5 text-green-600" />
                </div>
                <div className="ml-3">
                  <p className="text-sm font-medium text-gray-500">Active</p>
                  <p className="text-xl font-bold">{stats.active_brackets}</p>
                </div>
              </div>
            </CardContent>
          </Card>
          
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center">
                <div className="p-2 bg-blue-100 rounded-lg">
                  <CheckCircle className="w-5 h-5 text-blue-600" />
                </div>
                <div className="ml-3">
                  <p className="text-sm font-medium text-gray-500">Completed</p>
                  <p className="text-xl font-bold">{stats.completed_brackets}</p>
                </div>
              </div>
            </CardContent>
          </Card>
          
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center">
                <div className="p-2 bg-purple-100 rounded-lg">
                  <Trophy className="w-5 h-5 text-purple-600" />
                </div>
                <div className="ml-3">
                  <p className="text-sm font-medium text-gray-500">Total Matches</p>
                  <p className="text-xl font-bold">{stats.total_matches}</p>
                </div>
              </div>
            </CardContent>
          </Card>
          
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center">
                <div className="p-2 bg-yellow-100 rounded-lg">
                  <Award className="w-5 h-5 text-yellow-600" />
                </div>
                <div className="ml-3">
                  <p className="text-sm font-medium text-gray-500">Completed Matches</p>
                  <p className="text-xl font-bold">{stats.completed_matches}</p>
                </div>
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Brackets Tabs */}
        <Tabs value={activeTab} onValueChange={setActiveTab} className="space-y-4">
          <TabsList className="grid w-full grid-cols-3">
            <TabsTrigger value="overview">Overview</TabsTrigger>
            <TabsTrigger value="brackets">Brackets ({brackets.length})</TabsTrigger>
            <TabsTrigger value="visual">Visual View</TabsTrigger>
          </TabsList>

          <TabsContent value="overview">
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
              {/* Bracket Summary */}
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center space-x-2">
                    <Target className="h-5 w-5 text-blue-600" />
                    <span>Bracket Summary</span>
                  </CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div className="grid grid-cols-2 gap-4">
                    <div>
                      <Label className="text-sm font-medium text-gray-500">Tournament Type</Label>
                      <p className="text-sm font-medium">{tournament.type.replace('_', ' ')}</p>
                    </div>
                    <div>
                      <Label className="text-sm font-medium text-gray-500">Total Brackets</Label>
                      <p className="text-sm font-medium">{stats.total_brackets}</p>
                    </div>
                    <div>
                      <Label className="text-sm font-medium text-gray-500">Active Brackets</Label>
                      <p className="text-sm font-medium text-green-600">{stats.active_brackets}</p>
                    </div>
                    <div>
                      <Label className="text-sm font-medium text-gray-500">Completed Brackets</Label>
                      <p className="text-sm font-medium text-blue-600">{stats.completed_brackets}</p>
                    </div>
                  </div>
                  
                  <div className="pt-4 border-t">
                    <div className="flex items-center justify-between mb-2">
                      <span className="text-sm font-medium">Match Progress</span>
                      <span className="text-sm text-gray-500">{stats.completed_matches}/{stats.total_matches}</span>
                    </div>
                    <Progress value={progressPercentage} className="h-2" />
                  </div>
                </CardContent>
              </Card>

              {/* Quick Actions */}
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center space-x-2">
                    <Zap className="h-5 w-5 text-yellow-600" />
                    <span>Quick Actions</span>
                  </CardTitle>
                </CardHeader>
                <CardContent className="space-y-3">
                  <Button 
                    className="w-full justify-start" 
                    variant="outline"
                    onClick={handleGenerateBracket}
                  >
                    <Plus className="w-4 h-4 mr-2" />
                    Generate New Bracket
                  </Button>
                  <Button 
                    className="w-full justify-start" 
                    variant="outline"
                    onClick={handleAutoAdvance}
                  >
                    <Zap className="w-4 h-4 mr-2" />
                    Auto Advance Winners
                  </Button>
                  <Button 
                    className="w-full justify-start" 
                    variant="outline"
                    onClick={() => router.get(route('admin.tournaments.matches', tournament.id))}
                  >
                    <Play className="w-4 h-4 mr-2" />
                    Manage Matches
                  </Button>
                  <Button 
                    className="w-full justify-start" 
                    variant="outline"
                  >
                    <Download className="w-4 h-4 mr-2" />
                    Export Brackets
                  </Button>
                </CardContent>
              </Card>
            </div>
          </TabsContent>

          <TabsContent value="brackets">
            <Card>
              <CardContent className="p-0">
                {brackets.length === 0 ? (
                  <div className="p-8 text-center text-gray-500">
                    <Target className="h-12 w-12 mx-auto mb-4 text-gray-300" />
                    <p>No brackets found for this tournament.</p>
                    <Button 
                      className="mt-4" 
                      onClick={handleGenerateBracket}
                    >
                      <Plus className="w-4 h-4 mr-2" />
                      Generate Bracket
                    </Button>
                  </div>
                ) : (
                  <div className="divide-y">
                    {brackets.map((bracket) => (
                      <div key={bracket.id} className="p-4 hover:bg-gray-50">
                        <div className="flex items-center justify-between">
                          <div className="flex-1">
                            <div className="flex items-center space-x-4">
                              <h3 className="font-semibold">{bracket.name}</h3>
                              <Badge className={getStatusColor(bracket.status)}>
                                {bracket.status}
                              </Badge>
                              <Badge className={getBracketTypeColor(bracket.type)}>
                                {bracket.type.replace('_', ' ')}
                              </Badge>
                            </div>
                            <p className="text-sm text-gray-600 mt-1">
                              Created: {formatDate(bracket.created_at)}
                            </p>
                            <div className="flex items-center space-x-4 mt-2">
                              <span className="text-sm text-gray-500">
                                {bracket.rounds.length} rounds
                              </span>
                              <span className="text-sm text-gray-500">
                                {bracket.rounds.reduce((total, round) => total + round.matches.length, 0)} matches
                              </span>
                            </div>
                          </div>
                          <div className="flex items-center space-x-2">
                            <Button 
                              variant="outline" 
                              size="sm"
                              onClick={() => setSelectedBracket(bracket)}
                            >
                              <Eye className="h-4 w-4 mr-1" />
                              View
                            </Button>
                            <DropdownMenu>
                              <DropdownMenuTrigger asChild>
                                <Button variant="ghost" size="sm">
                                  <MoreHorizontal className="h-4 w-4" />
                                </Button>
                              </DropdownMenuTrigger>
                              <DropdownMenuContent align="end">
                                <DropdownMenuItem onClick={() => router.get(route('admin.tournaments.brackets.edit', { 
                                  tournament: tournament.id, 
                                  bracket: bracket.id 
                                }))}>
                                  <Edit className="w-4 h-4 mr-2" />
                                  Edit Bracket
                                </DropdownMenuItem>
                                <DropdownMenuItem>
                                  <Download className="w-4 h-4 mr-2" />
                                  Export
                                </DropdownMenuItem>
                                <DropdownMenuSeparator />
                                <DropdownMenuItem 
                                  onClick={() => router.delete(route('admin.tournaments.brackets.destroy', { 
                                    tournament: tournament.id, 
                                    bracket: bracket.id 
                                  }))}
                                  className="text-red-600"
                                >
                                  <Trash2 className="w-4 h-4 mr-2" />
                                  Delete
                                </DropdownMenuItem>
                              </DropdownMenuContent>
                            </DropdownMenu>
                          </div>
                        </div>
                      </div>
                    ))}
                  </div>
                )}
              </CardContent>
            </Card>
          </TabsContent>

          <TabsContent value="visual">
            <Card>
              <CardContent className="p-8">
                {selectedBracket ? (
                  <div className="space-y-6">
                    <div className="flex items-center justify-between">
                      <h3 className="text-lg font-semibold">{selectedBracket.name}</h3>
                      <Button 
                        variant="outline" 
                        size="sm"
                        onClick={() => setSelectedBracket(null)}
                      >
                        Back to List
                      </Button>
                    </div>
                    
                    {/* Visual Bracket Display */}
                    <div className="border rounded-lg p-4 bg-gray-50">
                      <div className="text-center text-gray-500">
                        <Target className="h-12 w-12 mx-auto mb-4 text-gray-300" />
                        <p>Visual bracket display coming soon</p>
                        <p className="text-sm">This will show a visual representation of the bracket with match results</p>
                      </div>
                    </div>
                  </div>
                ) : (
                  <div className="text-center text-gray-500">
                    <Target className="h-12 w-12 mx-auto mb-4 text-gray-300" />
                    <p>Select a bracket to view its visual representation</p>
                  </div>
                )}
              </CardContent>
            </Card>
          </TabsContent>
        </Tabs>

        {/* Info Alert */}
        <Alert className="mt-6">
          <Info className="h-4 w-4" />
          <AlertDescription>
            <strong>Bracket Management:</strong> Generate brackets automatically based on tournament participants. 
            Use auto-advance to automatically move winners to the next round. 
            Brackets can be exported and shared with participants.
          </AlertDescription>
        </Alert>
      </div>
    </AppLayout>
  );
};

export default TournamentBrackets; 