import React, { useState } from 'react';
import { Head, router } from '@inertiajs/react';
import AppLayout from '@/layouts/app-layout';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Separator } from '@/components/ui/separator';
import { Switch } from '@/components/ui/switch';
import { Alert, AlertDescription } from '@/components/ui/alert';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { TagsInput } from '@/components/ui/tags-input';
import { 
  Settings, 
  Users, 
  Calendar, 
  CreditCard, 
  Shield, 
  Save, 
  AlertCircle,
  CheckCircle,
  Trophy,
  Clock,
  DollarSign
} from 'lucide-react';
import { toast } from 'sonner';

interface CategoryLocation {
  restrict_locations: boolean;
  locations: string[];
}

interface TournamentSettings {
  registration_fee: number;
  max_players_per_team: number;
  min_players_per_team: number;
  max_officials_per_team: number;
  registration_start: string | null;
  registration_end: string | null;
  tournament_start: string | null;
  tournament_end: string | null;
  terms_version: string;
  terms_last_updated: string;
  allow_jersey_number_change: boolean;
  require_team_logo: boolean;
  payment_deadline_days: number;
  allow_partial_payments: boolean;
  otp_expiry_hours: number;
  max_otp_resends: number;
  default_first_approver_id: number | null;
  default_second_approver_id: number | null;
  available_countries: string;
  available_counties: string;
  category_locations: Record<string, CategoryLocation>;
  categories: Record<string, any>;
}

interface Props {
  settings: TournamentSettings;
  firstApprovers: Array<{
    id: number;
    name: string;
    email: string;
  }>;
  secondApprovers: Array<{
    id: number;
    name: string;
    email: string;
  }>;
  currentFirstApproverId: string | null;
  currentSecondApproverId: string | null;
}

export default function TournamentSettingsIndex({ settings, firstApprovers, secondApprovers, currentFirstApproverId, currentSecondApproverId }: Props) {
  const [form, setForm] = useState<TournamentSettings>({
    ...settings,
    default_first_approver_id: currentFirstApproverId ? parseInt(currentFirstApproverId) : null,
    default_second_approver_id: currentSecondApproverId ? parseInt(currentSecondApproverId) : null,
  });
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [errors, setErrors] = useState<Record<string, string>>({});

  const handleInputChange = (field: keyof TournamentSettings, value: any) => {
    setForm(prev => ({ ...prev, [field]: value }));
    // Clear error for this field
    if (errors[field]) {
      setErrors(prev => ({ ...prev, [field]: '' }));
    }
  };

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    setIsSubmitting(true);
    setErrors({});

    router.put(route('admin.tournament-settings.update'), form as any, {
      onError: (errors) => {
        setErrors(errors);
        setIsSubmitting(false);
        toast.error('Failed to update tournament settings!');
      },
      onSuccess: () => {
        setIsSubmitting(false);
        toast.success('Tournament settings updated successfully!');
      },
      onFinish: () => {
        setIsSubmitting(false);
      }
    });
  };

  const formatDate = (dateString: string | null) => {
    if (!dateString) return '';
    return new Date(dateString).toISOString().split('T')[0];
  };

  const handleDateChange = (field: keyof TournamentSettings, value: string) => {
    handleInputChange(field, value || null);
  };

  return (
    <AppLayout breadcrumbs={[
      { title: 'Tournament Settings', href: '/admin/tournament-settings' }
    ]}>
      <Head title="Tournament Settings" />
      
      <div className="space-y-6 p-4">
        {/* Header */}
        <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
          <div>
            <h1 className="text-2xl sm:text-3xl font-bold flex items-center gap-2">
              <Settings className="h-6 w-6 sm:h-8 sm:w-8 text-blue-600" />
              Tournament Settings
            </h1>
            <p className="text-muted-foreground text-sm sm:text-base mt-2">
              Configure tournament parameters and system settings
            </p>
          </div>
        </div>

        <form onSubmit={handleSubmit} className="space-y-6">
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
            {/* Global Location Settings */}
            <Card className="lg:col-span-2">
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Settings className="h-5 w-5 text-blue-600" />
                  Location Settings
                </CardTitle>
                <CardDescription>
                  Configure global countries and counties. These apply when category restrictions are disabled.
                </CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <TagsInput
                  label="Available Countries"
                  value={form.available_countries}
                  onChange={(value) => handleInputChange('available_countries', value)}
                  placeholder="Type country name and press Enter"
                  error={errors.available_countries}
                  helpText="Add countries that teams can register from (applies globally)"
                />

                <TagsInput
                  label="Available Counties"
                  value={form.available_counties}
                  onChange={(value) => handleInputChange('available_counties', value)}
                  placeholder="Type county name and press Enter"
                  error={errors.available_counties}
                  helpText="Add counties/regions for team registration (used when category restrictions are off)"
                />
              </CardContent>
            </Card>

            {/* Category Location Settings */}
            <Card className="lg:col-span-2">
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Settings className="h-5 w-5 text-green-600" />
                  Location Settings
                </CardTitle>
                <CardDescription>
                  Configure locations for each tournament category. Enable restriction to limit teams to specific locations.
                </CardDescription>
              </CardHeader>
              <CardContent className="space-y-6">
                {Object.keys(form.categories || {}).map((categoryKey) => {
                  const category = form.categories[categoryKey];
                  const categoryLocation = form.category_locations?.[categoryKey] || { restrict_locations: false, locations: [] };
                  
                  return (
                    <div key={categoryKey} className="border rounded-lg p-4 space-y-4">
                      <div className="flex items-center justify-between">
                        <div className="flex items-center gap-2">
                          <span className="text-2xl">{category.icon}</span>
                          <div>
                            <h4 className="font-semibold text-lg">{category.name}</h4>
                            <p className="text-sm text-muted-foreground">{category.description}</p>
                          </div>
                        </div>
                        <div className="flex items-center space-x-2">
                          <Label htmlFor={`restrict_${categoryKey}`} className="text-sm">
                            Restrict Locations
                          </Label>
                          <Switch
                            id={`restrict_${categoryKey}`}
                            checked={categoryLocation.restrict_locations || false}
                            onCheckedChange={(checked) => {
                              const newCategoryLocations = { ...form.category_locations };
                              if (!newCategoryLocations[categoryKey]) {
                                newCategoryLocations[categoryKey] = { restrict_locations: false, locations: [] };
                              }
                              newCategoryLocations[categoryKey].restrict_locations = checked;
                              handleInputChange('category_locations', newCategoryLocations);
                            }}
                          />
                        </div>
                      </div>
                      
                      <div>
                        <TagsInput
                          label={`${category.name} Locations`}
                          value={categoryLocation.locations?.length ? categoryLocation.locations.join(', ') : ''}
                          onChange={(value) => {
                            const newCategoryLocations = { ...form.category_locations };
                            if (!newCategoryLocations[categoryKey]) {
                              newCategoryLocations[categoryKey] = { restrict_locations: false, locations: [] };
                            }
                            newCategoryLocations[categoryKey].locations = value.split(',').map(l => l.trim()).filter(l => l);
                            handleInputChange('category_locations', newCategoryLocations);
                          }}
                          placeholder={categoryLocation.restrict_locations ? "Type location and press Enter" : "Restriction disabled - locations not restricted"}
                          disabled={!categoryLocation.restrict_locations}
                          helpText={categoryLocation.restrict_locations 
                            ? `Add specific locations allowed for ${category.name} teams` 
                            : `Enable restriction to limit ${category.name} teams to specific locations`
                          }
                        />
                      </div>
                    </div>
                  );
                })}
              </CardContent>
            </Card>
          </div>

          {/* System Information */}
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Trophy className="h-5 w-5 text-yellow-600" />
                System Information
              </CardTitle>
              <CardDescription>
                Read-only system information
              </CardDescription>
            </CardHeader>
            <CardContent>
              <div className="grid grid-cols-1 sm:grid-cols-2 gap-4 text-sm">
                <div>
                  <Label className="text-muted-foreground">Terms Version</Label>
                  <div className="font-medium">{form.terms_version}</div>
                </div>
                <div>
                  <Label className="text-muted-foreground">Terms Last Updated</Label>
                  <div className="font-medium">{form.terms_last_updated}</div>
                </div>
              </div>
            </CardContent>
          </Card>

          {/* Save Button */}
          <div className="flex justify-end">
            <Button 
              type="submit" 
              disabled={isSubmitting}
              className="w-full sm:w-auto min-w-[120px]"
            >
              {isSubmitting ? (
                <>
                  <Clock className="w-4 h-4 mr-2 animate-spin" />
                  Saving...
                </>
              ) : (
                <>
                  <Save className="w-4 h-4 mr-2" />
                  Save Settings
                </>
              )}
            </Button>
          </div>
        </form>

        {/* Warning Alert */}
        <Alert>
          <AlertCircle className="h-4 w-4" />
          <AlertDescription>
            <strong>Important:</strong> Changes to these settings will affect the entire tournament system. 
            Some changes may require a cache clear to take effect immediately.
          </AlertDescription>
        </Alert>
      </div>
    </AppLayout>
  );
}