import React, { useState } from 'react';
import AppLayout from '@/layouts/app-layout';
import { Head, router } from '@inertiajs/react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Dialog, DialogContent, DialogHeader, DialogTitle } from '@/components/ui/dialog';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import { Input } from '@/components/ui/input';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { usePermissions } from '@/hooks/use-permissions';
import { 
  Users, 
  Trophy, 
  User, 
  Calendar,
  Edit,
  Trash2,
  ArrowLeft,
  CheckCircle,
  XCircle,
  DollarSign,
  CreditCard
} from 'lucide-react';

interface Props {
  team: any;
  tournament?: any;
  captain?: any;
  members?: any[];
  tournaments?: any[];
  tournamentSettings?: { min_players_per_team: number, max_players_per_team: number };
}

const TeamShow: React.FC<Props> = ({ team, tournament, captain, members = [], tournaments, tournamentSettings }) => {
  // Permission checks
  const { hasPermission } = usePermissions();
  const canApproveTeams = hasPermission(['teams.approve', 'tournaments.approve_participants']);
  const canFirstApprove = hasPermission('tournaments.first_approve_participants');
  const canSecondApprove = hasPermission('tournaments.second_approve_participants');
  const canReject = hasPermission(['teams.approve', 'tournaments.approve_participants']);
  const canEditTeams = hasPermission('teams.edit');
  const canDeleteTeams = hasPermission('teams.delete');
  const canManagePayments = hasPermission('teams.manage_payments');

  // Approval modals state
  const [approvalModal, setApprovalModal] = useState<{
    isOpen: boolean;
    team: any;
    tournament: any;
    action: 'approve' | 'reject' | 'first-approve' | 'second-approve';
  }>({
    isOpen: false,
    team: null,
    tournament: null,
    action: 'approve'
  });
  const [approvalNotes, setApprovalNotes] = useState('');
  const [rejectionReason, setRejectionReason] = useState('');

  // Payment modal state
  const [isPaymentModalOpen, setIsPaymentModalOpen] = useState(false);
  const [paymentForm, setPaymentForm] = useState({
    tournament_id: team.tournaments?.[0]?.id || '',
    amount: '',
    currency: 'KES',
    payment_method: 'mpesa',
    reference_number: '',
    transaction_id: '',
    description: '',
    paid_at: new Date().toISOString().split('T')[0],
  });
  const [paymentFormErrors, setPaymentFormErrors] = useState<Record<string, string>>({});

  // Format approval status
  const formatApprovalStatus = (status: string) => {
    return status
      .split('_')
      .map(word => word.charAt(0).toUpperCase() + word.slice(1))
      .join(' ');
  };

  // Get team requirements based on tournament
  const getTeamRequirements = () => {
    const playerCount = team.players?.length || 0;
    
    let minPlayers = tournamentSettings?.min_players_per_team || 11;
    let recommendedPlayers = tournamentSettings?.max_players_per_team || 23;
    
    if (team.tournaments && team.tournaments.length > 0 && Array.isArray(tournaments)) {
      const teamTournamentId = team.tournaments[0].id;
      const tournamentConfig = tournaments.find((t: any) => t.id === teamTournamentId);
      
      if (tournamentConfig) {
        minPlayers = tournamentConfig.min_players_per_team || minPlayers;
        recommendedPlayers = tournamentConfig.max_players_per_team || recommendedPlayers;
      }
    }
    
    return {
      playerCount,
      minPlayers,
      recommendedPlayers,
      meetsMinimum: playerCount >= minPlayers,
      meetsRecommended: playerCount >= recommendedPlayers,
      percentage: Math.min((playerCount / recommendedPlayers) * 100, 100)
    };
  };

  // Get payment status
  const getPaymentStatus = () => {
    if (team.payment_status === 'completed') {
      return { paid: true, status: 'completed' };
    }
    
    if (team.payments && team.payments.length > 0) {
      const completedPayment = team.payments.find((p: any) => p.status === 'completed');
      if (completedPayment) {
        return { paid: true, status: 'completed', amount: completedPayment.amount };
      }
    }
    
    return { paid: false, status: team.payment_status || 'pending' };
  };

  const getStatusColor = (status: string) => {
    const colors: Record<string, string> = {
      'active': 'bg-green-100 text-green-800 border-green-200',
      'eliminated': 'bg-red-100 text-red-800 border-red-200',
      'withdrawn': 'bg-gray-100 text-gray-800 border-gray-200',
    };
    return colors[status] || 'bg-gray-100 text-gray-800 border-gray-200';
  };

  const getApprovalStatusColor = (status: string) => {
    const colors: Record<string, string> = {
      'pending': 'bg-yellow-100 text-yellow-800 border-yellow-200',
      'first_approved': 'bg-blue-100 text-blue-800 border-blue-200',
      'second_approved': 'bg-green-100 text-green-800 border-green-200',
      'rejected': 'bg-red-100 text-red-800 border-red-200',
      'active': 'bg-green-100 text-green-800 border-green-200',
    };
    return colors[status] || 'bg-gray-100 text-gray-800 border-gray-200';
  };

  const handleDelete = () => {
    if (confirm(`Are you sure you want to delete "${team.name}"? This action cannot be undone.`)) {
      router.delete(route('admin.teams.destroy', { team: team.id }));
    }
  };

  // Handle manual payment
  const openPaymentModal = () => {
    // Set default tournament and amount
    if (team.tournaments && team.tournaments.length > 0 && tournaments) {
      const teamTournament = tournaments.find((t: any) => t.id === team.tournaments[0].id);
      setPaymentForm({
        ...paymentForm,
        tournament_id: team.tournaments[0].id.toString(),
        amount: teamTournament?.entry_fee || '',
        currency: teamTournament?.currency || 'KES',
      });
    }
    setIsPaymentModalOpen(true);
    setPaymentFormErrors({});
  };

  const handlePaymentSubmit = (e: React.FormEvent<HTMLFormElement>) => {
    e.preventDefault();
    router.post(route('admin.teams.payments.manual', { team: team.id }), paymentForm, {
      onError: (errors) => setPaymentFormErrors(errors),
      onSuccess: () => {
        setIsPaymentModalOpen(false);
        setPaymentForm({
          tournament_id: team.tournaments?.[0]?.id || '',
          amount: '',
          currency: 'KES',
          payment_method: 'mpesa',
          reference_number: '',
          transaction_id: '',
          description: '',
          paid_at: new Date().toISOString().split('T')[0],
        });
        setPaymentFormErrors({});
      },
    });
  };

  // Handle approval actions
  const openApprovalModal = (team: any, tournament: any, action: 'approve' | 'reject' | 'first-approve' | 'second-approve') => {
    setApprovalModal({
      isOpen: true,
      team,
      tournament,
      action
    });
    setApprovalNotes('');
    setRejectionReason('');
  };

  const handleApprovalAction = () => {
    const { team, tournament, action } = approvalModal;
    
    if (action === 'reject' && !rejectionReason.trim()) {
      return;
    }

    const data: any = {
      tournament_id: tournament.id,
    };

    if (action === 'reject') {
      data.reason = rejectionReason;
    } else {
      data.notes = approvalNotes;
    }

    const routeName = `admin.teams.${action}`;
    router.post(route(routeName, { team: team.id }), data, {
      onSuccess: () => {
        setApprovalModal({ isOpen: false, team: null, tournament: null, action: 'approve' });
        setApprovalNotes('');
        setRejectionReason('');
      },
    });
  };

  return (
    <AppLayout breadcrumbs={[
      { title: 'Teams', href: '/admin/teams' },
      { title: team.name, href: `/admin/teams/${team.id}` }
    ]}> 
      <Head title={`Team: ${team.name}`} />
      
      <div className="space-y-6 p-4">
        {/* Header */}
        <div className="flex flex-col sm:flex-row items-start sm:items-center justify-between gap-4">
          <div className="flex items-center gap-4">
            <Button
              variant="outline"
              size="sm"
              onClick={() => router.get(route('admin.teams.index'))}
            >
              <ArrowLeft className="w-4 h-4 mr-2" />
              Back to Teams
            </Button>
           
          </div>
          <div className="flex flex-wrap gap-2">
            {canEditTeams && (
              <Button
                variant="outline"
                onClick={() => router.get(route('admin.teams.edit', { team: team.id }))}
              >
                <Edit className="w-4 h-4 mr-2" />
                Edit Team
              </Button>
            )}
            {canDeleteTeams && (
              <Button
                variant="destructive"
                onClick={handleDelete}
              >
                <Trash2 className="w-4 h-4 mr-2" />
                Delete Team
              </Button>
            )}
          </div>
        </div>

        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
          {/* Team Details */}
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Users className="w-5 h-5" />
                Team Details
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="space-y-3">
                <div>
                  <label className="text-sm font-medium text-gray-500">Name</label>
                  <p className="text-lg font-semibold">{team.name}</p>
                </div>
                
                <div>
                  <label className="text-sm font-medium text-gray-500">Status</label>
                  <div className="mt-1">
                    <Badge className={getStatusColor(team.status)}>
                      {team.status.charAt(0).toUpperCase() + team.status.slice(1)}
                    </Badge>
                  </div>
                </div>

                {team.description && (
                  <div>
                    <label className="text-sm font-medium text-gray-500">Description</label>
                    <p className="text-sm mt-1">{team.description}</p>
                  </div>
                )}

                <div>
                  <label className="text-sm font-medium text-gray-500">Created</label>
                  <p className="text-sm mt-1">
                    {new Date(team.created_at).toLocaleDateString()} at {new Date(team.created_at).toLocaleTimeString()}
                  </p>
                </div>

                <div>
                  <label className="text-sm font-medium text-gray-500">Last Updated</label>
                  <p className="text-sm mt-1">
                    {new Date(team.updated_at).toLocaleDateString()} at {new Date(team.updated_at).toLocaleTimeString()}
                  </p>
                </div>
              </div>
            </CardContent>
          </Card>

          {/* Tournament & Captain */}
          <div className="space-y-6">
            {/* Tournament */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Trophy className="w-5 h-5" />
                  Tournament
                </CardTitle>
              </CardHeader>
              <CardContent>
                {tournament ? (
                  <div className="space-y-2">
                    <p className="font-semibold">{tournament.name}</p>
                    <Badge variant="outline">{tournament.status}</Badge>
                  </div>
                ) : (
                  <p className="text-muted-foreground">Tournament information not available</p>
                )}
              </CardContent>
            </Card>          

            {/* Captain */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <User className="w-5 h-5" />
                  Team Captain
                </CardTitle>
              </CardHeader>
              <CardContent>
                {captain ? (
                  <div className="space-y-2">
                    <p className="font-semibold">{captain.name}</p>
                    <p className="text-sm text-muted-foreground">{captain.email}</p>
                  </div>
                ) : (
                  <p className="text-muted-foreground">Captain information not available</p>
                )}
              </CardContent>
            </Card>

            {/* Team Requirements */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Users className="w-5 h-5" />
                  Team Requirements
                </CardTitle>
              </CardHeader>
              <CardContent>
                {(() => {
                  const requirements = getTeamRequirements();
                  return (
                    <div className="space-y-3">
                      <div className="flex justify-between items-center">
                        <span className="text-sm text-gray-600">Players</span>
                        <span className="font-semibold">{requirements.playerCount}</span>
                      </div>
                      <div className="flex justify-between items-center">
                        <span className="text-sm text-gray-600">Minimum Required</span>
                        <Badge className={requirements.meetsMinimum ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'}>
                          {requirements.minPlayers} {requirements.meetsMinimum ? '✓' : '✗'}
                        </Badge>
                      </div>
                      <div className="flex justify-between items-center">
                        <span className="text-sm text-gray-600">Recommended</span>
                        <Badge className={requirements.meetsRecommended ? 'bg-green-100 text-green-800' : 'bg-yellow-100 text-yellow-800'}>
                          {requirements.recommendedPlayers} {requirements.meetsRecommended ? '✓' : '○'}
                        </Badge>
                      </div>
                      <div className="pt-2">
                        <div className="flex justify-between text-sm mb-1">
                          <span className="text-gray-600">Progress</span>
                          <span className="font-medium">{Math.round(requirements.percentage)}%</span>
                        </div>
                        <div className="w-full bg-gray-200 rounded-full h-2">
                          <div 
                            className={`h-2 rounded-full ${requirements.meetsRecommended ? 'bg-green-600' : requirements.meetsMinimum ? 'bg-yellow-500' : 'bg-red-600'}`}
                            style={{ width: `${requirements.percentage}%` }}
                          ></div>
                        </div>
                      </div>
                    </div>
                  );
                })()}
              </CardContent>
            </Card>

            {/* Payment Status */}
            <Card>
              <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                <CardTitle className="flex items-center gap-2">
                  <DollarSign className="w-5 h-5" />
                  Payment Status
                </CardTitle>
                {canManagePayments && !getPaymentStatus().paid && (
                  <Button size="sm" onClick={openPaymentModal} className="bg-green-600 hover:bg-green-700">
                    <CreditCard className="w-4 h-4 mr-1" />
                    Add Payment
                  </Button>
                )}
              </CardHeader>
              <CardContent>
                {(() => {
                  const payment = getPaymentStatus();
                  return (
                    <div className="space-y-3">
                      <div className="flex justify-between items-center">
                        <span className="text-sm text-gray-600">Status</span>
                        <Badge className={payment.paid ? 'bg-green-600 text-white' : 'bg-yellow-100 text-yellow-800'}>
                          {payment.paid ? '✓ Paid' : '○ Unpaid'}
                        </Badge>
                      </div>
                      {payment.amount && (
                        <div className="flex justify-between items-center">
                          <span className="text-sm text-gray-600">Amount</span>
                          <span className="font-semibold">{payment.amount}</span>
                        </div>
                      )}
                      {team.payment_method && (
                        <div className="flex justify-between items-center">
                          <span className="text-sm text-gray-600">Method</span>
                          <Badge variant="outline" className="text-xs">
                            <CreditCard className="w-3 h-3 mr-1" />
                            {team.payment_method.toUpperCase()}
                          </Badge>
                        </div>
                      )}
                      {team.transaction_id && (
                        <div className="flex flex-col gap-1">
                          <span className="text-xs text-gray-500">Transaction ID</span>
                          <span className="text-xs font-mono bg-gray-100 p-1 rounded">{team.transaction_id}</span>
                        </div>
                      )}
                      {team.payment_completed_at && (
                        <div className="flex justify-between items-center">
                          <span className="text-sm text-gray-600">Completed</span>
                          <span className="text-sm">{new Date(team.payment_completed_at).toLocaleDateString()}</span>
                        </div>
                      )}
                    </div>
                  );
                })()}
              </CardContent>
            </Card>


            {/* Tournament Approval Status */}
            {team.tournaments && team.tournaments.length > 0 && (
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <Calendar className="w-5 h-5" />
                    Approval Status
                  </CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="space-y-3">
                    {team.tournaments.map((t: any) => (
                      <div key={t.id} className="border rounded-lg p-3">
                        <div className="flex justify-between items-center mb-2">
                          <span className="font-medium">{t.name}</span>
                          <Badge className={getApprovalStatusColor(t.pivot?.approval_status || 'pending')}>
                            {formatApprovalStatus(t.pivot?.approval_status || 'pending')}
                          </Badge>
                        </div>
                        <div className="text-sm text-gray-600 space-y-1 mb-3">
                          <div>Status: {t.pivot?.status || 'registered'}</div>
                          {t.pivot?.registered_at && (
                            <div>Registered: {new Date(t.pivot.registered_at).toLocaleDateString()}</div>
                          )}
                          {t.pivot?.first_approved_at && (
                            <div>First Approved: {new Date(t.pivot.first_approved_at).toLocaleDateString()}</div>
                          )}
                          {t.pivot?.second_approved_at && (
                            <div>Second Approved: {new Date(t.pivot.second_approved_at).toLocaleDateString()}</div>
                          )}
                          {t.pivot?.rejected_at && (
                            <div>Rejected: {new Date(t.pivot.rejected_at).toLocaleDateString()}</div>
                          )}
                          {t.pivot?.first_approval_notes && (
                            <div>First Approval Notes: {t.pivot.first_approval_notes}</div>
                          )}
                          {t.pivot?.second_approval_notes && (
                            <div>Second Approval Notes: {t.pivot.second_approval_notes}</div>
                          )}
                          {t.pivot?.rejection_reason && (
                            <div className="text-red-600 font-medium">Rejection Reason: {t.pivot.rejection_reason}</div>
                          )}
                        </div>
                        
                        {/* Approval Actions */}
                        {(canApproveTeams || canFirstApprove || canSecondApprove || canReject) && (
                          <div className="flex flex-wrap gap-2 mt-3 pt-3 border-t">
                            {(canApproveTeams || canFirstApprove || canSecondApprove) && (
                              <Button 
                                size="sm" 
                                variant="outline"
                                className="bg-purple-600 hover:bg-purple-700 text-white"
                                onClick={() => router.get(route('admin.teams.approval-view', { team: team.id }), { tournament_id: t.id })}
                              >
                                <CheckCircle className="w-4 h-4 mr-1" />
                                Review Details
                              </Button>
                            )}
                            {canFirstApprove && t.pivot?.approval_status === 'pending' && (
                              <Button 
                                size="sm" 
                                className="bg-blue-600 hover:bg-blue-700 text-white"
                                onClick={() => openApprovalModal(team, t, 'first-approve')}
                              >
                                <CheckCircle className="w-4 h-4 mr-1" />
                                1st Approve
                              </Button>
                            )}
                            {canSecondApprove && t.pivot?.approval_status === 'first_approved' && (
                              <Button 
                                size="sm" 
                                className="bg-green-600 hover:bg-green-700 text-white"
                                onClick={() => openApprovalModal(team, t, 'second-approve')}
                              >
                                <CheckCircle className="w-4 h-4 mr-1" />
                                2nd Approve
                              </Button>
                            )}
                            {canReject && (
                              <Button 
                                size="sm" 
                                variant="destructive"
                                onClick={() => openApprovalModal(team, t, 'reject')}
                              >
                                <XCircle className="w-4 h-4 mr-1" />
                                Reject
                              </Button>
                            )}
                          </div>
                        )}
                      </div>
                    ))}
                  </div>
                </CardContent>
              </Card>
            )}
          </div>
        </div>

        {/* Payment History */}
        {team.payments && team.payments.length > 0 && (
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <CreditCard className="w-5 h-5" />
                Payment History
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-3">
                {team.payments.map((payment: any) => (
                  <div key={payment.id} className="flex items-center justify-between p-3 border rounded-lg">
                    <div className="space-y-1">
                      <div className="flex items-center gap-2">
                        <Badge className={
                          payment.status === 'completed' ? 'bg-green-100 text-green-800' :
                          payment.status === 'pending' ? 'bg-yellow-100 text-yellow-800' :
                          payment.status === 'processing' ? 'bg-blue-100 text-blue-800' :
                          'bg-red-100 text-red-800'
                        }>
                          {payment.status}
                        </Badge>
                        <span className="text-sm font-medium">{payment.currency || 'KES'} {payment.amount}</span>
                      </div>
                      {payment.payment_method && (
                        <p className="text-xs text-gray-600">Method: {payment.payment_method.toUpperCase()}</p>
                      )}
                      {payment.reference_number && (
                        <p className="text-xs text-gray-500">Ref: {payment.reference_number}</p>
                      )}
                    </div>
                    <div className="text-right text-xs text-gray-500">
                      {payment.paid_at ? new Date(payment.paid_at).toLocaleDateString() : new Date(payment.created_at).toLocaleDateString()}
                    </div>
                  </div>
                ))}
              </div>
            </CardContent>
          </Card>
        )}

        {/* Team Members */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Users className="w-5 h-5" />
              Team Members ({members.length + 1})
            </CardTitle>
          </CardHeader>
          <CardContent>
            {members.length > 0 ? (
              <div className="space-y-3">
                {members.map((member) => (
                  <div key={member.id} className="flex items-center justify-between p-3 border rounded-lg">
                    <div>
                      <p className="font-medium">{member.name}</p>
                      <p className="text-sm text-muted-foreground">{member.email}</p>
                    </div>
                    <Badge variant="outline">{member.pivot?.role || 'member'}</Badge>
                  </div>
                ))}
              </div>
            ) : (
              <p className="text-muted-foreground">No additional team members</p>
            )}
          </CardContent>
        </Card>

        {/* Approval Modal */}
        <Dialog open={approvalModal.isOpen} onOpenChange={() => setApprovalModal({ isOpen: false, team: null, tournament: null, action: 'approve' })}>
          <DialogContent>
            <DialogHeader>
              <DialogTitle>
                {approvalModal.action === 'approve' && 'Approve Team'}
                {approvalModal.action === 'reject' && 'Reject Team'}
                {approvalModal.action === 'first-approve' && 'First Approve Team'}
                {approvalModal.action === 'second-approve' && 'Second Approve Team'}
              </DialogTitle>
            </DialogHeader>
            <div className="space-y-4">
              <div>
                <p className="text-sm text-gray-600">
                  Team: <span className="font-semibold">{approvalModal.team?.name}</span>
                </p>
                <p className="text-sm text-gray-600">
                  Tournament: <span className="font-semibold">{approvalModal.tournament?.name}</span>
                </p>
              </div>
              
              {approvalModal.action === 'reject' ? (
                <div>
                  <Label htmlFor="rejection_reason">Rejection Reason *</Label>
                  <Textarea
                    id="rejection_reason"
                    value={rejectionReason}
                    onChange={(e) => setRejectionReason(e.target.value)}
                    placeholder="Please provide a reason for rejection..."
                    required
                    rows={3}
                  />
                </div>
              ) : (
                <div>
                  <Label htmlFor="approval_notes">Notes (Optional)</Label>
                  <Textarea
                    id="approval_notes"
                    value={approvalNotes}
                    onChange={(e) => setApprovalNotes(e.target.value)}
                    placeholder="Add any notes about this approval..."
                    rows={3}
                  />
                </div>
              )}
              
              <div className="flex gap-2">
                <Button 
                  onClick={handleApprovalAction}
                  className={
                    approvalModal.action === 'reject' 
                      ? 'bg-red-600 hover:bg-red-700' 
                      : 'bg-green-600 hover:bg-green-700'
                  }
                >
                  {approvalModal.action === 'approve' && 'Approve Team'}
                  {approvalModal.action === 'reject' && 'Reject Team'}
                  {approvalModal.action === 'first-approve' && 'First Approve'}
                  {approvalModal.action === 'second-approve' && 'Second Approve'}
                </Button>
                <Button variant="outline" onClick={() => setApprovalModal({ isOpen: false, team: null, tournament: null, action: 'approve' })}>
                  Cancel
                </Button>
              </div>
            </div>
          </DialogContent>
        </Dialog>

        {/* Manual Payment Modal */}
        <Dialog open={isPaymentModalOpen} onOpenChange={setIsPaymentModalOpen}>
          <DialogContent className="max-w-lg">
            <DialogHeader>
              <DialogTitle className="flex items-center gap-2">
                <DollarSign className="w-5 h-5" />
                Add Manual Payment
              </DialogTitle>
            </DialogHeader>
            <form onSubmit={handlePaymentSubmit} className="space-y-4">
              <div>
                <Label htmlFor="tournament_id">Tournament *</Label>
                <Select 
                  value={paymentForm.tournament_id} 
                  onValueChange={(value) => {
                    setPaymentForm({ ...paymentForm, tournament_id: value });
                    // Auto-fill amount from tournament
                    const tournament = tournaments?.find((t: any) => t.id.toString() === value);
                    if (tournament) {
                      setPaymentForm({ 
                        ...paymentForm, 
                        tournament_id: value,
                        amount: tournament.entry_fee || '',
                        currency: tournament.currency || 'KES'
                      });
                    }
                  }}
                >
                  <SelectTrigger>
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    {tournaments && Array.isArray(tournaments) ? tournaments.map((tournament: any) => (
                      <SelectItem key={tournament.id} value={tournament.id.toString()}>
                        {tournament.name}
                      </SelectItem>
                    )) : null}
                  </SelectContent>
                </Select>
                {paymentFormErrors.tournament_id && <p className="text-red-500 text-sm">{paymentFormErrors.tournament_id}</p>}
              </div>

              <div className="grid grid-cols-2 gap-4">
                <div>
                  <Label htmlFor="amount">Amount *</Label>
                  <Input
                    id="amount"
                    type="number"
                    step="0.01"
                    value={paymentForm.amount}
                    onChange={(e) => setPaymentForm({ ...paymentForm, amount: e.target.value })}
                    required
                  />
                  {paymentFormErrors.amount && <p className="text-red-500 text-sm">{paymentFormErrors.amount}</p>}
                </div>

                <div>
                  <Label htmlFor="currency">Currency</Label>
                  <Select value={paymentForm.currency} onValueChange={(value) => setPaymentForm({ ...paymentForm, currency: value })}>
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="KES">KES</SelectItem>
                      <SelectItem value="USD">USD</SelectItem>
                      <SelectItem value="EUR">EUR</SelectItem>
                      <SelectItem value="GBP">GBP</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
              </div>

              <div>
                <Label htmlFor="payment_method">Payment Method *</Label>
                <Select value={paymentForm.payment_method} onValueChange={(value) => setPaymentForm({ ...paymentForm, payment_method: value })}>
                  <SelectTrigger>
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="mpesa">M-Pesa</SelectItem>
                    <SelectItem value="cash">Cash</SelectItem>
                    <SelectItem value="bank_transfer">Bank Transfer</SelectItem>
                    <SelectItem value="card">Card</SelectItem>
                    <SelectItem value="other">Other</SelectItem>
                  </SelectContent>
                </Select>
                {paymentFormErrors.payment_method && <p className="text-red-500 text-sm">{paymentFormErrors.payment_method}</p>}
              </div>

              <div className="grid grid-cols-2 gap-4">
                <div>
                  <Label htmlFor="reference_number">Reference Number</Label>
                  <Input
                    id="reference_number"
                    value={paymentForm.reference_number}
                    onChange={(e) => setPaymentForm({ ...paymentForm, reference_number: e.target.value })}
                    placeholder="Payment reference"
                  />
                </div>

                <div>
                  <Label htmlFor="transaction_id">Transaction ID</Label>
                  <Input
                    id="transaction_id"
                    value={paymentForm.transaction_id}
                    onChange={(e) => setPaymentForm({ ...paymentForm, transaction_id: e.target.value })}
                    placeholder="Transaction ID"
                  />
                </div>
              </div>

              <div>
                <Label htmlFor="paid_at">Payment Date</Label>
                <Input
                  id="paid_at"
                  type="date"
                  value={paymentForm.paid_at}
                  onChange={(e) => setPaymentForm({ ...paymentForm, paid_at: e.target.value })}
                />
              </div>

              <div>
                <Label htmlFor="description">Description</Label>
                <Textarea
                  id="description"
                  value={paymentForm.description}
                  onChange={(e) => setPaymentForm({ ...paymentForm, description: e.target.value })}
                  placeholder="Add notes about this payment..."
                  rows={2}
                />
              </div>

              <div className="flex gap-2">
                <Button type="submit" className="bg-green-600 hover:bg-green-700">
                  <CreditCard className="w-4 h-4 mr-2" />
                  Add Payment
                </Button>
                <Button type="button" variant="outline" onClick={() => setIsPaymentModalOpen(false)}>
                  Cancel
                </Button>
              </div>
            </form>
          </DialogContent>
        </Dialog>
      </div>
    </AppLayout>
  );
};

export default TeamShow; 