import AppLayout from '@/layouts/app-layout';
import { Head, Link, router } from '@inertiajs/react';
import { Card } from '@/components/ui/card';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger } from '@/components/ui/dialog';
import { Label } from '@/components/ui/label';
import { Badge } from '@/components/ui/badge';
import { Textarea } from '@/components/ui/textarea';
import { DropdownMenu, DropdownMenuContent, DropdownMenuItem, DropdownMenuLabel, DropdownMenuSeparator, DropdownMenuTrigger } from '@/components/ui/dropdown-menu';
import { Progress } from '@/components/ui/progress';
import { Tooltip, TooltipContent, TooltipProvider, TooltipTrigger } from '@/components/ui/tooltip';
import { useState } from 'react';
import { usePermissions } from '@/hooks/use-permissions';
import { MoreVertical, Eye, Edit, CheckCircle, XCircle, Trash2, Users, AlertCircle, Shield, Trophy } from 'lucide-react';

export default function AdminTeams({ teams, filters, tournaments, users, tournamentSettings }: { teams: any, filters: any, tournaments: any, users: any, tournamentSettings?: { min_players_per_team: number, max_players_per_team: number } }) {
  
  // Permission checks
  const { hasPermission } = usePermissions();
  const canApproveTeams = hasPermission(['teams.approve', 'tournaments.approve_participants']);
  const canFirstApprove = hasPermission('tournaments.first_approve_participants');
  const canSecondApprove = hasPermission('tournaments.second_approve_participants');
  const canReject = hasPermission(['teams.approve', 'tournaments.approve_participants']);
  const canViewTeams = hasPermission('teams.view');
  const canEditTeams = hasPermission('teams.edit');
  const canDeleteTeams = hasPermission('teams.delete');

  const [editingTeam, setEditingTeam] = useState<any>(null);
  const [selectedTeams, setSelectedTeams] = useState<number[]>([]);
  const [isDeleteModalOpen, setIsDeleteModalOpen] = useState(false);
  const [isMultipleDeleteModalOpen, setIsMultipleDeleteModalOpen] = useState(false);
  const [teamToDelete, setTeamToDelete] = useState<any>(null);
  const [localFilters, setLocalFilters] = useState({
    search: filters?.search || '',
    status: filters?.status || 'all',
    tournament_id: filters?.tournament_id || '',
  });

  // Approval modals state
  const [approvalModal, setApprovalModal] = useState<{
    isOpen: boolean;
    team: any;
    tournament: any;
    action: 'approve' | 'reject' | 'first-approve' | 'second-approve';
  }>({
    isOpen: false,
    team: null,
    tournament: null,
    action: 'approve'
  });
  const [approvalNotes, setApprovalNotes] = useState('');
  const [rejectionReason, setRejectionReason] = useState('');

  // Format approval status
  const formatApprovalStatus = (status: string) => {
    return status
      .split('_')
      .map(word => word.charAt(0).toUpperCase() + word.slice(1))
      .join(' ');
  };

  // Get team player count
  const getTeamPlayerCount = (team: any) => {
    return team.players?.length || 0;
  };

  // Get team requirements status based on tournament
  const getTeamRequirements = (team: any) => {
    const playerCount = getTeamPlayerCount(team);
    
    // If team has tournaments, use the first tournament's settings
    let minPlayers = tournamentSettings?.min_players_per_team || 11;
    let recommendedPlayers = tournamentSettings?.max_players_per_team || 23;
    
    if (team.tournaments && team.tournaments.length > 0 && Array.isArray(tournaments)) {
      const teamTournamentId = team.tournaments[0].id;
      const tournamentConfig = tournaments.find((t: any) => t.id === teamTournamentId);
      
      if (tournamentConfig) {
        minPlayers = tournamentConfig.min_players_per_team || minPlayers;
        recommendedPlayers = tournamentConfig.max_players_per_team || recommendedPlayers;
      }
    }
    
    return {
      playerCount,
      minPlayers,
      recommendedPlayers,
      meetsMinimum: playerCount >= minPlayers,
      meetsRecommended: playerCount >= recommendedPlayers,
      percentage: Math.min((playerCount / recommendedPlayers) * 100, 100)
    };
  };

  // Get payment status for team
  const getPaymentStatus = (team: any) => {
    // Check team payment_status field first
    if (team.payment_status === 'completed') {
      return { paid: true, status: 'completed' };
    }
    
    // Check if team has completed payments
    if (team.payments && team.payments.length > 0) {
      const completedPayment = team.payments.find((p: any) => p.status === 'completed');
      if (completedPayment) {
        return { paid: true, status: 'completed', amount: completedPayment.amount };
      }
    }
    
    return { paid: false, status: team.payment_status || 'pending' };
  };

  // Render payment status badge
  const renderPaymentBadge = (team: any) => {
    const payment = getPaymentStatus(team);
    
    return (
      <TooltipProvider>
        <Tooltip>
          <TooltipTrigger>
            <Badge 
              variant={payment.paid ? 'default' : 'outline'}
              className={payment.paid ? 'bg-green-600 hover:bg-green-700' : 'bg-yellow-50 text-yellow-800 border-yellow-300'}
            >
              {payment.paid ? '✓ Paid' : '○ Unpaid'}
            </Badge>
          </TooltipTrigger>
          <TooltipContent>
            <div className="text-xs">
              <p className="font-semibold">Payment Status</p>
              <p>{payment.paid ? 'Registration fee paid' : 'Payment pending'}</p>
              {payment.amount && <p>Amount: {payment.amount}</p>}
            </div>
          </TooltipContent>
        </Tooltip>
      </TooltipProvider>
    );
  };

  // Render tournament badges
  const renderTournamentBadges = (team: any) => {
    if (!team.tournaments || team.tournaments.length === 0) {
      return <span className="text-gray-400 text-sm">No tournaments</span>;
    }

    return (
      <div className="flex flex-wrap gap-1">
        {team.tournaments.map((tournament: any, index: number) => (
          <TooltipProvider key={tournament.id}>
            <Tooltip>
              <TooltipTrigger>
                <Badge 
                  variant="outline" 
                  className="text-xs cursor-help"
                >
                  <Trophy className="w-3 h-3 mr-1" />
                  {tournament.name.length > 15 ? `${tournament.name.substring(0, 15)}...` : tournament.name}
                </Badge>
              </TooltipTrigger>
              <TooltipContent>
                <div className="text-xs">
                  <p className="font-semibold">{tournament.name}</p>
                  <p className="text-gray-400">Status: {tournament.pivot?.status || 'registered'}</p>
                  <p className="text-gray-400">Approval: {formatApprovalStatus(tournament.pivot?.approval_status || 'pending')}</p>
                </div>
              </TooltipContent>
            </Tooltip>
          </TooltipProvider>
        ))}
      </div>
    );
  };

  // Render requirements badge
  const renderRequirementsBadge = (team: any) => {
    const requirements = getTeamRequirements(team);
    
    return (
      <TooltipProvider>
        <Tooltip>
          <TooltipTrigger>
            <div className="flex items-center gap-2">
              <div className="flex items-center gap-1">
                <Users className={`w-4 h-4 ${requirements.meetsMinimum ? 'text-green-600' : 'text-red-600'}`} />
                <span className={`text-sm font-medium ${requirements.meetsMinimum ? 'text-green-600' : 'text-red-600'}`}>
                  {requirements.playerCount}
                </span>
              </div>
              {requirements.meetsRecommended ? (
                <CheckCircle className="w-4 h-4 text-green-600" />
              ) : requirements.meetsMinimum ? (
                <AlertCircle className="w-4 h-4 text-yellow-600" />
              ) : (
                <XCircle className="w-4 h-4 text-red-600" />
              )}
            </div>
          </TooltipTrigger>
          <TooltipContent>
            <div className="text-xs space-y-1">
              <p className="font-semibold">Team Requirements</p>
              <p className={requirements.meetsMinimum ? 'text-green-400' : 'text-red-400'}>
                Minimum ({requirements.minPlayers}): {requirements.meetsMinimum ? '✓ Met' : '✗ Not Met'}
              </p>
              <p className={requirements.meetsRecommended ? 'text-green-400' : 'text-yellow-400'}>
                Recommended ({requirements.recommendedPlayers}): {requirements.meetsRecommended ? '✓ Met' : '○ Not Met'}
              </p>
              <div className="pt-1">
                <Progress value={requirements.percentage} className="h-1" />
              </div>
            </div>
          </TooltipContent>
        </Tooltip>
      </TooltipProvider>
    );
  };

  // Edit form state
  const [editForm, setEditForm] = useState({
    id: null,
    name: '',
    status: '',
    tournament_id: '',
    captain_id: '',
  });
  const [editFormErrors, setEditFormErrors] = useState<Record<string, string>>({});

  // Helper function to get tournament names for a team
  const getTeamTournaments = (team: any) => {
    if (team && team.tournaments && Array.isArray(team.tournaments)) {
      return team.tournaments.map((t: any) => t.name).join(', ');
    }
    return 'No tournaments';
  };

  // Helper function to get tournament status for a team
  const getTeamTournamentStatus = (team: any) => {
    if (team && team.tournaments && Array.isArray(team.tournaments)) {
      const statuses = team.tournaments.map((t: any) => t.pivot?.status || 'unknown');
      return statuses.join(', ');
    }
    return 'No tournaments';
  };

  // Helper function to get approval status for a team
  const getTeamApprovalStatus = (team: any, tournamentId: number) => {
    if (team && team.tournaments && Array.isArray(team.tournaments)) {
      const tournament = team.tournaments.find((t: any) => t.id === tournamentId);
      return tournament?.pivot?.approval_status || 'pending';
    }
    return 'pending';
  };

  // Helper function to get approval status badge color
  const getApprovalStatusColor = (status: string) => {
    const colors: Record<string, string> = {
      'pending': 'bg-yellow-100 text-yellow-800 border-yellow-200',
      'first_approved': 'bg-blue-100 text-blue-800 border-blue-200',
      'second_approved': 'bg-green-100 text-green-800 border-green-200',
      'rejected': 'bg-red-100 text-red-800 border-red-200',
      'active': 'bg-green-100 text-green-800 border-green-200',
    };
    return colors[status] || 'bg-gray-100 text-gray-800 border-gray-200';
  };

  // Helper function to check if team can be approved
  const canApproveTeam = (team: any, tournamentId: number) => {
    const status = getTeamApprovalStatus(team, tournamentId);
    return status === 'pending' || status === 'first_approved';
  };

  // Helper function to check if team can be first approved
  const canFirstApproveTeam = (team: any, tournamentId: number) => {
    const status = getTeamApprovalStatus(team, tournamentId);
    return status === 'pending';
  };

  // Helper function to check if team can be second approved
  const canSecondApproveTeam = (team: any, tournamentId: number) => {
    const status = getTeamApprovalStatus(team, tournamentId);
    return status === 'first_approved';
  };

  // Handle filter form submit
  const handleFilterSubmit = (e: React.FormEvent<HTMLFormElement>) => {
    e.preventDefault();
    const filters: any = {};
    
    if (localFilters.search) {
      filters.search = localFilters.search;
    }
    
    if (localFilters.tournament_id && localFilters.tournament_id !== '0') {
      filters.tournament_id = localFilters.tournament_id;
    }
    
    router.get(route('admin.teams.index'), {
      ...filters,
      page: 1,
    }, { preserveState: true, replace: true });
  };

  // Handle pagination
  const handlePageChange = (page: number) => {
    const filters: any = {};
    
    if (localFilters.search) {
      filters.search = localFilters.search;
    }
    
    if (localFilters.tournament_id && localFilters.tournament_id !== '0') {
      filters.tournament_id = localFilters.tournament_id;
    }
    
    console.log('Pagination request:', { ...filters, page });
    
    router.get(route('admin.teams.index'), {
      ...filters,
      page,
    }, { preserveState: true, replace: true });
  };



  // Open edit modal and populate form
  const openEditModal = (t: any) => {
    setEditingTeam(t);
    setEditForm({
      id: t.id,
      name: t.name,
      status: t.status,
      tournament_id: t.tournaments?.[0]?.id || '',
      captain_id: t.captain_id,
    });
    setEditFormErrors({});
  };

  // Handle edit form submit
  const handleEditSubmit = (e: React.FormEvent<HTMLFormElement>) => {
    e.preventDefault();
    router.put(route('admin.teams.update', { team: editForm.id }), editForm, {
      onError: (errors) => setEditFormErrors(errors),
      onSuccess: () => {
        setEditingTeam(null);
        setEditForm({ id: null, name: '', status: '', tournament_id: '', captain_id: '' });
        setEditFormErrors({});
      },
    });
  };

  // Handle single team selection
  const handleTeamSelect = (teamId: number, checked: boolean) => {
    if (checked) {
      setSelectedTeams([...selectedTeams, teamId]);
    } else {
      setSelectedTeams(selectedTeams.filter(id => id !== teamId));
    }
  };

  // Handle select all teams
  const handleSelectAll = (checked: boolean) => {
    if (checked) {
      setSelectedTeams((teams?.data || []).map((t: any) => t.id));
    } else {
      setSelectedTeams([]);
    }
  };

  // Handle delete single team
  const handleDelete = (team: any) => {
    setTeamToDelete(team);
    setIsDeleteModalOpen(true);
  };

  // Confirm delete
  const confirmDelete = () => {
    if (teamToDelete) {
      router.delete(route('admin.teams.destroy', { team: teamToDelete.id }), {
        onSuccess: () => {
          setIsDeleteModalOpen(false);
          setTeamToDelete(null);
        },
      });
    }
  };

  // Handle multiple delete
  const handleMultipleDelete = () => {
    if (selectedTeams.length > 0) {
      setIsMultipleDeleteModalOpen(true);
    }
  };

  // Confirm multiple delete
  const confirmMultipleDelete = () => {
    router.delete(route('admin.teams.destroyMultiple'), {
      data: { ids: selectedTeams },
      onSuccess: () => {
        setIsMultipleDeleteModalOpen(false);
        setSelectedTeams([]);
      },
    });
  };

  // Handle approval actions
  const openApprovalModal = (team: any, tournament: any, action: 'approve' | 'reject' | 'first-approve' | 'second-approve') => {
    setApprovalModal({
      isOpen: true,
      team,
      tournament,
      action
    });
    setApprovalNotes('');
    setRejectionReason('');
  };

  const handleApprovalAction = () => {
    const { team, tournament, action } = approvalModal;
    
    if (action === 'reject' && !rejectionReason.trim()) {
      return;
    }

    const data: any = {
      tournament_id: tournament.id,
    };

    if (action === 'reject') {
      data.reason = rejectionReason;
    } else {
      data.notes = approvalNotes;
    }

    const routeName = `admin.teams.${action}`;
    router.post(route(routeName, { team: team.id }), data, {
      onSuccess: () => {
        setApprovalModal({ isOpen: false, team: null, tournament: null, action: 'approve' });
        setApprovalNotes('');
        setRejectionReason('');
      },
    });
  };

  const handleBulkApprove = () => {
    if (selectedTeams.length === 0) return;
    
    const tournamentId = localFilters.tournament_id || tournaments?.[0]?.id;
    if (!tournamentId) return;

    router.post(route('admin.teams.bulk-approve'), {
      team_ids: selectedTeams,
      tournament_id: tournamentId,
      notes: 'Bulk approval by admin'
    }, {
      onSuccess: () => {
        setSelectedTeams([]);
      },
    });
  };

  return (
    <AppLayout>
      <Head title="Teams" />
      
      <div className="p-6">
          <div className="flex justify-between items-center mb-6">         
           <Button asChild>
             <Link href={route('admin.teams.create')}>Create Team</Link>
           </Button>
         </div>

        {/* Filters */}
        <Card className="p-4 mb-6">
          <form onSubmit={handleFilterSubmit} className="flex flex-wrap gap-4">
            <div className="flex-1 min-w-[200px]">
              <Label htmlFor="search">Search</Label>
              <Input
                id="search"
                value={localFilters.search}
                onChange={(e) => setLocalFilters({ ...localFilters, search: e.target.value })}
                placeholder="Search teams..."
              />
            </div>
            <div className="w-[150px]">
              <Label htmlFor="status">Status</Label>
              <Select value={localFilters.status} onValueChange={(value) => setLocalFilters({ ...localFilters, status: value })}>
                <SelectTrigger>
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Status</SelectItem>
                  <SelectItem value="active">Active</SelectItem>
                  <SelectItem value="eliminated">Eliminated</SelectItem>
                  <SelectItem value="withdrawn">Withdrawn</SelectItem>
                </SelectContent>
              </Select>
            </div>
            <div className="w-[200px]">
              <Label htmlFor="tournament">Tournament</Label>
              <Select value={localFilters.tournament_id} onValueChange={(value) => setLocalFilters({ ...localFilters, tournament_id: value })}>
                <SelectTrigger>
                  <SelectValue placeholder="All tournaments" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="0">All tournaments</SelectItem>
                  {tournaments && Array.isArray(tournaments) ? tournaments.map((tournament: any) => (
                    <SelectItem key={tournament.id} value={tournament.id.toString()}>
                      {tournament.name}
                    </SelectItem>
                  )) : null}
                </SelectContent>
              </Select>
            </div>
            <div className="flex items-end">
              <Button type="submit">Filter</Button>
            </div>
          </form>
        </Card>

        {/* Bulk Actions */}
        {selectedTeams.length > 0 && (canApproveTeams || canFirstApprove || canSecondApprove || canDeleteTeams) && (
          <div className="mb-4 p-4 bg-blue-50 border border-blue-200 rounded-lg">
            <div className="flex items-center justify-between">
              <span className="text-blue-800">
                {selectedTeams.length} team(s) selected
              </span>
              <div className="flex gap-2">
                {(canApproveTeams || canFirstApprove || canSecondApprove) && (
                <Button 
                  variant="default" 
                  size="sm" 
                  onClick={handleBulkApprove}
                  className="bg-green-600 hover:bg-green-700"
                >
                  Approve Selected
                </Button>
                )}
                {canDeleteTeams && (
                <Button variant="destructive" size="sm" onClick={handleMultipleDelete}>
                  Delete Selected
                </Button>
                )}
              </div>
            </div>
          </div>
        )}

        {/* Teams List */}
        <Card>
          {/* Mobile Card View */}
          <div className="sm:hidden p-4">
                          <div className="space-y-4">
              {teams?.data?.map((t: any) => {
                const requirements = getTeamRequirements(t);
                return (
                  <div key={t.id} className="border rounded-lg p-4 space-y-3">
                    <div className="flex items-center justify-between">
                      <h3 className="font-semibold text-lg">{t.name}</h3>
                    <Badge variant={t.status === 'active' ? 'default' : 'secondary'}>
                      {t.status}
                    </Badge>
                  </div>

                    {/* Tournaments */}
                    <div>
                      <span className="text-xs text-gray-500 mb-1 block">Tournaments</span>
                      {renderTournamentBadges(t)}
                    </div>

                    {/* Requirements */}
                    <div className="flex items-center justify-between p-2 bg-gray-50 rounded-md">
                      <span className="text-xs text-gray-500">Team Requirements</span>
                      <div className="flex items-center gap-3">
                        <div className="flex items-center gap-1">
                          <Users className={`w-4 h-4 ${requirements.meetsMinimum ? 'text-green-600' : 'text-red-600'}`} />
                          <span className={`text-sm font-semibold ${requirements.meetsMinimum ? 'text-green-600' : 'text-red-600'}`}>
                            {requirements.playerCount}/{requirements.minPlayers}
                          </span>
                        </div>
                        {requirements.meetsRecommended ? (
                          <CheckCircle className="w-5 h-5 text-green-600" />
                        ) : requirements.meetsMinimum ? (
                          <AlertCircle className="w-5 h-5 text-yellow-600" />
                        ) : (
                          <XCircle className="w-5 h-5 text-red-600" />
                        )}
                      </div>
                    </div>

                    {/* Progress bar */}
                    <div>
                      <div className="flex justify-between text-xs text-gray-500 mb-1">
                        <span>Progress to recommended ({requirements.recommendedPlayers})</span>
                        <span>{Math.round(requirements.percentage)}%</span>
                      </div>
                      <Progress value={requirements.percentage} className="h-2" />
                    </div>

                    {/* Payment Status */}
                    <div className="flex justify-between items-center">
                      <span className="text-xs text-gray-500">Payment Status</span>
                      {renderPaymentBadge(t)}
                    </div>

                    {/* Approval Status */}
                    {t.tournaments && t.tournaments.length > 0 && (
                      <div className="flex justify-between items-center">
                        <span className="text-xs text-gray-500">Approval Status</span>
                        <Badge className={getApprovalStatusColor(getTeamApprovalStatus(t, t.tournaments[0].id))}>
                          {formatApprovalStatus(getTeamApprovalStatus(t, t.tournaments[0].id))}
                        </Badge>
                      </div>
                    )}

                    {/* Team Manager */}
                    <div className="flex justify-between items-center text-sm">
                      <span className="text-gray-500">Team Manager:</span>
                      <span className="font-medium">{users && Array.isArray(users) ? users.find((u: any) => u.id === t.captain_id)?.name || t.captain_id : t.captain_id}</span>
                    </div>
                  <div className="mt-3">
                    <DropdownMenu>
                      <DropdownMenuTrigger asChild>
                        <Button variant="outline" size="sm" className="w-full">
                          <MoreVertical className="h-4 w-4 mr-2" />
                          Actions
                      </Button>
                      </DropdownMenuTrigger>
                      <DropdownMenuContent align="end" className="w-48">
                        <DropdownMenuLabel>Team Actions</DropdownMenuLabel>
                        <DropdownMenuSeparator />
                        
                        {canViewTeams && (
                          <DropdownMenuItem onClick={() => router.get(route('admin.teams.show', { team: t.id }))}>
                            <Eye className="mr-2 h-4 w-4" />
                            View Details
                          </DropdownMenuItem>
                        )}
                        
                        {canEditTeams && (
                          <DropdownMenuItem onClick={() => router.get(route('admin.teams.edit', { team: t.id }))}>
                            <Edit className="mr-2 h-4 w-4" />
                            Edit Team
                          </DropdownMenuItem>
                        )}
                        
                        {t.tournaments && t.tournaments.length > 0 && (canApproveTeams || canFirstApprove || canSecondApprove || canReject) && (
                          <>
                            <DropdownMenuSeparator />
                            <DropdownMenuLabel>Approval Actions</DropdownMenuLabel>
                            
                            {(canApproveTeams || canFirstApprove || canSecondApprove) && (
                              <DropdownMenuItem onClick={() => router.get(route('admin.teams.approval-view', { team: t.id }), { tournament_id: t.tournaments[0].id })}>
                                <CheckCircle className="mr-2 h-4 w-4 text-purple-600" />
                                Review Details
                              </DropdownMenuItem>
                            )}
                            
                            {canFirstApprove && getTeamApprovalStatus(t, t.tournaments[0].id) === 'pending' && (
                              <DropdownMenuItem onClick={() => openApprovalModal(t, t.tournaments[0], 'first-approve')}>
                                <CheckCircle className="mr-2 h-4 w-4 text-blue-600" />
                            1st Approve
                              </DropdownMenuItem>
                            )}
                            
                            {canSecondApprove && getTeamApprovalStatus(t, t.tournaments[0].id) === 'first_approved' && (
                              <DropdownMenuItem onClick={() => openApprovalModal(t, t.tournaments[0], 'second-approve')}>
                                <CheckCircle className="mr-2 h-4 w-4 text-green-600" />
                            2nd Approve
                              </DropdownMenuItem>
                            )}
                            
                            {canReject && (
                              <DropdownMenuItem onClick={() => openApprovalModal(t, t.tournaments[0], 'reject')}>
                                <XCircle className="mr-2 h-4 w-4 text-red-600" />
                          Reject
                              </DropdownMenuItem>
                            )}
                          </>
                        )}
                        
                        {canDeleteTeams && (
                          <>
                            <DropdownMenuSeparator />
                            <DropdownMenuItem 
                              onClick={() => handleDelete(t)}
                              className="text-red-600 focus:text-red-600"
                            >
                              <Trash2 className="mr-2 h-4 w-4" />
                              Delete Team
                            </DropdownMenuItem>
                          </>
                        )}
                      </DropdownMenuContent>
                    </DropdownMenu>
                  </div>
                </div>
                );
              })}
            </div>
          </div>

          {/* Desktop Table View */}
          <div className="hidden sm:block">
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead className="w-12">
                                            <input
                          type="checkbox"
                          checked={selectedTeams.length === (teams?.data?.length || 0)}
                          onChange={(e: React.ChangeEvent<HTMLInputElement>) => handleSelectAll(e.target.checked)}
                          className="rounded border-gray-300"
                        />
                  </TableHead>
                  <TableHead>Name</TableHead>
                  <TableHead>Status</TableHead>
                  <TableHead>Tournaments</TableHead>
                  <TableHead>Requirements</TableHead>
                  <TableHead>Payment</TableHead>
                  <TableHead>Approval Status</TableHead>
                  <TableHead>Team Manager</TableHead>
                  <TableHead>Actions</TableHead>
                </TableRow>
              </TableHeader>
                <TableBody>
                    {teams?.data?.map((t: any) => (
                  <TableRow key={t.id}>
                    <TableCell>
                      <input
                        type="checkbox"
                        checked={selectedTeams.includes(t.id)}
                        onChange={(e: React.ChangeEvent<HTMLInputElement>) => handleTeamSelect(t.id, e.target.checked)}
                        className="rounded border-gray-300"
                      />
                    </TableCell>
                    <TableCell className="font-medium">{t.name}</TableCell>
                    <TableCell>
                      <Badge variant={t.status === 'active' ? 'default' : 'secondary'}>
                        {t.status}
                      </Badge>
                    </TableCell>
                    <TableCell>{renderTournamentBadges(t)}</TableCell>
                    <TableCell>{renderRequirementsBadge(t)}</TableCell>
                    <TableCell>{renderPaymentBadge(t)}</TableCell>
                    <TableCell>
                      {t.tournaments && t.tournaments.length > 0 ? (
                        <Badge className={getApprovalStatusColor(getTeamApprovalStatus(t, t.tournaments[0].id))+'b-radius-sm p-2'} variant="outline">
                          {formatApprovalStatus(getTeamApprovalStatus(t, t.tournaments[0].id))}
                        </Badge>
                      ) : (
                        <span className="text-gray-500">N/A</span>
                      )}
                    </TableCell>
                    <TableCell>{users && Array.isArray(users) ? users.find((u: any) => u.id === t.captain_id)?.name || t.captain_id : t.captain_id}</TableCell>
                    <TableCell>
                      <DropdownMenu>
                        <DropdownMenuTrigger asChild>
                          <Button variant="ghost" size="sm">
                            <MoreVertical className="h-4 w-4" />
                        </Button>
                        </DropdownMenuTrigger>
                        <DropdownMenuContent align="end" className="w-48">
                          <DropdownMenuLabel>Team Actions</DropdownMenuLabel>
                          <DropdownMenuSeparator />
                          
                          {canViewTeams && (
                            <DropdownMenuItem onClick={() => router.get(route('admin.teams.show', { team: t.id }))}>
                              <Eye className="mr-2 h-4 w-4" />
                              View Details
                            </DropdownMenuItem>
                          )}
                          
                          {canEditTeams && (
                            <DropdownMenuItem onClick={() => router.get(route('admin.teams.edit', { team: t.id }))}>
                              <Edit className="mr-2 h-4 w-4" />
                              Edit Team
                            </DropdownMenuItem>
                          )}
                          
                          {t.tournaments && t.tournaments.length > 0 && (canApproveTeams || canFirstApprove || canSecondApprove || canReject) && (
                            <>
                              <DropdownMenuSeparator />
                              <DropdownMenuLabel>Approval Actions</DropdownMenuLabel>
                              
                              {(canApproveTeams || canFirstApprove || canSecondApprove) && (
                                <DropdownMenuItem onClick={() => router.get(route('admin.teams.approval-view', { team: t.id }), { tournament_id: t.tournaments[0].id })}>
                                  <CheckCircle className="mr-2 h-4 w-4 text-purple-600" />
                                  Review Details
                                </DropdownMenuItem>
                              )}
                              
                              {canFirstApprove && getTeamApprovalStatus(t, t.tournaments[0].id) === 'pending' && (
                                <DropdownMenuItem onClick={() => openApprovalModal(t, t.tournaments[0], 'first-approve')}>
                                  <CheckCircle className="mr-2 h-4 w-4 text-blue-600" />
                                1st Approve
                                </DropdownMenuItem>
                              )}
                              
                              {canSecondApprove && getTeamApprovalStatus(t, t.tournaments[0].id) === 'first_approved' && (
                                <DropdownMenuItem onClick={() => openApprovalModal(t, t.tournaments[0], 'second-approve')}>
                                  <CheckCircle className="mr-2 h-4 w-4 text-green-600" />
                                2nd Approve
                                </DropdownMenuItem>
                              )}
                              
                              {canReject && (
                                <DropdownMenuItem onClick={() => openApprovalModal(t, t.tournaments[0], 'reject')}>
                                  <XCircle className="mr-2 h-4 w-4 text-red-600" />
                              Reject
                                </DropdownMenuItem>
                              )}
                          </>
                        )}
                          
                          {canDeleteTeams && (
                            <>
                              <DropdownMenuSeparator />
                              <DropdownMenuItem 
                                onClick={() => handleDelete(t)}
                                className="text-red-600 focus:text-red-600"
                              >
                                <Trash2 className="mr-2 h-4 w-4" />
                                Delete Team
                              </DropdownMenuItem>
                            </>
                          )}
                        </DropdownMenuContent>
                      </DropdownMenu>
                    </TableCell>
                  </TableRow>
                ))}
              </TableBody>
            </Table>
          </div>

          {/* Pagination */}
          <div className="flex flex-col sm:flex-row justify-between items-center p-4 border-t gap-4">
            <div className="text-sm text-gray-500 text-center sm:text-left">
              Showing {(teams?.from || 0)} to {(teams?.to || 0)} of {teams?.total || 0} results
            </div>
            <div className="flex gap-2 items-center">
              <Button
                variant="outline"
                size="sm"
                disabled={teams?.current_page === 1}
                onClick={() => handlePageChange((teams?.current_page || 1) - 1)}
              >
                Previous
              </Button>
              
              <span className="text-sm text-gray-600 px-2">
                Page {teams?.current_page || 1} of {teams?.last_page || 1}
              </span>
              
              <Button
                variant="outline"
                size="sm"
                disabled={teams?.current_page === teams?.last_page}
                onClick={() => handlePageChange((teams?.current_page || 1) + 1)}
              >
                Next
              </Button>
            </div>
          </div>
        </Card>

     

        {/* Edit Team Modal */}
        <Dialog open={!!editingTeam} onOpenChange={() => setEditingTeam(null)}>
          <DialogContent>
            <DialogHeader>
              <DialogTitle>Edit Team</DialogTitle>
            </DialogHeader>
            <form onSubmit={handleEditSubmit} className="space-y-4">
              <div>
                <Label htmlFor="edit_name">Name</Label>
                <Input
                  id="edit_name"
                  value={editForm.name}
                  onChange={(e) => setEditForm({ ...editForm, name: e.target.value })}
                  required
                />
                {editFormErrors.name && <p className="text-red-500 text-sm">{editFormErrors.name}</p>}
              </div>
              <div>
                <Label htmlFor="edit_status">Status</Label>
                <Select value={editForm.status} onValueChange={(value) => setEditForm({ ...editForm, status: value })}>
                  <SelectTrigger>
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="active">Active</SelectItem>
                    <SelectItem value="eliminated">Eliminated</SelectItem>
                    <SelectItem value="withdrawn">Withdrawn</SelectItem>
                  </SelectContent>
                </Select>
                {editFormErrors.status && <p className="text-red-500 text-sm">{editFormErrors.status}</p>}
              </div>
              <div>
                <Label htmlFor="edit_tournament_id">Tournament</Label>
                <Select value={editForm.tournament_id} onValueChange={(value) => setEditForm({ ...editForm, tournament_id: value })}>
                  <SelectTrigger>
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    {tournaments && Array.isArray(tournaments) ? tournaments.map((tournament: any) => (
                      <SelectItem key={tournament.id} value={tournament.id.toString()}>
                        {tournament.name}
                      </SelectItem>
                    )) : null}
                  </SelectContent>
                </Select>
                {editFormErrors.tournament_id && <p className="text-red-500 text-sm">{editFormErrors.tournament_id}</p>}
              </div>
              <div>
                <Label htmlFor="edit_captain_id">Captain</Label>
                <Select value={editForm.captain_id} onValueChange={(value) => setEditForm({ ...editForm, captain_id: value })}>
                  <SelectTrigger>
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    {users && Array.isArray(users) ? users.map((user: any) => (
                      <SelectItem key={user.id} value={user.id.toString()}>
                        {user.name}
                      </SelectItem>
                    )) : null}
                  </SelectContent>
                </Select>
                {editFormErrors.captain_id && <p className="text-red-500 text-sm">{editFormErrors.captain_id}</p>}
              </div>
              <div className="flex gap-2">
                <Button type="submit">Update</Button>
                <Button type="button" variant="outline" onClick={() => setEditingTeam(null)}>
                  Cancel
                </Button>
              </div>
            </form>
          </DialogContent>
        </Dialog>

        {/* Delete Confirmation Modal */}
        <Dialog open={isDeleteModalOpen} onOpenChange={setIsDeleteModalOpen}>
          <DialogContent>
            <DialogHeader>
              <DialogTitle>Delete Team</DialogTitle>
            </DialogHeader>
            <p>Are you sure you want to delete "{teamToDelete?.name}"?</p>
            <div className="flex gap-2">
              <Button variant="destructive" onClick={confirmDelete}>
                Delete
              </Button>
              <Button variant="outline" onClick={() => setIsDeleteModalOpen(false)}>
                Cancel
              </Button>
            </div>
          </DialogContent>
        </Dialog>

        {/* Multiple Delete Confirmation Modal */}
        <Dialog open={isMultipleDeleteModalOpen} onOpenChange={setIsMultipleDeleteModalOpen}>
          <DialogContent>
            <DialogHeader>
              <DialogTitle>Delete Multiple Teams</DialogTitle>
            </DialogHeader>
            <p>Are you sure you want to delete {selectedTeams.length} team(s)?</p>
            <div className="flex gap-2">
              <Button variant="destructive" onClick={confirmMultipleDelete}>
                Delete
              </Button>
              <Button variant="outline" onClick={() => setIsMultipleDeleteModalOpen(false)}>
                Cancel
              </Button>
            </div>
          </DialogContent>
        </Dialog>

        {/* Approval Modal */}
        <Dialog open={approvalModal.isOpen} onOpenChange={() => setApprovalModal({ isOpen: false, team: null, tournament: null, action: 'approve' })}>
          <DialogContent>
            <DialogHeader>
              <DialogTitle>
                {approvalModal.action === 'approve' && 'Approve Team'}
                {approvalModal.action === 'reject' && 'Reject Team'}
                {approvalModal.action === 'first-approve' && 'First Approve Team'}
                {approvalModal.action === 'second-approve' && 'Second Approve Team'}
              </DialogTitle>
            </DialogHeader>
            <div className="space-y-4">
              <div>
                <p className="text-sm text-gray-600">
                  Team: <span className="font-semibold">{approvalModal.team?.name}</span>
                </p>
                <p className="text-sm text-gray-600">
                  Tournament: <span className="font-semibold">{approvalModal.tournament?.name}</span>
                </p>
              </div>
              
              {approvalModal.action === 'reject' ? (
                <div>
                  <Label htmlFor="rejection_reason">Rejection Reason *</Label>
                  <Textarea
                    id="rejection_reason"
                    value={rejectionReason}
                    onChange={(e) => setRejectionReason(e.target.value)}
                    placeholder="Please provide a reason for rejection..."
                    required
                    rows={3}
                  />
                </div>
              ) : (
                <div>
                  <Label htmlFor="approval_notes">Notes (Optional)</Label>
                  <Textarea
                    id="approval_notes"
                    value={approvalNotes}
                    onChange={(e) => setApprovalNotes(e.target.value)}
                    placeholder="Add any notes about this approval..."
                    rows={3}
                  />
                </div>
              )}
              
              <div className="flex gap-2">
                <Button 
                  onClick={handleApprovalAction}
                  className={
                    approvalModal.action === 'reject' 
                      ? 'bg-red-600 hover:bg-red-700' 
                      : 'bg-green-600 hover:bg-green-700'
                  }
                >
                  {approvalModal.action === 'approve' && 'Approve Team'}
                  {approvalModal.action === 'reject' && 'Reject Team'}
                  {approvalModal.action === 'first-approve' && 'First Approve'}
                  {approvalModal.action === 'second-approve' && 'Second Approve'}
                </Button>
                <Button variant="outline" onClick={() => setApprovalModal({ isOpen: false, team: null, tournament: null, action: 'approve' })}>
                  Cancel
                </Button>
              </div>
            </div>
          </DialogContent>
        </Dialog>
      </div>
    </AppLayout>
  );
} 