import React, { useState, useEffect } from 'react';
import { Head, router } from '@inertiajs/react';
import AppLayout from '@/layouts/app-layout';
import TeamForm from '@/components/teams/team-form';
import { Button } from '@/components/ui/button';
import { ArrowLeft, Save } from 'lucide-react';

interface Props {
  team: any;
  tournaments: any[];
  users: any[];
  errors: Record<string, string>;
}

const TeamEdit: React.FC<Props> = ({ team, tournaments, users, errors }) => {
  const [form, setForm] = useState({
    name: '',
    description: '',
    tournament_id: '',
    captain_id: '',
    status: 'active',
  });

  useEffect(() => {
    if (team) {
      setForm({
        name: team.name || '',
        description: team.description || '',
        tournament_id: team.tournament_id ? String(team.tournament_id) : '',
        captain_id: team.captain_id ? String(team.captain_id) : '',
        status: team.status || 'active',
      });
    }
  }, [team]);

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    router.put(route('admin.teams.update', { team: team.id }), form, {
      onError: (errors) => {
      },
      onSuccess: () => {
      },
    });
  };

  return (
    <AppLayout breadcrumbs={[
      { title: 'Teams', href: '/admin/teams' },
      { title: team.name, href: `/admin/teams/${team.id}` },
      { title: 'Edit', href: `/admin/teams/${team.id}/edit` }
    ]}>
      <Head title={`Edit Team: ${team.name}`} />
      
      <div className="space-y-6 p-4">
        {/* Header */}
        <div className="flex items-center justify-between">
          <div className="flex items-center gap-4">
            <Button
              variant="outline"
              size="sm"
              onClick={() => router.get(route('admin.teams.show', { team: team.id }))}
            >
              <ArrowLeft className="w-4 h-4 mr-2" />
              Back to Team
            </Button>
            
          </div>
        </div>

        {/* Form */}
        <form onSubmit={handleSubmit} className="space-y-6">
          <TeamForm
            form={form}
            setForm={setForm}
            errors={errors}
            tournaments={tournaments}
            users={users}
            isEditing={true}
          />

          {/* Form Actions */}
          <div className="flex justify-end gap-4 pt-6 border-t">
            <Button
              type="button"
              variant="outline"
              onClick={() => router.get(route('admin.teams.show', { team: team.id }))}
            >
              Cancel
            </Button>
            <Button type="submit">
              <Save className="w-4 h-4 mr-2" />
              Update Team
            </Button>
          </div>
        </form>
      </div>
    </AppLayout>
  );
};

export default TeamEdit; 