import React, { useState, useEffect } from 'react';
import { Head, router } from '@inertiajs/react';
import AppLayout from '@/layouts/app-layout';
import TeamFormSteps from '@/components/teams/team-form-steps';
import BasicTeamInfoStep from '@/components/teams/form-steps/basic-team-info-step';
import CaptainSelectionStep from '@/components/teams/form-steps/captain-selection-step';
import PlayersSelectionStep from '@/components/teams/form-steps/players-selection-step';
import MPesaPaymentStep from '@/components/teams/form-steps/mpesa-payment-step';
import TeamSettingsStep from '@/components/teams/form-steps/team-settings-step';
import { Button } from '@/components/ui/button';
import { ArrowLeft, Save, Users, Crown, Shield, Smartphone, CheckCircle } from 'lucide-react';

interface Props {
  tournaments: any[];
  users: any[];
  registration_fee: number;
  errors: Record<string, string>;
}

const TeamCreate: React.FC<Props> = ({ tournaments, users, registration_fee, errors }) => {
  const [currentStep, setCurrentStep] = useState(0);
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [draftId, setDraftId] = useState<string | null>(null);
  
  const [form, setForm] = useState({
    name: '',
    description: '',
    tournament_id: '',
    captain_id: '',
    status: 'active',
    logo: '',
    max_players: '',
    notes: '',
    notify_match_updates: true,
    notify_tournament_updates: true,
    // Players array
    players: Array.from({ length: 11 }, () => ({
      name: '',
      email: '',
      phone: '',
      position: 'Midfielder',
      jersey_number: '',
      existing_user_id: null
    })),
    // Payment fields
    phone_number: '',
    payment_method: 'mpesa',
    payment_status: 'pending',
    transaction_id: '',
    payment_amount: registration_fee, // Use tournament registration fee
  });

  const [formErrors, setFormErrors] = useState<any>({});

  const steps = [
    {
      id: 'foundation',
      title: '🏗️ Build Your Foundation',
      description: 'Establish your team identity and tournament selection',
      icon: <Users className="w-5 h-5" />,
    },
    {
      id: 'leadership',
      title: '👑 Choose Your Champion',
      description: 'Select the captain who will lead your warriors to victory',
      icon: <Crown className="w-5 h-5" />,
    },
    {
      id: 'warriors',
      title: '⚽ Assemble Your Squad',
      description: 'Recruit your team of football heroes',
      icon: <Shield className="w-5 h-5" />,
    },
    {
      id: 'investment',
      title: '💳 Secure Your Spot',
      description: 'Complete your tournament registration with M-Pesa',
      icon: <Smartphone className="w-5 h-5" />,
    },
    {
      id: 'victory',
      title: '🎯 Final Preparations',
      description: 'Review everything and prepare for glory',
      icon: <CheckCircle className="w-5 h-5" />,
    },
  ];

  // Auto-save draft functionality
  useEffect(() => {
    const generateDraftId = () => {
      const id = `team_draft_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;
      setDraftId(id);
      return id;
    };

    // Try to load existing draft
    const savedDraft = localStorage.getItem(`team_creation_draft`);
    if (savedDraft) {
      try {
        const parsedDraft = JSON.parse(savedDraft);
        setForm(parsedDraft.form);
        setCurrentStep(parsedDraft.currentStep || 0);
        setDraftId(parsedDraft.draftId);
      } catch (error) {
        generateDraftId();
      }
    } else {
      generateDraftId();
    }
  }, []);

  // Save draft whenever form changes
  useEffect(() => {
    if (draftId && Object.keys(form).some(key => form[key as keyof typeof form] !== '')) {
      const draftData = {
        form,
        currentStep,
        draftId,
        timestamp: Date.now()
      };
      localStorage.setItem(`team_creation_draft`, JSON.stringify(draftData));
    }
  }, [form, currentStep, draftId]);

  const clearDraft = () => {
    localStorage.removeItem(`team_creation_draft`);
    setDraftId(null);
  };

  const handleInputChange = (field: string, value: any) => {
    setForm((prev: any) => ({ ...prev, [field]: value }));
    if (formErrors[field]) {
      setFormErrors((prev: any) => ({ ...prev, [field]: null }));
    }
  };

  const validateCurrentStep = () => {
    const newErrors: any = {};
    
    switch (currentStep) {
      case 0: // Foundation - Basic Information
        if (!form.name) newErrors.name = 'Team name is required';
        if (!form.tournament_id) newErrors.tournament_id = 'Tournament selection is required';
        if (!form.status) newErrors.status = 'Team status is required';
        break;
      case 1: // Leadership - Captain Selection
        if (!form.captain_id) newErrors.captain_id = 'Team captain selection is required';
        break;
      case 2: // Warriors - Players Selection
        const validPlayers = form.players.filter(p => p.name && p.email && p.phone);
        if (validPlayers.length < 11) {
          newErrors.players = 'At least 11 players are required';
        }
        // Validate each player
        form.players.forEach((player, index) => {
          if (player.name || player.email || player.phone) { // If any field is filled, all are required
            if (!player.name) newErrors[`players.${index}.name`] = 'Player name is required';
            if (!player.email) newErrors[`players.${index}.email`] = 'Player email is required';
            if (!player.phone) newErrors[`players.${index}.phone`] = 'Player phone is required';
          }
        });
        break;
      case 3: // Investment - M-Pesa Payment
        if (!form.phone_number) newErrors.phone_number = 'Phone number is required for M-Pesa payment';
        // Validate Kenyan phone number format
        const phoneRegex = /^254[0-9]{9}$/;
        if (form.phone_number && !phoneRegex.test(form.phone_number.replace(/[\s\-\+]/g, ''))) {
          newErrors.phone_number = 'Please enter a valid Kenyan phone number (e.g., 254712345678)';
        }
        break;
      case 4: // Victory - Final Review
        // No additional validation needed for this step
        break;
    }

    setFormErrors(newErrors);
    return Object.keys(newErrors).length === 0;
  };

  const handleNext = () => {
    if (validateCurrentStep() && currentStep < steps.length - 1) {
      setCurrentStep(currentStep + 1);
    }
  };

  const handlePrevious = () => {
    if (currentStep > 0) {
      setCurrentStep(currentStep - 1);
    }
  };

  const handleStepChange = (step: number) => {
    // Allow going back to any previous step, but validate current step before going forward
    if (step <= currentStep || validateCurrentStep()) {
      setCurrentStep(step);
    }
  };

  const handlePaymentComplete = (paymentData: any) => {
    // Update form with payment data
    setForm(prev => ({
      ...prev,
      payment_status: 'completed',
      transaction_id: paymentData.transaction_id,
      payment_amount: paymentData.amount
    }));

    // Move to final review step
    setCurrentStep(4);
  };

  const handleSubmit = () => {
    // For payment step, payment completion is handled separately
    if (currentStep === 3) {
      return;
    }

    if (!validateCurrentStep()) return;

    setIsSubmitting(true);
    router.post(route('admin.teams.store'), form, {
      onError: (errors) => {
        setFormErrors(errors);
        setIsSubmitting(false);
        // Go back to the step with errors
        if (errors.name || errors.tournament_id || errors.status) setCurrentStep(0);
        else if (errors.captain_id) setCurrentStep(1);
        else if (errors.players) setCurrentStep(2);
        else if (errors.phone_number) setCurrentStep(3);
      },
      onSuccess: () => {
        clearDraft(); // Clear the saved draft on successful submission
        router.get(route('admin.teams.index'));
      },
      onFinish: () => {
        setIsSubmitting(false);
      }
    });
  };

  const renderCurrentStep = () => {
    switch (currentStep) {
      case 0: // Foundation
        return (
          <BasicTeamInfoStep
            form={form}
            errors={formErrors}
            onInputChange={handleInputChange}
            tournaments={tournaments}
          />
        );
      case 1: // Leadership
        return (
          <CaptainSelectionStep
            form={form}
            errors={formErrors}
            onInputChange={handleInputChange}
            users={users}
          />
        );
      case 2: // Warriors
        return (
          <PlayersSelectionStep
            form={form}
            errors={formErrors}
            onInputChange={handleInputChange}
            users={users}
          />
        );
      case 3: // Investment
        return (
          <MPesaPaymentStep
            form={form}
            errors={formErrors}
            onInputChange={handleInputChange}
            onPaymentComplete={handlePaymentComplete}
            teamData={{
              name: form.name,
              captain_name: users.find(u => u.id === parseInt(form.captain_id))?.name || 'Not selected',
              tournament: tournaments.find(t => t.id === parseInt(form.tournament_id))?.name || 'Not selected',
              players_count: form.players.filter(p => p.name && p.email).length
            }}
            registrationFee={registration_fee}
          />
        );
      case 4: // Victory
        return (
          <TeamSettingsStep
            form={form}
            errors={formErrors}
            onInputChange={handleInputChange}
            tournaments={tournaments}
            users={users}
          />
        );
      default:
        return null;
    }
  };

  return (
    <AppLayout breadcrumbs={[
      { title: 'Teams', href: '/admin/teams' },
      { title: 'Create Team', href: '/admin/teams/create' }
    ]}>
      <Head title="Create Team" />
      
      <div className="space-y-6 p-4">
        {/* Header */}
        <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
          <div className="flex flex-col sm:flex-row sm:items-center gap-4">
            <Button
              variant="outline"
              size="sm"
              onClick={() => router.get(route('admin.teams.index'))}
              className="w-fit"
            >
              <ArrowLeft className="w-4 h-4 mr-2" />
              Back to Teams
            </Button>          
          </div>
        </div>

        {/* Form Steps */}
        <TeamFormSteps
          steps={steps}
          currentStep={currentStep}
          onStepChange={handleStepChange}
          onNext={handleNext}
          onPrevious={handlePrevious}
          onSubmit={handleSubmit}
          isSubmitting={isSubmitting}
          canProceed={true}
          isLastStep={currentStep === steps.length - 1}
        >
          {renderCurrentStep()}
        </TeamFormSteps>
      </div>
    </AppLayout>
  );
};

export default TeamCreate; 