import React, { useState } from 'react';
import AppLayout from '@/layouts/app-layout';
import { Head, router } from '@inertiajs/react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Progress } from '@/components/ui/progress';
import { Dialog, DialogContent, DialogHeader, DialogTitle } from '@/components/ui/dialog';
import { Textarea } from '@/components/ui/textarea';
import { Label } from '@/components/ui/label';
import { Input } from '@/components/ui/input';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { usePermissions } from '@/hooks/use-permissions';
import { 
  Users, 
  Trophy, 
  User, 
  Calendar,
  ArrowLeft,
  CheckCircle,
  XCircle,
  AlertCircle,
  FileText,
  Shield,
  UserCheck,
  UserX,
  Download,
  Eye,
  DollarSign,
  CreditCard
} from 'lucide-react';

interface Props {
  team: any;
  tournament: any;
  approvalSummary: any;
  userApprovalPermissions: any;
  tournaments: any[];
  tournamentSettings?: { min_players_per_team: number, max_players_per_team: number };
}

const TeamApprovalView: React.FC<Props> = ({ team, tournament, approvalSummary, userApprovalPermissions, tournaments, tournamentSettings }) => {
  // Permission checks
  const { hasPermission } = usePermissions();
  const canApproveTeams = hasPermission(['teams.approve', 'tournaments.approve_participants']);
  const canFirstApprove = hasPermission('tournaments.first_approve_participants');
  const canSecondApprove = hasPermission('tournaments.second_approve_participants');
  const canReject = hasPermission(['teams.approve', 'tournaments.approve_participants']);
  const canManagePayments = hasPermission('teams.manage_payments');

  // Get tournament-specific requirements directly from tournament object
  const requirements = {
    minPlayers: tournament?.min_players_per_team ?? tournamentSettings?.min_players_per_team ?? 11,
    maxPlayers: tournament?.max_players_per_team ?? tournamentSettings?.max_players_per_team ?? 23,
  };

  // Get payment status
  const getPaymentStatus = () => {
    if (team.payment_status === 'completed') {
      return { paid: true, status: 'completed' };
    }
    
    if (team.payments && team.payments.length > 0) {
      const completedPayment = team.payments.find((p: any) => p.status === 'completed');
      if (completedPayment) {
        return { paid: true, status: 'completed', amount: completedPayment.amount, currency: completedPayment.currency };
      }
    }
    
    return { paid: false, status: team.payment_status || 'pending' };
  };

  const [playerApprovalModal, setPlayerApprovalModal] = useState<{
    isOpen: boolean;
    player: any;
    action: 'approve' | 'reject';
  }>({
    isOpen: false,
    player: null,
    action: 'approve'
  });
  const [approvalNotes, setApprovalNotes] = useState('');
  const [rejectionReason, setRejectionReason] = useState('');

  // Payment modal state
  const [isPaymentModalOpen, setIsPaymentModalOpen] = useState(false);
  const [paymentForm, setPaymentForm] = useState({
    tournament_id: tournament.id.toString(),
    amount: tournament.entry_fee || '',
    currency: tournament.currency || 'KES',
    payment_method: 'mpesa',
    reference_number: '',
    transaction_id: '',
    description: '',
    paid_at: new Date().toISOString().split('T')[0],
  });
  const [paymentFormErrors, setPaymentFormErrors] = useState<Record<string, string>>({});

  // Format approval status
  const formatApprovalStatus = (status: string) => {
    return status
      .split('_')
      .map(word => word.charAt(0).toUpperCase() + word.slice(1))
      .join(' ');
  };

  const getStatusColor = (status: string) => {
    const colors: Record<string, string> = {
      'pending': 'bg-yellow-100 text-yellow-800 border-yellow-200',
      'first_approved': 'bg-blue-100 text-blue-800 border-blue-200',
      'second_approved': 'bg-green-100 text-green-800 border-green-200',
      'rejected': 'bg-red-100 text-red-800 border-red-200',
      'active': 'bg-green-100 text-green-800 border-green-200',
    };
    return colors[status] || 'bg-gray-100 text-gray-800 border-gray-200';
  };

  const getPlayerStatusColor = (status: string) => {
    const colors: Record<string, string> = {
      'approved': 'bg-green-100 text-green-800 border-green-200',
      'rejected': 'bg-red-100 text-red-800 border-red-200',
      'pending': 'bg-yellow-100 text-yellow-800 border-yellow-200',
      'verified': 'bg-green-100 text-green-800 border-green-200',
      'unverified': 'bg-gray-100 text-gray-800 border-gray-200',
    };
    return colors[status] || 'bg-gray-100 text-gray-800 border-gray-200';
  };

  const openPlayerApprovalModal = (player: any, action: 'approve' | 'reject') => {
    setPlayerApprovalModal({
      isOpen: true,
      player,
      action
    });
    setApprovalNotes('');
    setRejectionReason('');
  };

  const handlePlayerApprovalAction = () => {
    const { player, action } = playerApprovalModal;
    
    if (action === 'reject' && !rejectionReason.trim()) {
      return;
    }

    const data: any = {
      tournament_id: tournament.id,
    };

    if (action === 'reject') {
      data.reason = rejectionReason;
    } else {
      data.notes = approvalNotes;
    }

    const routeName = `admin.teams.player.${action}`;
    router.post(route(routeName, { team: team.id, player: player.id }), data, {
      onSuccess: () => {
        setPlayerApprovalModal({ isOpen: false, player: null, action: 'approve' });
        setApprovalNotes('');
        setRejectionReason('');
      },
    });
  };

  const handleBulkApprovePlayers = () => {
    router.post(route('admin.teams.players.bulk-approve', { team: team.id }), {
      tournament_id: tournament.id,
      notes: 'Bulk approval of complete players'
    });
  };

  // Handle manual payment
  const handlePaymentSubmit = (e: React.FormEvent<HTMLFormElement>) => {
    e.preventDefault();
    router.post(route('admin.teams.payments.manual', { team: team.id }), paymentForm, {
      onError: (errors) => setPaymentFormErrors(errors),
      onSuccess: () => {
        setIsPaymentModalOpen(false);
        setPaymentFormErrors({});
      },
    });
  };

  const handleTeamApproval = (action: 'approve' | 'reject' | 'first-approve' | 'second-approve') => {
    const data: any = {
      tournament_id: tournament.id,
    };

    if (action === 'reject') {
      data.reason = rejectionReason;
    } else {
      data.notes = approvalNotes;
    }

    const routeName = `admin.teams.${action}`;
    router.post(route(routeName, { team: team.id }), data);
  };

  return (
    <AppLayout breadcrumbs={[
      { title: 'Teams', href: '/admin/teams' },
      { title: team.name, href: `/admin/teams/${team.id}` },
      { title: 'Approval Review', href: `/admin/teams/${team.id}/approval-view` }
    ]}> 
      <Head title={`Team Approval: ${team.name}`} />
      
      <div className="space-y-6 p-4">
        {/* Header */}
        <div className="flex items-center justify-between">
          <div className="flex items-center gap-4">
            <Button
              variant="outline"
              size="sm"
              onClick={() => router.get(route('admin.teams.index'))}
            >
              <ArrowLeft className="w-4 h-4 mr-2" />
              Back to Teams
            </Button>
            <div>
              <h1 className="text-2xl font-bold">{team.name}</h1>
              <p className="text-gray-600">Tournament: {tournament.name}</p>
            </div>
          </div>
          
          <div className="flex gap-2">
            <Badge className={getStatusColor(approvalSummary.team.status)}>
              {formatApprovalStatus(approvalSummary.team.status)}
            </Badge>
          </div>
        </div>

        {/* Team Overview Cards */}
        <div className="grid grid-cols-1 md:grid-cols-5 gap-4">
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center gap-2">
                <Users className="w-5 h-5 text-blue-600" />
                <div>
                  <p className="text-sm text-gray-600">Total Players</p>
                  <p className="text-2xl font-bold">{approvalSummary.players.total}</p>
                </div>
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardContent className="p-4">
              <div className="flex items-center gap-2">
                <UserCheck className="w-5 h-5 text-green-600" />
                <div>
                  <p className="text-sm text-gray-600">Complete</p>
                  <p className="text-2xl font-bold">{approvalSummary.players.complete}</p>
                </div>
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardContent className="p-4">
              <div className="flex items-center gap-2">
                <AlertCircle className="w-5 h-5 text-yellow-600" />
                <div>
                  <p className="text-sm text-gray-600">Incomplete</p>
                  <p className="text-2xl font-bold">{approvalSummary.players.incomplete}</p>
                </div>
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardContent className="p-4">
              <div className="flex items-center gap-2">
                <Shield className="w-5 h-5 text-purple-600" />
                <div>
                  <p className="text-sm text-gray-600">Completeness</p>
                  <p className="text-2xl font-bold">{approvalSummary.players.completeness_percentage}%</p>
                </div>
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardContent className="p-4">
              <div className="flex items-center gap-2">
                <CheckCircle className={`w-5 h-5 ${getPaymentStatus().paid ? 'text-green-600' : 'text-yellow-600'}`} />
                <div>
                  <p className="text-sm text-gray-600">Payment</p>
                  <p className="text-lg font-bold">{getPaymentStatus().paid ? 'Paid' : 'Unpaid'}</p>
                </div>
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Team Requirements & Payment Status */}
        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Shield className="w-5 h-5" />
                Team Requirements Status
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                <div className="flex items-center justify-between">
                  <span>Minimum Players ({requirements.minPlayers})</span>
                  <div className="flex items-center gap-2">
                    {approvalSummary.requirements.minimum_players_met ? (
                      <CheckCircle className="w-5 h-5 text-green-600" />
                    ) : (
                      <XCircle className="w-5 h-5 text-red-600" />
                    )}
                    <span className={approvalSummary.requirements.minimum_players_met ? 'text-green-600 font-semibold' : 'text-red-600 font-semibold'}>
                      {approvalSummary.requirements.minimum_players_met ? 'Met' : 'Not Met'}
                    </span>
                  </div>
                </div>

                <div className="flex items-center justify-between">
                  <span>Recommended Players ({requirements.maxPlayers})</span>
                  <div className="flex items-center gap-2">
                    {approvalSummary.requirements.recommended_players_met ? (
                      <CheckCircle className="w-5 h-5 text-green-600" />
                    ) : (
                      <AlertCircle className="w-5 h-5 text-yellow-600" />
                    )}
                    <span className={approvalSummary.requirements.recommended_players_met ? 'text-green-600 font-semibold' : 'text-yellow-600 font-semibold'}>
                      {approvalSummary.requirements.recommended_players_met ? 'Met' : 'Not Met'}
                    </span>
                  </div>
                </div>

                <div className="flex items-center justify-between">
                  <span>All Players Complete</span>
                  <div className="flex items-center gap-2">
                    {approvalSummary.requirements.all_players_complete ? (
                      <CheckCircle className="w-5 h-5 text-green-600" />
                    ) : (
                      <XCircle className="w-5 h-5 text-red-600" />
                    )}
                    <span className={approvalSummary.requirements.all_players_complete ? 'text-green-600 font-semibold' : 'text-red-600 font-semibold'}>
                      {approvalSummary.requirements.all_players_complete ? 'Complete' : 'Incomplete'}
                    </span>
                  </div>
                </div>

                <div className="mt-4 space-y-2">
                  <div className="flex justify-between text-sm">
                    <span className="font-medium">Player Completeness</span>
                    <span className="font-bold">{approvalSummary.players.completeness_percentage}%</span>
                  </div>
                  <Progress value={approvalSummary.players.completeness_percentage} className="h-3" />
                  <div className="flex justify-between text-xs text-gray-500">
                    <span>{approvalSummary.players.complete} complete</span>
                    <span>{approvalSummary.players.incomplete} incomplete</span>
                  </div>
                </div>
              </div>
            </CardContent>
          </Card>

          {/* Payment Details Card */}
          <Card>
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="flex items-center gap-2">
                <CheckCircle className="w-5 h-5" />
                Payment Details
              </CardTitle>
              {canManagePayments && !getPaymentStatus().paid && (
                <Button size="sm" onClick={() => setIsPaymentModalOpen(true)} className="bg-green-600 hover:bg-green-700">
                  <CreditCard className="w-4 h-4 mr-1" />
                  Add Payment
                </Button>
              )}
            </CardHeader>
            <CardContent>
              {(() => {
                const payment = getPaymentStatus();
                return (
                  <div className="space-y-4">
                    <div className="flex items-center justify-between">
                      <span>Registration Fee</span>
                      <Badge className={payment.paid ? 'bg-green-600 text-white' : 'bg-yellow-100 text-yellow-800 border-yellow-300'}>
                        {payment.paid ? '✓ Paid' : '○ Unpaid'}
                      </Badge>
                    </div>

                    {tournament.entry_fee && (
                      <div className="flex items-center justify-between">
                        <span>Required Amount</span>
                        <span className="font-semibold">{tournament.currency || 'KES'} {tournament.entry_fee}</span>
                      </div>
                    )}

                    {payment.amount && (
                      <div className="flex items-center justify-between">
                        <span>Amount Paid</span>
                        <span className="font-semibold text-green-600">{payment.currency || 'KES'} {payment.amount}</span>
                      </div>
                    )}

                    {team.payment_method && (
                      <div className="flex items-center justify-between">
                        <span>Payment Method</span>
                        <Badge variant="outline">{team.payment_method.toUpperCase()}</Badge>
                      </div>
                    )}

                    {team.transaction_id && (
                      <div className="flex items-center justify-between">
                        <span className="text-xs text-gray-500">Transaction ID</span>
                        <span className="text-xs font-mono">{team.transaction_id}</span>
                      </div>
                    )}

                    {team.payment_completed_at && (
                      <div className="flex items-center justify-between">
                        <span>Completed</span>
                        <span className="text-sm">{new Date(team.payment_completed_at).toLocaleDateString()}</span>
                      </div>
                    )}

                    {!payment.paid && (
                      <div className="bg-yellow-50 border border-yellow-200 rounded p-3 mt-3">
                        <p className="text-sm text-yellow-800">⚠️ Payment required before approval</p>
                      </div>
                    )}
                  </div>
                );
              })()}
            </CardContent>
          </Card>
        </div>

        {/* Payment History */}
        {team.payments && team.payments.length > 0 && (
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <FileText className="w-5 h-5" />
                Payment History
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-3">
                {team.payments.map((payment: any) => (
                  <div key={payment.id} className="flex flex-col sm:flex-row sm:items-center sm:justify-between p-3 border rounded-lg gap-2">
                    <div className="space-y-1">
                      <div className="flex items-center gap-2 flex-wrap">
                        <Badge className={
                          payment.status === 'completed' ? 'bg-green-100 text-green-800' :
                          payment.status === 'pending' ? 'bg-yellow-100 text-yellow-800' :
                          payment.status === 'processing' ? 'bg-blue-100 text-blue-800' :
                          'bg-red-100 text-red-800'
                        }>
                          {payment.status.toUpperCase()}
                        </Badge>
                        <span className="text-sm font-semibold">{payment.currency || tournament.currency || 'KES'} {payment.amount}</span>
                        {payment.payment_method && (
                          <Badge variant="outline" className="text-xs">
                            {payment.payment_method.toUpperCase()}
                          </Badge>
                        )}
                      </div>
                      {payment.reference_number && (
                        <p className="text-xs text-gray-500">Reference: {payment.reference_number}</p>
                      )}
                      {payment.description && (
                        <p className="text-xs text-gray-600">{payment.description}</p>
                      )}
                    </div>
                    <div className="text-right text-xs text-gray-500">
                      {payment.paid_at ? (
                        <div>
                          <p className="font-medium text-green-600">Paid</p>
                          <p>{new Date(payment.paid_at).toLocaleDateString()}</p>
                        </div>
                      ) : (
                        <p>{new Date(payment.created_at).toLocaleDateString()}</p>
                      )}
                    </div>
                  </div>
                ))}
              </div>
            </CardContent>
          </Card>
        )}

        {/* Team Approval Actions */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Trophy className="w-5 h-5" />
              Team Approval Actions
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-4">
              {!approvalSummary.approval.can_approve && (
                <div className="bg-red-50 border border-red-200 rounded-lg p-4">
                  <h4 className="font-semibold text-red-800 mb-2">Cannot Approve Team</h4>
                  <ul className="text-red-700 text-sm space-y-1">
                    {approvalSummary.approval.reasons.map((reason: string, index: number) => (
                      <li key={index}>• {reason}</li>
                    ))}
                  </ul>
                </div>
              )}

              {approvalSummary.approval.can_approve && (
                <div className="bg-green-50 border border-green-200 rounded-lg p-4">
                  <h4 className="font-semibold text-green-800 mb-2">Team Ready for Approval</h4>
                  <p className="text-green-700 text-sm">All requirements have been met. You can proceed with team approval.</p>
                </div>
              )}

              <div className="flex gap-2 flex-wrap">
                {/* Super Admin or General Approve for non-two-tier systems */}
                {canApproveTeams && userApprovalPermissions.can_approve && (
                  <Button 
                    onClick={() => handleTeamApproval('approve')}
                    className="bg-green-600 hover:bg-green-700"
                    disabled={!approvalSummary.approval.can_approve}
                  >
                    <CheckCircle className="w-4 h-4 mr-2" />
                    {userApprovalPermissions.is_super_admin ? 'Super Approve' : 'Approve Team'}
                  </Button>
                )}
                
                {/* First Approve - Only show if user has permission AND team status is pending */}
                {canFirstApprove && approvalSummary.team.status === 'pending' && (
                  <Button 
                    onClick={() => handleTeamApproval('first-approve')}
                    variant="outline"
                    className="border-blue-600 text-blue-600 hover:bg-blue-50"
                    disabled={!approvalSummary.approval.can_approve}
                  >
                    <CheckCircle className="w-4 h-4 mr-2" />
                    1st Approve
                  </Button>
                )}
                
                {/* Second Approve - Only show if user has permission AND team status is first_approved */}
                {canSecondApprove && approvalSummary.team.status === 'first_approved' && (
                  <Button 
                    onClick={() => handleTeamApproval('second-approve')}
                    variant="outline"
                    className="border-green-600 text-green-600 hover:bg-green-50"
                    disabled={!approvalSummary.approval.can_approve}
                  >
                    <CheckCircle className="w-4 h-4 mr-2" />
                    2nd Approve
                  </Button>
                )}
                
                {/* Reject - Show if user has reject permission */}
                {canReject && (
                  <Button 
                    onClick={() => handleTeamApproval('reject')}
                    variant="destructive"
                  >
                    <XCircle className="w-4 h-4 mr-2" />
                    Reject Team
                  </Button>
                )}

                {/* Bulk Approve Players - Show if user has approval permissions */}
                {approvalSummary.players.incomplete > 0 && (canApproveTeams || canFirstApprove || canSecondApprove) && (
                  <Button 
                    onClick={handleBulkApprovePlayers}
                    variant="outline"
                    className="border-purple-600 text-purple-600 hover:bg-purple-50"
                  >
                    <UserCheck className="w-4 h-4 mr-2" />
                    Approve Complete Players
                  </Button>
                )}
              </div>
            </div>
          </CardContent>
        </Card>

        {/* Players List */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Users className="w-5 h-5" />
              Team Players ({team.players?.length || 0})
            </CardTitle>
          </CardHeader>
          <CardContent>
            {team.players && team.players.length > 0 ? (
              <div className="space-y-4">
                {team.players.map((player: any) => {
                  const completeness = approvalSummary.players.details.find((p: any) => p.player.id === player.id);
                  const isComplete = !completeness || completeness.completeness_percentage === 100;
                  
                  return (
                    <div key={player.id} className="border rounded-lg p-4">
                      <div className="flex items-center justify-between mb-3">
                        <div className="flex items-center gap-3">
                          <div className="w-10 h-10 bg-gray-200 rounded-full flex items-center justify-center">
                            {player.avatar ? (
                              <img src={player.avatar} alt={player.display_name} className="w-10 h-10 rounded-full" />
                            ) : (
                              <User className="w-5 h-5 text-gray-500" />
                            )}
                          </div>
                          <div>
                            <h4 className="font-semibold">{player.display_name}</h4>
                            <p className="text-sm text-gray-600">
                              {player.football_position} • #{player.jersey_number || 'N/A'}
                            </p>
                          </div>
                        </div>
                        
                        <div className="flex items-center gap-2">
                          <Badge className={getPlayerStatusColor(player.approval_status || 'pending')}>
                            {formatApprovalStatus(player.approval_status || 'pending')}
                          </Badge>
                          <Badge className={getPlayerStatusColor(player.verification_status || 'unverified')}>
                            {formatApprovalStatus(player.verification_status || 'unverified')}
                          </Badge>
                        </div>
                      </div>

                      <div className="grid grid-cols-1 md:grid-cols-3 gap-4 mb-3">
                        <div>
                          <p className="text-sm text-gray-600">Phone</p>
                          <p className="font-medium">{player.phone || 'Not provided'}</p>
                        </div>
                        <div>
                          <p className="text-sm text-gray-600">Date of Birth</p>
                          <p className="font-medium">{player.date_of_birth ? new Date(player.date_of_birth).toLocaleDateString() : 'Not provided'}</p>
                        </div>
                        <div>
                          <p className="text-sm text-gray-600">ID Number</p>
                          <p className="font-medium">{player.id_number || 'Not provided'}</p>
                        </div>
                      </div>

                      {/* Player Completeness */}
                      {completeness && (
                        <div className="mb-3">
                          <div className="flex justify-between items-center mb-1">
                            <span className="text-xs text-gray-600">Profile Completeness</span>
                            <span className="text-xs font-semibold">{completeness.completeness_percentage}%</span>
                          </div>
                          <Progress value={completeness.completeness_percentage} className="h-2" />
                        </div>
                      )}

                      {!isComplete && completeness && completeness.missing_fields && completeness.missing_fields.length > 0 && (
                        <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-3 mb-3">
                          <h5 className="font-semibold text-yellow-800 mb-2">Missing Information</h5>
                          <div className="flex flex-wrap gap-2">
                            {completeness.missing_fields.map((field: string, index: number) => (
                              <Badge key={index} variant="outline" className="text-yellow-700 border-yellow-300">
                                {field}
                              </Badge>
                            ))}
                          </div>
                        </div>
                      )}

                      <div className="flex gap-2">
                        <Button 
                          size="sm" 
                          variant="outline"
                          onClick={() => router.get(route('admin.players.show', { player: player.id }))}
                        >
                          <Eye className="w-4 h-4 mr-2" />
                          View Details
                        </Button>
                        
                        {(canApproveTeams || canFirstApprove || canSecondApprove) && isComplete && (player.approval_status !== 'approved') && (
                          <Button 
                            size="sm" 
                            className="bg-green-600 hover:bg-green-700"
                            onClick={() => openPlayerApprovalModal(player, 'approve')}
                          >
                            <CheckCircle className="w-4 h-4 mr-2" />
                            Approve Player
                          </Button>
                        )}
                        
                        {canReject && (
                          <Button 
                            size="sm" 
                            variant="destructive"
                            onClick={() => openPlayerApprovalModal(player, 'reject')}
                          >
                            <XCircle className="w-4 h-4 mr-2" />
                            Reject Player
                          </Button>
                        )}
                      </div>
                    </div>
                  );
                })}
              </div>
            ) : (
              <p className="text-gray-500 text-center py-8">No players found for this team.</p>
            )}
          </CardContent>
        </Card>

        {/* Player Approval Modal */}
        <Dialog open={playerApprovalModal.isOpen} onOpenChange={() => setPlayerApprovalModal({ isOpen: false, player: null, action: 'approve' })}>
          <DialogContent>
            <DialogHeader>
              <DialogTitle>
                {playerApprovalModal.action === 'approve' && 'Approve Player'}
                {playerApprovalModal.action === 'reject' && 'Reject Player'}
              </DialogTitle>
            </DialogHeader>
            <div className="space-y-4">
              <div>
                <p className="text-sm text-gray-600">
                  Player: <span className="font-semibold">{playerApprovalModal.player?.display_name}</span>
                </p>
                <p className="text-sm text-gray-600">
                  Team: <span className="font-semibold">{team.name}</span>
                </p>
              </div>
              
              {playerApprovalModal.action === 'reject' ? (
                <div>
                  <Label htmlFor="rejection_reason">Rejection Reason *</Label>
                  <Textarea
                    id="rejection_reason"
                    value={rejectionReason}
                    onChange={(e) => setRejectionReason(e.target.value)}
                    placeholder="Please provide a reason for rejection..."
                    required
                    rows={3}
                  />
                </div>
              ) : (
                <div>
                  <Label htmlFor="approval_notes">Notes (Optional)</Label>
                  <Textarea
                    id="approval_notes"
                    value={approvalNotes}
                    onChange={(e) => setApprovalNotes(e.target.value)}
                    placeholder="Add any notes about this approval..."
                    rows={3}
                  />
                </div>
              )}
              
              <div className="flex gap-2">
                <Button 
                  onClick={handlePlayerApprovalAction}
                  className={
                    playerApprovalModal.action === 'reject' 
                      ? 'bg-red-600 hover:bg-red-700' 
                      : 'bg-green-600 hover:bg-green-700'
                  }
                >
                  {playerApprovalModal.action === 'approve' && 'Approve Player'}
                  {playerApprovalModal.action === 'reject' && 'Reject Player'}
                </Button>
                <Button variant="outline" onClick={() => setPlayerApprovalModal({ isOpen: false, player: null, action: 'approve' })}>
                  Cancel
                </Button>
              </div>
            </div>
          </DialogContent>
        </Dialog>

        {/* Manual Payment Modal */}
        <Dialog open={isPaymentModalOpen} onOpenChange={setIsPaymentModalOpen}>
          <DialogContent className="max-w-lg">
            <DialogHeader>
              <DialogTitle className="flex items-center gap-2">
                <DollarSign className="w-5 h-5" />
                Add Manual Payment
              </DialogTitle>
            </DialogHeader>
            <form onSubmit={handlePaymentSubmit} className="space-y-4">
              <div className="bg-blue-50 border border-blue-200 rounded p-3">
                <p className="text-sm text-blue-800">
                  <strong>Team:</strong> {team.name}<br/>
                  <strong>Tournament:</strong> {tournament.name}
                </p>
              </div>

              <div className="grid grid-cols-2 gap-4">
                <div>
                  <Label htmlFor="amount">Amount *</Label>
                  <Input
                    id="amount"
                    type="number"
                    step="0.01"
                    value={paymentForm.amount}
                    onChange={(e) => setPaymentForm({ ...paymentForm, amount: e.target.value })}
                    required
                  />
                  {paymentFormErrors.amount && <p className="text-red-500 text-sm">{paymentFormErrors.amount}</p>}
                </div>

                <div>
                  <Label htmlFor="currency">Currency</Label>
                  <Select value={paymentForm.currency} onValueChange={(value) => setPaymentForm({ ...paymentForm, currency: value })}>
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="KES">KES</SelectItem>
                      <SelectItem value="USD">USD</SelectItem>
                      <SelectItem value="EUR">EUR</SelectItem>
                      <SelectItem value="GBP">GBP</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
              </div>

              <div>
                <Label htmlFor="payment_method">Payment Method *</Label>
                <Select value={paymentForm.payment_method} onValueChange={(value) => setPaymentForm({ ...paymentForm, payment_method: value })}>
                  <SelectTrigger>
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="mpesa">M-Pesa</SelectItem>
                    <SelectItem value="cash">Cash</SelectItem>
                    <SelectItem value="bank_transfer">Bank Transfer</SelectItem>
                    <SelectItem value="card">Card</SelectItem>
                    <SelectItem value="other">Other</SelectItem>
                  </SelectContent>
                </Select>
                {paymentFormErrors.payment_method && <p className="text-red-500 text-sm">{paymentFormErrors.payment_method}</p>}
              </div>

              <div className="grid grid-cols-2 gap-4">
                <div>
                  <Label htmlFor="reference_number">Reference Number</Label>
                  <Input
                    id="reference_number"
                    value={paymentForm.reference_number}
                    onChange={(e) => setPaymentForm({ ...paymentForm, reference_number: e.target.value })}
                    placeholder="Payment reference"
                  />
                </div>

                <div>
                  <Label htmlFor="transaction_id">Transaction ID</Label>
                  <Input
                    id="transaction_id"
                    value={paymentForm.transaction_id}
                    onChange={(e) => setPaymentForm({ ...paymentForm, transaction_id: e.target.value })}
                    placeholder="Transaction ID"
                  />
                </div>
              </div>

              <div>
                <Label htmlFor="paid_at">Payment Date</Label>
                <Input
                  id="paid_at"
                  type="date"
                  value={paymentForm.paid_at}
                  onChange={(e) => setPaymentForm({ ...paymentForm, paid_at: e.target.value })}
                />
              </div>

              <div>
                <Label htmlFor="description">Description</Label>
                <Textarea
                  id="description"
                  value={paymentForm.description}
                  onChange={(e) => setPaymentForm({ ...paymentForm, description: e.target.value })}
                  placeholder="Add notes about this payment..."
                  rows={2}
                />
              </div>

              <div className="flex gap-2">
                <Button type="submit" className="bg-green-600 hover:bg-green-700">
                  <CreditCard className="w-4 h-4 mr-2" />
                  Add Payment
                </Button>
                <Button type="button" variant="outline" onClick={() => setIsPaymentModalOpen(false)}>
                  Cancel
                </Button>
              </div>
            </form>
          </DialogContent>
        </Dialog>
      </div>
    </AppLayout>
  );
};

export default TeamApprovalView;
