import React, { useState } from 'react';
import { Head, router } from '@inertiajs/react';
import AdminLayout from '@/layouts/admin-layout';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Input } from '@/components/ui/input';
import { Switch } from '@/components/ui/switch';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Alert, AlertDescription } from '@/components/ui/alert';
import { 
    Bell, 
    Search, 
    RefreshCw, 
    CheckCircle2, 
    XCircle, 
    ChevronDown, 
    ChevronRight,
    Save,
    AlertTriangle,
    Info
} from 'lucide-react';
import { cn } from '@/lib/utils';

interface NotificationSetting {
    id: number;
    notification_type: string;
    category: string;
    enabled: boolean;
    description: string;
    is_critical: boolean;
    channels: string[];
}

interface CategoryCount {
    category: string;
    total: number;
    enabled_count: number;
}

interface Props {
    notifications: Record<string, NotificationSetting[]>;
    categories: Record<string, CategoryCount>;
    categoryCounts: Record<string, number>;
}

export default function NotificationSettings({ notifications, categories, categoryCounts }: Props) {
    const [searchTerm, setSearchTerm] = useState('');
    const [expandedCategories, setExpandedCategories] = useState<Set<string>>(new Set());
    const [pendingChanges, setPendingChanges] = useState<Map<number, boolean>>(new Map());
    const [isSyncing, setIsSyncing] = useState(false);
    const [selectedCategory, setSelectedCategory] = useState<string>('all');

    // Category metadata
    const categoryInfo: Record<string, { icon: string; color: string; description: string }> = {
        auth: { icon: '🔐', color: 'bg-blue-500', description: 'Authentication & security notifications' },
        team: { icon: '👥', color: 'bg-green-500', description: 'Team management notifications' },
        player: { icon: '⚽', color: 'bg-purple-500', description: 'Player management notifications' },
        payment: { icon: '💳', color: 'bg-yellow-500', description: 'Payment & transaction notifications' },
        tournament: { icon: '🏆', color: 'bg-orange-500', description: 'Tournament lifecycle notifications' },
        match: { icon: '🎮', color: 'bg-red-500', description: 'Match & game notifications' },
        official: { icon: '👔', color: 'bg-indigo-500', description: 'Officials & staff notifications' },
        user: { icon: '👤', color: 'bg-gray-500', description: 'User account notifications' },
        admin: { icon: '⚙️', color: 'bg-slate-500', description: 'Admin-only notifications' },
        certificate: { icon: '🎓', color: 'bg-teal-500', description: 'Certificates & achievements' },
        invitation: { icon: '✉️', color: 'bg-pink-500', description: 'Invitation notifications' },
        system: { icon: '🔔', color: 'bg-cyan-500', description: 'System-wide announcements' },
    };

    // Toggle category expansion
    const toggleCategory = (category: string) => {
        const newExpanded = new Set(expandedCategories);
        if (newExpanded.has(category)) {
            newExpanded.delete(category);
        } else {
            newExpanded.add(category);
        }
        setExpandedCategories(newExpanded);
    };

    // Toggle individual notification
    const toggleNotification = (id: number, currentValue: boolean) => {
        const newChanges = new Map(pendingChanges);
        newChanges.set(id, !currentValue);
        setPendingChanges(newChanges);
    };

    // Toggle entire category
    const toggleCategoryEnabled = (category: string, enabled: boolean) => {
        router.post(`/admin/settings/notifications/category/${category}/toggle`, {
            enabled: !enabled
        }, {
            preserveScroll: true,
            onSuccess: () => {
                setPendingChanges(new Map());
            }
        });
    };

    // Save pending changes
    const saveChanges = () => {
        const updates = Array.from(pendingChanges.entries()).map(([id, enabled]) => ({
            id,
            enabled
        }));

        router.post('/admin/settings/notifications/bulk-update', {
            updates
        }, {
            preserveScroll: true,
            onSuccess: () => {
                setPendingChanges(new Map());
            }
        });
    };

    // Sync notifications from code
    const syncNotifications = () => {
        setIsSyncing(true);
        router.post('/admin/settings/notifications/sync', {}, {
            preserveScroll: true,
            onFinish: () => setIsSyncing(false)
        });
    };

    // Filter notifications
    const filteredNotifications = Object.entries(notifications).reduce((acc, [category, items]) => {
        if (selectedCategory !== 'all' && category !== selectedCategory) {
            return acc;
        }

        const filtered = items.filter(notif => 
            notif.notification_type.toLowerCase().includes(searchTerm.toLowerCase()) ||
            notif.description?.toLowerCase().includes(searchTerm.toLowerCase())
        );

        if (filtered.length > 0) {
            acc[category] = filtered;
        }

        return acc;
    }, {} as Record<string, NotificationSetting[]>);

    // Calculate statistics
    const stats = {
        total: Object.values(notifications).flat().length,
        enabled: Object.values(notifications).flat().filter(n => {
            const pendingValue = pendingChanges.get(n.id);
            return pendingValue !== undefined ? pendingValue : n.enabled;
        }).length,
        categories: Object.keys(notifications).length
    };

    stats.disabled = stats.total - stats.enabled;

    return (
        <AdminLayout>
            <Head title="Notification Settings" />

            <div className="space-y-6">
                {/* Header */}
                <div className="flex items-center justify-between">
                    <div>
                        <h1 className="text-3xl font-bold">Notification Settings</h1>
                        <p className="text-muted-foreground mt-1">
                            Manage system-wide notification settings and control which notifications are sent
                        </p>
                    </div>
                    <div className="flex items-center gap-2">
                        {pendingChanges.size > 0 && (
                            <Button onClick={saveChanges} className="gap-2">
                                <Save className="w-4 h-4" />
                                Save {pendingChanges.size} Changes
                            </Button>
                        )}
                        <Button onClick={syncNotifications} variant="outline" disabled={isSyncing} className="gap-2">
                            <RefreshCw className={cn("w-4 h-4", isSyncing && "animate-spin")} />
                            Sync from Code
                        </Button>
                    </div>
                </div>

                {/* Alert for pending changes */}
                {pendingChanges.size > 0 && (
                    <Alert>
                        <AlertTriangle className="h-4 w-4" />
                        <AlertDescription>
                            You have {pendingChanges.size} unsaved change{pendingChanges.size !== 1 ? 's' : ''}. Click "Save Changes" to apply them.
                        </AlertDescription>
                    </Alert>
                )}

                {/* Statistics Cards */}
                <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
                    <Card>
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium">Total Notifications</CardTitle>
                            <Bell className="h-4 w-4 text-muted-foreground" />
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold">{stats.total}</div>
                        </CardContent>
                    </Card>
                    <Card>
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium">Enabled</CardTitle>
                            <CheckCircle2 className="h-4 w-4 text-green-600" />
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold text-green-600">{stats.enabled}</div>
                        </CardContent>
                    </Card>
                    <Card>
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium">Disabled</CardTitle>
                            <XCircle className="h-4 w-4 text-red-600" />
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold text-red-600">{stats.disabled}</div>
                        </CardContent>
                    </Card>
                    <Card>
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium">Categories</CardTitle>
                            <Info className="h-4 w-4 text-muted-foreground" />
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold">{stats.categories}</div>
                        </CardContent>
                    </Card>
                </div>

                {/* Search and Filter */}
                <Card>
                    <CardContent className="pt-6">
                        <div className="flex gap-4">
                            <div className="flex-1 relative">
                                <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-muted-foreground w-4 h-4" />
                                <Input
                                    placeholder="Search notifications..."
                                    value={searchTerm}
                                    onChange={(e) => setSearchTerm(e.target.value)}
                                    className="pl-10"
                                />
                            </div>
                        </div>

                        {/* Category Pills */}
                        <div className="flex flex-wrap gap-2 mt-4">
                            <Button
                                variant={selectedCategory === 'all' ? 'default' : 'outline'}
                                size="sm"
                                onClick={() => setSelectedCategory('all')}
                            >
                                All ({stats.total})
                            </Button>
                            {Object.entries(categoryInfo).map(([category, info]) => {
                                const count = categoryCounts[category] || 0;
                                if (count === 0) return null;

                                return (
                                    <Button
                                        key={category}
                                        variant={selectedCategory === category ? 'default' : 'outline'}
                                        size="sm"
                                        onClick={() => setSelectedCategory(category)}
                                        className="gap-1"
                                    >
                                        <span>{info.icon}</span>
                                        {category.charAt(0).toUpperCase() + category.slice(1)} ({count})
                                    </Button>
                                );
                            })}
                        </div>
                    </CardContent>
                </Card>

                {/* Notifications by Category */}
                <div className="space-y-4">
                    {Object.entries(filteredNotifications).map(([category, items]) => {
                        const isExpanded = expandedCategories.has(category);
                        const info = categoryInfo[category] || { icon: '🔔', color: 'bg-gray-500', description: 'Notifications' };
                        const enabledCount = items.filter(n => {
                            const pendingValue = pendingChanges.get(n.id);
                            return pendingValue !== undefined ? pendingValue : n.enabled;
                        }).length;
                        const allEnabled = enabledCount === items.length;

                        return (
                            <Card key={category}>
                                <CardHeader>
                                    <div className="flex items-center justify-between">
                                        <div className="flex items-center gap-3 flex-1 cursor-pointer" onClick={() => toggleCategory(category)}>
                                            <div className={cn("w-10 h-10 rounded-lg flex items-center justify-center text-white text-xl", info.color)}>
                                                {info.icon}
                                            </div>
                                            <div className="flex-1">
                                                <CardTitle className="flex items-center gap-2">
                                                    {category.charAt(0).toUpperCase() + category.slice(1)}
                                                    <Badge variant="secondary">{items.length}</Badge>
                                                    <Badge variant={allEnabled ? "default" : "destructive"}>
                                                        {enabledCount}/{items.length} enabled
                                                    </Badge>
                                                </CardTitle>
                                                <CardDescription>{info.description}</CardDescription>
                                            </div>
                                            {isExpanded ? <ChevronDown className="w-5 h-5" /> : <ChevronRight className="w-5 h-5" />}
                                        </div>
                                        <div className="flex items-center gap-2">
                                            <span className="text-sm text-muted-foreground">
                                                {allEnabled ? 'All enabled' : 'Toggle all'}
                                            </span>
                                            <Switch
                                                checked={allEnabled}
                                                onCheckedChange={() => toggleCategoryEnabled(category, allEnabled)}
                                            />
                                        </div>
                                    </div>
                                </CardHeader>

                                {isExpanded && (
                                    <CardContent>
                                        <div className="space-y-2">
                                            {items.map((notification) => {
                                                const isEnabled = pendingChanges.has(notification.id) 
                                                    ? pendingChanges.get(notification.id)! 
                                                    : notification.enabled;
                                                const hasChanges = pendingChanges.has(notification.id);

                                                return (
                                                    <div
                                                        key={notification.id}
                                                        className={cn(
                                                            "flex items-center justify-between p-3 rounded-lg border transition-colors",
                                                            hasChanges && "bg-yellow-50 border-yellow-300"
                                                        )}
                                                    >
                                                        <div className="flex-1">
                                                            <div className="flex items-center gap-2">
                                                                <code className="text-sm font-mono bg-muted px-2 py-1 rounded">
                                                                    {notification.notification_type}
                                                                </code>
                                                                {hasChanges && (
                                                                    <Badge variant="outline" className="text-xs">
                                                                        Pending
                                                                    </Badge>
                                                                )}
                                                            </div>
                                                            {notification.description && (
                                                                <p className="text-sm text-muted-foreground mt-1">
                                                                    {notification.description}
                                                                </p>
                                                            )}
                                                            <div className="flex items-center gap-2 mt-1">
                                                                {notification.channels?.map((channel) => (
                                                                    <Badge key={channel} variant="secondary" className="text-xs">
                                                                        {channel}
                                                                    </Badge>
                                                                ))}
                                                            </div>
                                                        </div>
                                                        <div className="flex items-center gap-2">
                                                            <span className={cn("text-sm font-medium", isEnabled ? "text-green-600" : "text-red-600")}>
                                                                {isEnabled ? 'Enabled' : 'Disabled'}
                                                            </span>
                                                            <Switch
                                                                checked={isEnabled}
                                                                onCheckedChange={() => toggleNotification(notification.id, notification.enabled)}
                                                            />
                                                        </div>
                                                    </div>
                                                );
                                            })}
                                        </div>
                                    </CardContent>
                                )}
                            </Card>
                        );
                    })}
                </div>

                {Object.keys(filteredNotifications).length === 0 && (
                    <Card>
                        <CardContent className="py-12 text-center">
                            <Bell className="w-12 h-12 text-muted-foreground mx-auto mb-4" />
                            <p className="text-lg font-medium">No notifications found</p>
                            <p className="text-sm text-muted-foreground">
                                Try adjusting your search or filter criteria
                            </p>
                        </CardContent>
                    </Card>
                )}
            </div>
        </AdminLayout>
    );
}

