import React, { useState } from 'react';
import { Head, Link, router } from '@inertiajs/react';
import AppLayout from '@/layouts/app-layout';
import { Badge } from '@/components/ui/badge';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import UserAccessDialog from '@/components/UserAccessDialog';
import { 
  Mail, 
  Users, 
  Settings,
  ArrowLeft,
  Edit,
  TestTube,
  CheckCircle,
  XCircle,
  UserPlus,
  UserMinus,
  Smartphone,
  Monitor,
  Laptop,
  Copy,
  Eye,
  EyeOff
} from 'lucide-react';

interface User {
  id: number;
  name: string;
  email: string;
  pivot: {
    can_send: boolean;
    can_receive: boolean;
    permissions: any;
    granted_at: string;
    expires_at?: string;
  };
}

interface SetupInstructions {
  android: {
    title: string;
    steps: string[];
  };
  ios: {
    title: string;
    steps: string[];
  };
  windows_outlook: {
    title: string;
    steps: string[];
  };
}

interface EmailCredential {
  id: number;
  name: string;
  email: string;
  provider: string;
  smtp_host: string;
  smtp_port: number;
  smtp_username: string;
  smtp_encryption: string;
  is_active: boolean;
  instructions: string;
  metadata: any;
  created_at: string;
  updated_at: string;
  users: User[];
}

interface Props {
  emailCredential: EmailCredential;
  setupInstructions: SetupInstructions;
}

const EmailShow: React.FC<Props> = ({ emailCredential, setupInstructions }) => {
  const [showPassword, setShowPassword] = useState(false);
  const [selectedInstructions, setSelectedInstructions] = useState<'android' | 'ios' | 'windows_outlook'>('android');

  const getProviderIcon = (provider: string) => {
    const iconMap: Record<string, any> = {
      gmail: Mail,
      outlook: Mail,
      yahoo: Mail,
      icloud: Mail,
      exchange: Mail,
    };
    const IconComponent = iconMap[provider.toLowerCase()] || Mail;
    return <IconComponent className="h-6 w-6" />;
  };

  const getProviderColor = (provider: string) => {
    const colorMap: Record<string, string> = {
      gmail: 'bg-red-500',
      outlook: 'bg-blue-500',
      yahoo: 'bg-purple-500',
      icloud: 'bg-gray-500',
      exchange: 'bg-green-500',
    };
    return colorMap[provider.toLowerCase()] || 'bg-gray-500';
  };

  const handleToggleStatus = () => {
    router.patch(route('admin.settings.emails.toggle-status', emailCredential.id), {
      is_active: !emailCredential.is_active,
    });
  };

  const handleTestConnection = () => {
    router.post(route('admin.settings.emails.test-connection', emailCredential.id), {}, {
      onSuccess: (page) => {
        // Handle success response
      },
      onError: (errors) => {
        // Handle error response
      }
    });
  };

  const handleRemoveUser = (userId: number) => {
    if (confirm('Are you sure you want to remove this user from the email account?')) {
      router.post(route('admin.settings.emails.remove-users', emailCredential.id), {
        user_ids: [userId],
      });
    }
  };

  const copyToClipboard = (text: string) => {
    navigator.clipboard.writeText(text);
  };

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric',
      hour: '2-digit',
      minute: '2-digit',
    });
  };

  const isExpired = (expiresAt: string | null) => {
    if (!expiresAt) return false;
    return new Date(expiresAt) < new Date();
  };

  return (
    <AppLayout breadcrumbs={[
      { title: 'Settings', href: '/admin/settings' },
      { title: 'Emails', href: '/admin/settings/emails' },
      { title: emailCredential.name, href: `/admin/settings/emails/${emailCredential.id}` }
    ]}>
      <Head title={`${emailCredential.name} - Email Details`} />
      
      <div className="space-y-6 p-4 flex flex-col gap-4">
        {/* Header */}
        <div className="flex justify-between items-center">
          <div className="flex items-center gap-4">
            <Button variant="ghost" size="sm" asChild>
              <Link href={route('admin.settings.emails.index')}>
                <ArrowLeft className="h-4 w-4 mr-2" />
                Back to Emails
              </Link>
            </Button>
            <div className="flex items-center gap-3">
              <div className={`p-3 rounded-lg ${getProviderColor(emailCredential.provider)}`}>
                {getProviderIcon(emailCredential.provider)}
              </div>
              <div>
                <h1 className="text-3xl font-bold">{emailCredential.name}</h1>
                <p className="text-muted-foreground">{emailCredential.email}</p>
              </div>
            </div>
          </div>
          <div className="flex gap-2">
            <Button variant="outline" onClick={handleTestConnection}>
              <TestTube className="h-4 w-4 mr-2" />
              Test Connection
            </Button>
            <Button variant="outline" onClick={handleToggleStatus}>
              {emailCredential.is_active ? (
                <>
                  <XCircle className="h-4 w-4 mr-2" />
                  Deactivate
                </>
              ) : (
                <>
                  <CheckCircle className="h-4 w-4 mr-2" />
                  Activate
                </>
              )}
            </Button>
            <Button asChild>
              <Link href={route('admin.settings.emails.edit', emailCredential.id)}>
                <Edit className="h-4 w-4 mr-2" />
                Edit
              </Link>
            </Button>
          </div>
        </div>

        {/* Email Information */}
        <div className="grid gap-6 md:grid-cols-2">
          <Card>
            <CardHeader>
              <CardTitle>Email Information</CardTitle>
              <CardDescription>Basic details about this email account</CardDescription>
            </CardHeader>
            <CardContent className="space-y-4">
              <div>
                <label className="text-sm font-medium text-muted-foreground">Email Address</label>
                <div className="flex items-center gap-2 mt-1">
                  <p className="font-mono">{emailCredential.email}</p>
                  <Button
                    variant="ghost"
                    size="sm"
                    onClick={() => copyToClipboard(emailCredential.email)}
                  >
                    <Copy className="h-4 w-4" />
                  </Button>
                </div>
              </div>
              
              <div>
                <label className="text-sm font-medium text-muted-foreground">Provider</label>
                <div className="mt-1">
                  <Badge variant="outline" className="capitalize">
                    {emailCredential.provider}
                  </Badge>
                </div>
              </div>

              <div>
                <label className="text-sm font-medium text-muted-foreground">Status</label>
                <div className="mt-1">
                  <Badge variant={emailCredential.is_active ? 'default' : 'secondary'}>
                    {emailCredential.is_active ? 'Active' : 'Inactive'}
                  </Badge>
                </div>
              </div>

              {emailCredential.instructions && (
                <div>
                  <label className="text-sm font-medium text-muted-foreground">Instructions</label>
                  <p className="mt-1 text-sm">{emailCredential.instructions}</p>
                </div>
              )}
            </CardContent>
          </Card>

          <Card>
            <CardHeader>
              <CardTitle>SMTP Configuration</CardTitle>
              <CardDescription>Server settings for this email account</CardDescription>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <label className="text-sm font-medium text-muted-foreground">SMTP Host</label>
                  <div className="flex items-center gap-2 mt-1">
                    <p className="font-mono text-sm">{emailCredential.smtp_host}</p>
                    <Button
                      variant="ghost"
                      size="sm"
                      onClick={() => copyToClipboard(emailCredential.smtp_host)}
                    >
                      <Copy className="h-3 w-3" />
                    </Button>
                  </div>
                </div>
                <div>
                  <label className="text-sm font-medium text-muted-foreground">Port</label>
                  <p className="mt-1 font-mono text-sm">{emailCredential.smtp_port}</p>
                </div>
              </div>

              <div>
                <label className="text-sm font-medium text-muted-foreground">Username</label>
                <div className="flex items-center gap-2 mt-1">
                  <p className="font-mono text-sm">{emailCredential.smtp_username}</p>
                  <Button
                    variant="ghost"
                    size="sm"
                    onClick={() => copyToClipboard(emailCredential.smtp_username)}
                  >
                    <Copy className="h-3 w-3" />
                  </Button>
                </div>
              </div>

              <div>
                <label className="text-sm font-medium text-muted-foreground">Encryption</label>
                <p className="mt-1 font-mono text-sm">{emailCredential.smtp_encryption.toUpperCase()}</p>
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Setup Instructions */}
        <Card>
          <CardHeader>
            <CardTitle>Setup Instructions</CardTitle>
            <CardDescription>
              Instructions for setting up this email account on different devices
            </CardDescription>
          </CardHeader>
          <CardContent>
            <div className="space-y-4">
              <div className="flex gap-2">
                <Button
                  variant={selectedInstructions === 'android' ? 'default' : 'outline'}
                  size="sm"
                  onClick={() => setSelectedInstructions('android')}
                >
                  <Smartphone className="h-4 w-4 mr-2" />
                  Android
                </Button>
                <Button
                  variant={selectedInstructions === 'ios' ? 'default' : 'outline'}
                  size="sm"
                  onClick={() => setSelectedInstructions('ios')}
                >
                  <Smartphone className="h-4 w-4 mr-2" />
                  iOS
                </Button>
                <Button
                  variant={selectedInstructions === 'windows_outlook' ? 'default' : 'outline'}
                  size="sm"
                  onClick={() => setSelectedInstructions('windows_outlook')}
                >
                  <Monitor className="h-4 w-4 mr-2" />
                  Windows Outlook
                </Button>
              </div>

              <div className="border rounded-lg p-4">
                <h4 className="font-medium mb-3">
                  {setupInstructions[selectedInstructions].title}
                </h4>
                <ol className="space-y-2">
                  {setupInstructions[selectedInstructions].steps.map((step, index) => (
                    <li key={index} className="flex items-start gap-2">
                      <span className="flex-shrink-0 w-6 h-6 bg-primary text-primary-foreground rounded-full flex items-center justify-center text-sm font-medium">
                        {index + 1}
                      </span>
                      <span className="text-sm">{step}</span>
                    </li>
                  ))}
                </ol>
              </div>
            </div>
          </CardContent>
        </Card>

        {/* Assigned Users */}
        <Card>
          <CardHeader>
            <div className="flex justify-between items-center">
              <div>
                <CardTitle>Assigned Users</CardTitle>
                <CardDescription>
                  {emailCredential.users.length} user(s) have access to this email account
                </CardDescription>
              </div>
              <UserAccessDialog
                emailCredentialId={emailCredential.id}
                assignedUsers={emailCredential.users}
                onUserAssigned={() => {
                  // Refresh the page to show updated user list
                  router.reload();
                }}
                onUserRemoved={() => {
                  // Refresh the page to show updated user list
                  router.reload();
                }}
                onPermissionsUpdated={() => {
                  // Refresh the page to show updated permissions
                  router.reload();
                }}
              />
            </div>
          </CardHeader>
          <CardContent>
            {emailCredential.users.length === 0 ? (
              <div className="text-center py-8 text-muted-foreground">
                <Users className="h-12 w-12 mx-auto mb-4 opacity-50" />
                <p>No users assigned to this email account</p>
              </div>
            ) : (
              <div className="overflow-x-auto">
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead>User</TableHead>
                      <TableHead>Permissions</TableHead>
                      <TableHead>Granted Date</TableHead>
                      <TableHead>Expires</TableHead>
                      <TableHead className="text-right">Actions</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {emailCredential.users.map((user) => (
                      <TableRow key={user.id}>
                        <TableCell>
                          <div>
                            <div className="font-medium">{user.name}</div>
                            <div className="text-sm text-muted-foreground">{user.email}</div>
                          </div>
                        </TableCell>
                        <TableCell>
                          <div className="flex gap-1">
                            {user.pivot.can_send && (
                              <Badge variant="outline" className="text-xs">Send</Badge>
                            )}
                            {user.pivot.can_receive && (
                              <Badge variant="outline" className="text-xs">Receive</Badge>
                            )}
                          </div>
                        </TableCell>
                        <TableCell>
                          <div className="text-sm">
                            {formatDate(user.pivot.granted_at)}
                          </div>
                        </TableCell>
                        <TableCell>
                          {user.pivot.expires_at ? (
                            <div className={`text-sm ${isExpired(user.pivot.expires_at) ? 'text-red-600' : 'text-muted-foreground'}`}>
                              {formatDate(user.pivot.expires_at)}
                              {isExpired(user.pivot.expires_at) && (
                                <Badge variant="destructive" className="ml-2">Expired</Badge>
                              )}
                            </div>
                          ) : (
                            <span className="text-muted-foreground">Never</span>
                          )}
                        </TableCell>
                        <TableCell className="text-right">
                          <Button
                            variant="ghost"
                            size="sm"
                            onClick={() => handleRemoveUser(user.id)}
                            className="text-red-600 hover:text-red-700"
                          >
                            <UserMinus className="h-4 w-4" />
                          </Button>
                        </TableCell>
                      </TableRow>
                    ))}
                  </TableBody>
                </Table>
              </div>
            )}
          </CardContent>
        </Card>
      </div>
    </AppLayout>
  );
};

export default EmailShow;
