import React, { useState } from 'react';
import { Head, Link, router } from '@inertiajs/react';
import AppLayout from '@/layouts/app-layout';
import { Badge } from '@/components/ui/badge';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { Checkbox } from '@/components/ui/checkbox';
import { 
  Mail, 
  Users, 
  Settings,
  Search,
  Filter,
  Eye,
  Edit,
  Trash2,
  Plus,
  MoreHorizontal,
  CheckCircle,
  XCircle,
  Key,
  UserPlus,
  UserMinus,
  TestTube,
  Smartphone,
  Monitor,
  Laptop
} from 'lucide-react';
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuLabel,
  DropdownMenuSeparator,
  DropdownMenuTrigger,
} from '@/components/ui/dropdown-menu';

interface EmailCredential {
  id: number;
  name: string;
  email: string;
  provider: string;
  smtp_host: string;
  smtp_port: number;
  smtp_encryption: string;
  is_active: boolean;
  users_count: number;
  created_at: string;
  updated_at: string;
}

interface Props {
  emailCredentials: {
    data: EmailCredential[];
    current_page: number;
    last_page: number;
    per_page: number;
    total: number;
  };
  providers: string[];
  filters: {
    search: string;
    provider: string;
    is_active: string;
    sort_by: string;
    sort_order: string;
  };
}

const EmailsIndex: React.FC<Props> = ({ emailCredentials, providers, filters }) => {
  const [searchTerm, setSearchTerm] = useState(filters.search || '');
  const [selectedEmails, setSelectedEmails] = useState<number[]>([]);

  const getProviderIcon = (provider: string) => {
    const iconMap: Record<string, any> = {
      gmail: Mail,
      outlook: Mail,
      yahoo: Mail,
      icloud: Mail,
      exchange: Mail,
    };
    const IconComponent = iconMap[provider.toLowerCase()] || Mail;
    return <IconComponent className="h-4 w-4" />;
  };

  const getProviderColor = (provider: string) => {
    const colorMap: Record<string, string> = {
      gmail: 'bg-red-500',
      outlook: 'bg-blue-500',
      yahoo: 'bg-purple-500',
      icloud: 'bg-gray-500',
      exchange: 'bg-green-500',
    };
    return colorMap[provider.toLowerCase()] || 'bg-gray-500';
  };

  const handleSearch = (e: React.FormEvent) => {
    e.preventDefault();
    router.get(route('admin.settings.emails.index'), {
      ...filters,
      search: searchTerm,
    });
  };

  const handleFilter = (key: string, value: string) => {
    router.get(route('admin.settings.emails.index'), {
      ...filters,
      [key]: value,
    });
  };

  const handleSort = (column: string) => {
    const newOrder = filters.sort_by === column && filters.sort_order === 'asc' ? 'desc' : 'asc';
    router.get(route('admin.settings.emails.index'), {
      ...filters,
      sort_by: column,
      sort_order: newOrder,
    });
  };

  const handleSelectAll = (checked: boolean) => {
    if (checked) {
      setSelectedEmails(emailCredentials.data.map(e => e.id));
    } else {
      setSelectedEmails([]);
    }
  };

  const handleSelectEmail = (emailId: number, checked: boolean) => {
    if (checked) {
      setSelectedEmails([...selectedEmails, emailId]);
    } else {
      setSelectedEmails(selectedEmails.filter(id => id !== emailId));
    }
  };

  const handleBulkAction = (action: string) => {
    if (selectedEmails.length === 0) return;

    router.post(route('admin.settings.emails.bulk-update'), {
      ids: selectedEmails,
      action,
    }, {
      onSuccess: () => {
        setSelectedEmails([]);
      }
    });
  };

  const handleDelete = (emailId: number) => {
    if (confirm('Are you sure you want to delete this email credential?')) {
      router.delete(route('admin.settings.emails.destroy', emailId));
    }
  };

  const handleToggleStatus = (emailId: number, currentStatus: boolean) => {
    router.patch(route('admin.settings.emails.toggle-status', emailId), {
      is_active: !currentStatus,
    });
  };

  const handleTestConnection = (emailId: number) => {
    router.post(route('admin.settings.emails.test-connection', emailId), {}, {
      onSuccess: (page) => {
        // Handle success response
      },
      onError: (errors) => {
        // Handle error response
      }
    });
  };

  // Calculate statistics
  const stats = {
    total: emailCredentials.total,
    active: emailCredentials.data.filter(e => e.is_active).length,
    inactive: emailCredentials.data.filter(e => !e.is_active).length,
    providers: providers.length,
  };

  const statCards = [
    {
      title: 'Total Email Accounts',
      value: stats.total,
      icon: Mail,
      description: 'All email accounts',
      color: 'text-blue-600',
    },
    {
      title: 'Active Accounts',
      value: stats.active,
      icon: CheckCircle,
      description: 'Currently active',
      color: 'text-green-600',
    },
    {
      title: 'Providers',
      value: stats.providers,
      icon: Settings,
      description: 'Different providers',
      color: 'text-purple-600',
    },
    {
      title: 'Inactive Accounts',
      value: stats.inactive,
      icon: XCircle,
      description: 'Currently inactive',
      color: 'text-red-600',
    },
  ];

  return (
    <AppLayout breadcrumbs={[
      { title: 'Settings', href: '/admin/settings' },
      { title: 'Emails', href: '/admin/settings/emails' }
    ]}>
      <Head title="Email Management" />
      
      <div className="space-y-6 p-4 flex flex-col gap-4">
        {/* Header */}
        <div className="flex justify-between items-center">
          <div>
            <h1 className="text-3xl font-bold">Email Management</h1>
            <p className="text-muted-foreground">Manage email credentials and user access</p>
          </div>
          <Button asChild>
            <Link href={route('admin.settings.emails.create')}>
              <Plus className="h-4 w-4 mr-2" />
              Add Email Account
            </Link>
          </Button>
        </div>

        {/* Statistics Grid */}
        <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
          {statCards.map((stat) => {
            const IconComponent = stat.icon;
            return (
              <Card key={stat.title}>
                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                  <CardTitle className="text-sm font-medium">
                    {stat.title}
                  </CardTitle>
                  <IconComponent className={`h-4 w-4 ${stat.color}`} />
                </CardHeader>
                <CardContent>
                  <div className="text-2xl font-bold">{stat.value}</div>
                  <p className="text-xs text-muted-foreground">
                    {stat.description}
                  </p>
                </CardContent>
              </Card>
            );
          })}
        </div>

        {/* Filters and Search */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Filter className="h-5 w-5" />
              Filters & Search
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="flex flex-col sm:flex-row gap-4">
              <form onSubmit={handleSearch} className="flex gap-2 flex-1">
                <div className="relative flex-1">
                  <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 h-4 w-4" />
                  <Input
                    placeholder="Search email accounts..."
                    value={searchTerm}
                    onChange={(e) => setSearchTerm(e.target.value)}
                    className="pl-10"
                  />
                </div>
                <Button type="submit">Search</Button>
              </form>

              <div className="flex gap-2">
                <Select
                  value={filters.provider || 'all'}
                  onValueChange={(value) => handleFilter('provider', value)}
                >
                  <SelectTrigger className="w-40">
                    <SelectValue placeholder="Provider" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">All Providers</SelectItem>
                    {providers.map((provider) => (
                      <SelectItem key={provider} value={provider}>{provider}</SelectItem>
                    ))}
                  </SelectContent>
                </Select>

                <Select
                  value={filters.is_active || 'all'}
                  onValueChange={(value) => handleFilter('is_active', value)}
                >
                  <SelectTrigger className="w-32">
                    <SelectValue placeholder="Status" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">All Status</SelectItem>
                    <SelectItem value="true">Active</SelectItem>
                    <SelectItem value="false">Inactive</SelectItem>
                  </SelectContent>
                </Select>
              </div>
            </div>
          </CardContent>
        </Card>

        {/* Bulk Actions */}
        {selectedEmails.length > 0 && (
          <Card>
            <CardContent className="py-4">
              <div className="flex items-center justify-between">
                <span className="text-sm text-muted-foreground">
                  {selectedEmails.length} email account(s) selected
                </span>
                <div className="flex gap-2">
                  <Button 
                    variant="outline" 
                    size="sm"
                    onClick={() => handleBulkAction('activate')}
                  >
                    <CheckCircle className="h-4 w-4 mr-2" />
                    Activate
                  </Button>
                  <Button 
                    variant="outline" 
                    size="sm"
                    onClick={() => handleBulkAction('deactivate')}
                  >
                    <XCircle className="h-4 w-4 mr-2" />
                    Deactivate
                  </Button>
                  <Button 
                    variant="destructive" 
                    size="sm"
                    onClick={() => {
                      if (confirm('Are you sure you want to delete selected email accounts?')) {
                        router.post(route('admin.settings.emails.bulk-delete'), {
                          ids: selectedEmails
                        });
                      }
                    }}
                  >
                    <Trash2 className="h-4 w-4 mr-2" />
                    Delete
                  </Button>
                </div>
              </div>
            </CardContent>
          </Card>
        )}

        {/* Email Accounts Table */}
        <Card>
          <CardHeader>
            <CardTitle>Email Accounts</CardTitle>
            <CardDescription>
              Showing {emailCredentials.data.length} of {emailCredentials.total} email accounts
            </CardDescription>
          </CardHeader>
          <CardContent>
            <div className="overflow-x-auto">
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead className="w-12">
                      <Checkbox
                        checked={selectedEmails.length === emailCredentials.data.length}
                        onCheckedChange={handleSelectAll}
                      />
                    </TableHead>
                    <TableHead 
                      className="cursor-pointer hover:bg-muted/50"
                      onClick={() => handleSort('name')}
                    >
                      Email Account
                    </TableHead>
                    <TableHead 
                      className="cursor-pointer hover:bg-muted/50"
                      onClick={() => handleSort('provider')}
                    >
                      Provider
                    </TableHead>
                    <TableHead>SMTP Details</TableHead>
                    <TableHead>Users</TableHead>
                    <TableHead>Status</TableHead>
                    <TableHead className="text-right">Actions</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {emailCredentials.data.map((email) => (
                    <TableRow key={email.id}>
                      <TableCell>
                        <Checkbox
                          checked={selectedEmails.includes(email.id)}
                          onCheckedChange={(checked) => handleSelectEmail(email.id, !!checked)}
                        />
                      </TableCell>
                      <TableCell>
                        <div className="flex items-center gap-3">
                          <div className={`p-2 rounded-lg ${getProviderColor(email.provider)}`}>
                            {getProviderIcon(email.provider)}
                          </div>
                          <div>
                            <div className="font-medium">{email.name}</div>
                            <div className="text-sm text-muted-foreground">{email.email}</div>
                          </div>
                        </div>
                      </TableCell>
                      <TableCell>
                        <Badge variant="outline" className="capitalize">
                          {email.provider}
                        </Badge>
                      </TableCell>
                      <TableCell>
                        <div className="text-sm">
                          <div className="font-mono">{email.smtp_host}:{email.smtp_port}</div>
                          <div className="text-muted-foreground">{email.smtp_encryption.toUpperCase()}</div>
                        </div>
                      </TableCell>
                      <TableCell>
                        <div className="flex items-center gap-1">
                          <Users className="h-4 w-4 text-muted-foreground" />
                          <span>{email.users_count}</span>
                        </div>
                      </TableCell>
                      <TableCell>
                        <Badge variant={email.is_active ? 'default' : 'secondary'}>
                          {email.is_active ? 'Active' : 'Inactive'}
                        </Badge>
                      </TableCell>
                      <TableCell className="text-right">
                        <DropdownMenu>
                          <DropdownMenuTrigger asChild>
                            <Button variant="ghost" className="h-8 w-8 p-0">
                              <MoreHorizontal className="h-4 w-4" />
                            </Button>
                          </DropdownMenuTrigger>
                          <DropdownMenuContent align="end">
                            <DropdownMenuLabel>Actions</DropdownMenuLabel>
                            <DropdownMenuItem asChild>
                              <Link href={route('admin.settings.emails.show', email.id)}>
                                <Eye className="h-4 w-4 mr-2" />
                                View Details
                              </Link>
                            </DropdownMenuItem>
                            <DropdownMenuItem asChild>
                              <Link href={route('admin.settings.emails.edit', email.id)}>
                                <Edit className="h-4 w-4 mr-2" />
                                Edit
                              </Link>
                            </DropdownMenuItem>
                            <DropdownMenuItem 
                              onClick={() => handleTestConnection(email.id)}
                            >
                              <TestTube className="h-4 w-4 mr-2" />
                              Test Connection
                            </DropdownMenuItem>
                            <DropdownMenuItem 
                              onClick={() => handleToggleStatus(email.id, email.is_active)}
                            >
                              {email.is_active ? (
                                <>
                                  <XCircle className="h-4 w-4 mr-2" />
                                  Deactivate
                                </>
                              ) : (
                                <>
                                  <CheckCircle className="h-4 w-4 mr-2" />
                                  Activate
                                </>
                              )}
                            </DropdownMenuItem>
                            <DropdownMenuSeparator />
                            <DropdownMenuItem 
                              onClick={() => handleDelete(email.id)}
                              className="text-red-600"
                            >
                              <Trash2 className="h-4 w-4 mr-2" />
                              Delete
                            </DropdownMenuItem>
                          </DropdownMenuContent>
                        </DropdownMenu>
                      </TableCell>
                    </TableRow>
                  ))}
                </TableBody>
              </Table>
            </div>

            {/* Pagination */}
            {emailCredentials.last_page > 1 && (
              <div className="flex items-center justify-between mt-4">
                <div className="text-sm text-muted-foreground">
                  Showing {((emailCredentials.current_page - 1) * emailCredentials.per_page) + 1} to{' '}
                  {Math.min(emailCredentials.current_page * emailCredentials.per_page, emailCredentials.total)} of{' '}
                  {emailCredentials.total} email accounts
                </div>
                <div className="flex gap-2">
                  {emailCredentials.current_page > 1 && (
                    <Button
                      variant="outline"
                      size="sm"
                      onClick={() => router.get(route('admin.settings.emails.index'), {
                        ...filters,
                        page: emailCredentials.current_page - 1,
                      })}
                    >
                      Previous
                    </Button>
                  )}
                  {emailCredentials.current_page < emailCredentials.last_page && (
                    <Button
                      variant="outline"
                      size="sm"
                      onClick={() => router.get(route('admin.settings.emails.index'), {
                        ...filters,
                        page: emailCredentials.current_page + 1,
                      })}
                    >
                      Next
                    </Button>
                  )}
                </div>
              </div>
            )}
          </CardContent>
        </Card>
      </div>
    </AppLayout>
  );
};

export default EmailsIndex;
