import React from 'react';
import { Head, router, Link } from '@inertiajs/react';
import AppLayout from '@/layouts/app-layout';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Switch } from '@/components/ui/switch';
import { 
  Mail, 
  ArrowLeft,
  Save,
  TestTube
} from 'lucide-react';

interface Provider {
  [key: string]: string;
}

interface Props {
  providers: Provider;
}

const EmailCreate: React.FC<Props> = ({ providers }) => {
  const [formData, setFormData] = React.useState({
    name: '',
    email: '',
    provider: '',
    smtp_host: '',
    smtp_port: 587,
    smtp_username: '',
    smtp_password: '',
    smtp_encryption: 'tls',
    is_active: true,
    instructions: '',
  });

  const [errors, setErrors] = React.useState<Record<string, string>>({});
  const [isSubmitting, setIsSubmitting] = React.useState(false);

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    setIsSubmitting(true);
    setErrors({});

    router.post(route('admin.settings.emails.store'), formData, {
      onSuccess: () => {
        // Redirect will be handled by the controller
      },
      onError: (errors) => {
        setErrors(errors);
        setIsSubmitting(false);
      },
      onFinish: () => {
        setIsSubmitting(false);
      }
    });
  };

  const handleTestConnection = () => {
    // Test connection logic would go here
    console.log('Testing connection...');
  };

  const getDefaultSmtpSettings = (provider: string) => {
    const settings: Record<string, any> = {
      'Gmail': {
        smtp_host: 'smtp.gmail.com',
        smtp_port: 587,
        smtp_encryption: 'tls',
      },
      'Outlook': {
        smtp_host: 'smtp-mail.outlook.com',
        smtp_port: 587,
        smtp_encryption: 'tls',
      },
      'Yahoo': {
        smtp_host: 'smtp.mail.yahoo.com',
        smtp_port: 587,
        smtp_encryption: 'tls',
      },
      'iCloud': {
        smtp_host: 'smtp.mail.me.com',
        smtp_port: 587,
        smtp_encryption: 'tls',
      },
      'Exchange': {
        smtp_host: 'smtp.office365.com',
        smtp_port: 587,
        smtp_encryption: 'tls',
      },
    };

    return settings[provider] || {};
  };

  const handleProviderChange = (provider: string) => {
    const settings = getDefaultSmtpSettings(provider);
    setFormData(prev => ({
      ...prev,
      provider,
      ...settings,
    }));
  };

  return (
    <AppLayout breadcrumbs={[
      { title: 'Settings', href: '/admin/settings' },
      { title: 'Emails', href: '/admin/settings/emails' },
      { title: 'Create Email Account', href: '/admin/settings/emails/create' }
    ]}>
      <Head title="Create Email Account" />
      
      <div className="space-y-6 p-4">
        <div className="flex items-center gap-4">
          <Button variant="ghost" size="sm" asChild>
            <Link href={route('admin.settings.emails.index')}>
              <ArrowLeft className="h-4 w-4 mr-2" />
              Back to Emails
            </Link>
          </Button>
          <div>
            <h1 className="text-3xl font-bold">Create Email Account</h1>
            <p className="text-muted-foreground">Add a new email account for system use</p>
          </div>
        </div>

        <form onSubmit={handleSubmit} className="space-y-6">
          <div className="grid gap-6 md:grid-cols-2">
            {/* Basic Information */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Mail className="h-5 w-5" />
                  Basic Information
                </CardTitle>
                <CardDescription>
                  Basic details about the email account
                </CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="space-y-2">
                  <Label htmlFor="name">Account Name *</Label>
                  <Input
                    id="name"
                    value={formData.name}
                    onChange={(e) => setFormData(prev => ({ ...prev, name: e.target.value }))}
                    placeholder="e.g., Main Support Email"
                    className={errors.name ? 'border-red-500' : ''}
                  />
                  {errors.name && (
                    <p className="text-sm text-red-500">{errors.name}</p>
                  )}
                </div>

                <div className="space-y-2">
                  <Label htmlFor="email">Email Address *</Label>
                  <Input
                    id="email"
                    type="email"
                    value={formData.email}
                    onChange={(e) => setFormData(prev => ({ ...prev, email: e.target.value }))}
                    placeholder="e.g., support@example.com"
                    className={errors.email ? 'border-red-500' : ''}
                  />
                  {errors.email && (
                    <p className="text-sm text-red-500">{errors.email}</p>
                  )}
                </div>

                <div className="space-y-2">
                  <Label htmlFor="provider">Email Provider *</Label>
                  <Select 
                    value={formData.provider} 
                    onValueChange={handleProviderChange}
                  >
                    <SelectTrigger className={errors.provider ? 'border-red-500' : ''}>
                      <SelectValue placeholder="Select provider" />
                    </SelectTrigger>
                    <SelectContent>
                      {Object.entries(providers).map(([key, value]) => (
                        <SelectItem key={key} value={key}>{value}</SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                  {errors.provider && (
                    <p className="text-sm text-red-500">{errors.provider}</p>
                  )}
                </div>

                <div className="flex items-center space-x-2">
                  <Switch
                    id="is_active"
                    checked={formData.is_active}
                    onCheckedChange={(checked) => setFormData(prev => ({ ...prev, is_active: checked }))}
                  />
                  <Label htmlFor="is_active">Active</Label>
                </div>
              </CardContent>
            </Card>

            {/* SMTP Configuration */}
            <Card>
              <CardHeader>
                <CardTitle>SMTP Configuration</CardTitle>
                <CardDescription>
                  SMTP server settings for sending emails
                </CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="space-y-2">
                  <Label htmlFor="smtp_host">SMTP Host *</Label>
                  <Input
                    id="smtp_host"
                    value={formData.smtp_host}
                    onChange={(e) => setFormData(prev => ({ ...prev, smtp_host: e.target.value }))}
                    placeholder="e.g., smtp.gmail.com"
                    className={errors.smtp_host ? 'border-red-500' : ''}
                  />
                  {errors.smtp_host && (
                    <p className="text-sm text-red-500">{errors.smtp_host}</p>
                  )}
                </div>

                <div className="grid grid-cols-2 gap-4">
                  <div className="space-y-2">
                    <Label htmlFor="smtp_port">Port *</Label>
                    <Input
                      id="smtp_port"
                      type="number"
                      value={formData.smtp_port}
                      onChange={(e) => setFormData(prev => ({ ...prev, smtp_port: parseInt(e.target.value) || 587 }))}
                      min="1"
                      max="65535"
                      className={errors.smtp_port ? 'border-red-500' : ''}
                    />
                    {errors.smtp_port && (
                      <p className="text-sm text-red-500">{errors.smtp_port}</p>
                    )}
                  </div>

                  <div className="space-y-2">
                    <Label htmlFor="smtp_encryption">Encryption *</Label>
                    <Select 
                      value={formData.smtp_encryption} 
                      onValueChange={(value) => setFormData(prev => ({ ...prev, smtp_encryption: value }))}
                    >
                      <SelectTrigger className={errors.smtp_encryption ? 'border-red-500' : ''}>
                        <SelectValue />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="tls">TLS</SelectItem>
                        <SelectItem value="ssl">SSL</SelectItem>
                        <SelectItem value="none">None</SelectItem>
                      </SelectContent>
                    </Select>
                    {errors.smtp_encryption && (
                      <p className="text-sm text-red-500">{errors.smtp_encryption}</p>
                    )}
                  </div>
                </div>

                <div className="space-y-2">
                  <Label htmlFor="smtp_username">Username *</Label>
                  <Input
                    id="smtp_username"
                    value={formData.smtp_username}
                    onChange={(e) => setFormData(prev => ({ ...prev, smtp_username: e.target.value }))}
                    placeholder="e.g., support@example.com"
                    className={errors.smtp_username ? 'border-red-500' : ''}
                  />
                  {errors.smtp_username && (
                    <p className="text-sm text-red-500">{errors.smtp_username}</p>
                  )}
                </div>

                <div className="space-y-2">
                  <Label htmlFor="smtp_password">Password *</Label>
                  <Input
                    id="smtp_password"
                    type="password"
                    value={formData.smtp_password}
                    onChange={(e) => setFormData(prev => ({ ...prev, smtp_password: e.target.value }))}
                    placeholder="Enter email password"
                    className={errors.smtp_password ? 'border-red-500' : ''}
                  />
                  {errors.smtp_password && (
                    <p className="text-sm text-red-500">{errors.smtp_password}</p>
                  )}
                </div>

                <Button
                  type="button"
                  variant="outline"
                  onClick={handleTestConnection}
                  className="w-full"
                >
                  <TestTube className="h-4 w-4 mr-2" />
                  Test Connection
                </Button>
              </CardContent>
            </Card>
          </div>

          {/* Additional Settings */}
          <Card>
            <CardHeader>
              <CardTitle>Additional Settings</CardTitle>
              <CardDescription>
                Optional settings and instructions
              </CardDescription>
            </CardHeader>
            <CardContent>
              <div className="space-y-2">
                <Label htmlFor="instructions">Setup Instructions</Label>
                <Textarea
                  id="instructions"
                  value={formData.instructions}
                  onChange={(e) => setFormData(prev => ({ ...prev, instructions: e.target.value }))}
                  placeholder="Add any specific setup instructions or notes..."
                  rows={4}
                  className={errors.instructions ? 'border-red-500' : ''}
                />
                {errors.instructions && (
                  <p className="text-sm text-red-500">{errors.instructions}</p>
                )}
                <p className="text-xs text-muted-foreground">
                  These instructions will be shown to users when they set up email access on their devices.
                </p>
              </div>
            </CardContent>
          </Card>

          {/* Actions */}
          <div className="flex justify-end gap-4">
            <Button type="button" variant="outline" asChild>
              <Link href={route('admin.settings.emails.index')}>
                Cancel
              </Link>
            </Button>
            <Button type="submit" disabled={isSubmitting}>
              {isSubmitting ? (
                <>
                  <div className="mr-2 h-4 w-4 animate-spin rounded-full border-2 border-white border-t-transparent"></div>
                  Creating...
                </>
              ) : (
                <>
                  <Save className="h-4 w-4 mr-2" />
                  Create Email Account
                </>
              )}
            </Button>
          </div>
        </form>
      </div>
    </AppLayout>
  );
};

export default EmailCreate;
