import React, { useState } from 'react';
import { Head, Link, router } from '@inertiajs/react';
import AppLayout from '@/layouts/app-layout';
import { Badge } from '@/components/ui/badge';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { 
  Shield, 
  Users, 
  Settings,
  Key,
  Edit,
  ArrowLeft,
  UserPlus,
  UserMinus,
  CheckCircle,
  XCircle,
  Calendar,
  Clock
} from 'lucide-react';

interface User {
  id: number;
  name: string;
  email: string;
  pivot: {
    is_primary: boolean;
    assigned_at: string;
    expires_at?: string;
  };
}

interface Permission {
  id: number;
  name: string;
  display_name: string;
  description: string;
  category: string;
}

interface Role {
  id: number;
  name: string;
  display_name: string;
  description: string;
  is_system: boolean;
  is_active: boolean;
  priority: number;
  color: string;
  icon: string;
  metadata: any;
  created_at: string;
  updated_at: string;
  users: User[];
  permissions: Permission[];
}

interface Props {
  role: Role;
}

const RoleShow: React.FC<Props> = ({ role }) => {
  const [showUserAssignment, setShowUserAssignment] = useState(false);

  const getRoleIcon = (icon: string) => {
    const iconMap: Record<string, any> = {
      shield: Shield,
      users: Users,
      settings: Settings,
      key: Key,
    };
    const IconComponent = iconMap[icon] || Shield;
    return <IconComponent className="h-6 w-6" />;
  };

  const handleToggleStatus = () => {
    router.patch(route('admin.roles.toggle-status', role.id), {
      is_active: !role.is_active,
    });
  };

  const handleRemoveUser = (userId: number) => {
    if (confirm('Are you sure you want to remove this user from the role?')) {
      router.post(route('admin.roles.remove-users', role.id), {
        user_ids: [userId],
      });
    }
  };

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric',
      hour: '2-digit',
      minute: '2-digit',
    });
  };

  const isExpired = (expiresAt: string | null) => {
    if (!expiresAt) return false;
    return new Date(expiresAt) < new Date();
  };

  return (
    <AppLayout breadcrumbs={[
      { title: 'Roles', href: '/admin/roles' },
      { title: role.display_name, href: `/admin/roles/${role.id}` }
    ]}>
      <Head title={`${role.display_name} - Role Details`} />
      
      <div className="space-y-6 p-4 flex flex-col gap-4">
        {/* Header */}
        <div className="flex justify-between items-center">
          <div className="flex items-center gap-4">
            <Button variant="ghost" size="sm" asChild>
              <Link href={route('admin.roles.index')}>
                <ArrowLeft className="h-4 w-4 mr-2" />
                Back to Roles
              </Link>
            </Button>
            <div className="flex items-center gap-3">
              <div className={`p-3 rounded-lg ${role.color || 'bg-gray-100'}`}>
                {getRoleIcon(role.icon)}
              </div>
              <div>
                <h1 className="text-3xl font-bold">{role.display_name}</h1>
                <p className="text-muted-foreground font-mono">{role.name}</p>
              </div>
            </div>
          </div>
          <div className="flex gap-2">
            <Button variant="outline" onClick={handleToggleStatus}>
              {role.is_active ? (
                <>
                  <XCircle className="h-4 w-4 mr-2" />
                  Deactivate
                </>
              ) : (
                <>
                  <CheckCircle className="h-4 w-4 mr-2" />
                  Activate
                </>
              )}
            </Button>
            <Button asChild>
              <Link href={route('admin.roles.edit', role.id)}>
                <Edit className="h-4 w-4 mr-2" />
                Edit Role
              </Link>
            </Button>
          </div>
        </div>

        {/* Role Information */}
        <div className="grid gap-6 md:grid-cols-2">
          <Card>
            <CardHeader>
              <CardTitle>Role Information</CardTitle>
              <CardDescription>Basic details about this role</CardDescription>
            </CardHeader>
            <CardContent className="space-y-4">
              <div>
                <label className="text-sm font-medium text-muted-foreground">Description</label>
                <p className="mt-1">{role.description || 'No description provided'}</p>
              </div>
              
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <label className="text-sm font-medium text-muted-foreground">Priority</label>
                  <p className="mt-1 font-mono">{role.priority}</p>
                </div>
                <div>
                  <label className="text-sm font-medium text-muted-foreground">Status</label>
                  <div className="mt-1">
                    <Badge variant={role.is_active ? 'default' : 'secondary'}>
                      {role.is_active ? 'Active' : 'Inactive'}
                    </Badge>
                  </div>
                </div>
              </div>

              <div className="grid grid-cols-2 gap-4">
                <div>
                  <label className="text-sm font-medium text-muted-foreground">Type</label>
                  <div className="mt-1">
                    <Badge variant={role.is_system ? 'destructive' : 'outline'}>
                      {role.is_system ? 'System Role' : 'Custom Role'}
                    </Badge>
                  </div>
                </div>
                <div>
                  <label className="text-sm font-medium text-muted-foreground">Color</label>
                  <div className="mt-1 flex items-center gap-2">
                    <div className={`w-4 h-4 rounded ${role.color || 'bg-gray-300'}`}></div>
                    <span className="text-sm font-mono">{role.color || 'default'}</span>
                  </div>
                </div>
              </div>

              <div>
                <label className="text-sm font-medium text-muted-foreground">Created</label>
                <p className="mt-1 flex items-center gap-2">
                  <Calendar className="h-4 w-4 text-muted-foreground" />
                  {formatDate(role.created_at)}
                </p>
              </div>

              <div>
                <label className="text-sm font-medium text-muted-foreground">Last Updated</label>
                <p className="mt-1 flex items-center gap-2">
                  <Clock className="h-4 w-4 text-muted-foreground" />
                  {formatDate(role.updated_at)}
                </p>
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardHeader>
              <CardTitle>Statistics</CardTitle>
              <CardDescription>Role usage and permissions</CardDescription>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="grid grid-cols-2 gap-4">
                <div className="text-center p-4 border rounded-lg">
                  <div className="text-2xl font-bold text-blue-600">{role.users.length}</div>
                  <div className="text-sm text-muted-foreground">Assigned Users</div>
                </div>
                <div className="text-center p-4 border rounded-lg">
                  <div className="text-2xl font-bold text-green-600">{role?.permissions?.length}</div>
                  <div className="text-sm text-muted-foreground">Permissions</div>
                </div>
              </div>

              <div className="space-y-2">
                <div className="flex justify-between text-sm">
                  <span>Primary Users</span>
                  <span className="font-medium">
                    {role?.users?.filter(u => u.pivot.is_primary).length}
                  </span>
                </div>
                <div className="flex justify-between text-sm">
                  <span>Expired Assignments</span>
                  <span className="font-medium">
                    {role?.users?.filter(u => u.pivot.expires_at && isExpired(u.pivot.expires_at)).length}
                  </span>
                </div>
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Assigned Users */}
        <Card>
          <CardHeader>
            <div className="flex justify-between items-center">
              <div>
                <CardTitle>Assigned Users</CardTitle>
                <CardDescription>
                  {role?.users?.length} user(s) assigned to this role
                </CardDescription>
              </div>
              <Button 
                variant="outline" 
                size="sm"
                onClick={() => setShowUserAssignment(!showUserAssignment)}
              >
                <UserPlus className="h-4 w-4 mr-2" />
                Manage Users
              </Button>
            </div>
          </CardHeader>
          <CardContent>
            {role?.users?.length === 0 ? (
              <div className="text-center py-8 text-muted-foreground">
                <Users className="h-12 w-12 mx-auto mb-4 opacity-50" />
                <p>No users assigned to this role</p>
              </div>
            ) : (
              <div className="overflow-x-auto">
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead>User</TableHead>
                      <TableHead>Email</TableHead>
                      <TableHead>Assignment Type</TableHead>
                      <TableHead>Assigned Date</TableHead>
                      <TableHead>Expires</TableHead>
                      <TableHead className="text-right">Actions</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {role?.users?.map((user) => (
                      <TableRow key={user.id}>
                        <TableCell>
                          <div className="font-medium">{user.name}</div>
                        </TableCell>
                        <TableCell>
                          <div className="text-muted-foreground">{user.email}</div>
                        </TableCell>
                        <TableCell>
                          <Badge variant={user.pivot.is_primary ? 'default' : 'outline'}>
                            {user.pivot.is_primary ? 'Primary' : 'Secondary'}
                          </Badge>
                        </TableCell>
                        <TableCell>
                          <div className="text-sm">
                            {formatDate(user.pivot.assigned_at)}
                          </div>
                        </TableCell>
                        <TableCell>
                          {user.pivot.expires_at ? (
                            <div className={`text-sm ${isExpired(user.pivot.expires_at) ? 'text-red-600' : 'text-muted-foreground'}`}>
                              {formatDate(user.pivot.expires_at)}
                              {isExpired(user.pivot.expires_at) && (
                                <Badge variant="destructive" className="ml-2">Expired</Badge>
                              )}
                            </div>
                          ) : (
                            <span className="text-muted-foreground">Never</span>
                          )}
                        </TableCell>
                        <TableCell className="text-right">
                          <Button
                            variant="ghost"
                            size="sm"
                            onClick={() => handleRemoveUser(user.id)}
                            className="text-red-600 hover:text-red-700"
                          >
                            <UserMinus className="h-4 w-4" />
                          </Button>
                        </TableCell>
                      </TableRow>
                    ))}
                  </TableBody>
                </Table>
              </div>
            )}
          </CardContent>
        </Card>

        {/* Permissions */}
        <Card>
          <CardHeader>
            <CardTitle>Permissions</CardTitle>
            <CardDescription>
              {role?.permissions?.length} permission(s) assigned to this role
            </CardDescription>
          </CardHeader>
          <CardContent>
            {role?.permissions?.length === 0 ? (
              <div className="text-center py-8 text-muted-foreground">
                <Key className="h-12 w-12 mx-auto mb-4 opacity-50" />
                <p>No permissions assigned to this role</p>
              </div>
            ) : (
              <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-3">
                {role?.permissions?.map((permission) => (
                  <div key={permission.id} className="p-4 border rounded-lg">
                    <div className="flex items-start justify-between">
                      <div className="flex-1">
                        <h4 className="font-medium">{permission.display_name}</h4>
                        <p className="text-sm text-muted-foreground font-mono">
                          {permission.name}
                        </p>
                        <p className="text-sm text-muted-foreground mt-1">
                          {permission.description}
                        </p>
                      </div>
                      <Badge variant="outline" className="ml-2">
                        {permission.category}
                      </Badge>
                    </div>
                  </div>
                ))}
              </div>
            )}
          </CardContent>
        </Card>
      </div>
    </AppLayout>
  );
};

export default RoleShow;
