import React, { useState } from 'react';
import { Head, Link, router } from '@inertiajs/react';
import AppLayout from '@/layouts/app-layout';
import { Badge } from '@/components/ui/badge';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { Checkbox } from '@/components/ui/checkbox';
import { 
  Shield, 
  Users, 
  Settings,
  Search,
  Filter,
  Eye,
  Edit,
  Trash2,
  Plus,
  MoreHorizontal,
  CheckCircle,
  XCircle,
  Key,
  UserPlus,
  UserMinus
} from 'lucide-react';
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuLabel,
  DropdownMenuSeparator,
  DropdownMenuTrigger,
} from '@/components/ui/dropdown-menu';

interface Role {
  id: number;
  name: string;
  display_name: string;
  description: string;
  is_system: boolean;
  is_active: boolean;
  priority: number;
  color: string;
  icon: string;
  users_count: number;
  permissions_count: number;
  created_at: string;
  updated_at: string;
}

interface Props {
  roles: {
    data: Role[];
    current_page: number;
    last_page: number;
    per_page: number;
    total: number;
  };
  filters: {
    search: string;
    is_active: string;
    is_system: string;
    sort_by: string;
    sort_order: string;
  };
}

const RolesIndex: React.FC<Props> = ({ roles, filters }) => {
  const [searchTerm, setSearchTerm] = useState(filters.search || '');
  const [selectedRoles, setSelectedRoles] = useState<number[]>([]);

  const getRoleIcon = (icon: string) => {
    const iconMap: Record<string, any> = {
      shield: Shield,
      users: Users,
      settings: Settings,
      key: Key,
    };
    const IconComponent = iconMap[icon] || Shield;
    return <IconComponent className="h-4 w-4" />;
  };

  const handleSearch = (e: React.FormEvent) => {
    e.preventDefault();
    router.get(route('admin.roles.index'), {
      ...filters,
      search: searchTerm,
    });
  };

  const handleFilter = (key: string, value: string) => {
    router.get(route('admin.roles.index'), {
      ...filters,
      [key]: value,
    });
  };

  const handleSort = (column: string) => {
    const newOrder = filters.sort_by === column && filters.sort_order === 'asc' ? 'desc' : 'asc';
    router.get(route('admin.roles.index'), {
      ...filters,
      sort_by: column,
      sort_order: newOrder,
    });
  };

  const handleSelectAll = (checked: boolean) => {
    if (checked) {
      setSelectedRoles(roles.data.map(r => r.id));
    } else {
      setSelectedRoles([]);
    }
  };

  const handleSelectRole = (roleId: number, checked: boolean) => {
    if (checked) {
      setSelectedRoles([...selectedRoles, roleId]);
    } else {
      setSelectedRoles(selectedRoles.filter(id => id !== roleId));
    }
  };

  const handleBulkAction = (action: string) => {
    if (selectedRoles.length === 0) return;

    router.post(route('admin.roles.bulk-update'), {
      ids: selectedRoles,
      action,
    }, {
      onSuccess: () => {
        setSelectedRoles([]);
      }
    });
  };

  const handleDelete = (roleId: number) => {
    if (confirm('Are you sure you want to delete this role?')) {
      router.delete(route('admin.roles.destroy', roleId));
    }
  };

  const handleToggleStatus = (roleId: number, currentStatus: boolean) => {
    router.patch(route('admin.roles.toggle-status', roleId), {
      is_active: !currentStatus,
    });
  };

  // Calculate statistics
  const stats = {
    total: roles.total,
    active: roles.data.filter(r => r.is_active).length,
    inactive: roles.data.filter(r => !r.is_active).length,
    system: roles.data.filter(r => r.is_system).length,
    custom: roles.data.filter(r => !r.is_system).length,
  };

  const statCards = [
    {
      title: 'Total Roles',
      value: stats.total,
      icon: Shield,
      description: 'All roles',
      color: 'text-blue-600',
    },
    {
      title: 'Active Roles',
      value: stats.active,
      icon: CheckCircle,
      description: 'Currently active',
      color: 'text-green-600',
    },
    {
      title: 'System Roles',
      value: stats.system,
      icon: Settings,
      description: 'Core system roles',
      color: 'text-red-600',
    },
    {
      title: 'Custom Roles',
      value: stats.custom,
      icon: Users,
      description: 'User-defined roles',
      color: 'text-purple-600',
    },
  ];

  return (
    <AppLayout breadcrumbs={[{ title: 'Roles', href: '/admin/roles' }]}>
      <Head title="Role Management" />
      
      <div className="space-y-6 p-4 flex flex-col gap-4">
        {/* Header */}
        <div className="flex justify-between items-center">
          <div>
            <h1 className="text-3xl font-bold">Roles</h1>
            <p className="text-muted-foreground">Manage user roles and permissions</p>
          </div>
          <Button asChild>
            <Link href={route('admin.roles.create')}>
              <Plus className="h-4 w-4 mr-2" />
              Create Role
            </Link>
          </Button>
        </div>

        {/* Statistics Grid */}
        <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
          {statCards.map((stat) => {
            const IconComponent = stat.icon;
            return (
              <Card key={stat.title}>
                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                  <CardTitle className="text-sm font-medium">
                    {stat.title}
                  </CardTitle>
                  <IconComponent className={`h-4 w-4 ${stat.color}`} />
                </CardHeader>
                <CardContent>
                  <div className="text-2xl font-bold">{stat.value}</div>
                  <p className="text-xs text-muted-foreground">
                    {stat.description}
                  </p>
                </CardContent>
              </Card>
            );
          })}
        </div>

        {/* Filters and Search */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Filter className="h-5 w-5" />
              Filters & Search
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="flex flex-col sm:flex-row gap-4">
              <form onSubmit={handleSearch} className="flex gap-2 flex-1">
                <div className="relative flex-1">
                  <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 h-4 w-4" />
                  <Input
                    placeholder="Search roles..."
                    value={searchTerm}
                    onChange={(e) => setSearchTerm(e.target.value)}
                    className="pl-10"
                  />
                </div>
                <Button type="submit">Search</Button>
              </form>

              <div className="flex gap-2">
                <Select
                  value={filters.is_active || 'all'}
                  onValueChange={(value) => handleFilter('is_active', value)}
                >
                  <SelectTrigger className="w-32">
                    <SelectValue placeholder="Status" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">All Status</SelectItem>
                    <SelectItem value="true">Active</SelectItem>
                    <SelectItem value="false">Inactive</SelectItem>
                  </SelectContent>
                </Select>

                <Select
                  value={filters.is_system || 'all'}
                  onValueChange={(value) => handleFilter('is_system', value)}
                >
                  <SelectTrigger className="w-32">
                    <SelectValue placeholder="Type" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">All Types</SelectItem>
                    <SelectItem value="true">System</SelectItem>
                    <SelectItem value="false">Custom</SelectItem>
                  </SelectContent>
                </Select>
              </div>
            </div>
          </CardContent>
        </Card>

        {/* Bulk Actions */}
        {selectedRoles.length > 0 && (
          <Card>
            <CardContent className="py-4">
              <div className="flex items-center justify-between">
                <span className="text-sm text-muted-foreground">
                  {selectedRoles.length} role(s) selected
                </span>
                <div className="flex gap-2">
                  <Button 
                    variant="outline" 
                    size="sm"
                    onClick={() => handleBulkAction('activate')}
                  >
                    <CheckCircle className="h-4 w-4 mr-2" />
                    Activate
                  </Button>
                  <Button 
                    variant="outline" 
                    size="sm"
                    onClick={() => handleBulkAction('deactivate')}
                  >
                    <XCircle className="h-4 w-4 mr-2" />
                    Deactivate
                  </Button>
                  <Button 
                    variant="destructive" 
                    size="sm"
                    onClick={() => {
                      if (confirm('Are you sure you want to delete selected roles?')) {
                        router.post(route('admin.roles.bulk-delete'), {
                          ids: selectedRoles
                        });
                      }
                    }}
                  >
                    <Trash2 className="h-4 w-4 mr-2" />
                    Delete
                  </Button>
                </div>
              </div>
            </CardContent>
          </Card>
        )}

        {/* Roles Table */}
        <Card>
          <CardHeader>
            <CardTitle>Roles</CardTitle>
            <CardDescription>
              Showing {roles.data.length} of {roles.total} roles
            </CardDescription>
          </CardHeader>
          <CardContent>
            <div className="overflow-x-auto">
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead className="w-12">
                      <Checkbox
                        checked={selectedRoles.length === roles.data.length}
                        onCheckedChange={handleSelectAll}
                      />
                    </TableHead>
                    <TableHead 
                      className="cursor-pointer hover:bg-muted/50"
                      onClick={() => handleSort('display_name')}
                    >
                      Role
                    </TableHead>
                    <TableHead>Description</TableHead>
                    <TableHead 
                      className="cursor-pointer hover:bg-muted/50"
                      onClick={() => handleSort('priority')}
                    >
                      Priority
                    </TableHead>
                    <TableHead>Users</TableHead>
                    <TableHead>Permissions</TableHead>
                    <TableHead>Status</TableHead>
                    <TableHead>Type</TableHead>
                    <TableHead className="text-right">Actions</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {roles.data.map((role) => (
                    <TableRow key={role.id}>
                      <TableCell>
                        <Checkbox
                          checked={selectedRoles.includes(role.id)}
                          onCheckedChange={(checked) => handleSelectRole(role.id, !!checked)}
                        />
                      </TableCell>
                      <TableCell>
                        <div className="flex items-center gap-3">
                          <div className={`p-2 rounded-lg ${role.color || 'bg-gray-100'}`}>
                            {getRoleIcon(role.icon)}
                          </div>
                          <div>
                            <div className="font-medium">{role.display_name}</div>
                            <div className="text-sm text-muted-foreground font-mono">
                              {role.name}
                            </div>
                          </div>
                        </div>
                      </TableCell>
                      <TableCell>
                        <div className="max-w-xs truncate" title={role.description}>
                          {role.description || 'No description'}
                        </div>
                      </TableCell>
                      <TableCell>
                        <Badge variant="outline">{role.priority}</Badge>
                      </TableCell>
                      <TableCell>
                        <div className="flex items-center gap-1">
                          <Users className="h-4 w-4 text-muted-foreground" />
                          <span>{role.users_count}</span>
                        </div>
                      </TableCell>
                      <TableCell>
                        <div className="flex items-center gap-1">
                          <Key className="h-4 w-4 text-muted-foreground" />
                          <span>{role.permissions_count}</span>
                        </div>
                      </TableCell>
                      <TableCell>
                        <Badge variant={role.is_active ? 'default' : 'secondary'}>
                          {role.is_active ? 'Active' : 'Inactive'}
                        </Badge>
                      </TableCell>
                      <TableCell>
                        <Badge variant={role.is_system ? 'destructive' : 'outline'}>
                          {role.is_system ? 'System' : 'Custom'}
                        </Badge>
                      </TableCell>
                      <TableCell className="text-right">
                        <DropdownMenu>
                          <DropdownMenuTrigger asChild>
                            <Button variant="ghost" className="h-8 w-8 p-0">
                              <MoreHorizontal className="h-4 w-4" />
                            </Button>
                          </DropdownMenuTrigger>
                          <DropdownMenuContent align="end">
                            <DropdownMenuLabel>Actions</DropdownMenuLabel>
                            <DropdownMenuItem asChild>
                              <Link href={route('admin.roles.show', role.id)}>
                                <Eye className="h-4 w-4 mr-2" />
                                View
                              </Link>
                            </DropdownMenuItem>
                            <DropdownMenuItem asChild>
                              <Link href={route('admin.roles.edit', role.id)}>
                                <Edit className="h-4 w-4 mr-2" />
                                Edit
                              </Link>
                            </DropdownMenuItem>
                            <DropdownMenuItem 
                              onClick={() => handleToggleStatus(role.id, role.is_active)}
                            >
                              {role.is_active ? (
                                <>
                                  <XCircle className="h-4 w-4 mr-2" />
                                  Deactivate
                                </>
                              ) : (
                                <>
                                  <CheckCircle className="h-4 w-4 mr-2" />
                                  Activate
                                </>
                              )}
                            </DropdownMenuItem>
                            <DropdownMenuSeparator />
                            {!role.is_system && (
                              <DropdownMenuItem 
                                onClick={() => handleDelete(role.id)}
                                className="text-red-600"
                              >
                                <Trash2 className="h-4 w-4 mr-2" />
                                Delete
                              </DropdownMenuItem>
                            )}
                          </DropdownMenuContent>
                        </DropdownMenu>
                      </TableCell>
                    </TableRow>
                  ))}
                </TableBody>
              </Table>
            </div>

            {/* Pagination */}
            {roles.last_page > 1 && (
              <div className="flex items-center justify-between mt-4">
                <div className="text-sm text-muted-foreground">
                  Showing {((roles.current_page - 1) * roles.per_page) + 1} to{' '}
                  {Math.min(roles.current_page * roles.per_page, roles.total)} of{' '}
                  {roles.total} roles
                </div>
                <div className="flex gap-2">
                  {roles.current_page > 1 && (
                    <Button
                      variant="outline"
                      size="sm"
                      onClick={() => router.get(route('admin.roles.index'), {
                        ...filters,
                        page: roles.current_page - 1,
                      })}
                    >
                      Previous
                    </Button>
                  )}
                  {roles.current_page < roles.last_page && (
                    <Button
                      variant="outline"
                      size="sm"
                      onClick={() => router.get(route('admin.roles.index'), {
                        ...filters,
                        page: roles.current_page + 1,
                      })}
                    >
                      Next
                    </Button>
                  )}
                </div>
              </div>
            )}
          </CardContent>
        </Card>
      </div>
    </AppLayout>
  );
};

export default RolesIndex;
