import React, { useState, useEffect } from 'react';
import { useForm } from '@inertiajs/react';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import { Switch } from '@/components/ui/switch';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Checkbox } from '@/components/ui/checkbox';
import { Badge } from '@/components/ui/badge';
import { 
  Shield, 
  Users, 
  Settings,
  Key,
  Palette,
  Info
} from 'lucide-react';

interface Permission {
  id: number;
  name: string;
  display_name: string;
  description: string;
  category: string;
  is_system: boolean;
  is_active: boolean;
}

interface Role {
  id?: number;
  name: string;
  display_name: string;
  description: string;
  is_system: boolean;
  is_active: boolean;
  priority: number;
  color: string;
  icon: string;
  permissions: Permission[];
}

interface Props {
  role?: Role;
  permissions: Record<string, Permission[]>;
  categories: Record<string, string>;
  isEditing?: boolean;
  onSubmit: (data: any) => void;
  onCancel: () => void;
  loading?: boolean;
}

const RoleForm: React.FC<Props> = ({ 
  role, 
  permissions, 
  categories, 
  isEditing = false, 
  onSubmit, 
  onCancel, 
  loading = false 
}) => {
  const { data, setData, errors, processing } = useForm({
    name: role?.name || '',
    display_name: role?.display_name || '',
    description: role?.description || '',
    is_system: role?.is_system || false,
    is_active: role?.is_active ?? true,
    priority: role?.priority || 0,
    color: role?.color || 'bg-blue-500',
    icon: role?.icon || 'shield',
    permission_ids: role?.permissions?.map(p => p.id) || [],
  });

  const [selectedPermissions, setSelectedPermissions] = useState<number[]>(data.permission_ids);

  const iconOptions = [
    { value: 'shield', label: 'Shield', icon: Shield },
    { value: 'users', label: 'Users', icon: Users },
    { value: 'settings', label: 'Settings', icon: Settings },
    { value: 'key', label: 'Key', icon: Key },
  ];

  const colorOptions = [
    { value: 'bg-blue-500', label: 'Blue', color: 'bg-blue-500' },
    { value: 'bg-green-500', label: 'Green', color: 'bg-green-500' },
    { value: 'bg-red-500', label: 'Red', color: 'bg-red-500' },
    { value: 'bg-yellow-500', label: 'Yellow', color: 'bg-yellow-500' },
    { value: 'bg-purple-500', label: 'Purple', color: 'bg-purple-500' },
    { value: 'bg-pink-500', label: 'Pink', color: 'bg-pink-500' },
    { value: 'bg-indigo-500', label: 'Indigo', color: 'bg-indigo-500' },
    { value: 'bg-gray-500', label: 'Gray', color: 'bg-gray-500' },
  ];

  const getIconComponent = (iconName: string) => {
    const iconMap: Record<string, any> = {
      shield: Shield,
      users: Users,
      settings: Settings,
      key: Key,
    };
    return iconMap[iconName] || Shield;
  };

  const handlePermissionToggle = (permissionId: number, checked: boolean) => {
    let newSelected;
    if (checked) {
      newSelected = [...selectedPermissions, permissionId];
    } else {
      newSelected = selectedPermissions.filter(id => id !== permissionId);
    }
    setSelectedPermissions(newSelected);
    setData('permission_ids', newSelected);
  };

  const handleCategoryToggle = (category: string, checked: boolean) => {
    const categoryPermissions = permissions[category] || [];
    const categoryPermissionIds = categoryPermissions.map(p => p.id);
    
    let newSelected;
    if (checked) {
      // Add all permissions from this category
      newSelected = [...new Set([...selectedPermissions, ...categoryPermissionIds])];
    } else {
      // Remove all permissions from this category
      newSelected = selectedPermissions.filter(id => !categoryPermissionIds.includes(id));
    }
    setSelectedPermissions(newSelected);
    setData('permission_ids', newSelected);
  };

  const isCategorySelected = (category: string) => {
    const categoryPermissions = permissions[category] || [];
    const categoryPermissionIds = categoryPermissions.map(p => p.id);
    return categoryPermissionIds.every(id => selectedPermissions.includes(id));
  };

  const isCategoryPartiallySelected = (category: string) => {
    const categoryPermissions = permissions[category] || [];
    const categoryPermissionIds = categoryPermissions.map(p => p.id);
    const selectedInCategory = categoryPermissionIds.filter(id => selectedPermissions.includes(id));
    return selectedInCategory.length > 0 && selectedInCategory.length < categoryPermissionIds.length;
  };

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    onSubmit(data);
  };

  return (
    <form onSubmit={handleSubmit} className="space-y-6">
      {/* Basic Information */}
      <Card>
        <CardHeader>
          <CardTitle>Basic Information</CardTitle>
          <CardDescription>
            Configure the basic details of the role
          </CardDescription>
        </CardHeader>
        <CardContent className="space-y-4">
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div className="space-y-2">
              <Label htmlFor="name">Role Name *</Label>
              <Input
                id="name"
                value={data.name}
                onChange={(e) => setData('name', e.target.value)}
                placeholder="e.g., tournament_manager"
                className={errors.name ? 'border-red-500' : ''}
                disabled={isEditing && role?.is_system}
              />
              {errors.name && (
                <p className="text-sm text-red-500">{errors.name}</p>
              )}
              <p className="text-xs text-muted-foreground">
                Unique identifier for the role (lowercase, underscores)
              </p>
            </div>

            <div className="space-y-2">
              <Label htmlFor="display_name">Display Name *</Label>
              <Input
                id="display_name"
                value={data.display_name}
                onChange={(e) => setData('display_name', e.target.value)}
                placeholder="e.g., Tournament Manager"
                className={errors.display_name ? 'border-red-500' : ''}
              />
              {errors.display_name && (
                <p className="text-sm text-red-500">{errors.display_name}</p>
              )}
            </div>
          </div>

          <div className="space-y-2">
            <Label htmlFor="description">Description</Label>
            <Textarea
              id="description"
              value={data.description}
              onChange={(e) => setData('description', e.target.value)}
              placeholder="Describe what this role is for..."
              className={errors.description ? 'border-red-500' : ''}
              rows={3}
            />
            {errors.description && (
              <p className="text-sm text-red-500">{errors.description}</p>
            )}
          </div>

          <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
            <div className="space-y-2">
              <Label htmlFor="priority">Priority</Label>
              <Input
                id="priority"
                type="number"
                value={data.priority}
                onChange={(e) => setData('priority', parseInt(e.target.value) || 0)}
                className={errors.priority ? 'border-red-500' : ''}
              />
              {errors.priority && (
                <p className="text-sm text-red-500">{errors.priority}</p>
              )}
              <p className="text-xs text-muted-foreground">
                Higher numbers = higher priority
              </p>
            </div>

            <div className="space-y-2">
              <Label htmlFor="icon">Icon</Label>
              <Select value={data.icon} onValueChange={(value) => setData('icon', value)}>
                <SelectTrigger>
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  {iconOptions.map((option) => {
                    const IconComponent = option.icon;
                    return (
                      <SelectItem key={option.value} value={option.value}>
                        <div className="flex items-center gap-2">
                          <IconComponent className="h-4 w-4" />
                          {option.label}
                        </div>
                      </SelectItem>
                    );
                  })}
                </SelectContent>
              </Select>
            </div>

            <div className="space-y-2">
              <Label htmlFor="color">Color</Label>
              <Select value={data.color} onValueChange={(value) => setData('color', value)}>
                <SelectTrigger>
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  {colorOptions.map((option) => (
                    <SelectItem key={option.value} value={option.value}>
                      <div className="flex items-center gap-2">
                        <div className={`w-4 h-4 rounded ${option.color}`}></div>
                        {option.label}
                      </div>
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
          </div>

          <div className="flex items-center space-x-4">
            <div className="flex items-center space-x-2">
              <Switch
                id="is_active"
                checked={data.is_active}
                onCheckedChange={(checked) => setData('is_active', checked)}
                disabled={isEditing && role?.is_system}
              />
              <Label htmlFor="is_active">Active</Label>
            </div>

            {!isEditing && (
              <div className="flex items-center space-x-2">
                <Switch
                  id="is_system"
                  checked={data.is_system}
                  onCheckedChange={(checked) => setData('is_system', checked)}
                />
                <Label htmlFor="is_system">System Role</Label>
              </div>
            )}
          </div>
        </CardContent>
      </Card>

      {/* Permissions */}
      <Card>
        <CardHeader>
          <CardTitle>Permissions</CardTitle>
          <CardDescription>
            Select the permissions to assign to this role
          </CardDescription>
        </CardHeader>
        <CardContent>
          <div className="space-y-6">
            {Object.entries(permissions).map(([category, categoryPermissions]) => (
              <div key={category} className="space-y-3">
                <div className="flex items-center justify-between">
                  <div className="flex items-center gap-2">
                    <Checkbox
                      checked={isCategorySelected(category)}
                      ref={(el) => {
                        if (el) {
                          el.indeterminate = isCategoryPartiallySelected(category);
                        }
                      }}
                      onCheckedChange={(checked) => handleCategoryToggle(category, !!checked)}
                    />
                    <h4 className="font-medium capitalize">
                      {categories[category] || category}
                    </h4>
                    <Badge variant="outline" className="text-xs">
                      {categoryPermissions.length}
                    </Badge>
                  </div>
                </div>

                <div className="grid gap-2 md:grid-cols-2 lg:grid-cols-3 ml-6">
                  {categoryPermissions.map((permission) => (
                    <div key={permission.id} className="flex items-center space-x-2">
                      <Checkbox
                        id={`permission-${permission.id}`}
                        checked={selectedPermissions.includes(permission.id)}
                        onCheckedChange={(checked) => handlePermissionToggle(permission.id, !!checked)}
                      />
                      <Label 
                        htmlFor={`permission-${permission.id}`}
                        className="text-sm font-normal cursor-pointer flex-1"
                      >
                        <div>
                          <div className="font-medium">{permission.display_name}</div>
                          <div className="text-xs text-muted-foreground font-mono">
                            {permission.name}
                          </div>
                        </div>
                      </Label>
                    </div>
                  ))}
                </div>
              </div>
            ))}
          </div>

          {Object.keys(permissions).length === 0 && (
            <div className="text-center py-8 text-muted-foreground">
              <Key className="h-12 w-12 mx-auto mb-4 opacity-50" />
              <p>No permissions available</p>
            </div>
          )}
        </CardContent>
      </Card>

      {/* Preview */}
      <Card>
        <CardHeader>
          <CardTitle>Preview</CardTitle>
          <CardDescription>How this role will appear in the system</CardDescription>
        </CardHeader>
        <CardContent>
          <div className="flex items-center gap-3 p-4 border rounded-lg">
            <div className={`p-2 rounded-lg ${data.color}`}>
              {React.createElement(getIconComponent(data.icon), { className: "h-6 w-6 text-white" })}
            </div>
            <div className="flex-1">
              <div className="font-medium">{data.display_name || 'Display Name'}</div>
              <div className="text-sm text-muted-foreground font-mono">
                {data.name || 'role_name'}
              </div>
              <div className="text-sm text-muted-foreground">
                {data.description || 'No description'}
              </div>
            </div>
            <div className="flex gap-2">
              <Badge variant={data.is_active ? 'default' : 'secondary'}>
                {data.is_active ? 'Active' : 'Inactive'}
              </Badge>
              <Badge variant={data.is_system ? 'destructive' : 'outline'}>
                {data.is_system ? 'System' : 'Custom'}
              </Badge>
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Actions */}
      <div className="flex justify-end gap-4">
        <Button type="button" variant="outline" onClick={onCancel}>
          Cancel
        </Button>
        <Button type="submit" disabled={processing || loading}>
          {processing || loading ? 'Saving...' : (isEditing ? 'Update Role' : 'Create Role')}
        </Button>
      </div>
    </form>
  );
};

export default RoleForm;
