import React from 'react';
import { Head, router } from '@inertiajs/react';
import AppLayout from '@/layouts/app-layout';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import RoleForm from './form';

interface Permission {
  id: number;
  name: string;
  display_name: string;
  description: string;
  category: string;
  is_system: boolean;
  is_active: boolean;
}

interface Role {
  id: number;
  name: string;
  display_name: string;
  description: string;
  is_system: boolean;
  is_active: boolean;
  priority: number;
  color: string;
  icon: string;
  permissions: Permission[];
}

interface Props {
  role: Role;
  permissions: Record<string, Permission[]>;
  categories: Record<string, string>;
}

const RoleEdit: React.FC<Props> = ({ role, permissions, categories }) => {
  const handleSubmit = (data: any) => {
    router.put(route('admin.roles.update', role.id), data, {
      onSuccess: () => {
        // Redirect will be handled by the controller
      },
      onError: (errors) => {
        console.error('Validation errors:', errors);
      }
    });
  };

  const handleCancel = () => {
    router.visit(route('admin.roles.show', role.id));
  };

  return (
    <AppLayout breadcrumbs={[
      { title: 'Roles', href: '/admin/roles' },
      { title: role.display_name, href: `/admin/roles/${role.id}` },
      { title: 'Edit', href: `/admin/roles/${role.id}/edit` }
    ]}>
      <Head title={`Edit ${role.display_name}`} />
      
      <div className="space-y-6 p-4">
        <div>
          <h1 className="text-3xl font-bold">Edit Role</h1>
          <p className="text-muted-foreground">
            Modify the role settings and permissions
          </p>
        </div>

        {role.is_system && (
          <Card className="border-amber-200 bg-amber-50">
            <CardContent className="pt-6">
              <div className="flex items-center gap-2 text-amber-800">
                <div className="h-2 w-2 bg-amber-500 rounded-full"></div>
                <p className="text-sm font-medium">
                  This is a system role. Some settings cannot be modified.
                </p>
              </div>
            </CardContent>
          </Card>
        )}

        <Card>
          <CardHeader>
            <CardTitle>Role Configuration</CardTitle>
            <CardDescription>
              Update the role details and permissions. Fields marked with * are required.
            </CardDescription>
          </CardHeader>
          <CardContent>
            <RoleForm
              role={role}
              permissions={permissions}
              categories={categories}
              isEditing={true}
              onSubmit={handleSubmit}
              onCancel={handleCancel}
            />
          </CardContent>
        </Card>
      </div>
    </AppLayout>
  );
};

export default RoleEdit;
