import { Head, Link } from "@inertiajs/react";
import AppLayout from "@/layouts/app-layout";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { ArrowLeft, Edit, Trophy, Medal, Award, Star, Crown, Gift, DollarSign, Users, Target, Calendar, Clock } from "lucide-react";

interface PrizeAward {
  id: number;
  tournament_id: number;
  category: string;
  position: string;
  title: string;
  description: string;
  prize_value: number;
  prize_type: string;
  additional_benefits: string;
  is_active: boolean;
  sort_order: number;
  created_at: string;
  updated_at: string;
  tournament: {
    id: number;
    name: string;
  };
}

interface ShowPrizeAwardPageProps {
  prizeAward: PrizeAward;
  categories: Record<string, string>;
  prizeTypes: Record<string, string>;
  positions: Record<string, string>;
}

const ShowPrizeAward = ({ prizeAward, categories, prizeTypes, positions }: ShowPrizeAwardPageProps) => {
  const categoryIcons = {
    team: Trophy,
    individual: Medal,
    special: Award,
    performance: Star,
    fair_play: Crown,
  };

  const prizeTypeIcons = {
    cash: DollarSign,
    trophy: Trophy,
    medal: Medal,
    certificate: Award,
    merchandise: Gift,
    experience: Star,
    mixed: Crown,
  };

  const categoryColors = {
    team: 'bg-blue-100 text-blue-800',
    individual: 'bg-green-100 text-green-800',
    special: 'bg-purple-100 text-purple-800',
    performance: 'bg-yellow-100 text-yellow-800',
    fair_play: 'bg-indigo-100 text-indigo-800',
  };

  const getCategoryIcon = (category: string) => {
    const IconComponent = categoryIcons[category as keyof typeof categoryIcons] || Trophy;
    return <IconComponent className="w-5 h-5" />;
  };

  const getPrizeTypeIcon = (prizeType: string) => {
    const IconComponent = prizeTypeIcons[prizeType as keyof typeof prizeTypeIcons] || Gift;
    return <IconComponent className="w-5 h-5" />;
  };

  const formatPrizeValue = (value: number, prizeType: string) => {
    if (!value) return '';
    if (prizeType === 'cash') {
      return `$${value.toLocaleString()}`;
    }
    return `Value: $${value.toLocaleString()}`;
  };

  return (
    <AppLayout>
      <Head title={`Prize Award - ${prizeAward.title}`} />
      
      <div className="space-y-6">
        {/* Header */}
        <div className="flex items-center justify-between">
          <div className="flex items-center space-x-4">
            <Link href={route('admin.prize-awards.index')}>
              <Button variant="outline" size="sm">
                <ArrowLeft className="w-4 h-4 mr-2" />
                Back to Prize Awards
              </Button>
            </Link>
            <div>
              <h1 className="text-3xl font-bold text-gray-900">{prizeAward.title}</h1>
              <p className="text-gray-600 mt-1">Prize Award Details</p>
            </div>
          </div>
          <Link href={route('admin.prize-awards.edit', prizeAward.id)}>
            <Button>
              <Edit className="w-4 h-4 mr-2" />
              Edit Prize Award
            </Button>
          </Link>
        </div>

        <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
          {/* Main Details */}
          <div className="lg:col-span-2 space-y-6">
            {/* Basic Information */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center space-x-2">
                  {getCategoryIcon(prizeAward.category)}
                  <span>Prize Award Information</span>
                </CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <div>
                    <label className="text-sm font-medium text-gray-500">Title</label>
                    <p className="text-lg font-semibold text-gray-900">{prizeAward.title}</p>
                  </div>
                  <div>
                    <label className="text-sm font-medium text-gray-500">Position</label>
                    <p className="text-lg font-semibold text-gray-900">{positions[prizeAward.position]}</p>
                  </div>
                  <div>
                    <label className="text-sm font-medium text-gray-500">Category</label>
                    <Badge className={`${categoryColors[prizeAward.category as keyof typeof categoryColors]} flex items-center space-x-1 w-fit`}>
                      {getCategoryIcon(prizeAward.category)}
                      <span>{categories[prizeAward.category]}</span>
                    </Badge>
                  </div>
                  <div>
                    <label className="text-sm font-medium text-gray-500">Prize Type</label>
                    <div className="flex items-center space-x-2">
                      {getPrizeTypeIcon(prizeAward.prize_type)}
                      <span className="font-medium">{prizeTypes[prizeAward.prize_type]}</span>
                    </div>
                  </div>
                </div>

                {prizeAward.description && (
                  <div>
                    <label className="text-sm font-medium text-gray-500">Description</label>
                    <p className="text-gray-700 mt-1">{prizeAward.description}</p>
                  </div>
                )}

                {prizeAward.additional_benefits && (
                  <div>
                    <label className="text-sm font-medium text-gray-500">Additional Benefits</label>
                    <p className="text-gray-700 mt-1">{prizeAward.additional_benefits}</p>
                  </div>
                )}
              </CardContent>
            </Card>

            {/* Prize Value */}
            {prizeAward.prize_value && (
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center space-x-2">
                    <DollarSign className="w-5 h-5" />
                    <span>Prize Value</span>
                  </CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="text-center">
                    <div className="text-4xl font-bold text-green-600 mb-2">
                      {formatPrizeValue(prizeAward.prize_value, prizeAward.prize_type)}
                    </div>
                    <p className="text-gray-600">Prize Value</p>
                  </div>
                </CardContent>
              </Card>
            )}
          </div>

          {/* Sidebar */}
          <div className="space-y-6">
            {/* Status */}
            <Card>
              <CardHeader>
                <CardTitle>Status</CardTitle>
              </CardHeader>
              <CardContent>
                <div className="space-y-3">
                  <div className="flex items-center justify-between">
                    <span className="text-sm font-medium text-gray-500">Status</span>
                    <Badge variant={prizeAward.is_active ? "default" : "secondary"}>
                      {prizeAward.is_active ? "Active" : "Inactive"}
                    </Badge>
                  </div>
                  <div className="flex items-center justify-between">
                    <span className="text-sm font-medium text-gray-500">Sort Order</span>
                    <span className="font-medium">{prizeAward.sort_order}</span>
                  </div>
                </div>
              </CardContent>
            </Card>

            {/* Tournament */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center space-x-2">
                  <Trophy className="w-5 h-5" />
                  <span>Tournament</span>
                </CardTitle>
              </CardHeader>
              <CardContent>
                <div className="space-y-2">
                  <p className="font-semibold text-gray-900">{prizeAward.tournament.name}</p>
                  <p className="text-sm text-gray-600">Tournament ID: {prizeAward.tournament.id}</p>
                </div>
              </CardContent>
            </Card>

            {/* Timestamps */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center space-x-2">
                  <Clock className="w-5 h-5" />
                  <span>Timestamps</span>
                </CardTitle>
              </CardHeader>
              <CardContent>
                <div className="space-y-3">
                  <div>
                    <label className="text-sm font-medium text-gray-500">Created</label>
                    <p className="text-sm text-gray-900">
                      {new Date(prizeAward.created_at).toLocaleDateString()} at {new Date(prizeAward.created_at).toLocaleTimeString()}
                    </p>
                  </div>
                  <div>
                    <label className="text-sm font-medium text-gray-500">Last Updated</label>
                    <p className="text-sm text-gray-900">
                      {new Date(prizeAward.updated_at).toLocaleDateString()} at {new Date(prizeAward.updated_at).toLocaleTimeString()}
                    </p>
                  </div>
                </div>
              </CardContent>
            </Card>

            {/* Actions */}
            <Card>
              <CardHeader>
                <CardTitle>Actions</CardTitle>
              </CardHeader>
              <CardContent>
                <div className="space-y-2">
                  <Link href={route('admin.prize-awards.edit', prizeAward.id)} className="w-full">
                    <Button variant="outline" className="w-full">
                      <Edit className="w-4 h-4 mr-2" />
                      Edit Prize Award
                    </Button>
                  </Link>
                  <Link href={route('admin.prize-awards.index')} className="w-full">
                    <Button variant="outline" className="w-full">
                      <ArrowLeft className="w-4 h-4 mr-2" />
                      Back to List
                    </Button>
                  </Link>
                </div>
              </CardContent>
            </Card>
          </div>
        </div>
      </div>
    </AppLayout>
  );
};

export default ShowPrizeAward;
