import { useState } from "react";
import { Head, Link, router } from "@inertiajs/react";
import AppLayout from "@/layouts/app-layout";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Checkbox } from "@/components/ui/checkbox";
import { Badge } from "@/components/ui/badge";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { DropdownMenu, DropdownMenuContent, DropdownMenuItem, DropdownMenuTrigger } from "@/components/ui/dropdown-menu";
import { Dialog, DialogFooter, DialogHeader, DialogTitle, DialogContent, DialogClose, DialogDescription } from "@/components/ui/dialog";
import { Plus, Search, Filter, MoreHorizontal, Edit, Trash2, Eye, ToggleLeft, ToggleRight, Trophy, Medal, Award, Star, Crown, Gift, DollarSign, Users, Target, Calendar } from "lucide-react";

interface PrizeAward {
  id: number;
  tournament_id: number;
  category: string;
  position: string;
  title: string;
  description: string;
  prize_value: number;
  prize_type: string;
  additional_benefits: string;
  is_active: boolean;
  sort_order: number;
  created_at: string;
  updated_at: string;
  tournament: {
    id: number;
    name: string;
  };
}

interface PrizeAwardsPageProps {
  prizeAwards: {
    data: PrizeAward[];
    links: any[];
    meta: any;
  };
  filters: {
    search: string;
    category: string;
    tournament: string;
    status: string;
  };
  categories: Record<string, string>;
  prizeTypes: Record<string, string>;
  positions: Record<string, string>;
  tournaments: Array<{ id: number; name: string }>;
}

const PrizeAwardsIndex = ({ prizeAwards, filters, categories, prizeTypes, positions, tournaments }: PrizeAwardsPageProps) => {
  const [searchTerm, setSearchTerm] = useState(filters.search || '');
  const [selectedCategory, setSelectedCategory] = useState(filters.category || '');
  const [selectedTournament, setSelectedTournament] = useState(filters.tournament || '');
  const [selectedStatus, setSelectedStatus] = useState(filters.status || '');
  const [selectedPrizeAwards, setSelectedPrizeAwards] = useState<number[]>([]);
  const [deleteDialogOpen, setDeleteDialogOpen] = useState(false);
  const [prizeAwardToDelete, setPrizeAwardToDelete] = useState<PrizeAward | null>(null);
  const [bulkActionDialogOpen, setBulkActionDialogOpen] = useState(false);
  const [bulkAction, setBulkAction] = useState<string>('');

  const categoryIcons = {
    team: Trophy,
    individual: Medal,
    special: Award,
    performance: Star,
    fair_play: Crown,
  };

  const categoryColors = {
    team: 'bg-blue-100 text-blue-800',
    individual: 'bg-green-100 text-green-800',
    special: 'bg-purple-100 text-purple-800',
    performance: 'bg-yellow-100 text-yellow-800',
    fair_play: 'bg-indigo-100 text-indigo-800',
  };

  const prizeTypeIcons = {
    cash: DollarSign,
    trophy: Trophy,
    medal: Medal,
    certificate: Award,
    merchandise: Gift,
    experience: Star,
    mixed: Crown,
  };

  const handleSearch = () => {
    router.get(route('admin.prize-awards.index'), {
      search: searchTerm,
      category: selectedCategory,
      tournament: selectedTournament,
      status: selectedStatus,
    }, {
      preserveState: true,
      replace: true,
    });
  };

  const handleClearFilters = () => {
    setSearchTerm('');
    setSelectedCategory('');
    setSelectedTournament('');
    setSelectedStatus('');
    router.get(route('admin.prize-awards.index'), {}, {
      preserveState: true,
      replace: true,
    });
  };

  const handleSelectAll = () => {
    if (selectedPrizeAwards.length === prizeAwards.data.length) {
      setSelectedPrizeAwards([]);
    } else {
      setSelectedPrizeAwards(prizeAwards.data.map(prizeAward => prizeAward.id));
    }
  };

  const handleSelectPrizeAward = (prizeAwardId: number) => {
    setSelectedPrizeAwards(prev => 
      prev.includes(prizeAwardId) 
        ? prev.filter(id => id !== prizeAwardId)
        : [...prev, prizeAwardId]
    );
  };

  const handleDelete = (prizeAward: PrizeAward) => {
    setPrizeAwardToDelete(prizeAward);
    setDeleteDialogOpen(true);
  };

  const confirmDelete = () => {
    if (prizeAwardToDelete) {
      router.delete(route('admin.prize-awards.destroy', prizeAwardToDelete.id), {
        onSuccess: () => {
          setDeleteDialogOpen(false);
          setPrizeAwardToDelete(null);
        },
      });
    }
  };

  const handleToggleStatus = (prizeAward: PrizeAward) => {
    router.patch(route('admin.prize-awards.toggle-status', prizeAward.id), {}, {
      preserveState: true,
    });
  };

  const handleBulkAction = () => {
    if (selectedPrizeAwards.length === 0 || !bulkAction) return;

    router.post(route('admin.prize-awards.bulk-action'), {
      action: bulkAction,
      prize_award_ids: selectedPrizeAwards,
    }, {
      onSuccess: () => {
        setSelectedPrizeAwards([]);
        setBulkActionDialogOpen(false);
        setBulkAction('');
      },
    });
  };

  const getCategoryIcon = (category: string) => {
    const IconComponent = categoryIcons[category as keyof typeof categoryIcons] || Trophy;
    return <IconComponent className="w-4 h-4" />;
  };

  const getPrizeTypeIcon = (prizeType: string) => {
    const IconComponent = prizeTypeIcons[prizeType as keyof typeof prizeTypeIcons] || Gift;
    return <IconComponent className="w-4 h-4" />;
  };

  const formatPrizeValue = (value: number, prizeType: string) => {
    if (!value) return '';
    if (prizeType === 'cash') {
      return `$${value.toLocaleString()}`;
    }
    return `Value: $${value.toLocaleString()}`;
  };

  return (
    <AppLayout>
      <Head title="Prize Awards Management" />
      
      <div className="space-y-6">
        {/* Header */}
        <div className="flex justify-between items-center">
          <div>
            <h1 className="text-3xl font-bold text-gray-900">Prize Awards Management</h1>
            <p className="text-gray-600 mt-1">Manage tournament prize awards and recognition</p>
          </div>
          <Link href={route('admin.prize-awards.create')}>
            <Button className="flex items-center space-x-2">
              <Plus className="w-4 h-4" />
              <span>Add Prize Award</span>
            </Button>
          </Link>
        </div>

        {/* Filters */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center space-x-2">
              <Filter className="w-5 h-5" />
              <span>Filters</span>
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="grid grid-cols-1 md:grid-cols-5 gap-4">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Search</label>
                <div className="relative">
                  <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-4 h-4" />
                  <Input
                    placeholder="Search awards..."
                    value={searchTerm}
                    onChange={(e) => setSearchTerm(e.target.value)}
                    className="pl-10"
                    onKeyPress={(e) => e.key === 'Enter' && handleSearch()}
                  />
                </div>
              </div>
              
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Category</label>
                <Select value={selectedCategory} onValueChange={setSelectedCategory}>
                  <SelectTrigger>
                    <SelectValue placeholder="All categories" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">All categories</SelectItem>
                    {Object.entries(categories).map(([key, value]) => (
                      <SelectItem key={key} value={key}>
                        {value}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Tournament</label>
                <Select value={selectedTournament} onValueChange={setSelectedTournament}>
                  <SelectTrigger>
                    <SelectValue placeholder="All tournaments" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">All tournaments</SelectItem>
                    {tournaments.map(tournament => (
                      <SelectItem key={tournament.id} value={tournament.id.toString()}>
                        {tournament.name}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Status</label>
                <Select value={selectedStatus} onValueChange={setSelectedStatus}>
                  <SelectTrigger>
                    <SelectValue placeholder="All statuses" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">All statuses</SelectItem>
                    <SelectItem value="active">Active</SelectItem>
                    <SelectItem value="inactive">Inactive</SelectItem>
                  </SelectContent>
                </Select>
              </div>
              
              <div className="flex items-end space-x-2">
                <Button onClick={handleSearch} className="flex-1">
                  Apply Filters
                </Button>
                <Button variant="outline" onClick={handleClearFilters}>
                  Clear
                </Button>
              </div>
            </div>
          </CardContent>
        </Card>

        {/* Bulk Actions */}
        {selectedPrizeAwards.length > 0 && (
          <Card>
            <CardContent className="pt-6">
              <div className="flex items-center justify-between">
                <div className="flex items-center space-x-4">
                  <span className="text-sm text-gray-600">
                    {selectedPrizeAwards.length} Prize Award{selectedPrizeAwards.length !== 1 ? 's' : ''} selected
                  </span>
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() => setBulkActionDialogOpen(true)}
                  >
                    Bulk Actions
                  </Button>
                </div>
                <Button
                  variant="ghost"
                  size="sm"
                  onClick={() => setSelectedPrizeAwards([])}
                >
                  Clear Selection
                </Button>
              </div>
            </CardContent>
          </Card>
        )}

        {/* Prize Awards List */}
        <Card>
          <CardHeader>
            <div className="flex items-center justify-between">
              <CardTitle>Prize Awards ({prizeAwards?.meta?.total})</CardTitle>
              <div className="flex items-center space-x-2">
                <Checkbox
                  checked={selectedPrizeAwards.length === prizeAwards?.data?.length && prizeAwards?.data?.length > 0}
                  onCheckedChange={handleSelectAll}
                />
                <span className="text-sm text-gray-600">Select All</span>
              </div>
            </div>
          </CardHeader>
          <CardContent>
            <div className="space-y-4">
              {prizeAwards.data.map((prizeAward) => (
                <div key={prizeAward.id} className="border rounded-lg p-4 hover:bg-gray-50 transition-colors">
                  <div className="flex items-start space-x-4">
                    <Checkbox
                      checked={selectedPrizeAwards.includes(prizeAward.id)}
                      onCheckedChange={() => handleSelectPrizeAward(prizeAward.id)}
                    />
                    
                    <div className="flex-1 min-w-0">
                      <div className="flex items-center space-x-3 mb-2">
                        <Badge className={`${categoryColors[prizeAward.category as keyof typeof categoryColors]} flex items-center space-x-1`}>
                          {getCategoryIcon(prizeAward.category)}
                          <span>{categories[prizeAward.category]}</span>
                        </Badge>
                        <Badge variant={prizeAward.is_active ? "default" : "secondary"}>
                          {prizeAward.is_active ? "Active" : "Inactive"}
                        </Badge>
                        <Badge variant="outline" className="flex items-center space-x-1">
                          {getPrizeTypeIcon(prizeAward.prize_type)}
                          <span>{prizeTypes[prizeAward.prize_type]}</span>
                        </Badge>
                        <span className="text-sm text-gray-500">Order: {prizeAward.sort_order}</span>
                      </div>
                      
                      <div className="flex items-center space-x-4 mb-2">
                        <h3 className="text-lg font-semibold text-gray-900">{prizeAward.title}</h3>
                        <Badge variant="outline">{positions[prizeAward.position]}</Badge>
                        {prizeAward.prize_value && (
                          <span className="text-sm font-medium text-green-600">
                            {formatPrizeValue(prizeAward.prize_value, prizeAward.prize_type)}
                          </span>
                        )}
                      </div>

                      <p className="text-gray-600 text-sm line-clamp-2 mb-3">{prizeAward.description}</p>
                      
                      <div className="flex items-center space-x-4 text-xs text-gray-500">
                        <span>Tournament: {prizeAward.tournament.name}</span>
                        <span>Created: {new Date(prizeAward.created_at).toLocaleDateString()}</span>
                        <span>Updated: {new Date(prizeAward.updated_at).toLocaleDateString()}</span>
                      </div>
                    </div>
                    
                    <div className="flex items-center space-x-2">
                      <Button
                        variant="ghost"
                        size="sm"
                        onClick={() => handleToggleStatus(prizeAward)}
                        className="flex items-center space-x-1"
                      >
                        {prizeAward.is_active ? (
                          <ToggleRight className="w-4 h-4 text-green-600" />
                        ) : (
                          <ToggleLeft className="w-4 h-4 text-gray-400" />
                        )}
                      </Button>
                      
                      <DropdownMenu>
                        <DropdownMenuTrigger asChild>
                          <Button variant="ghost" size="sm">
                            <MoreHorizontal className="w-4 h-4" />
                          </Button>
                        </DropdownMenuTrigger>
                        <DropdownMenuContent align="end">
                          <DropdownMenuItem asChild>
                            <Link href={route('admin.prize-awards.show', prizeAward.id)} className="flex items-center space-x-2">
                              <Eye className="w-4 h-4" />
                              <span>View</span>
                            </Link>
                          </DropdownMenuItem>
                          <DropdownMenuItem asChild>
                            <Link href={route('admin.prize-awards.edit', prizeAward.id)} className="flex items-center space-x-2">
                              <Edit className="w-4 h-4" />
                              <span>Edit</span>
                            </Link>
                          </DropdownMenuItem>
                          <DropdownMenuItem
                            onClick={() => handleDelete(prizeAward)}
                            className="flex items-center space-x-2 text-red-600"
                          >
                            <Trash2 className="w-4 h-4" />
                            <span>Delete</span>
                          </DropdownMenuItem>
                        </DropdownMenuContent>
                      </DropdownMenu>
                    </div>
                  </div>
                </div>
              ))}
              
              {prizeAwards.data.length === 0 && (
                <div className="text-center py-12">
                  <Trophy className="w-12 h-12 text-gray-400 mx-auto mb-4" />
                  <h3 className="text-lg font-semibold text-gray-900 mb-2">No Prize Awards found</h3>
                  <p className="text-gray-600 mb-4">Get started by creating your first prize award.</p>
                  <Link href={route('admin.prize-awards.create')}>
                    <Button>Create Prize Award</Button>
                  </Link>
                </div>
              )}
            </div>
          </CardContent>
        </Card>

        {/* Pagination */}
        {prizeAwards.links && prizeAwards.links.length > 3 && (
          <div className="flex justify-center">
            <nav className="flex space-x-2">
              {prizeAwards.links.map((link, index) => (
                <Link
                  key={index}
                  href={link.url || '#'}
                  className={`px-3 py-2 rounded-md text-sm ${
                    link.active
                      ? 'bg-blue-600 text-white'
                      : 'bg-white text-gray-700 hover:bg-gray-50 border'
                  } ${!link.url ? 'opacity-50 cursor-not-allowed' : ''}`}
                  dangerouslySetInnerHTML={{ __html: link.label }}
                />
              ))}
            </nav>
          </div>
        )}
      </div>

      {/* Delete Confirmation Dialog */}
      <Dialog open={deleteDialogOpen} onOpenChange={setDeleteDialogOpen}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Delete Prize Award</DialogTitle>
            <DialogDescription>
              Are you sure you want to delete this prize award? This action cannot be undone.
            </DialogDescription>
          </DialogHeader>
          <DialogFooter>
            <DialogClose>Cancel</DialogClose>
            <DialogClose onClick={confirmDelete} className="bg-red-600 hover:bg-red-700">
              Delete
            </DialogClose>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Bulk Action Dialog */}
      <Dialog open={bulkActionDialogOpen} onOpenChange={setBulkActionDialogOpen}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Bulk Action</DialogTitle>
            <DialogDescription>
              Choose an action to perform on {selectedPrizeAwards.length} selected Prize Award{selectedPrizeAwards.length !== 1 ? 's' : ''}.
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4">
            <Select value={bulkAction} onValueChange={setBulkAction}>
              <SelectTrigger>
                <SelectValue placeholder="Select action" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="activate">Activate</SelectItem>
                <SelectItem value="deactivate">Deactivate</SelectItem>
                <SelectItem value="delete">Delete</SelectItem>
              </SelectContent>
            </Select>
          </div>
          <DialogFooter>
            <DialogClose>Cancel</DialogClose>
            <DialogClose 
              onClick={handleBulkAction}
              disabled={!bulkAction}
              className={bulkAction === 'delete' ? 'bg-red-600 hover:bg-red-700' : ''}
            >
              {bulkAction === 'activate' ? 'Activate' : 
               bulkAction === 'deactivate' ? 'Deactivate' : 
               bulkAction === 'delete' ? 'Delete' : 'Confirm'}
            </DialogClose>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </AppLayout>
  );
};

export default PrizeAwardsIndex;
