import { useState } from "react";
import { Head, Link, useForm } from "@inertiajs/react";
import AppLayout from "@/layouts/app-layout";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Switch } from "@/components/ui/switch";
import { ArrowLeft, Trophy, Medal, Award, Star, Crown, Gift, DollarSign, Users, Target, Calendar } from "lucide-react";

interface PrizeAward {
  id: number;
  tournament_id: number;
  category: string;
  position: string;
  title: string;
  description: string;
  prize_value: number;
  prize_type: string;
  additional_benefits: string;
  is_active: boolean;
  sort_order: number;
  created_at: string;
  updated_at: string;
  tournament: {
    id: number;
    name: string;
  };
}

interface EditPrizeAwardPageProps {
  prizeAward: PrizeAward;
  categories: Record<string, string>;
  prizeTypes: Record<string, string>;
  positions: Record<string, string>;
  tournaments: Array<{ id: number; name: string }>;
}

const EditPrizeAward = ({ prizeAward, categories, prizeTypes, positions, tournaments }: EditPrizeAwardPageProps) => {
  const { data, setData, put, processing, errors } = useForm({
    tournament_id: prizeAward.tournament_id.toString(),
    category: prizeAward.category,
    position: prizeAward.position,
    title: prizeAward.title,
    description: prizeAward.description || '',
    prize_value: prizeAward.prize_value?.toString() || '',
    prize_type: prizeAward.prize_type,
    additional_benefits: prizeAward.additional_benefits || '',
    is_active: prizeAward.is_active,
    sort_order: prizeAward.sort_order,
  });

  const [previewMode, setPreviewMode] = useState(false);

  const categoryIcons = {
    team: Trophy,
    individual: Medal,
    special: Award,
    performance: Star,
    fair_play: Crown,
  };

  const prizeTypeIcons = {
    cash: DollarSign,
    trophy: Trophy,
    medal: Medal,
    certificate: Award,
    merchandise: Gift,
    experience: Star,
    mixed: Crown,
  };

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    put(route('admin.prize-awards.update', prizeAward.id));
  };

  const getCategoryIcon = (category: string) => {
    const IconComponent = categoryIcons[category as keyof typeof categoryIcons] || Trophy;
    return <IconComponent className="w-5 h-5" />;
  };

  const getPrizeTypeIcon = (prizeType: string) => {
    const IconComponent = prizeTypeIcons[prizeType as keyof typeof prizeTypeIcons] || Gift;
    return <IconComponent className="w-5 h-5" />;
  };

  const formatPrizeValue = (value: string, prizeType: string) => {
    if (!value) return '';
    const numValue = parseFloat(value);
    if (prizeType === 'cash') {
      return `$${numValue.toLocaleString()}`;
    }
    return `Value: $${numValue.toLocaleString()}`;
  };

  return (
    <AppLayout>
      <Head title={`Edit Prize Award - ${prizeAward.title}`} />
      
      <div className="space-y-6">
        {/* Header */}
        <div className="flex items-center space-x-4">
          <Link href={route('admin.prize-awards.index')}>
            <Button variant="outline" size="sm">
              <ArrowLeft className="w-4 h-4 mr-2" />
              Back to Prize Awards
            </Button>
          </Link>
          <div>
            <h1 className="text-3xl font-bold text-gray-900">Edit Prize Award</h1>
            <p className="text-gray-600 mt-1">Update prize award details</p>
          </div>
        </div>

        <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
          {/* Form */}
          <Card>
            <CardHeader>
              <CardTitle>Prize Award Details</CardTitle>
            </CardHeader>
            <CardContent>
              <form onSubmit={handleSubmit} className="space-y-6">
                {/* Tournament Selection */}
                <div className="space-y-2">
                  <Label htmlFor="tournament_id">Tournament *</Label>
                  <Select value={data.tournament_id} onValueChange={(value) => setData('tournament_id', value)}>
                    <SelectTrigger>
                      <SelectValue placeholder="Select tournament" />
                    </SelectTrigger>
                    <SelectContent>
                      {tournaments.map(tournament => (
                        <SelectItem key={tournament.id} value={tournament.id.toString()}>
                          {tournament.name}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                  {errors.tournament_id && (
                    <p className="text-sm text-red-600">{errors.tournament_id}</p>
                  )}
                </div>

                {/* Category */}
                <div className="space-y-2">
                  <Label htmlFor="category">Category *</Label>
                  <Select value={data.category} onValueChange={(value) => setData('category', value)}>
                    <SelectTrigger>
                      <SelectValue placeholder="Select category" />
                    </SelectTrigger>
                    <SelectContent>
                      {Object.entries(categories).map(([key, value]) => (
                        <SelectItem key={key} value={key}>
                          <div className="flex items-center space-x-2">
                            {getCategoryIcon(key)}
                            <span>{value}</span>
                          </div>
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                  {errors.category && (
                    <p className="text-sm text-red-600">{errors.category}</p>
                  )}
                </div>

                {/* Position */}
                <div className="space-y-2">
                  <Label htmlFor="position">Position *</Label>
                  <Select value={data.position} onValueChange={(value) => setData('position', value)}>
                    <SelectTrigger>
                      <SelectValue placeholder="Select position" />
                    </SelectTrigger>
                    <SelectContent>
                      {Object.entries(positions).map(([key, value]) => (
                        <SelectItem key={key} value={key}>
                          {value}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                  {errors.position && (
                    <p className="text-sm text-red-600">{errors.position}</p>
                  )}
                </div>

                {/* Title */}
                <div className="space-y-2">
                  <Label htmlFor="title">Title *</Label>
                  <Input
                    id="title"
                    value={data.title}
                    onChange={(e) => setData('title', e.target.value)}
                    placeholder="Enter prize award title"
                  />
                  {errors.title && (
                    <p className="text-sm text-red-600">{errors.title}</p>
                  )}
                </div>

                {/* Description */}
                <div className="space-y-2">
                  <Label htmlFor="description">Description</Label>
                  <Textarea
                    id="description"
                    value={data.description}
                    onChange={(e) => setData('description', e.target.value)}
                    placeholder="Enter detailed description"
                    rows={4}
                  />
                  {errors.description && (
                    <p className="text-sm text-red-600">{errors.description}</p>
                  )}
                </div>

                {/* Prize Type */}
                <div className="space-y-2">
                  <Label htmlFor="prize_type">Prize Type *</Label>
                  <Select value={data.prize_type} onValueChange={(value) => setData('prize_type', value)}>
                    <SelectTrigger>
                      <SelectValue placeholder="Select prize type" />
                    </SelectTrigger>
                    <SelectContent>
                      {Object.entries(prizeTypes).map(([key, value]) => (
                        <SelectItem key={key} value={key}>
                          <div className="flex items-center space-x-2">
                            {getPrizeTypeIcon(key)}
                            <span>{value}</span>
                          </div>
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                  {errors.prize_type && (
                    <p className="text-sm text-red-600">{errors.prize_type}</p>
                  )}
                </div>

                {/* Prize Value */}
                <div className="space-y-2">
                  <Label htmlFor="prize_value">Prize Value ($)</Label>
                  <Input
                    id="prize_value"
                    type="number"
                    step="0.01"
                    min="0"
                    value={data.prize_value}
                    onChange={(e) => setData('prize_value', e.target.value)}
                    placeholder="Enter monetary value"
                  />
                  {errors.prize_value && (
                    <p className="text-sm text-red-600">{errors.prize_value}</p>
                  )}
                </div>

                {/* Additional Benefits */}
                <div className="space-y-2">
                  <Label htmlFor="additional_benefits">Additional Benefits</Label>
                  <Textarea
                    id="additional_benefits"
                    value={data.additional_benefits}
                    onChange={(e) => setData('additional_benefits', e.target.value)}
                    placeholder="Enter additional benefits or perks"
                    rows={3}
                  />
                  {errors.additional_benefits && (
                    <p className="text-sm text-red-600">{errors.additional_benefits}</p>
                  )}
                </div>

                {/* Sort Order */}
                <div className="space-y-2">
                  <Label htmlFor="sort_order">Sort Order</Label>
                  <Input
                    id="sort_order"
                    type="number"
                    min="0"
                    value={data.sort_order}
                    onChange={(e) => setData('sort_order', parseInt(e.target.value) || 0)}
                    placeholder="Display order"
                  />
                  {errors.sort_order && (
                    <p className="text-sm text-red-600">{errors.sort_order}</p>
                  )}
                </div>

                {/* Active Status */}
                <div className="flex items-center space-x-2">
                  <Switch
                    id="is_active"
                    checked={data.is_active}
                    onCheckedChange={(checked) => setData('is_active', checked)}
                  />
                  <Label htmlFor="is_active">Active</Label>
                </div>

                {/* Submit Button */}
                <div className="flex space-x-4">
                  <Button type="submit" disabled={processing}>
                    {processing ? 'Updating...' : 'Update Prize Award'}
                  </Button>
                  <Button
                    type="button"
                    variant="outline"
                    onClick={() => setPreviewMode(!previewMode)}
                  >
                    {previewMode ? 'Hide Preview' : 'Show Preview'}
                  </Button>
                </div>
              </form>
            </CardContent>
          </Card>

          {/* Preview */}
          {previewMode && (
            <Card>
              <CardHeader>
                <CardTitle>Preview</CardTitle>
              </CardHeader>
              <CardContent>
                <div className="space-y-4">
                  <div className="flex items-center space-x-3">
                    {data.category && getCategoryIcon(data.category)}
                    <span className="font-semibold text-lg">{data.title || 'Prize Award Title'}</span>
                    {data.position && (
                      <span className="px-2 py-1 bg-blue-100 text-blue-800 text-sm rounded-full">
                        {positions[data.position]}
                      </span>
                    )}
                  </div>

                  {data.description && (
                    <p className="text-gray-600">{data.description}</p>
                  )}

                  <div className="flex items-center space-x-4">
                    {data.prize_type && (
                      <div className="flex items-center space-x-2">
                        {getPrizeTypeIcon(data.prize_type)}
                        <span className="text-sm">{prizeTypes[data.prize_type]}</span>
                      </div>
                    )}
                    {data.prize_value && (
                      <span className="text-sm font-medium text-green-600">
                        {formatPrizeValue(data.prize_value, data.prize_type)}
                      </span>
                    )}
                  </div>

                  {data.additional_benefits && (
                    <div>
                      <h4 className="font-medium text-gray-900 mb-2">Additional Benefits:</h4>
                      <p className="text-sm text-gray-600">{data.additional_benefits}</p>
                    </div>
                  )}

                  <div className="flex items-center space-x-2">
                    <span className={`px-2 py-1 text-xs rounded-full ${
                      data.is_active ? 'bg-green-100 text-green-800' : 'bg-gray-100 text-gray-800'
                    }`}>
                      {data.is_active ? 'Active' : 'Inactive'}
                    </span>
                    <span className="text-xs text-gray-500">Order: {data.sort_order}</span>
                  </div>
                </div>
              </CardContent>
            </Card>
          )}
        </div>
      </div>
    </AppLayout>
  );
};

export default EditPrizeAward;
