import AppLayout from '@/layouts/app-layout';
import { Head, router } from '@inertiajs/react';
import { Card } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { 
  Trophy, 
  Target, 
  TrendingUp, 
  MapPin, 
  Calendar, 
  Phone, 
  Globe, 
  Shield,
  Star,
  Award,
  Users,
  User,
  Activity,
  CreditCard,
  QrCode,
  FileText,
  Eye,
  Download,
  Printer,
  EyeOff
} from 'lucide-react';
import PlayerDigitalId from './digital-id';
import Swal from 'sweetalert2';
import CertificateManager from '@/components/CertificateManager';
import CertificatePreviewButton, { CertificateDownloadButton } from '@/components/CertificatePreviewButton';
import { useCertificatePreview } from '@/hooks/useCertificatePreview';
import CertificatePreviewDialog from '@/components/CertificatePreviewDialog';
import { showSuccessToast, showWarningToast } from '@/frontend/components/ui/toast-utils';

export default function AdminPlayerShow({   player, stats, performanceMetrics, socialLinks, achievements }: { player: any, stats: any, performanceMetrics: any, socialLinks: any, achievements: any }) {
  // Certificate preview hook
  const { isOpen, certificateData, playerId, openPreview, closePreview } = useCertificatePreview();

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'active': return 'bg-green-100 text-green-800';
      case 'eliminated': return 'bg-red-100 text-red-800';
      case 'withdrawn': return 'bg-yellow-100 text-yellow-800';
      case 'suspended': return 'bg-orange-100 text-orange-800';
      case 'banned': return 'bg-red-100 text-red-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  const getVerificationColor = (status: string) => {
    switch (status) {
      case 'verified': return 'bg-green-100 text-green-800';
      case 'pending': return 'bg-yellow-100 text-yellow-800';
      case 'rejected': return 'bg-red-100 text-red-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  const getPlayerTypeColor = (type: string) => {
    switch (type) {
      case 'legend': return 'bg-purple-100 text-purple-800';
      case 'professional': return 'bg-blue-100 text-blue-800';
      case 'semi_pro': return 'bg-indigo-100 text-indigo-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  const handleGenerateDigitalId = async () => {
    try {
      const response = await fetch(route('admin.players.digital-id.generate', player.id), {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || '',
        },
      });

      const data = await response.json();
      
      if (data.success) {
        // Reload the page
        showSuccessToast('Digital ID generated successfully!');
        window.location.reload();
      } else {
        // SweetAlert2
        Swal.fire({
          title: 'Failed to generate Digital ID',
          text: 'Please try again.',
          icon: 'error',
        });
      }
    } catch (error) {
      // SweetAlert2
      Swal.fire({
        title: 'Error generating Digital ID',
        text: 'Please try again.',
        icon: 'error',
      });
    }
  };

  const handleViewDigitalId = () => {
    if (player.digital_id) {
      showSuccessToast('Digital ID generated successfully!');  
      // router.get(route('admin.players.digital-id', player.id));
    } else {
        // SweetAlert2
        showWarningToast('Digital ID not generated yet');
    }
  };

  const handleGenerateCertificate = () => {
    // Generate certificate data
    const certificateId = `CERT-${String(player.id).padStart(6, '0')}-${Math.random().toString(36).substr(2, 8).toUpperCase()}`;
    const issuedDate = new Date().toLocaleDateString('en-US', { 
      year: 'numeric', 
      month: 'long', 
      day: 'numeric' 
    });

    const certificateData = {
      player_name: player.display_name || player.user?.name || 'Player Name',
      team_name: player.team_name || '',
      position: player.position || 'Participant',
      jersey_number: player.jersey_number || undefined,
      tournament_matches_played: player.tournament_matches_played || 0,
      tournament_win_rate: player.tournament_win_rate || 0,
      global_rating: player.global_rating || 0,
      digital_id: certificateId,
      issued_date: issuedDate,
      tournament_name: player.tournament?.name || 'The Milimus Tournament Cup',
      verification_status: player.verification_status || 'verified',
    };

    openPreview(player.id, certificateData);
  };

  const handleDirectDownload = () => {
    // Use direct link approach for PDF download
    router.get(route('admin.players.certificate.download', player.id));

  };

  const handleDirectPrint = () => {  
      router.get(route('admin.players.certificate.print', player.id));
  };

  const handleResendDigitalId = async () => {
    if (!player.digital_id) {
      //SweetAlert2
      Swal.fire({
        title: 'Digital ID not generated yet',
        text: 'Please generate it first.',
        icon: 'warning',
      });
      return;
    } else {
      //SweetAlert2
      Swal.fire({
        title: 'Digital ID generated successfully',
        text: 'Digital ID has been resent to the player via email with PDF attachment!',
      });
      return;
    }

    //SweetAlert2

    //Are you sure you want to resend the Digital ID?
    Swal.fire({
      title: 'Are you sure?',
      text: 'This action cannot be undone.',
      icon: 'warning',
      showCancelButton: true,
    }).then(async (result) => {
      if (result.isConfirmed) {
        try {
          const response = await fetch(route('admin.players.digital-id.resend', player.id), {
            method: 'POST',
            headers: {
              'Content-Type': 'application/json',
              'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || '',
            },
          });

          if (response.status === 200) {
            //SweetAlert2
            Swal.fire({
              title: 'Digital ID generated successfully',
              text: 'Digital ID has been resent to the player via email with PDF attachment!',
            });
          } else {
            //SweetAlert2
            Swal.fire({
              title: 'Failed to resend Digital ID',
              text: 'Please try again.',
              icon: 'error',
            });
          }
        } catch (error) {
          //SweetAlert2
          Swal.fire({
            title: 'Error resending Digital ID',
            text: 'Please try again.',
            icon: 'error',
          });
        }
      }
    });
  };

  return (
    <AppLayout breadcrumbs={[
      { title: 'Players', href: '/admin/players' },
      { title: player.display_name, href: `/admin/players/${player.id}` }
    ]}>
      <Head title={`Player: ${player.display_name}`} />
      
      <div className="flex flex-col gap-6 p-4">
        {/* Header */}
        <div className="flex justify-between items-start">
          <div className="flex items-center gap-4">
            <div className="w-20 h-20 rounded-full overflow-hidden border-4 border-primary/20">
              {player.avatar ? (
                <img 
                  src={player.avatar} 
                  alt={player.display_name}
                  className="w-full h-full object-cover"
                />
              ) : (
                <div className="w-full h-full bg-gradient-to-br from-green-400 to-green-600 flex items-center justify-center text-white text-2xl font-bold">
                  {player.display_name.charAt(0).toUpperCase()}
                </div>
              )}
            </div>
            <div>
              <h1 className="text-3xl font-bold">{player.display_name}</h1>
              <div className="flex items-center gap-2 mt-2">
                <Badge className={getPlayerTypeColor(player.player_type)}>
                  {player.player_type?.replace('_', ' ').toUpperCase()}
                </Badge>
                <Badge className={getStatusColor(player.status)}>
                  {player.status}
                </Badge>
                <Badge className={getVerificationColor(player.verification_status)}>
                  {player.verification_status}
                </Badge>
                {player.digital_id && (
                  <Badge className="bg-primary/10 text-primary border-primary/20">
                    <QrCode className="w-3 h-3 mr-1" />
                    Digital ID
                  </Badge>
                )}
              </div>
            </div>
          </div>
          <div className="flex gap-2">
            
            <Button variant="outline" onClick={() => router.get(route('admin.players.edit', { player: player.id }))}>
              Edit Player
            </Button>
            <Button variant="outline" onClick={() => router.get(route('admin.players.index'))}>
              Back to Players
            </Button>
          </div>
        </div>

        {/* Main Content */}
        <Tabs defaultValue="overview" className="space-y-6">
          <TabsList>
            <TabsTrigger value="overview">Overview</TabsTrigger>
            <TabsTrigger value="statistics">Statistics</TabsTrigger>
            <TabsTrigger value="achievements">Achievements</TabsTrigger>
            <TabsTrigger value="social">Social & Contact</TabsTrigger>
            <TabsTrigger value="performance">Performance</TabsTrigger>
            <TabsTrigger value="digital-id">Digital ID</TabsTrigger>
            <TabsTrigger value="certificate">Certificate</TabsTrigger>
          </TabsList>

          {/* Overview Tab */}
          <TabsContent value="overview" className="space-y-6">
            <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
              {/* Basic Information */}
              <Card className="p-6">
                <h3 className="text-lg font-semibold mb-4 flex items-center gap-2">
                  <Users className="w-5 h-5" />
                  Basic Information
                </h3>
                <div className="space-y-3">
                  <div>
                    <label className="text-sm font-medium text-gray-500">
                      Display Name <span className="text-red-500">*</span>
                    </label>
                    <p className={`text-lg ${!player.display_name ? 'text-red-600 font-semibold bg-red-50 p-2 rounded' : ''}`}>
                      {player.display_name || '⚠️ Not set - Required for approval'}
                    </p>
                  </div>
                  {player.bio && (
                    <div>
                      <label className="text-sm font-medium text-gray-500">Bio</label>
                      <p className="text-sm">{player.bio}</p>
                    </div>
                  )}
                  <div className="flex gap-4">
                    <div>
                      <label className="text-sm font-medium text-gray-500">Country</label>
                      <p className="text-sm">{player.country || 'Not specified'}</p>
                    </div>
                    <div>
                      <label className="text-sm font-medium text-gray-500">City</label>
                      <p className="text-sm">{player.city || 'Not specified'}</p>
                    </div>
                  </div>
                  <div>
                    <label className="text-sm font-medium text-gray-500">
                      Date of Birth <span className="text-red-500">*</span>
                    </label>
                    <p className={`text-sm ${!player.date_of_birth ? 'text-red-600 font-semibold bg-red-50 p-2 rounded' : ''}`}>
                      {player.date_of_birth ? new Date(player.date_of_birth).toLocaleDateString() : '⚠️ Not provided - Required for approval'}
                    </p>
                  </div>
                  <div>
                    <label className="text-sm font-medium text-gray-500">
                      Gender <span className="text-red-500">*</span>
                    </label>
                    <p className={`text-sm capitalize ${!player.gender ? 'text-red-600 font-semibold bg-red-50 p-2 rounded' : ''}`}>
                      {player.gender ? player.gender.replace('_', ' ') : '⚠️ Not specified - Required for approval'}
                    </p>
                  </div>
                </div>
              </Card>

              {/* Player Classification */}
              <Card className="p-6">
                <h3 className="text-lg font-semibold mb-4 flex items-center gap-2">
                  <Star className="w-5 h-5" />
                  Player Classification
                </h3>
                <div className="space-y-3">
                  <div>
                    <label className="text-sm font-medium text-gray-500">Player Type</label>
                    <Badge className={getPlayerTypeColor(player.player_type)}>
                      {player.player_type?.replace('_', ' ').toUpperCase()}
                    </Badge>
                  </div>
                  <div>
                    <label className="text-sm font-medium text-gray-500">Experience Level</label>
                    <p className="text-sm capitalize">{player.experience_level}</p>
                  </div>
                  {player.primary_game && (
                    <div>
                      <label className="text-sm font-medium text-gray-500">Primary Game</label>
                      <p className="text-sm">{player.primary_game}</p>
                    </div>
                  )}
                  <div>
                    <label className="text-sm font-medium text-gray-500">
                      Football Position <span className="text-red-500">*</span>
                    </label>
                    <p className={`text-sm ${!player.football_position && !player.position ? 'text-red-600 font-semibold bg-red-50 p-2 rounded' : ''}`}>
                      {player.football_position || player.position || '⚠️ Not assigned - Required for approval'}
                    </p>
                  </div>
                  <div>
                    <label className="text-sm font-medium text-gray-500">
                      ID Number <span className="text-red-500">*</span>
                    </label>
                    <p className={`text-sm font-mono ${!player.id_number ? 'text-red-600 font-semibold bg-red-50 p-2 rounded' : 'bg-gray-100 p-2'} rounded`}>
                      {player.id_number || '⚠️ Not provided - Required for approval'}
                    </p>
                  </div>
                  <div>
                    <label className="text-sm font-medium text-gray-500">Registration Date</label>
                    <p className="text-sm">{new Date(player.registered_at).toLocaleDateString()}</p>
                  </div>
                  <div>
                    <label className="text-sm font-medium text-gray-500">Last Active</label>
                    <p className="text-sm">{new Date(player.last_active_at).toLocaleDateString()}</p>
                  </div>
                </div>
              </Card>

              {/* Quick Stats */}
              <Card className="p-6">
                <h3 className="text-lg font-semibold mb-4 flex items-center gap-2">
                  <TrendingUp className="w-5 h-5" />
                  Quick Stats
                </h3>
                <div className="space-y-3">
                  <div className="flex justify-between">
                    <span className="text-sm font-medium">Global Rating</span>
                    <span className="text-lg font-bold">{player.global_rating || 'N/A'}</span>
                  </div>
                  <div className="flex justify-between">
                    <span className="text-sm font-medium">Tournament Rating</span>
                    <span className="text-lg font-bold">{player.tournament_rating || 'N/A'}</span>
                  </div>
                  <div className="flex justify-between">
                    <span className="text-sm font-medium">Win Rate</span>
                    <span className="text-lg font-bold">{player.tournament_win_rate || 0}%</span>
                  </div>
                  <div className="flex justify-between">
                    <span className="text-sm font-medium">Matches Played</span>
                    <span className="text-lg font-bold">{player.tournament_matches_played || 0}</span>
                  </div>
                  <div className="flex justify-between">
                    <span className="text-sm font-medium">Total Earnings</span>
                    <span className="text-lg font-bold">${player.total_earnings || 0}</span>
                  </div>
                </div>
              </Card>
            </div>
          </TabsContent>

          {/* Statistics Tab */}
          <TabsContent value="statistics" className="space-y-6">
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
              {/* Tournament Statistics */}
              <Card className="p-6">
                <h3 className="text-lg font-semibold mb-4 flex items-center gap-2">
                  <Trophy className="w-5 h-5" />
                  Tournament Statistics
                </h3>
                <div className="space-y-4">
                  <div className="grid grid-cols-2 gap-4">
                    <div className="text-center p-4 bg-green-50 rounded-lg">
                      <div className="text-2xl font-bold text-green-600">{stats.matches_won}</div>
                      <div className="text-sm text-gray-600">Wins</div>
                    </div>
                    <div className="text-center p-4 bg-red-50 rounded-lg">
                      <div className="text-2xl font-bold text-red-600">{stats.matches_played - stats.matches_won}</div>
                      <div className="text-sm text-gray-600">Losses</div>
                    </div>
                  </div>
                  <div className="space-y-2">
                    <div className="flex justify-between">
                      <span>Current Streak</span>
                      <span className="font-medium">{stats.current_streak || 0}</span>
                    </div>
                    <div className="flex justify-between">
                      <span>Longest Streak</span>
                      <span className="font-medium">{stats.longest_streak || 0}</span>
                    </div>
                    <div className="flex justify-between">
                      <span>Tournaments Won</span>
                      <span className="font-medium">{player.tournaments_won || 0}</span>
                    </div>
                    <div className="flex justify-between">
                      <span>Prizes Won</span>
                      <span className="font-medium">{player.prizes_won || 0}</span>
                    </div>
                  </div>
                </div>
              </Card>

              {/* Performance Metrics */}
              <Card className="p-6">
                <h3 className="text-lg font-semibold mb-4 flex items-center gap-2">
                  <Target className="w-5 h-5" />
                  Performance Metrics
                </h3>
                <div className="space-y-3">
                  {performanceMetrics && Object.entries(performanceMetrics).map(([key, value]) => (
                    <div key={key} className="flex justify-between">
                      <span className="text-sm font-medium capitalize">
                        {key.replace('_', ' ')}
                      </span>
                      <span className="text-sm">
                          {typeof value === 'number' && key.includes('rate') ? `${value}%` : 
                         typeof value === 'number' && key.includes('time') ? `${value}ms` :
                         typeof value === 'number' && key.includes('hours') ? `${value}h` :
                         String(value)}
                      </span>
                    </div>
                  ))}
                </div>
              </Card>
            </div>
          </TabsContent>

          {/* Achievements Tab */}
          <TabsContent value="achievements" className="space-y-6">
            <Card className="p-6">
              <h3 className="text-lg font-semibold mb-4 flex items-center gap-2">
                <Award className="w-5 h-5" />
                Achievements ({achievements?.length || 0})
              </h3>
              {achievements && achievements.length > 0 ? (
                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                  {achievements.map((achievement: any, index: number) => (
                    <div key={index} className="p-4 border rounded-lg bg-gradient-to-br from-yellow-50 to-orange-50">
                      <div className="flex items-center gap-2 mb-2">
                        <Star className="w-4 h-4 text-yellow-600" />
                        <h4 className="font-medium">{achievement.name}</h4>
                      </div>
                      {achievement.data?.description && (
                        <p className="text-sm text-gray-600 mb-2">{achievement.data.description}</p>
                      )}
                      <div className="flex justify-between items-center text-xs text-gray-500">
                        <span>{achievement.data?.difficulty || 'Unknown'}</span>
                        <span>{new Date(achievement.earned_at).toLocaleDateString()}</span>
                      </div>
                    </div>
                  ))}
                </div>
              ) : (
                <div className="text-center py-8 text-gray-500">
                  <Award className="w-12 h-12 mx-auto mb-4 text-gray-300" />
                  <p>No achievements earned yet.</p>
                </div>
              )}
            </Card>
          </TabsContent>

          {/* Social & Contact Tab */}
          <TabsContent value="social" className="space-y-6">
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
              {/* Contact Information */}
              <Card className="p-6">
                <h3 className="text-lg font-semibold mb-4 flex items-center gap-2">
                  <Phone className="w-5 h-5" />
                  Contact Information
                </h3>
                <div className="space-y-3">
                  <div>
                    <label className="text-sm font-medium text-gray-500">
                      Phone <span className="text-red-500">*</span>
                    </label>
                    <p className={`text-sm ${!player.phone ? 'text-red-600 font-semibold bg-red-50 p-2 rounded' : ''}`}>
                      {player.phone || '⚠️ Not provided - Required for approval'}
                    </p>
                  </div>
                  <div>
                    <label className="text-sm font-medium text-gray-500">Email</label>
                    <p className="text-sm">{player.user?.email}</p>
                  </div>
                  {player.website && (
                    <div>
                      <label className="text-sm font-medium text-gray-500">Website</label>
                      <a href={player.website} target="_blank" rel="noopener noreferrer" className="text-sm text-blue-600 hover:underline">
                        {player.website}
                      </a>
                    </div>
                  )}
                </div>
              </Card>

              {/* Social Media */}
              <Card className="p-6">
                <h3 className="text-lg font-semibold mb-4 flex items-center gap-2">
                  <Globe className="w-5 h-5" />
                  Social Media & Gaming
                </h3>
                <div className="space-y-3">
                  {socialLinks && Object.entries(socialLinks).map(([platform, username]) => (
                    <div key={platform}>
                      <label className="text-sm font-medium text-gray-500 capitalize">{platform}</label>
                      <p className="text-sm">{String(username)}</p>
                    </div>
                  ))}
                  {(!socialLinks || Object.keys(socialLinks).length === 0) && (
                    <p className="text-sm text-gray-500">No social media links provided.</p>
                  )}
                </div>
              </Card>
            </div>
          </TabsContent>

          {/* Performance Tab */}
          <TabsContent value="performance" className="space-y-6">
            <Card className="p-6">
              <h3 className="text-lg font-semibold mb-4 flex items-center gap-2">
                <Activity className="w-5 h-5" />
                Performance Analytics
              </h3>
              <div className="space-y-4">
                <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                  <div className="text-center p-4 bg-blue-50 rounded-lg">
                    <div className="text-2xl font-bold text-blue-600">{player.global_rank || 'N/A'}</div>
                    <div className="text-sm text-gray-600">Global Rank</div>
                  </div>
                  <div className="text-center p-4 bg-green-50 rounded-lg">
                    <div className="text-2xl font-bold text-green-600">{player.tournament_rank || 'N/A'}</div>
                    <div className="text-sm text-gray-600">Tournament Rank</div>
                  </div>
                  <div className="text-center p-4 bg-purple-50 rounded-lg">
                    <div className="text-2xl font-bold text-purple-600">{player.seeding_position || 'N/A'}</div>
                    <div className="text-sm text-gray-600">Seeding Position</div>
                  </div>
                </div>
                
                <div className="space-y-2">
                  <div className="flex justify-between">
                    <span>Privacy Settings</span>
                    <div className="flex gap-2">
                      <Badge variant={player.is_public_profile ? 'default' : 'secondary'}>
                        {player.is_public_profile ? 'Public Profile' : 'Private Profile'}
                      </Badge>
                      <Badge variant={player.allow_messages ? 'default' : 'secondary'}>
                        {player.allow_messages ? 'Messages Allowed' : 'Messages Blocked'}
                      </Badge>
                    </div>
                  </div>
                </div>
              </div>
            </Card>
          </TabsContent>

          {/* Digital ID Tab */}
          <TabsContent value="digital-id" className="space-y-6">
            <Card className="p-6">
              <h3 className="text-lg font-semibold mb-4 flex items-center gap-2">
                <CreditCard className="w-5 h-5" />
                Digital ID
              </h3>
              <div className="space-y-3">
                <p className="text-sm text-gray-600">
                  Your Digital ID is a unique QR code that can be scanned to verify your identity.
                  This helps ensure the authenticity of your account and prevents unauthorized access.
                </p>
                {player.digital_id ? (
                  <div className="flex flex-col items-center justify-center p-6 bg-blue-50 rounded-lg">
                    <h4 className="text-xl font-bold text-blue-600 mb-2">Your Digital ID</h4>
                    <div className="w-48 h-48 bg-white rounded-lg flex items-center justify-center shadow-lg">
                    <img src={`https://api.qrserver.com/v1/create-qr-code/?size=150x150&data=${player.digital_id}&format=png&margin=2`} alt="QR Code" className="mx-auto" />
                    </div>
                    <p className="text-sm text-gray-600 mt-4">
                      This QR code can be scanned by other users to verify your identity.
                      It is valid for a limited time.
                    </p>
                    <div className="flex gap-2 mt-4">
                      <Button variant="outline" onClick={handleGenerateDigitalId} className="bg-blue-50 text-blue-700 border-blue-200 hover:bg-blue-100">
                      <QrCode className="w-4 h-4 mr-2" />
                      Generate New Digital ID
                    </Button>
                      <Button variant="outline" onClick={handleResendDigitalId} className="bg-green-50 text-green-700 border-green-200 hover:bg-green-100">
                        <QrCode className="w-4 h-4 mr-2" />
                        Resend Digital ID
                      </Button>
                    </div>
                  </div>
                ) : (
                  <div className="flex gap-2">
                  <Button 
                    variant="outline" 
                    onClick={handleGenerateDigitalId}
                    className="bg-green-50 text-green-700 border-green-200 hover:bg-green-100"
                  >
                    <QrCode className="w-4 h-4 mr-2" />
                    Generate Digital ID
                  </Button>
                    <Button 
                      variant="outline" 
                      disabled
                      className="bg-gray-50 text-gray-400 border-gray-200 cursor-not-allowed"
                      title="Generate Digital ID first to enable resend"
                    >
                      <QrCode className="w-4 h-4 mr-2" />
                      Resend Digital ID
                    </Button>
                  </div>
                )}
              </div>
            </Card>

            {/* ID Card Template */}
            {player.digital_id ? (
            <Card className="p-6">
              <h3 className="text-lg font-semibold mb-4 flex items-center gap-2">
                <Eye className="w-5 h-5" />
                Player ID Card Template
              </h3>
              <div className="space-y-4">
                <p className="text-sm text-gray-600">
                  Official ID card template showing both front and back sides with player information.
                </p>
                
                {/* Blade Template Preview */}
                <div className="border rounded-lg overflow-hidden">
                  <iframe
                    src={route('admin.players.id-card', player.id)}
                    className="w-full h-[600px] border-0"
                    title="Player ID Card Template"
                  />
                </div>

                {/* Action Buttons */}
                <div className="flex gap-2 pt-4 border-t">                 
                  <Button 
                    variant="outline" 
                    onClick={() => window.open(route('admin.players.digital-id.print', player.id))}
                    className="bg-green-50 text-green-700 border-green-200 hover:bg-green-100"
                  >
                    <Download className="w-4 h-4 mr-2" />
                    Download ID Card
                  </Button>
                </div>
              </div>
            </Card>
            ) : (
              <div className="text-center py-8 text-gray-500">
                <EyeOff className="w-12 h-12 mx-auto mb-4 text-gray-300" />
                <p>No ID card generated yet.</p>
              </div>
            )}
          </TabsContent>

          {/* Certificate Tab */}
          <TabsContent value="certificate" className="space-y-6">
            <Card className="p-6">
              <h3 className="text-lg font-semibold mb-4 flex items-center gap-2">
                <FileText className="w-5 h-5" />
                Certificate
              </h3>
              <div className="space-y-4">
                <p className="text-sm text-gray-600">
                  Your certificate is a digital document that proves your identity and achievements.
                  It can be downloaded and shared for verification purposes.
                </p>
                
                {/* Certificate Actions */}
                <div className="flex flex-col sm:flex-row gap-3">
                  <Button 
                    variant="outline" 
                    onClick={handleGenerateCertificate}
                    className="bg-purple-50 text-purple-700 border-purple-200 hover:bg-purple-100 flex-1"
                  >
                    <Eye className="w-4 h-4 mr-2" />
                    Preview Certificate
                  </Button>
                  
                  {/* <Button 
                    variant="outline" 
                    onClick={handleDirectDownload}
                    className="bg-green-50 text-green-700 border-green-200 hover:bg-green-100 flex-1"
                  >
                    <Download className="w-4 h-4 mr-2" />
                    Download PDF
                  </Button> */}
                  
                  {/* <Button 
                    variant="outline" 
                    onClick={handleDirectPrint}
                    className="bg-blue-50 text-blue-700 border-blue-200 hover:bg-blue-100 flex-1"
                  >
                    <Printer className="w-4 h-4 mr-2" />
                    Print Certificate
                  </Button> */}
                </div>

                {/* Certificate Info */}
                <div className="bg-gray-50 p-4 rounded-lg">
                  <h4 className="font-semibold text-gray-900 mb-2">Certificate Information</h4>
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-3 text-sm">
                    <div>
                      <span className="text-gray-600">Player Name:</span>
                      <span className="ml-2 font-medium">{player.display_name || player.user?.name || 'N/A'}</span>
                    </div>
                    <div>
                      <span className="text-gray-600">Team:</span>
                      <span className="ml-2 font-medium">{player.team_name || 'N/A'}</span>
                    </div>
                    <div>
                      <span className="text-gray-600">Position:</span>
                      <span className="ml-2 font-medium">{player.position || 'Participant'}</span>
                    </div>
                    <div>
                      <span className="text-gray-600">Tournament:</span>
                      <span className="ml-2 font-medium">{player.tournament?.name || 'N/A'}</span>
                    </div>
                  </div>
                </div>
              </div>
            </Card>
          </TabsContent>
        </Tabs>
      </div>

      {/* Certificate Preview Dialog */}
      <CertificatePreviewDialog
        isOpen={isOpen}
        onClose={closePreview}
        certificateData={certificateData || {
          player_name: '',
          digital_id: '',
          issued_date: '',
          tournament_name: '',
        }}
        playerId={playerId || 0}
      />
    </AppLayout>
  );
} 