import AppLayout from '@/layouts/app-layout';
import { Head, router } from '@inertiajs/react';
import { Card } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger, DialogClose } from '@/components/ui/dialog';
import { Label   } from '@/components/ui/label';
import PlayersTable from '@/components/players/players-table';
import { useState } from 'react';
import { Plus } from 'lucide-react';
import { Trash } from 'lucide-react';

export default function AdminPlayers({ players, filters, tournaments, users }: any) {
  const [editingPlayer, setEditingPlayer] = useState<any>(null);
  const [selectedPlayers, setSelectedPlayers] = useState<number[]>([]);
  const [isDeleteModalOpen, setIsDeleteModalOpen] = useState(false);
  const [isMultipleDeleteModalOpen, setIsMultipleDeleteModalOpen] = useState(false);
  const [playerToDelete, setPlayerToDelete] = useState<any>(null);
  const [localFilters, setLocalFilters] = useState({
    search: filters?.search || '',
    status: filters?.status || 'all',
    verification_status: filters?.verification_status || 'all',
    tournament_id: filters?.tournament_id || 'all',
    player_type: filters?.player_type || 'all',
    experience_level: filters?.experience_level || 'all',
    country: filters?.country || 'all',
    gender: filters?.gender || 'all',
    min_rating: filters?.min_rating || '',
    max_rating: filters?.max_rating || '',
    min_win_rate: filters?.min_win_rate || '',
    max_win_rate: filters?.max_win_rate || '',
  });



  // Edit form state
  const [editForm, setEditForm] = useState({
    id: null,
    display_name: '',
    bio: '',
    country: '',
    city: '',
    date_of_birth: '',
    gender: '',
    phone: '',
    discord_username: '',
    steam_id: '',
    twitch_username: '',
    youtube_channel: '',
    twitter_handle: '',
    instagram_handle: '',
    website: '',
    player_type: '',
    experience_level: '',
    primary_game: '',
    status: '',
    verification_status: '',
    tournament_id: '',
    user_id: '',
    global_rating: '',
    tournament_rating: '',
    seeding_position: '',
    is_public_profile: true,
    allow_messages: true,
    show_contact_info: false,
  });
  const [editFormErrors, setEditFormErrors] = useState<any>({});

  // Handle filter form submit
  const handleFilterSubmit = (e: any) => {
    e.preventDefault();
    router.get(route('admin.players.index'), {
      ...localFilters,
      page: 1,
    }, { preserveState: true, replace: true });
  };

  // Handle pagination
  const handlePageChange = (page: any) => {
    router.get(route('admin.players.index'), {
      ...localFilters,
      page,
    }, { preserveState: true, replace: true });
  };



  // Open edit modal and populate form
  const openEditModal = (p: any) => {
    setEditingPlayer(p);
    setEditForm({
      id: p.id,
      display_name: p.display_name || '',
      bio: p.bio || '',
      country: p.country || '',
      city: p.city || '',
      date_of_birth: p.date_of_birth || '',
      gender: p.gender || '',
      phone: p.phone || '',
      discord_username: p.discord_username || '',
      steam_id: p.steam_id || '',
      twitch_username: p.twitch_username || '',
      youtube_channel: p.youtube_channel || '',
      twitter_handle: p.twitter_handle || '',
      instagram_handle: p.instagram_handle || '',
      website: p.website || '',
      player_type: p.player_type || '',
      experience_level: p.experience_level || '',
      primary_game: p.primary_game || '',
      status: p.status || '',
      verification_status: p.verification_status || '',
      tournament_id: p.tournament_id || '',
      user_id: p.user_id || '',
      global_rating: p.global_rating || '',
      tournament_rating: p.tournament_rating || '',
      seeding_position: p.seeding_position || '',
      is_public_profile: p.is_public_profile ?? true,
      allow_messages: p.allow_messages ?? true,
      show_contact_info: p.show_contact_info ?? false,
    });
    setEditFormErrors({});
  };

  // Handle edit form submit
  const handleEditSubmit = (e: any) => {
    e.preventDefault();
    router.put(route('admin.players.update', { player: editForm.id }), editForm, {
      onError: (errors) => setEditFormErrors(errors),
      onSuccess: () => {
        setEditingPlayer(null);
        setEditForm({
          id: null, display_name: '', bio: '', country: '', city: '', date_of_birth: '', gender: '',
          phone: '', discord_username: '', steam_id: '', twitch_username: '', youtube_channel: '',
          twitter_handle: '', instagram_handle: '', website: '', player_type: '', experience_level: '',
          primary_game: '', status: '', verification_status: '', tournament_id: '', user_id: '',
          global_rating: '', tournament_rating: '', seeding_position: '', is_public_profile: true,
          allow_messages: true, show_contact_info: false
        });
        setEditFormErrors({});
      },
    });
  };

  // Handle single player selection
  const handlePlayerSelect = (playerId: number, checked: boolean) => {
    if (checked) {
      setSelectedPlayers([...selectedPlayers, playerId]);
    } else {
      setSelectedPlayers(selectedPlayers.filter(id => id !== playerId));
    }
  };

  // Handle select all players
  const handleSelectAll = (checked: boolean) => {
    if (checked) {
      setSelectedPlayers(players.data.map(p => p.id) as any);
    } else {
      setSelectedPlayers([]);
    }
  };

  // Handle single delete
  const handleDelete = (player: any) => {
    setPlayerToDelete(player);
    setIsDeleteModalOpen(true);
  };

  // Confirm single delete
  const confirmDelete = () => {
    if (playerToDelete) {
      router.delete(route('admin.players.destroy', { player: playerToDelete.id }), {
        onSuccess: () => {
        setIsDeleteModalOpen(false);
        setPlayerToDelete(null);
        },
      });
    }
  };

  // Handle multiple delete
  const handleMultipleDelete = () => {
    if (selectedPlayers.length > 0) {
      setIsMultipleDeleteModalOpen(true);
    }
  };

  // Confirm multiple delete
  const confirmMultipleDelete = () => {
    router.delete(route('admin.players.destroyMultiple'), {
      data: { ids: selectedPlayers },
      onSuccess: () => {
        setIsMultipleDeleteModalOpen(false);
        setSelectedPlayers([]);
      },
    });
  };

  return (
    <AppLayout breadcrumbs={[{ title: 'Players', href: '/admin/players' }]}> 
      <Head title="Players" />
      <div className="flex flex-col gap-4 p-4">
        <div className="flex justify-between items-center">
          
          <div className="flex gap-2">
            {selectedPlayers.length > 0 && (
              <Button variant="destructive" onClick={handleMultipleDelete}>
                <Trash size={16} /> Delete Selected ({selectedPlayers.length})
              </Button>
            )}
            <Button onClick={() => router.get(route('admin.players.create'))}> <Plus size={16} /> Create Player</Button>
          </div>
        </div>

        {/* Enhanced Filters */}
        <Card className="p-4">
          <form className="space-y-4" onSubmit={handleFilterSubmit}>
            {/* Basic Filters Row */}
            <div className="flex gap-4">
              <div className="flex-1">
                <Input
                  placeholder="Search players, bio, user name, email..."
                  value={localFilters.search}
                  onChange={(e) => setLocalFilters({ ...localFilters, search: e.target.value })}
                  name="search"
                />
              </div>
              <Select value={localFilters.status} onValueChange={(value) => setLocalFilters({ ...localFilters, status: value })}>
                <SelectTrigger className="w-40">
                  <SelectValue placeholder="Status" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Status</SelectItem>
                  <SelectItem value="registered">Registered</SelectItem>
                  <SelectItem value="active">Active</SelectItem>
                  <SelectItem value="eliminated">Eliminated</SelectItem>
                  <SelectItem value="withdrawn">Withdrawn</SelectItem>
                  <SelectItem value="suspended">Suspended</SelectItem>
                  <SelectItem value="banned">Banned</SelectItem>
                </SelectContent>
              </Select>
              <Select value={localFilters.verification_status} onValueChange={(value) => setLocalFilters({ ...localFilters, verification_status: value })}>
                <SelectTrigger className="w-40">
                  <SelectValue placeholder="Verification" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Verification</SelectItem>
                  <SelectItem value="unverified">Unverified</SelectItem>
                  <SelectItem value="pending">Pending</SelectItem>
                  <SelectItem value="verified">Verified</SelectItem>
                  <SelectItem value="rejected">Rejected</SelectItem>
                </SelectContent>
              </Select>
              <Select value={localFilters.tournament_id} onValueChange={(value) => setLocalFilters({ ...localFilters, tournament_id: value })}>
                <SelectTrigger className="w-40">
                  <SelectValue placeholder="Tournament" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Tournaments</SelectItem>
                  {tournaments.map((t: any) => (
                    <SelectItem key={t.id} value={String(t.id)}>{t.name}</SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>

            {/* Advanced Filters Row */}
            <div className="flex gap-4">
              <Select value={localFilters.player_type} onValueChange={(value) => setLocalFilters({ ...localFilters, player_type: value })}>
                <SelectTrigger className="w-40">
                  <SelectValue placeholder="Player Type" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Types</SelectItem>
                  <SelectItem value="amateur">Amateur</SelectItem>
                  <SelectItem value="semi_pro">Semi Pro</SelectItem>
                  <SelectItem value="professional">Professional</SelectItem>
                  <SelectItem value="legend">Legend</SelectItem>
                </SelectContent>
              </Select>
              <Select value={localFilters.experience_level} onValueChange={(value) => setLocalFilters({ ...localFilters, experience_level: value })}>
                <SelectTrigger className="w-40">
                  <SelectValue placeholder="Experience" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Levels</SelectItem>
                  <SelectItem value="beginner">Beginner</SelectItem>
                  <SelectItem value="intermediate">Intermediate</SelectItem>
                  <SelectItem value="advanced">Advanced</SelectItem>
                  <SelectItem value="expert">Expert</SelectItem>
                </SelectContent>
              </Select>
              <Select value={localFilters.country} onValueChange={(value) => setLocalFilters({ ...localFilters, country: value })}>
                <SelectTrigger className="w-40">
                  <SelectValue placeholder="Country" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Countries</SelectItem>
                  <SelectItem value="Kenya">Kenya</SelectItem>
                  <SelectItem value="Canada">Canada</SelectItem>
                  <SelectItem value="United Kingdom">United Kingdom</SelectItem>
                  <SelectItem value="Germany">Germany</SelectItem>
                  <SelectItem value="France">France</SelectItem>
                </SelectContent>
              </Select>
              <Select value={localFilters.gender} onValueChange={(value) => setLocalFilters({ ...localFilters, gender: value })}>
                <SelectTrigger className="w-40">
                  <SelectValue placeholder="Gender" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Genders</SelectItem>
                  <SelectItem value="male">Male</SelectItem>
                  <SelectItem value="female">Female</SelectItem>
                  <SelectItem value="other">Other</SelectItem>
                  <SelectItem value="prefer_not_to_say">Prefer Not to Say</SelectItem>
                </SelectContent>
              </Select>
            </div>

            {/* Rating Filters Row */}
            <div className="flex gap-4">
              <Input
                placeholder="Min Rating"
                value={localFilters.min_rating}
                onChange={(e) => setLocalFilters({ ...localFilters, min_rating: e.target.value })}
                className="w-32"
                type="number"
              />
              <Input
                placeholder="Max Rating"
                value={localFilters.max_rating}
                onChange={(e) => setLocalFilters({ ...localFilters, max_rating: e.target.value })}
                className="w-32"
                type="number"
              />
              <Input
                placeholder="Min Win Rate %"
                value={localFilters.min_win_rate}
                onChange={(e) => setLocalFilters({ ...localFilters, min_win_rate: e.target.value })}
                className="w-32"
                type="number"
              />
              <Input
                placeholder="Max Win Rate %"
                value={localFilters.max_win_rate}
                onChange={(e) => setLocalFilters({ ...localFilters, max_win_rate: e.target.value })}
                className="w-32"
                type="number"
              />
              <Button type="submit" variant="outline">Apply Filters</Button>
              <Button type="button" variant="outline" onClick={() => setLocalFilters({
                search: '', status: 'all', verification_status: 'all', tournament_id: 'all',
                player_type: 'all', experience_level: 'all', country: 'all', gender: 'all',
                min_rating: '', max_rating: '', min_win_rate: '', max_win_rate: ''
              })}>
                Clear Filters
              </Button>
            </div>
          </form>
        </Card>

        <Card className="p-0 overflow-x-auto">
          <PlayersTable
            players={players}
            tournaments={tournaments}
            users={users}
            selectedPlayers={selectedPlayers}
            onPlayerSelect={handlePlayerSelect}
            onSelectAll={handleSelectAll}
            onEdit={(player: any) => router.get(route('admin.players.edit', { player: player.id }))}
            onDelete={handleDelete}
            onPageChange={handlePageChange}
          />
        </Card>

        {/* Edit Modal */}
        {editingPlayer && (
          <Dialog open={!!editingPlayer} onOpenChange={() => setEditingPlayer(null)}>
            <DialogContent>
              <DialogHeader>
                <DialogTitle>Edit Player</DialogTitle>
              </DialogHeader>
              <form onSubmit={handleEditSubmit} className="space-y-4">
                <div>
                  <Label htmlFor="edit-display-name">Display Name</Label>
                  <Input id="edit-display-name" name="display_name" value={editForm.display_name} onChange={e => setEditForm(f => ({ ...f, display_name: e.target.value }))} required />
                  {editFormErrors.display_name && <div className="text-red-500 text-xs">{editFormErrors.display_name as string}</div>}
                </div>
                <div>
                  <Label htmlFor="edit-status">Status</Label>
                  <Select value={editForm.status} onValueChange={value => setEditForm(f => ({ ...f, status: value }))} required>
                    <SelectTrigger>
                      <SelectValue placeholder="Select status" />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="registered">Registered</SelectItem>
                      <SelectItem value="active">Active</SelectItem>
                      <SelectItem value="eliminated">Eliminated</SelectItem>
                      <SelectItem value="withdrawn">Withdrawn</SelectItem>
                    </SelectContent>
                  </Select>
                  {editFormErrors.status && <div className="text-red-500 text-xs">{editFormErrors.status as string}</div>}
                </div>
                <div>
                  <Label htmlFor="edit-tournament-id">Tournament</Label>
                  <Select value={editForm.tournament_id} onValueChange={value => setEditForm(f => ({ ...f, tournament_id: value }))} required>
                    <SelectTrigger>
                      <SelectValue placeholder="Select tournament" />
                    </SelectTrigger>
                    <SelectContent>
                      {tournaments.map((t: any) => (
                        <SelectItem key={t.id} value={String(t.id)}>{t.name}</SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                  {editFormErrors.tournament_id && <div className="text-red-500 text-xs">{editFormErrors.tournament_id as string}</div>}
                </div>
                <div>
                  <Label htmlFor="edit-user-id">User</Label>
                  <Select value={editForm.user_id} onValueChange={value => setEditForm(f => ({ ...f, user_id: value }))} required>
                    <SelectTrigger>
                      <SelectValue placeholder="Select user" />
                    </SelectTrigger>
                    <SelectContent>
                      {users.map((u: any) => (
                        <SelectItem key={u.id} value={String(u.id)}>{u.name} ({u.email})</SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                  {editFormErrors.user_id && <div className="text-red-500 text-xs">{editFormErrors.user_id as string}</div>}
                </div>
                <Button type="submit">Update</Button>
              </form>
            </DialogContent>
          </Dialog>
        )}

        {/* Delete Confirmation Modal */}
        <Dialog open={isDeleteModalOpen} onOpenChange={setIsDeleteModalOpen}>
          <DialogContent>
            <DialogHeader>
              <DialogTitle>Delete Player</DialogTitle>
            </DialogHeader>
            <div className="space-y-4">
              <p>Are you sure you want to delete "{playerToDelete?.display_name}"? This action cannot be undone.</p>
              <div className="flex gap-2">
                <Button variant="destructive" onClick={confirmDelete}>Delete</Button>
                <Button variant="outline" onClick={() => setIsDeleteModalOpen(false)}>Cancel</Button>
              </div>
            </div>
          </DialogContent>
        </Dialog>

        {/* Multiple Delete Confirmation Modal */}
        <Dialog open={isMultipleDeleteModalOpen} onOpenChange={setIsMultipleDeleteModalOpen}>
          <DialogContent>
            <DialogHeader>
              <DialogTitle>Delete Selected Players</DialogTitle>
            </DialogHeader>
            <div className="space-y-4">
              <p>Are you sure you want to delete {selectedPlayers.length} player(s)? This action cannot be undone.</p>
              <div className="flex gap-2">
                <Button variant="destructive" onClick={confirmMultipleDelete}>Delete</Button>
                <Button variant="outline" onClick={() => setIsMultipleDeleteModalOpen(false)}>Cancel</Button>
              </div>
            </div>
          </DialogContent>
        </Dialog>
      </div>
    </AppLayout>
  );
} 